#include <iostream>
#include <sstream>
#include <fstream>
#include <ctime>
#include <iomanip>

#include <boost/filesystem.hpp>
#include <boost/date_time/posix_time/posix_time.hpp>
#include <boost/scoped_array.hpp>

#include <errno.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>

#include <crypto++/sha.h>

namespace fs = boost::filesystem;

int main(int argc, char **argv)
	{
	std::string datadir = "iotest";
	if(argc == 2)
		datadir = argv[1];

	if(argc > 2)
		{
		std::cerr << "Usage: iotester[ <datadir>]" << std::endl;
		exit(-1);
		}

	unsigned char buffer[1024 * 1024];
	srand(std::time(0));

	boost::posix_time::time_duration total_write_time;
	boost::posix_time::time_duration max_write_time;
	unsigned int write_count = 0;
	boost::posix_time::time_duration total_read_time;
	boost::posix_time::time_duration max_read_time;
	unsigned int read_count = 0;

	while(true)
		{
		bool diskfull = false;
		boost::posix_time::ptime loop_started = boost::posix_time::microsec_clock::universal_time();
		boost::posix_time::ptime hundred_started = boost::posix_time::microsec_clock::universal_time();
		int filecount = 0;
		while(!diskfull)
			{
			for(int i = 0 ; i < 1024 * 1024 ; ++i)
				buffer[i] = (unsigned char) rand();

			CryptoPP::SHA1 checksum;
			boost::scoped_array<unsigned char> digest(new unsigned char[checksum.DigestSize()]);
			checksum.CalculateDigest(digest.get(), buffer, 1024 * 1024);

			std::ostringstream digeststr;
			for(int i = 0 ; i < checksum.DigestSize() ; ++i)
				digeststr << std::hex << std::setw(2) << std::setfill('0') << (((unsigned int)digest[i]) & 0xff);

			int fd = open((datadir + "/" + digeststr.str()).c_str(), O_WRONLY | O_CREAT, 0666);
			if(fd == -1)
				{
				if(errno == ENOSPC)
					diskfull = true;
				else
					exit(-1);
				}
			ssize_t count;
			if(fd != -1 && (count = write(fd, (char*)buffer, 1024 * 1024)) == -1)
				{
				if(errno == ENOSPC)
					{
					diskfull = true;
					unlink((datadir + "/" + digeststr.str()).c_str());
					}
				else
					exit(-1);
				}

			if(count != 1024 * 1024)
					{
					diskfull = true;
					unlink((datadir + "/" + digeststr.str()).c_str());
					}
			close(fd);

			++filecount;
			if(filecount % 100 == 0)
				{
				boost::posix_time::time_duration write_time = boost::posix_time::microsec_clock::universal_time() - hundred_started;
				std::cout << "Wrote 100 MiB of data in " << write_time.total_milliseconds() << " milliseconds" << std::endl;
				hundred_started = boost::posix_time::microsec_clock::universal_time();
				total_write_time += write_time;
				write_count++;
				if(write_time > max_write_time)
					max_write_time = write_time;
				}
			}

		hundred_started = boost::posix_time::microsec_clock::universal_time();
		for(fs::directory_iterator i = fs::directory_iterator(datadir) ; i != fs::directory_iterator() ; ++i)
			{
			if(i->path().filename().string().size() == 40)
				{
				std::string ft = i->path().filename().string();
				unsigned char digest[20];
				int j = 0;
				while(!ft.empty())
					{
					std::istringstream curb(ft.substr(0, 2));
					unsigned int b;
					curb >> std::hex >> b;
					digest[j] = b;
					++j;
					ft.erase(0, 2);
					}

				std::ifstream file(i->path().c_str(), std::ios::in);
				file.read((char *)buffer, 1024 * 1024);
				CryptoPP::SHA1 checksum;
				if(checksum.VerifyDigest(digest, buffer, 1024 * 1024))
					{}
				else
					std::cout << "Digest wrong for file " << i->path() << std::endl;

				fs::remove(i->path());
				}
				++filecount;
				if(filecount % 100 == 0)
					{
					boost::posix_time::time_duration read_time = boost::posix_time::microsec_clock::universal_time() - hundred_started;
					std::cout << "Read 100 MiB of data in " << read_time.total_milliseconds() << " milliseconds" << std::endl;
					hundred_started = boost::posix_time::microsec_clock::universal_time();
					total_read_time += read_time;
					read_count++;
					if(read_time > max_read_time)
						max_read_time = read_time;
					}
			}
		std::cout << "Loop took " << (boost::posix_time::microsec_clock::universal_time() - loop_started).total_milliseconds() << " milliseconds" << std::endl;
		std::cout << "Average write time: " << total_write_time.total_milliseconds() / write_count << " milliseconds, max: " << max_write_time.total_milliseconds() << std::endl;
		std::cout << "Average read time:  " << total_read_time.total_milliseconds() / read_count << " milliseconds, max: " << max_read_time.total_milliseconds() << std::endl;
		}

	return 0;
	}
