// -*- mode:C++; tab-width:8; c-basic-offset:2; indent-tabs-mode:t -*-
// vim: ts=8 sw=2 smarttab
/*
 * Ceph - scalable distributed file system
 *
 * Copyright (C) 2011 New Dream Network
 *
 * This is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License version 2.1, as published by the Free Software
 * Foundation.  See file COPYING.
 *
 */

#include "include/types.h"
#include "include/rados/librgw.h"
#include "rgw/rgw_acl.h"
#include "rgw_acl.h"
#include "common/ceph_argparse.h"
#include "common/common_init.h"
#include "common/config.h"

#include <errno.h>
#include <sstream>
#include <string.h>

#define RGW_LOG(x) pdout(x, g_conf.rgw_log)

static Mutex librgw_init_mutex("librgw_init");
static int librgw_initialized = 0;

int librgw_create(librgw_t *rgw, const char * const id)
{
  librgw_init_mutex.Lock();
  CephContext *cct;
  if (!librgw_initialized) {
    CephInitParameters iparams(CEPH_ENTITY_TYPE_CLIENT, CEPH_CONF_FILE_DEFAULT);
    iparams.conf_file = "";
    if (id) {
      iparams.name.set(CEPH_ENTITY_TYPE_CLIENT, id);
    }
    CephContext *cct = common_preinit(iparams, CODE_ENVIRONMENT_LIBRARY, 0);
    cct->_conf->log_to_stderr = 1; // quiet by default
    cct->_conf->parse_env(); // environment variables override
    cct->_conf->apply_changes();

    ++librgw_initialized;
    common_init_finish(cct);
  }
  else {
    cct = &g_ceph_context;
  }
  librgw_init_mutex.Unlock();
  *rgw = &g_ceph_context;
  return 0;
}

int librgw_acl_bin2xml(librgw_t rgw, const char *bin, int bin_len, char **xml)
{
  try {
    // convert to bufferlist
    bufferlist bl;
    bl.append(bin, bin_len);

    // convert to RGWAccessControlPolicy
    RGWAccessControlPolicy acl;
    bufferlist::iterator bli(bl.begin());
    acl.decode(bli);

    // convert to XML stringstream
    stringstream ss;
    acl.to_xml(ss);

    // convert to XML C string
    *xml = strdup(ss.str().c_str());
    if (!*xml)
      return -ENOBUFS;
    return 0;
  }
  catch (const std::exception &e) {
    RGW_LOG(-1) << "librgw_acl_bin2xml: caught exception " << e.what() << dendl;
    return -2000;
  }
  catch (...) {
    RGW_LOG(-1) << "librgw_acl_bin2xml: caught unknown exception " << dendl;
    return -2000;
  }
}

void librgw_free_xml(librgw_t rgw, char *xml)
{
  free(xml);
}

int librgw_acl_xml2bin(librgw_t rgw, const char *xml, char **bin, int *bin_len)
{
  char *bin_ = NULL;
  try {
    RGWXMLParser parser;
    if (!parser.init()) {
      return -1000;
    }
    if (!parser.parse(xml, strlen(xml), true)) {
      return -EINVAL;
    }
    RGWAccessControlPolicy *policy =
      (RGWAccessControlPolicy *)parser.find_first("AccessControlPolicy");
    if (!policy) {
      return -1001;
    }
    bufferlist bl;
    policy->encode(bl);

    bin_ = (char*)malloc(bl.length());
    if (!bin_) {
      return -ENOBUFS;
    }
    int bin_len_ = bl.length();
    bl.copy(0, bin_len_, bin_);

    *bin = bin_;
    *bin_len = bin_len_;
    return 0;
  }
  catch (const std::exception &e) {
    RGW_LOG(-1) << "librgw_acl_bin2xml: caught exception " << e.what() << dendl;
  }
  catch (...) {
    RGW_LOG(-1) << "librgw_acl_bin2xml: caught unknown exception " << dendl;
  }
  if (!bin_)
    free(bin_);
  bin_ = NULL;
  return -2000;
}

void librgw_free_bin(librgw_t rgw, char *bin)
{
  free(bin);
}

void librgw_shutdown(librgw_t rgw)
{
  // TODO: free configuration, etc.
}
