/*
 * Diagnostics - a unified framework for code annotation, logging,
 * program monitoring, and unit-testing.
 *
 * Copyright (C) 2009 Christian Schallhart <christian@schallhart.net>,
 *                    Michael Tautschnig <tautschnig@forsyte.de>
 *               2008 model.in.tum.de group, FORSYTE group
 *               2006-2007 model.in.tum.de group
 *               2002-2005 Christian Schallhart
 *  
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */


/**
 * @file diagnostics/logger/intending_stream_logger.cpp
 *
 * @brief [LEVEL: beta] Implementation of the @ref diagnostics::Intending_Stream_Logger class
 *
 * $Id: intending_stream_logger.cpp,v 1.6 2005/06/23 09:54:21 esdentem Exp $
 *
 * @author Christian Schallhart
 */

#include <diagnostics/logger/intending_stream_logger.hpp>

// for DIAGNOSTICS_PANIC_LOG
#include <diagnostics/frame/logging_facility.hpp>

#include <diagnostics/frame/record.hpp>

#include <diagnostics/frame/logging_config_exception.hpp>

#include <diagnostics/util/to_string.hpp>

DIAGNOSTICS_NAMESPACE_BEGIN;

Intending_Stream_Logger::~Intending_Stream_Logger()
{
}

Intending_Stream_Logger::Intending_Stream_Logger(::std::ostream & stream) 
    : m_stream(stream),
      m_intend(0)
{
}

void Intending_Stream_Logger::log(Record const & record)
{
    using ::diagnostics::internal::to_string;
    switch(record.type()){
	case TYPE_BLOCK_ENTER:
	case TYPE_PROCEDURE_ENTER:
	case TYPE_METHOD_ENTER:
	case TYPE_TESTCASE_ENTER:
	    for(int i=0;i<m_intend;++i) m_stream << ' ';
	    m_stream << level_to_string(record.level()) << " "
		     << type_to_string(record.type()) << " "
		     << record.nr_what() << " " 
		     << record.str_what() 
		     << ::std::endl;
	    m_intend+=2;
	    break;

	case TYPE_BLOCK_EXIT:
	case TYPE_PROCEDURE_EXIT:
	case TYPE_METHOD_EXIT:
	case TYPE_TESTCASE_EXIT:
	    m_intend-=2;
	    for(int i=0;i<m_intend;++i) m_stream << ' ';
	    m_stream << level_to_string(record.level()) << " "
		     << type_to_string(record.type()) << " "
		     << record.nr_what() << " " 
		     << record.str_what() 
		     << ::std::endl;
	    break;
	    
	case TYPE_TRACE:
	case TYPE_TRACE_BINARY:
	case TYPE_FAILED_ASSERTION:
	case TYPE_FAILED_CHECK:
	case TYPE_UNEXPECTED_EXCEPTION:
	case TYPE_WRONG_EXCEPTION:
	case TYPE_MISSING_EXCEPTION:
	case TYPE_LOG_OPEN:
	case TYPE_LOG_CLOSE:
	    for(int i=0;i<m_intend;++i) m_stream << ' ';
	    m_stream << level_to_string(record.level()) << " "
		     << type_to_string(record.type()) << " "
		     << record.nr_what() << " " 
		     << record.str_what() 
		     << ::std::endl;
	    break;
    }
    
    if(!m_stream.good()) 
	DIAGNOSTICS_PANIC_LOG("Intending Stream Logger is unable to write record: " + to_string(record));
}

void Intending_Stream_Logger::close()
{
}


DIAGNOSTICS_NAMESPACE_END;

// vim:ts=4:sw=4
