#  Copyright (c) 2022, Manfred Moitzi
#  License: MIT License

import pytest
from ezdxf import shapefile, path


def test_filter_noise():
    lines = list(
        shapefile.filter_noise(
            b"""
    ;; comment
    
    *BIGFONT xxx, 234
    
    
    """.split(
                b"\n"
            )
        )
    )
    assert lines[0].startswith(b"*BIGFONT") is True


def test_merge_wrapped_spec_line():
    lines = list(
        shapefile.merge_lines(
            shapefile.filter_noise(
                b"""*000BB
,25,NAME
2,8,(6,4),1,8,(6,10),8,(-6,10),2,0C0,1,8,(6,-10),8,(-6,-10),2,
8,(12,-4),0
    """.split(
                    b"\n"
                )
            )
        )
    )
    assert len(lines) == 2
    assert lines[0] == b"*000BB,25,NAME"
    assert len(lines[1]) == 73


def test_do_not_merge_spec_without_name():
    lines = list(
        shapefile.merge_lines(
            shapefile.filter_noise(
                b"""*00025,29,
2,8,(6,2),1,8,(32,36),2,8,(-20,-6),1,10,(6,000),2,8,(8,-24),1,
10,(6,040),2,8,(18,-8),0
    """.split(
                    b"\n"
                )
            )
        )
    )
    assert len(lines) == 2
    assert lines[0] == b"*00025,29,"
    assert len(lines[1]) == 86


def test_big_font_not_supported():
    with pytest.raises(shapefile.UnsupportedShapeFile):
        shapefile.shp_load(
            b"""
*BIGFONT 7392,3,081,09F,0E0,0EA,0FD,0FE
*0,5,Comment
15,0,2,14,0
        """
        )


class TestFontShapeFile:
    @pytest.fixture(scope="class")
    def txt(self):
        return shapefile.shp_load(TXT)

    def test_shape_file_name(self, txt):
        assert txt.name == b"TXT"

    def test_cap_height(self, txt):
        assert txt.cap_height == 6

    def test_descender(self, txt):
        assert txt.descender == 2

    def test_mode(self, txt):
        assert txt.mode == shapefile.FontMode.BIDIRECT

    def test_encoding(self, txt):
        assert txt.encoding == shapefile.FontEncoding.UNICODE

    def test_embed(self, txt):
        assert txt.embed == shapefile.FontEmbedding.ALLOWED

    def test_is_font(self, txt):
        assert txt.is_font is True

    def test_shape_count(self, txt):
        assert len(txt) == 5

    def test_shape_by_number(self, txt):
        assert txt[32].name == b"spc"

    def test_find_shape_by_name(self, txt):
        assert txt.find(b"spc").number == 32

    def test_find_undefined_shape(self, txt):
        assert txt.find(b"undefined") is None


class TestShapeFile:
    """Any shape file without a font definition is a common shape file"""

    @pytest.fixture(scope="class")
    def shp(self):
        return shapefile.shp_load(FILE_1)

    def test_is_a_shape_file(self, shp):
        assert shp.is_shape_file is True

    def test_shape_by_number(self, shp):
        assert shp[0x53].name == b"S"

    def test_shape_by_name(self, shp):
        assert shp.find(b"S").number == 0x53


class TestShapeRenderer:
    @pytest.fixture(scope="class")
    def shp(self):
        return shapefile.shp_load(FILE_1)

    def test_render_only_lines(self, shp):
        p = shp.render_shape(0x41)  # uppercase letter A
        commands = p.commands()
        assert [c.type for c in commands] == [
            path.Command.LINE_TO,
            path.Command.LINE_TO,
            path.Command.MOVE_TO,
            path.Command.LINE_TO,
            path.Command.MOVE_TO,
        ]
        assert p.start.isclose((6, 2))
        assert p.end.isclose((36, 0))

    def test_render_bulges(self, shp):
        p = shp.render_shape(0x53)  # uppercase letter S
        commands = p.commands()
        assert [c.type for c in commands] == [
            path.Command.CURVE4_TO,
            path.Command.CURVE4_TO,
            path.Command.CURVE4_TO,
            path.Command.LINE_TO,
            path.Command.CURVE4_TO,
            path.Command.CURVE4_TO,
            path.Command.CURVE4_TO,
            path.Command.MOVE_TO,
        ]
        assert p.start.isclose((24, 36))
        assert p.end.isclose((32, 0))
        curve3 = commands[2]
        assert curve3[0].isclose((10.400000000000002, 22.8))
        assert curve3[1].isclose((5.3604119976499085, 28.205904025073004))
        assert curve3[2].isclose((7.089973469642831, 24.478591646965565))

        curve5 = commands[4]
        assert curve5[0].isclose((25.784251968503938, 8.182677165354324))
        assert curve5[1].isclose((24.910026530357175, 15.52140835303443))
        assert curve5[2].isclose((26.6395880023501, 11.794095974926988))

    def test_render_full_circle_by_octant_arc(self, shp):
        p = shp.render_shape(0x38)  # number 8
        commands = p.commands()
        assert [c.type for c in commands] == [
            path.Command.CURVE4_TO,
            path.Command.CURVE4_TO,
            path.Command.CURVE4_TO,
            path.Command.CURVE4_TO,
            path.Command.MOVE_TO,
            path.Command.CURVE4_TO,
            path.Command.CURVE4_TO,
            path.Command.LINE_TO,
            path.Command.CURVE4_TO,
            path.Command.CURVE4_TO,
            path.Command.LINE_TO,
            path.Command.MOVE_TO,
        ]
        assert p.start.isclose((14, 22))
        assert p.end.isclose((28, 0))

    def test_render_ccw_fractional_arcs_number_nine(self, shp):
        # This has the correct continuation
        p = shp.render_text("9")
        commands = p.commands()
        assert [c.type for c in commands] == [
            path.Command.CURVE4_TO,
            path.Command.LINE_TO,
            path.Command.CURVE4_TO,
            path.Command.CURVE4_TO,
            path.Command.LINE_TO,
            path.Command.CURVE4_TO,
            path.Command.LINE_TO,
            path.Command.MOVE_TO,
        ]
        assert p.start.isclose((10, 2))
        # solved by a hack:
        assert abs(p.end.y) < 1e-9, "has to be on the baseline"

        p = shp.render_shape(0x39)
        assert p.end.isclose(
            (28.144414741079267, 0.020931856849379926)
        ), "fractional arc rendering solved?"

    def test_render_ccw_fractional_arcs_ampersand(self, shp):
        p = shp.render_text("&")
        commands = p.commands()
        assert [c.type for c in commands] == [
            path.Command.LINE_TO,
            path.Command.CURVE4_TO,
            path.Command.CURVE4_TO,
            path.Command.LINE_TO,
            path.Command.CURVE4_TO,
            path.Command.CURVE4_TO,
            path.Command.CURVE4_TO,
            path.Command.LINE_TO,
            path.Command.MOVE_TO,
        ]
        assert p.start.isclose((30, 14))
        # solved by a hack:
        assert abs(p.end.y) < 1e-9, "has to be on the baseline"

        p = shp.render_shape(0x26)
        assert p.end.isclose(
            (36.031289001494976, -0.1845547786548103)
        ), "fractional arc rendering solved?"

    def test_render_clockwise_fractional_arcs_letter_c(self, shp):
        p = shp.render_text("C")  # bold.shp
        commands = p.commands()
        assert [c.type for c in commands] == [
            path.Command.CURVE4_TO,
            path.Command.CURVE4_TO,
            path.Command.LINE_TO,
            path.Command.MOVE_TO,
            path.Command.CURVE4_TO,
            path.Command.CURVE4_TO,
            path.Command.LINE_TO,
            path.Command.MOVE_TO,
            path.Command.CURVE4_TO,
            path.Command.CURVE4_TO,
            path.Command.MOVE_TO,
            path.Command.CURVE4_TO,
            path.Command.CURVE4_TO,
            path.Command.MOVE_TO,
            path.Command.CURVE4_TO,
            path.Command.CURVE4_TO,
            path.Command.MOVE_TO,
            path.Command.CURVE4_TO,
            path.Command.CURVE4_TO,
            path.Command.MOVE_TO,
            path.Command.CURVE4_TO,
            path.Command.CURVE4_TO,
            path.Command.MOVE_TO,
            path.Command.CURVE4_TO,
            path.Command.CURVE4_TO,
            path.Command.MOVE_TO,
            path.Command.CURVE4_TO,
            path.Command.CURVE4_TO,
            path.Command.MOVE_TO,
            path.Command.CURVE4_TO,
            path.Command.CURVE4_TO,
            path.Command.MOVE_TO,
            path.Command.CURVE4_TO,
            path.Command.CURVE4_TO,
            path.Command.MOVE_TO,
            path.Command.CURVE4_TO,
            path.Command.CURVE4_TO,
            path.Command.MOVE_TO,
            path.Command.CURVE4_TO,
            path.Command.CURVE4_TO,
            path.Command.MOVE_TO,
            path.Command.CURVE4_TO,
            path.Command.CURVE4_TO,
            path.Command.MOVE_TO,
        ]
        assert p.start.isclose((-2, 30))
        p2 = shp.render_shape(0x43)
        assert p2.end.isclose((64, 0))
        assert p2.end.isclose(
            p.end
        ), "should be perfect rendering - no placing-hack required"


FILE_1 = b"""
;; isocp.shp
*00026,41,&
2,8,(30,14),1,3,10,4,2,8,(-72,-54),3,2,12,(-89,76,-102),4,4,
8,(31,45),3,4,4,10,11,(68,187,0,6,076),8,(17,-27),2,8,(10,-2),0

;; isocp.shp
*00038,24,8
2,8,(14,22),1,10,(8,060),2,08A,1,10,(8,-044),04C,10,(8,-004),044,
2,8,(22,-14),0

;; isocp.shp
*00039,25,9
2,8,(10,2),1,11,(108,0,0,18,062),0B4,10,(8,004),04C,10,(8,042),
080,2,8,(6,-18),0

;; isocp.shp
*00041,28,A
2,8,(6,2),1,8,(12,36),8,(12,-36),3,2,2,8,(-7,20),1,8,(-34,0),2,
8,(53,-24),4,2,0

;; bold.shp
*00043,156,C
2,8,(-2,30),5,1,11,(0,125,0,30,044),078,2,6,5,1,
11,(0,125,0,30,-044),078,2,6,010,5,1,11,(0,119,0,29,044),2,6,5,1,
11,(0,119,0,29,-044),2,6,010,5,1,11,(0,114,0,28,044),2,6,5,1,
11,(0,114,0,28,-044),2,6,010,5,1,11,(0,108,0,27,044),2,6,5,1,
11,(0,108,0,27,-044),2,6,010,5,1,11,(0,102,0,26,044),2,6,5,1,
11,(0,102,0,26,-044),2,6,010,5,1,11,(0,97,0,25,044),2,6,5,1,
11,(0,97,0,25,-044),2,6,010,5,1,11,(0,90,0,24,044),2,6,5,1,
11,(0,90,0,24,-044),2,6,8,(60,-30),0

;; isocp.shp
*00053,37,S
2,8,(24,36),1,12,(-10,2,12),3,10,4,2,12,(-18,-76,100),8,(56,-28),
12,(-18,-76,-100),4,10,3,2,12,(-12,4,-18),2,8,(26,-6),0
"""

TXT = b"""

;;type, size, name
*UNIFONT,6,TXT
;;above, below, mode, encode, embed, 0
6,2,2,0,0,0

;;shapenumber, defbytes, shapename
*0000A,7,lf
2,0AC,14,8,(9,10),0

*00020,7,spc
2,060,14,8,(-6,-8),0

*00041,21,uca
2,14,8,(-2,-6),1,024,043,04D,02C,2,047,1,040,2,02E,14,8,(-4,-3),
0

*00042,29,ucb
2,14,8,(-2,-6),1,030,012,014,016,028,2,020,1,012,014,016,038,2,
010,1,06C,2,050,14,8,(-4,-3),0

*00043,23,ucc
2,14,8,(-2,-6),040,014,1,01A,028,016,044,012,020,01E,2,02E,03C,
14,8,(-4,-3),0

"""


class TestLoadShxShapeFile:
    @pytest.fixture(scope="class")
    def shx(self) -> shapefile.ShapeFile:
        return shapefile.shx_load(SHX0)

    def test_shape_count(self, shx):
        assert len(shx) == 6

    def test_shape_file_has_no_name(self, shx):
        assert shx.name == b""

    # fmt:off
    def test_shape_data(self, shx):
        assert shx[130].data == (0x14, 0x02, 0x1C, 0x01, 0x1C, 0)
        assert shx[130].name == b"TRACK1"
        assert shx[131].data == (0x12, 0x1E, 0)
        assert shx[132].data == (0x14, 0x20, 0x2C, 0x28, 0x14, 0)
        assert shx[133].data == (10, 1, int("-040", 16), 0)
        assert shx[134].data == (0x25, 10, 2, int("-044", 16), 0x2B, 0)
        assert shx[135].data == (
            9, 108, 0, 15, 56, 30, -112, 15, 56, 107, 0, 0, 0, 0,
        )
    # fmt:on


class TestLoadShxFontFile:
    """This is the old 'AutoCAD-86 shapes 1.0' format like the file ISO.shx"""

    @pytest.fixture(scope="class")
    def shx(self) -> shapefile.ShapeFile:
        return shapefile.shx_load(SHX1)

    def test_shape_count(self, shx):
        assert len(shx) == 126

    # fmt:off
    def test_some_shape_data(self, shx):
        # Ampersand "&" of ISO.shx
        # 2,14,8,(-6,-18),8,(12,5),1,8,(-5,-4),13,(-3,-1,-23),
        # (-4,3,-50),(1,2,-23),(6,7,0),(1,3,28),(-7,0,109),
        # (1,-3,28),(0,0),8,(8,-12),2,080,14,0C9,0
        assert shx[38].data == (
            2, 14, 8, -6, -18, 8, 12, 5, 1, 8, -5, -4, 13, -3, -1, -23,
            -4, 3, -50, 1, 2, -23, 6, 7, 0, 1, 3, 28, -7, 0, 109,
            1, -3, 28, 0, 0, 8, 8, -12, 2, 0x80, 14, 0xc9, 0
        )
    # fmt:on


class TestLoadUnifontFile:
    """This is the new format called 'AutoCAD-86 unifont 1.0'."""

    @pytest.fixture(scope="class")
    def shx(self) -> shapefile.ShapeFile:
        return shapefile.shx_load(SHX2)

    def test_shape_count(self, shx):
        assert len(shx) == 8

    # fmt:off
    def test_one_shape_data(self, shx):
        # "&" of isocp.shx
        # *00026,41,&
        # 2,8,(30,14),1,3,10,4,2,8,(-72,-54),3,2,12,(-89,76,-102),
        # 4,4,8,(31,45),3,4,4,10,11,(68,187,0,6,076),8,(17,-27),
        # 2,8,(10,-2),0
        shape = shx[0x26]
        assert shape.number == 0x26
        assert shape.name == b"&"
        assert shape.byte_count == 41
        assert shape.data == (
            2, 8, 30, 14, 1, 3, 10, 4, 2, 8, -72, -54, 3, 2, 12, -89, 76, -102,
            4, 4, 8, 31, 45, 3, 4, 4, 10, 11, 68, 187, 0, 6, 0x76, 8, 17, -27,
            2, 8, 10, -2, 0
        )
    # fmt:on


# fmt:off
SHX0 = bytes([
    0x41, 0x75, 0x74, 0x6F, 0x43, 0x41, 0x44, 0x2D, 0x38, 0x36, 0x20, 0x73,
    0x68, 0x61, 0x70, 0x65, 0x73, 0x20, 0x31, 0x2E, 0x30, 0x0D, 0x0A, 0x1A,
    0x82, 0x00, 0x87, 0x00, 0x06, 0x00, 0x82, 0x00, 0x0D, 0x00, 0x83, 0x00,
    0x07, 0x00, 0x84, 0x00, 0x0A, 0x00, 0x85, 0x00, 0x0A, 0x00, 0x86, 0x00,
    0x0A, 0x00, 0x87, 0x00, 0x17, 0x00, 0x54, 0x52, 0x41, 0x43, 0x4B, 0x31,
    0x00, 0x14, 0x02, 0x1C, 0x01, 0x1C, 0x00, 0x5A, 0x49, 0x47, 0x00, 0x12,
    0x1E, 0x00, 0x42, 0x4F, 0x58, 0x00, 0x14, 0x20, 0x2C, 0x28, 0x14, 0x00,
    0x43, 0x49, 0x52, 0x43, 0x31, 0x00, 0x0A, 0x01, 0xC0, 0x00, 0x42, 0x41,
    0x54, 0x00, 0x25, 0x0A, 0x02, 0xC4, 0x2B, 0x00, 0x41, 0x4D, 0x5A, 0x49,
    0x47, 0x5A, 0x41, 0x47, 0x00, 0x09, 0x6C, 0x00, 0x0F, 0x38, 0x1E, 0x90,
    0x0F, 0x38, 0x6B, 0x00, 0x00, 0x00, 0x00, 0x45, 0x4F, 0x46
])

SHX1 = bytes([
    0x41, 0x75, 0x74, 0x6F, 0x43, 0x41, 0x44, 0x2D, 0x38, 0x36, 0x20, 0x73,
    0x68, 0x61, 0x70, 0x65, 0x73, 0x20, 0x31, 0x2E, 0x30, 0x0D, 0x0A, 0x1A,
    0x00, 0x00, 0xFD, 0x00, 0x7F, 0x00, 0x00, 0x00, 0x4F, 0x00, 0x01, 0x00,
    0x03, 0x00, 0x0A, 0x00, 0x0A, 0x00, 0x0D, 0x00, 0x03, 0x00, 0x20, 0x00,
    0x0A, 0x00, 0x21, 0x00, 0x1D, 0x00, 0x22, 0x00, 0x26, 0x00, 0x23, 0x00,
    0x21, 0x00, 0x24, 0x00, 0x31, 0x00, 0x25, 0x00, 0x20, 0x00, 0x26, 0x00,
    0x2D, 0x00, 0x27, 0x00, 0x20, 0x00, 0x28, 0x00, 0x22, 0x00, 0x29, 0x00,
    0x20, 0x00, 0x2A, 0x00, 0x15, 0x00, 0x2B, 0x00, 0x15, 0x00, 0x2C, 0x00,
    0x1C, 0x00, 0x2D, 0x00, 0x10, 0x00, 0x2E, 0x00, 0x14, 0x00, 0x2F, 0x00,
    0x12, 0x00, 0x30, 0x00, 0x20, 0x00, 0x31, 0x00, 0x12, 0x00, 0x32, 0x00,
    0x1D, 0x00, 0x33, 0x00, 0x1C, 0x00, 0x34, 0x00, 0x1B, 0x00, 0x35, 0x00,
    0x1D, 0x00, 0x36, 0x00, 0x20, 0x00, 0x37, 0x00, 0x15, 0x00, 0x38, 0x00,
    0x2C, 0x00, 0x39, 0x00, 0x1E, 0x00, 0x3A, 0x00, 0x1F, 0x00, 0x3B, 0x00,
    0x1F, 0x00, 0x3C, 0x00, 0x14, 0x00, 0x3D, 0x00, 0x15, 0x00, 0x3E, 0x00,
    0x12, 0x00, 0x3F, 0x00, 0x2B, 0x00, 0x40, 0x00, 0x2E, 0x00, 0x41, 0x00,
    0x1D, 0x00, 0x42, 0x00, 0x1D, 0x00, 0x43, 0x00, 0x19, 0x00, 0x44, 0x00,
    0x1A, 0x00, 0x45, 0x00, 0x18, 0x00, 0x46, 0x00, 0x16, 0x00, 0x47, 0x00,
    0x1E, 0x00, 0x48, 0x00, 0x19, 0x00, 0x49, 0x00, 0x0F, 0x00, 0x4A, 0x00,
    0x14, 0x00, 0x4B, 0x00, 0x1A, 0x00, 0x4C, 0x00, 0x14, 0x00, 0x4D, 0x00,
    0x17, 0x00, 0x4E, 0x00, 0x18, 0x00, 0x4F, 0x00, 0x18, 0x00, 0x50, 0x00,
    0x16, 0x00, 0x51, 0x00, 0x28, 0x00, 0x52, 0x00, 0x19, 0x00, 0x53, 0x00,
    0x30, 0x00, 0x54, 0x00, 0x16, 0x00, 0x55, 0x00, 0x16, 0x00, 0x56, 0x00,
    0x15, 0x00, 0x57, 0x00, 0x1C, 0x00, 0x58, 0x00, 0x15, 0x00, 0x59, 0x00,
    0x18, 0x00, 0x5A, 0x00, 0x16, 0x00, 0x5B, 0x00, 0x22, 0x00, 0x5C, 0x00,
    0x21, 0x00, 0x5D, 0x00, 0x20, 0x00, 0x5E, 0x00, 0x13, 0x00, 0x5F, 0x00,
    0x10, 0x00, 0x60, 0x00, 0x20, 0x00, 0x61, 0x00, 0x21, 0x00, 0x62, 0x00,
    0x1C, 0x00, 0x63, 0x00, 0x18, 0x00, 0x64, 0x00, 0x1C, 0x00, 0x65, 0x00,
    0x19, 0x00, 0x66, 0x00, 0x17, 0x00, 0x67, 0x00, 0x1E, 0x00, 0x68, 0x00,
    0x18, 0x00, 0x69, 0x00, 0x1D, 0x00, 0x6A, 0x00, 0x1E, 0x00, 0x6B, 0x00,
    0x1E, 0x00, 0x6C, 0x00, 0x15, 0x00, 0x6D, 0x00, 0x14, 0x00, 0x6E, 0x00,
    0x14, 0x00, 0x6F, 0x00, 0x18, 0x00, 0x70, 0x00, 0x1A, 0x00, 0x71, 0x00,
    0x1A, 0x00, 0x72, 0x00, 0x14, 0x00, 0x73, 0x00, 0x2D, 0x00, 0x74, 0x00,
    0x16, 0x00, 0x75, 0x00, 0x15, 0x00, 0x76, 0x00, 0x15, 0x00, 0x77, 0x00,
    0x16, 0x00, 0x78, 0x00, 0x17, 0x00, 0x79, 0x00, 0x1C, 0x00, 0x7A, 0x00,
    0x15, 0x00, 0x7B, 0x00, 0x30, 0x00, 0x7C, 0x00, 0x12, 0x00, 0x7D, 0x00,
    0x2E, 0x00, 0x7E, 0x00, 0x17, 0x00, 0x7F, 0x00, 0x13, 0x00, 0x80, 0x00,
    0x18, 0x00, 0x81, 0x00, 0x20, 0x00, 0x84, 0x00, 0x2C, 0x00, 0x8E, 0x00,
    0x28, 0x00, 0x94, 0x00, 0x23, 0x00, 0x99, 0x00, 0x24, 0x00, 0x9A, 0x00,
    0x21, 0x00, 0x9C, 0x00, 0x28, 0x00, 0xD7, 0x00, 0x0D, 0x00, 0xE0, 0x00,
    0x2F, 0x00, 0xE1, 0x00, 0x1C, 0x00, 0xE3, 0x00, 0x21, 0x00, 0xE4, 0x00,
    0x19, 0x00, 0xE5, 0x00, 0x22, 0x00, 0xE6, 0x00, 0x25, 0x00, 0xE7, 0x00,
    0x1C, 0x00, 0xE9, 0x00, 0x21, 0x00, 0xEA, 0x00, 0x24, 0x00, 0xEC, 0x00,
    0x1B, 0x00, 0xED, 0x00, 0x18, 0x00, 0xF1, 0x00, 0x18, 0x00, 0xF2, 0x00,
    0x19, 0x00, 0xF3, 0x00, 0x19, 0x00, 0xF7, 0x00, 0x20, 0x00, 0xF8, 0x00,
    0x13, 0x00, 0xFB, 0x00, 0x1A, 0x00, 0xFD, 0x00, 0x1C, 0x00, 0x49, 0x53,
    0x4F, 0x2C, 0x20, 0x46, 0x6F, 0x6E, 0x74, 0x20, 0x30, 0x39, 0x2F, 0x32,
    0x32, 0x2F, 0x38, 0x36, 0x20, 0x73, 0x6F, 0x6E, 0x64, 0x65, 0x72, 0x7A,
    0x65, 0x69, 0x63, 0x68, 0x65, 0x6E, 0x20, 0x32, 0x32, 0x2E, 0x30, 0x39,
    0x2E, 0x38, 0x37, 0x20, 0x53, 0x69, 0x65, 0x67, 0x65, 0x6D, 0x75, 0x6E,
    0x64, 0x20, 0x69, 0x6E, 0x74, 0x65, 0x72, 0x6E, 0x61, 0x74, 0x69, 0x6F,
    0x6E, 0x61, 0x6C, 0x20, 0x32, 0x32, 0x2E, 0x31, 0x31, 0x2E, 0x39, 0x30,
    0x00, 0x12, 0x06, 0x02, 0x00, 0x00, 0x05, 0x00, 0x00, 0x02, 0x08, 0x00,
    0xE2, 0x0E, 0x08, 0x1A, 0x1E, 0x00, 0x00, 0x06, 0x00, 0x00, 0x02, 0x08,
    0x12, 0x00, 0x0E, 0x08, 0xEE, 0xEE, 0x00, 0x00, 0x02, 0x03, 0x02, 0x0E,
    0x08, 0xFF, 0xDC, 0x10, 0x01, 0x0A, 0x01, 0x60, 0x02, 0xC4, 0x01, 0xC4,
    0xC4, 0x02, 0x08, 0x0D, 0xDC, 0x0E, 0x08, 0xF3, 0xF4, 0x04, 0x02, 0x00,
    0x00, 0x02, 0x0E, 0x03, 0x02, 0x0E, 0x08, 0xF9, 0xDC, 0x0E, 0x04, 0x02,
    0xD4, 0x01, 0x08, 0x01, 0x05, 0x02, 0x60, 0x01, 0x08, 0xFF, 0xFB, 0x02,
    0x08, 0x07, 0xF3, 0x0E, 0x03, 0x02, 0x0E, 0x08, 0xED, 0x0E, 0x0E, 0x04,
    0x02, 0x00, 0x00, 0x02, 0x0E, 0x08, 0xF9, 0xF4, 0x20, 0x01, 0xC3, 0x02,
    0x40, 0x01, 0xCB, 0x02, 0x08, 0x06, 0x04, 0x01, 0xC8, 0x02, 0x08, 0x02,
    0x04, 0x01, 0xC0, 0x02, 0x6E, 0x2C, 0x0E, 0x08, 0xF3, 0xFA, 0x00, 0x00,
    0x02, 0x0E, 0x08, 0xFB, 0xEB, 0x14, 0x01, 0x0C, 0x03, 0xFF, 0x28, 0x40,
    0x0C, 0x01, 0x07, 0x47, 0x08, 0xF9, 0x05, 0x0C, 0x02, 0x06, 0xAB, 0x40,
    0x0C, 0x03, 0xFF, 0xD8, 0x02, 0x08, 0xFD, 0x04, 0x01, 0xCC, 0xCC, 0x02,
    0x48, 0x01, 0xC4, 0xC4, 0x02, 0xDE, 0x8C, 0x0E, 0x08, 0xF5, 0xF7, 0x00,
    0x00, 0x02, 0x0E, 0x08, 0xF4, 0xEE, 0x60, 0x01, 0x08, 0x0C, 0x12, 0x02,
    0xE8, 0x01, 0x0A, 0x04, 0x20, 0x02, 0x08, 0x10, 0xEE, 0x01, 0x0A, 0x04,
    0x60, 0x02, 0xA0, 0x0E, 0x08, 0xEE, 0xFA, 0x00, 0x00, 0x02, 0x0E, 0x08,
    0xFA, 0xEE, 0x08, 0x0C, 0x05, 0x01, 0x08, 0xFB, 0xFC, 0x0D, 0xFD, 0xFF,
    0xE9, 0xFC, 0x03, 0xCE, 0x01, 0x02, 0xE9, 0x06, 0x07, 0x00, 0x01, 0x03,
    0x1C, 0xF9, 0x00, 0x6D, 0x01, 0xFD, 0x1C, 0x00, 0x00, 0x08, 0x08, 0xF4,
    0x02, 0x80, 0x0E, 0xC9, 0x00, 0x00, 0x02, 0x0E, 0x03, 0x02, 0x0E, 0x08,
    0xFF, 0xDC, 0x0E, 0x04, 0x02, 0xD4, 0x01, 0x08, 0x01, 0x05, 0x02, 0x08,
    0x06, 0xEE, 0x0E, 0x03, 0x02, 0x0E, 0x08, 0xF3, 0x0E, 0x0E, 0x04, 0x02,
    0x00, 0x00, 0x02, 0x0E, 0x03, 0x02, 0x0E, 0x08, 0xFD, 0xDC, 0x0E, 0x04,
    0x02, 0x08, 0x03, 0xFE, 0x01, 0x0C, 0x00, 0x14, 0xDA, 0x02, 0x6E, 0xCC,
    0x0E, 0x03, 0x02, 0x0E, 0x08, 0xF1, 0xF0, 0x0E, 0x04, 0x02, 0x00, 0x00,
    0x02, 0x0E, 0x03, 0x02, 0x0E, 0x08, 0xFD, 0xDC, 0x0E, 0x04, 0x02, 0x2C,
    0x01, 0x0C, 0x00, 0x14, 0x26, 0x02, 0x9E, 0x9C, 0x0E, 0x03, 0x02, 0x0E,
    0x08, 0xF1, 0xF0, 0x0E, 0x04, 0x02, 0x00, 0x00, 0x02, 0x0E, 0x08, 0xFC,
    0xF6, 0x24, 0x01, 0x82, 0x02, 0x88, 0x01, 0x8E, 0x02, 0x2E, 0x40, 0x0E,
    0x08, 0xF6, 0xFC, 0x00, 0x00, 0x02, 0x0E, 0x08, 0xFC, 0xF6, 0x64, 0x01,
    0x80, 0x02, 0x46, 0x01, 0x8C, 0x02, 0x2E, 0x80, 0x0E, 0x08, 0xF6, 0xFC,
    0x00, 0x00, 0x02, 0x0E, 0x03, 0x02, 0x0E, 0x18, 0x0E, 0x04, 0x02, 0x5C,
    0x01, 0x08, 0x01, 0x05, 0x02, 0x60, 0x0E, 0x03, 0x02, 0x0E, 0x08, 0xF3,
    0xEA, 0x0E, 0x04, 0x02, 0x00, 0x00, 0x02, 0x0E, 0x08, 0xFC, 0xFA, 0x64,
    0x01, 0x80, 0x02, 0x6E, 0x0E, 0x08, 0xF6, 0x00, 0x00, 0x00, 0x02, 0x03,
    0x02, 0x0E, 0x2B, 0x10, 0x01, 0x0A, 0x01, 0x60, 0x02, 0xD0, 0x0E, 0x08,
    0xF3, 0xF4, 0x04, 0x02, 0x00, 0x00, 0x02, 0x0E, 0x08, 0xFD, 0xEE, 0x01,
    0x08, 0x06, 0x12, 0x02, 0x6E, 0xCC, 0x0E, 0x08, 0xF7, 0xFA, 0x00, 0x00,
    0x02, 0x0E, 0x08, 0xFC, 0xEE, 0x23, 0x01, 0x0D, 0x06, 0x00, 0x54, 0x00,
    0x0E, 0x12, 0xFA, 0x00, 0x55, 0x00, 0xF2, 0x12, 0x00, 0x00, 0x02, 0x08,
    0x0D, 0xFE, 0x0E, 0x08, 0xF6, 0xFA, 0x00, 0x00, 0x02, 0x0E, 0x08, 0xFE,
    0xEE, 0xE4, 0x01, 0x42, 0xFC, 0x3C, 0x02, 0x60, 0x0E, 0x08, 0xF8, 0xFA,
    0x00, 0x00, 0x02, 0x0E, 0x08, 0xFC, 0xEE, 0x80, 0x01, 0x88, 0x08, 0x07,
    0x0B, 0x0C, 0x01, 0x04, 0x19, 0x0C, 0xF8, 0x00, 0x5F, 0x02, 0x08, 0x0E,
    0xF1, 0x0E, 0x08, 0xF6, 0xFA, 0x00, 0x00, 0x02, 0x0E, 0x08, 0xFC, 0xEE,
    0x01, 0x30, 0x0A, 0x05, 0x64, 0x18, 0x02, 0x20, 0x01, 0x0A, 0x04, 0x64,
    0x48, 0x02, 0x08, 0x0E, 0xEE, 0x0E, 0x08, 0xF6, 0xFA, 0x00, 0x00, 0x02,
    0x0E, 0x08, 0xFB, 0xEE, 0x70, 0x01, 0x84, 0x02, 0x08, 0xFD, 0x0A, 0x01,
    0x08, 0xFC, 0xF2, 0xA0, 0x02, 0x08, 0x06, 0xFC, 0x0E, 0x08, 0xF5, 0xFA,
    0x00, 0x00, 0x02, 0x0E, 0x08, 0xFC, 0xEE, 0x01, 0x50, 0x0C, 0x03, 0x04,
    0x29, 0x24, 0x0C, 0xFD, 0x04, 0x29, 0x58, 0x84, 0x80, 0x02, 0x08, 0x06,
    0xEE, 0x0E, 0x08, 0xF6, 0xFA, 0x00, 0x00, 0x02, 0x0E, 0x08, 0xFC, 0xEE,
    0x08, 0x06, 0x12, 0x01, 0x0C, 0xFA, 0xF6, 0x27, 0x4C, 0x0A, 0x04, 0x44,
    0x24, 0x0A, 0x04, 0x02, 0x38, 0x02, 0x08, 0x0D, 0xF6, 0x0E, 0x08, 0xF6,
    0xFA, 0x00, 0x00, 0x02, 0x0E, 0x08, 0xFC, 0xEE, 0x30, 0x01, 0x08, 0x05,
    0x12, 0x88, 0x02, 0x08, 0x0E, 0xEE, 0x0E, 0x08, 0xF6, 0xFA, 0x00, 0x00,
    0x02, 0x0E, 0x08, 0xFC, 0xEE, 0x44, 0x01, 0x0A, 0x04, 0x44, 0x24, 0x0A,
    0x04, 0x04, 0x2C, 0x02, 0x94, 0x01, 0x0A, 0x03, 0x42, 0x20, 0x0A, 0x03,
    0x62, 0x24, 0x0A, 0x03, 0x02, 0x28, 0x0A, 0x03, 0x22, 0x2C, 0x02, 0x08,
    0x0E, 0xF3, 0x0E, 0x08, 0xF6, 0xFA, 0x00, 0x00, 0x02, 0x0E, 0x08, 0xFC,
    0xEE, 0x30, 0x01, 0x0C, 0x05, 0x0A, 0x17, 0x44, 0x0A, 0x04, 0x04, 0x2C,
    0x0A, 0x04, 0x42, 0x30, 0x02, 0x08, 0x07, 0xF8, 0x0E, 0x08, 0xF6, 0xFA,
    0x00, 0x00, 0x02, 0x03, 0x02, 0x0E, 0x08, 0xFF, 0xEE, 0x10, 0x01, 0x0A,
    0x01, 0x60, 0x02, 0x84, 0x84, 0x01, 0x0A, 0x01, 0x60, 0x02, 0x08, 0x0D,
    0xF0, 0x0E, 0x08, 0xF3, 0xF4, 0x04, 0x02, 0x00, 0x00, 0x02, 0x03, 0x02,
    0x0E, 0x08, 0xFF, 0xEE, 0xAC, 0x01, 0x08, 0x02, 0x0A, 0x02, 0xF4, 0x16,
    0x01, 0x0A, 0x01, 0x60, 0x02, 0x08, 0x0D, 0xF0, 0x0E, 0x08, 0xF3, 0xEA,
    0x04, 0x02, 0x00, 0x00, 0x02, 0x0E, 0x08, 0xFC, 0xF6, 0x08, 0x08, 0x02,
    0x01, 0x87, 0x81, 0x02, 0x6E, 0x4C, 0x0E, 0x08, 0xF6, 0xFC, 0x00, 0x00,
    0x02, 0x0E, 0x08, 0xFC, 0xF7, 0x34, 0x01, 0x80, 0x02, 0x64, 0x01, 0x88,
    0x02, 0x9E, 0x50, 0x0E, 0x08, 0xF6, 0xFD, 0x00, 0x00, 0x02, 0x0E, 0x08,
    0xFC, 0xF6, 0x24, 0x01, 0x81, 0x87, 0x02, 0xAE, 0x40, 0x0E, 0x08, 0xF6,
    0xFC, 0x00, 0x00, 0x02, 0x0E, 0x08, 0xFC, 0xEE, 0x40, 0x01, 0x03, 0x02,
    0x0A, 0x01, 0x60, 0x04, 0x02, 0x02, 0x54, 0x01, 0x34, 0x0D, 0x02, 0x03,
    0xCF, 0x02, 0x04, 0x28, 0xFC, 0x03, 0x39, 0xFC, 0xFE, 0x28, 0x00, 0x00,
    0x02, 0x08, 0x0E, 0xF0, 0x0E, 0x08, 0xF6, 0xFA, 0x00, 0x00, 0x02, 0x0E,
    0x08, 0xF6, 0xEC, 0x08, 0x12, 0x02, 0x01, 0x0C, 0xF8, 0xFE, 0xE7, 0x0A,
    0x0A, 0xE6, 0x0C, 0xFE, 0xFC, 0xE9, 0x0C, 0xFC, 0x00, 0xCD, 0x94, 0x48,
    0x0A, 0x04, 0x22, 0x4C, 0x0A, 0x03, 0x42, 0x20, 0x0A, 0x03, 0x62, 0x02,
    0x7E, 0x50, 0x0E, 0x08, 0xF0, 0xFA, 0x00, 0x00, 0x02, 0x0E, 0x08, 0xFA,
    0xEE, 0xC0, 0x01, 0x08, 0xFA, 0x12, 0x08, 0xFA, 0xEE, 0x02, 0x08, 0x02,
    0x05, 0x01, 0x80, 0x02, 0x08, 0x08, 0xFB, 0x0E, 0x08, 0xF4, 0xFA, 0x00,
    0x00, 0x02, 0x0E, 0x08, 0xFB, 0xEE, 0x01, 0xF4, 0x34, 0x50, 0x0A, 0x04,
    0xA4, 0x58, 0x02, 0x50, 0x01, 0x0A, 0x05, 0xA4, 0x58, 0x02, 0xF0, 0x10,
    0x0E, 0x08, 0xF5, 0xFA, 0x00, 0x00, 0x02, 0x0E, 0x08, 0xFC, 0xEE, 0x80,
    0x01, 0x48, 0x0A, 0x04, 0xE2, 0xA4, 0x0A, 0x04, 0xC2, 0x40, 0x02, 0xCC,
    0x6E, 0x0E, 0x08, 0xF6, 0xFA, 0x00, 0x00, 0x02, 0x0E, 0x08, 0xFB, 0xEE,
    0x01, 0x60, 0x0A, 0x04, 0x62, 0xA4, 0x0A, 0x04, 0x02, 0x68, 0xFC, 0x3C,
    0x02, 0xF0, 0x10, 0x0E, 0x08, 0xF5, 0xFA, 0x00, 0x00, 0x02, 0x0E, 0x08,
    0xFC, 0xEE, 0x80, 0x01, 0x88, 0xF4, 0x34, 0x80, 0x02, 0x8A, 0x01, 0x60,
    0x02, 0x2C, 0x8E, 0x0E, 0x08, 0xF6, 0xFA, 0x00, 0x00, 0x02, 0x0E, 0x08,
    0xFC, 0xEE, 0x01, 0xF4, 0x34, 0x80, 0x02, 0x8A, 0x01, 0x80, 0x02, 0x4C,
    0x6E, 0x0E, 0x08, 0xF6, 0xFA, 0x00, 0x00, 0x02, 0x0E, 0x08, 0xFB, 0xEE,
    0x08, 0x0A, 0x12, 0x01, 0x68, 0x0A, 0x04, 0x22, 0xAC, 0x0A, 0x04, 0x42,
    0x60, 0xA4, 0x38, 0x02, 0x08, 0x09, 0xF6, 0x0E, 0x08, 0xF5, 0xFA, 0x00,
    0x00, 0x02, 0x0E, 0x08, 0xFB, 0xEE, 0x01, 0xF4, 0x34, 0x02, 0x8C, 0x01,
    0xA0, 0x02, 0x84, 0x01, 0xFC, 0x3C, 0x02, 0x60, 0x0E, 0x08, 0xF5, 0xFA,
    0x00, 0x00, 0x02, 0x0E, 0x08, 0x00, 0xEE, 0x01, 0xF4, 0x34, 0x02, 0xCC,
    0x6E, 0x0E, 0x6A, 0x00, 0x00, 0x02, 0x0E, 0x08, 0xFD, 0xEE, 0x01, 0x20,
    0x0A, 0x04, 0x62, 0xE4, 0x02, 0xCC, 0x6E, 0x0E, 0x08, 0xF7, 0xFA, 0x00,
    0x00, 0x02, 0x0E, 0x08, 0xFB, 0xEE, 0x01, 0xF4, 0x34, 0x02, 0xA0, 0x01,
    0xAA, 0x02, 0x32, 0x01, 0x08, 0x07, 0xF5, 0x02, 0x60, 0x0E, 0x08, 0xF5,
    0xFA, 0x00, 0x00, 0x02, 0x0E, 0x08, 0xFC, 0xEE, 0x80, 0x01, 0x88, 0xF4,
    0x34, 0x02, 0x08, 0x0E, 0xEE, 0x0E, 0x08, 0xF6, 0xFA, 0x00, 0x00, 0x02,
    0x0E, 0x08, 0xFA, 0xEE, 0x01, 0x09, 0x00, 0x12, 0x06, 0xF6, 0x06, 0x0A,
    0x00, 0xEE, 0x00, 0x00, 0x02, 0x60, 0x0E, 0xC9, 0x00, 0x00, 0x02, 0x0E,
    0x08, 0xFB, 0xEE, 0x01, 0x09, 0x00, 0x12, 0x0A, 0xEE, 0x00, 0x12, 0x00,
    0x00, 0x02, 0xCC, 0x6E, 0x0E, 0x08, 0xF5, 0xFA, 0x00, 0x00, 0x02, 0x0E,
    0x08, 0xFB, 0xEE, 0x54, 0x01, 0x0A, 0x05, 0x44, 0x84, 0x0A, 0x05, 0x04,
    0x8C, 0x02, 0x5E, 0xB0, 0x0E, 0x08, 0xF5, 0xFA, 0x00, 0x00, 0x02, 0x0E,
    0x08, 0xFB, 0xEE, 0x01, 0xF4, 0x34, 0x50, 0x0A, 0x05, 0xA4, 0x58, 0x02,
    0x8E, 0x80, 0x0E, 0x08, 0xF5, 0xFA, 0x00, 0x00, 0x02, 0x0E, 0x03, 0x02,
    0x0E, 0x08, 0xF5, 0xDC, 0x0E, 0x04, 0x02, 0x54, 0x01, 0x0A, 0x05, 0x44,
    0x84, 0x0A, 0x05, 0x04, 0x8C, 0x02, 0x70, 0x1C, 0x01, 0x4E, 0x02, 0x60,
    0x0E, 0x03, 0x02, 0x0E, 0x08, 0xE9, 0xF4, 0x0E, 0x04, 0x02, 0x00, 0x00,
    0x02, 0x0E, 0x08, 0xFB, 0xEE, 0x01, 0xF4, 0x34, 0x50, 0x0A, 0x05, 0xA4,
    0x58, 0x02, 0x60, 0x01, 0x8D, 0x02, 0x60, 0x0E, 0x08, 0xF5, 0xFA, 0x00,
    0x00, 0x02, 0x0E, 0x03, 0x02, 0x0E, 0x08, 0xF7, 0xDC, 0x0E, 0x04, 0x02,
    0x14, 0x01, 0x0C, 0x05, 0xFF, 0x19, 0x0C, 0x02, 0x08, 0x5F, 0x08, 0xFB,
    0x03, 0x0C, 0xFF, 0x06, 0xC5, 0x0C, 0x08, 0x00, 0xE0, 0x02, 0x08, 0x06,
    0xEF, 0x0E, 0x03, 0x02, 0x0E, 0x08, 0xEB, 0xF4, 0x0E, 0x04, 0x02, 0x00,
    0x00, 0x02, 0x0E, 0x08, 0xFB, 0xEE, 0x50, 0x01, 0xF4, 0x34, 0x02, 0x58,
    0x01, 0xA0, 0x02, 0x6E, 0xCC, 0x0E, 0x08, 0xF5, 0xFA, 0x00, 0x00, 0x02,
    0x0E, 0x08, 0xFB, 0xEE, 0xF4, 0x34, 0x01, 0xDC, 0x0A, 0x05, 0x44, 0xD4,
    0x02, 0x6E, 0xCC, 0x0E, 0x08, 0xF5, 0xFA, 0x00, 0x00, 0x02, 0x0E, 0x08,
    0xFA, 0xEE, 0xF4, 0x34, 0x01, 0x08, 0x06, 0xEE, 0x08, 0x06, 0x12, 0x02,
    0x6E, 0xCC, 0x0E, 0xC9, 0x00, 0x00, 0x02, 0x0E, 0x08, 0xF8, 0xEE, 0xF4,
    0x34, 0x01, 0x09, 0x04, 0xEE, 0x04, 0x0C, 0x04, 0xF4, 0x04, 0x12, 0x00,
    0x00, 0x02, 0x6E, 0xCC, 0x0E, 0x08, 0xF2, 0xFA, 0x00, 0x00, 0x02, 0x0E,
    0x08, 0xFA, 0xEE, 0x01, 0x08, 0x0C, 0x12, 0x02, 0xC8, 0x01, 0x08, 0x0C,
    0xEE, 0x02, 0x60, 0x0E, 0xC9, 0x00, 0x00, 0x02, 0x0E, 0x08, 0xFA, 0xEE,
    0x60, 0x01, 0xA4, 0x08, 0xFA, 0x08, 0x02, 0xC0, 0x01, 0x08, 0xFA, 0xF8,
    0x02, 0xAE, 0x20, 0x0E, 0xC9, 0x00, 0x00, 0x02, 0x0E, 0x08, 0xFB, 0xEE,
    0xA0, 0x01, 0xA8, 0x08, 0x0A, 0x12, 0xA8, 0x02, 0x08, 0x10, 0xEE, 0x0E,
    0x08, 0xF5, 0xFA, 0x00, 0x00, 0x02, 0x0E, 0x03, 0x02, 0x0E, 0x08, 0xFD,
    0xDC, 0x0E, 0x04, 0x02, 0x08, 0x03, 0xFE, 0x01, 0x38, 0xA4, 0xA4, 0x30,
    0x02, 0x6E, 0xCC, 0x0E, 0x03, 0x02, 0x0E, 0x08, 0xF1, 0xF0, 0x0E, 0x04,
    0x02, 0x00, 0x00, 0x02, 0x0E, 0x03, 0x02, 0x0E, 0x08, 0xF9, 0xDC, 0x0E,
    0x04, 0x02, 0x08, 0x07, 0xFE, 0x01, 0x08, 0xF9, 0x14, 0x02, 0xDE, 0x5C,
    0x0E, 0x03, 0x02, 0x0E, 0x08, 0xED, 0xF0, 0x0E, 0x04, 0x02, 0x00, 0x00,
    0x02, 0x0E, 0x03, 0x02, 0x0E, 0x08, 0xFD, 0xDC, 0x0E, 0x04, 0x02, 0x2C,
    0x01, 0x30, 0xA4, 0xA4, 0x38, 0x02, 0x9E, 0x9C, 0x0E, 0x03, 0x02, 0x0E,
    0x08, 0xF1, 0xF0, 0x0E, 0x04, 0x02, 0x00, 0x00, 0x02, 0x0E, 0x08, 0xFC,
    0xEC, 0x84, 0x84, 0x01, 0x42, 0x4E, 0x02, 0x6E, 0xAC, 0x0E, 0x08, 0xF6,
    0x0A, 0x00, 0x00, 0x02, 0x0E, 0x08, 0xFB, 0x02, 0x2C, 0x01, 0xA0, 0x02,
    0x24, 0x0E, 0x08, 0xFB, 0xF8, 0x00, 0x00, 0x02, 0x0E, 0x03, 0x02, 0x0E,
    0x08, 0xFF, 0xDC, 0x0E, 0x04, 0x02, 0x94, 0x94, 0x01, 0x08, 0x01, 0xFB,
    0x02, 0x6E, 0x7C, 0x0E, 0x03, 0x02, 0x0E, 0x08, 0xF3, 0x0E, 0x0E, 0x04,
    0x02, 0x00, 0x00, 0x02, 0x0E, 0x08, 0xFB, 0xF4, 0x08, 0x08, 0x03, 0x01,
    0x0A, 0x03, 0x82, 0x28, 0x0A, 0x03, 0xE2, 0x54, 0x0A, 0x04, 0xC2, 0x40,
    0xAC, 0x0A, 0x02, 0x42, 0x02, 0x60, 0x0E, 0x08, 0xF5, 0xFA, 0x00, 0x00,
    0x02, 0x0E, 0x08, 0xFC, 0xEE, 0xF4, 0x34, 0x01, 0xFC, 0x3C, 0x50, 0x0A,
    0x03, 0x62, 0x64, 0x0A, 0x03, 0x02, 0x58, 0x02, 0xCE, 0x20, 0x0E, 0x08,
    0xF6, 0xFA, 0x00, 0x00, 0x02, 0x0E, 0x08, 0xFD, 0xF4, 0x60, 0x01, 0x38,
    0x0A, 0x03, 0xE2, 0x64, 0x0A, 0x03, 0xC2, 0x30, 0x02, 0xCD, 0x0E, 0x08,
    0xF7, 0xFA, 0x00, 0x00, 0x02, 0x0E, 0x08, 0xFC, 0xEE, 0x08, 0x08, 0x12,
    0x01, 0xFC, 0x3C, 0x58, 0x0A, 0x03, 0xE2, 0x64, 0x0A, 0x03, 0xC2, 0x50,
    0x02, 0xCD, 0x0E, 0x08, 0xF6, 0xFA, 0x00, 0x00, 0x02, 0x0E, 0x08, 0xFC,
    0xF4, 0x64, 0x01, 0x80, 0x24, 0x0A, 0x04, 0x04, 0x5C, 0x0A, 0x03, 0x42,
    0x50, 0x02, 0x60, 0x0E, 0x08, 0xF6, 0xFA, 0x00, 0x00, 0x02, 0x0E, 0x08,
    0xFD, 0xEE, 0x20, 0x01, 0xE4, 0x0A, 0x04, 0xC2, 0x02, 0x6A, 0x01, 0x60,
    0x02, 0xCD, 0x0E, 0x08, 0xF7, 0xFA, 0x00, 0x00, 0x02, 0x0E, 0x08, 0xFC,
    0xF4, 0x80, 0x01, 0x58, 0x0A, 0x03, 0xE2, 0x64, 0x0A, 0x03, 0xC2, 0x50,
    0xFC, 0x0A, 0x03, 0x82, 0x58, 0x02, 0x62, 0x80, 0x0E, 0x08, 0xF6, 0xF4,
    0x00, 0x00, 0x02, 0x0E, 0x08, 0xFC, 0xEE, 0x01, 0xF4, 0x34, 0x02, 0x6C,
    0x01, 0x50, 0x0A, 0x03, 0xA2, 0x9C, 0x02, 0x60, 0x0E, 0x08, 0xF6, 0xFA,
    0x00, 0x00, 0x02, 0x03, 0x02, 0x0E, 0x08, 0xFF, 0xDC, 0x10, 0x01, 0xC4,
    0xC4, 0x02, 0xC4, 0x01, 0x0A, 0x01, 0x20, 0x02, 0x08, 0x0D, 0xDC, 0x0E,
    0x08, 0xF3, 0xF4, 0x04, 0x02, 0x00, 0x00, 0x02, 0x0E, 0x08, 0xFE, 0xEE,
    0x6C, 0x01, 0x0A, 0x04, 0x62, 0xE4, 0x02, 0x64, 0x01, 0x03, 0x02, 0x0A,
    0x01, 0x20, 0x04, 0x02, 0x02, 0x6E, 0xCC, 0x0E, 0x08, 0xF8, 0xF4, 0x00,
    0x00, 0x02, 0x0E, 0x08, 0xFC, 0xEE, 0x01, 0xF4, 0x34, 0x02, 0xCC, 0x01,
    0x08, 0x08, 0x06, 0x02, 0x08, 0xFB, 0xFC, 0x01, 0x08, 0x05, 0xF8, 0x02,
    0x60, 0x0E, 0x08, 0xF6, 0xFA, 0x00, 0x00, 0x02, 0x0E, 0x08, 0xFE, 0xEE,
    0x40, 0x01, 0x18, 0x0A, 0x03, 0xE2, 0xF4, 0x02, 0xAE, 0x8C, 0x0E, 0x08,
    0xF8, 0xFA, 0x00, 0x00, 0x02, 0x0E, 0xCB, 0x01, 0xC4, 0x90, 0x0A, 0x03,
    0xA2, 0x9C, 0x02, 0x68, 0x01, 0xC4, 0x02, 0xCE, 0x0E, 0xC9, 0x00, 0x00,
    0x02, 0x0E, 0x08, 0xFC, 0xF4, 0x01, 0xC4, 0x50, 0x0A, 0x03, 0xA2, 0x9C,
    0x02, 0x60, 0x0E, 0x08, 0xF6, 0xFA, 0x00, 0x00, 0x02, 0x0E, 0x08, 0xFC,
    0xF4, 0x44, 0x01, 0x0A, 0x04, 0x44, 0x44, 0x0A, 0x04, 0x04, 0x4C, 0x02,
    0x4E, 0xA0, 0x0E, 0x08, 0xF6, 0xFA, 0x00, 0x00, 0x02, 0x0E, 0x08, 0xFC,
    0xF4, 0x6C, 0x01, 0xF4, 0x34, 0x50, 0x0A, 0x03, 0xA2, 0x6C, 0x0A, 0x03,
    0x82, 0x58, 0x02, 0xE0, 0x0E, 0x08, 0xF6, 0xF4, 0x00, 0x00, 0x02, 0x0E,
    0x08, 0xFC, 0xF4, 0x80, 0x01, 0x58, 0x0A, 0x03, 0xE2, 0x64, 0x0A, 0x03,
    0xC2, 0x50, 0xFC, 0x3C, 0x02, 0x62, 0x0E, 0x08, 0xF6, 0xF4, 0x00, 0x00,
    0x02, 0x0E, 0x08, 0xFD, 0xF4, 0x01, 0xC4, 0x40, 0x0A, 0x02, 0xA2, 0x02,
    0x6E, 0x4C, 0x0E, 0x08, 0xF7, 0xFA, 0x00, 0x00, 0x02, 0x0E, 0x08, 0xFC,
    0xF4, 0x24, 0x01, 0x0C, 0x03, 0xFE, 0x26, 0x20, 0x0C, 0x03, 0x02, 0x26,
    0x14, 0x0C, 0xFE, 0x02, 0x1A, 0x47, 0x0C, 0xFE, 0x02, 0xE6, 0x14, 0x0C,
    0x03, 0x02, 0xDA, 0x20, 0x0C, 0x03, 0xFE, 0xDA, 0x02, 0x6E, 0x4C, 0x0E,
    0x08, 0xF6, 0xFA, 0x00, 0x00, 0x02, 0x0E, 0x08, 0xFD, 0xEE, 0x20, 0x01,
    0xF4, 0x34, 0x02, 0x6C, 0x28, 0x01, 0x60, 0x02, 0xCD, 0x0E, 0x08, 0xF7,
    0xFA, 0x00, 0x00, 0x02, 0x0E, 0x08, 0xFC, 0xF4, 0xC4, 0x01, 0x9C, 0x0A,
    0x03, 0x42, 0x50, 0xC4, 0x02, 0xCD, 0x0E, 0x08, 0xF6, 0xFA, 0x00, 0x00,
    0x02, 0x0E, 0x08, 0xFC, 0xF4, 0xC4, 0x01, 0x08, 0x04, 0xF4, 0x08, 0x04,
    0x0C, 0x02, 0xCD, 0x0E, 0x08, 0xF6, 0xFA, 0x00, 0x00, 0x02, 0x0E, 0xCB,
    0xC4, 0x01, 0x09, 0x03, 0xF4, 0x03, 0x08, 0x03, 0xF8, 0x03, 0x0C, 0x00,
    0x00, 0x02, 0xCD, 0x0E, 0xC9, 0x00, 0x00, 0x02, 0x0E, 0x08, 0xFC, 0xF4,
    0x01, 0x08, 0x08, 0x0C, 0x02, 0x88, 0x01, 0x08, 0x08, 0xF4, 0x02, 0x60,
    0x0E, 0x08, 0xF6, 0xFA, 0x00, 0x00, 0x02, 0x0E, 0x08, 0xFC, 0xF4, 0x6C,
    0x01, 0x0C, 0x03, 0x03, 0x1E, 0x08, 0x05, 0x0F, 0x02, 0x88, 0x01, 0x08,
    0x04, 0xF4, 0x02, 0xA0, 0x0E, 0x08, 0xF6, 0xF4, 0x00, 0x00, 0x02, 0x0E,
    0x08, 0xFC, 0xF4, 0x80, 0x01, 0x88, 0x08, 0x08, 0x0C, 0x88, 0x02, 0xCE,
    0x20, 0x0E, 0x08, 0xF6, 0xFA, 0x00, 0x00, 0x02, 0x0E, 0x03, 0x02, 0x0E,
    0x08, 0xFB, 0xDC, 0x0E, 0x04, 0x02, 0x08, 0x05, 0xFE, 0x01, 0x0A, 0x03,
    0xE2, 0x44, 0x0C, 0xFF, 0x02, 0x1E, 0x16, 0x12, 0x0C, 0x01, 0x02, 0x1E,
    0x44, 0x0A, 0x03, 0xC2, 0x02, 0x6E, 0xCC, 0x0E, 0x03, 0x02, 0x0E, 0x08,
    0xEF, 0xF0, 0x0E, 0x04, 0x02, 0x00, 0x00, 0x02, 0x0E, 0x08, 0x00, 0xEE,
    0x2C, 0x01, 0xA4, 0xA4, 0x02, 0x6E, 0xCC, 0x0E, 0x08, 0xFA, 0xF8, 0x00,
    0x00, 0x02, 0x0E, 0x03, 0x02, 0x0E, 0x08, 0xFB, 0xDC, 0x0E, 0x04, 0x02,
    0x2C, 0x01, 0x0A, 0x03, 0x62, 0x44, 0x0C, 0x01, 0x02, 0xE2, 0x12, 0x16,
    0x0C, 0xFF, 0x02, 0xE2, 0x44, 0x0A, 0x03, 0x02, 0x02, 0xBE, 0x7C, 0x0E,
    0x03, 0x02, 0x0E, 0x08, 0xEF, 0xF0, 0x0E, 0x04, 0x02, 0x00, 0x00, 0x02,
    0x0E, 0x08, 0xFB, 0xFA, 0x64, 0x01, 0x0C, 0x05, 0x00, 0x9A, 0x0C, 0x05,
    0x00, 0x66, 0x02, 0x6E, 0x0E, 0x08, 0xF5, 0xFE, 0x00, 0x00, 0x02, 0x0E,
    0x08, 0xFD, 0xEE, 0xF4, 0x01, 0x0A, 0x03, 0x40, 0x02, 0xFC, 0xC0, 0x0E,
    0x08, 0xF7, 0x06, 0x00, 0x00, 0x02, 0x0E, 0x08, 0xFC, 0xF2, 0x01, 0x80,
    0x02, 0xA4, 0x01, 0x88, 0x02, 0x42, 0x01, 0x8C, 0x02, 0x6E, 0x40, 0x0E,
    0x08, 0xF6, 0xFA, 0x00, 0x00, 0x02, 0x0E, 0x08, 0xFC, 0xF4, 0xF4, 0x20,
    0x01, 0x34, 0x02, 0x40, 0x01, 0x3C, 0x02, 0x68, 0xFC, 0xC4, 0x01, 0x9C,
    0x0A, 0x03, 0x42, 0x50, 0xC4, 0x02, 0xCD, 0x0E, 0x08, 0xF6, 0xFA, 0x00,
    0x00, 0x02, 0x0E, 0x08, 0xFB, 0xF4, 0xF4, 0x20, 0x01, 0x34, 0x02, 0x40,
    0x01, 0x3C, 0x02, 0x68, 0xFC, 0x08, 0x08, 0x03, 0x01, 0x0A, 0x03, 0x82,
    0x28, 0x0A, 0x03, 0xE2, 0x54, 0x0A, 0x04, 0xC2, 0x40, 0xAC, 0x0A, 0x02,
    0x42, 0x02, 0x60, 0x0E, 0x08, 0xF5, 0xFA, 0x00, 0x00, 0x02, 0x0E, 0x08,
    0xFA, 0xEE, 0xF4, 0x10, 0x01, 0x34, 0x02, 0xA0, 0x01, 0x3C, 0x02, 0xB8,
    0xFC, 0xC0, 0x01, 0x08, 0xFA, 0x12, 0x08, 0xFA, 0xEE, 0x02, 0x08, 0x02,
    0x05, 0x01, 0x80, 0x02, 0x08, 0x08, 0xFB, 0x0E, 0x08, 0xF4, 0xFA, 0x00,
    0x00, 0x02, 0x0E, 0x08, 0xFC, 0xF4, 0xF4, 0x20, 0x01, 0x34, 0x02, 0x40,
    0x01, 0x3C, 0x02, 0x68, 0xFC, 0x44, 0x01, 0x0A, 0x04, 0x44, 0x44, 0x0A,
    0x04, 0x04, 0x4C, 0x02, 0x4E, 0xA0, 0x0E, 0x08, 0xF6, 0xFA, 0x00, 0x00,
    0x02, 0x0E, 0x08, 0xFB, 0xEE, 0xF4, 0x14, 0x01, 0x24, 0x02, 0xA0, 0x01,
    0x2C, 0x02, 0xA8, 0xFC, 0x1C, 0x54, 0x01, 0x0A, 0x05, 0x44, 0x84, 0x0A,
    0x05, 0x04, 0x8C, 0x02, 0x5E, 0xB0, 0x0E, 0x08, 0xF5, 0xFA, 0x00, 0x00,
    0x02, 0x0E, 0x08, 0xFB, 0xEE, 0xF4, 0x20, 0x01, 0x34, 0x02, 0x60, 0x01,
    0x3C, 0x02, 0xFC, 0x88, 0xF4, 0x34, 0x01, 0xDC, 0x0A, 0x05, 0x44, 0xD4,
    0x02, 0x6E, 0xCC, 0x0E, 0x08, 0xF5, 0xFA, 0x00, 0x00, 0x02, 0x0E, 0x08,
    0xF9, 0xEE, 0xE0, 0x01, 0x48, 0x0D, 0xFC, 0x02, 0xD9, 0xFC, 0x02, 0x27,
    0x00, 0xFC, 0x7F, 0x05, 0x04, 0x30, 0x00, 0x00, 0xB4, 0x0C, 0x07, 0x00,
    0x94, 0x02, 0xA9, 0x01, 0x60, 0x02, 0xAE, 0x0E, 0x08, 0xF3, 0xFA, 0x00,
    0x00, 0x01, 0xA0, 0x02, 0x58, 0x01, 0xF4, 0x34, 0x02, 0xFC, 0x3C, 0xA0,
    0x00, 0x00, 0x02, 0x0E, 0x08, 0xFB, 0xEE, 0x34, 0x05, 0x01, 0x64, 0x0A,
    0x03, 0xC2, 0x0D, 0x04, 0xFD, 0xC4, 0x02, 0xFA, 0x00, 0x03, 0xFD, 0x19,
    0x00, 0x00, 0x02, 0x06, 0x01, 0x0A, 0x03, 0x42, 0x0C, 0x03, 0x02, 0x19,
    0x08, 0x05, 0x0A, 0x02, 0x08, 0x04, 0xF4, 0x0E, 0x08, 0xF6, 0xFA, 0x00,
    0x00, 0x02, 0x0E, 0x08, 0xFC, 0xF4, 0x01, 0xE4, 0x0A, 0x04, 0xC4, 0x0A,
    0x02, 0x82, 0x0A, 0x02, 0xA2, 0x8C, 0x0A, 0x02, 0x82, 0x02, 0x80, 0x0E,
    0x08, 0xF6, 0xFA, 0x00, 0x00, 0x02, 0x0E, 0x08, 0xFB, 0xEE, 0x01, 0x0A,
    0x03, 0x62, 0x94, 0x02, 0x38, 0x01, 0x90, 0x0A, 0x01, 0x62, 0x02, 0x38,
    0x1C, 0x01, 0x9C, 0x0A, 0x03, 0x42, 0x02, 0x40, 0x0E, 0x08, 0xED, 0xFA,
    0x00, 0x00, 0x02, 0x0E, 0x08, 0xFA, 0xEE, 0x08, 0x0C, 0x12, 0x01, 0xC8,
    0x08, 0x04, 0xF7, 0x08, 0xFC, 0xF7, 0xC0, 0x02, 0x60, 0x0E, 0x08, 0xF4,
    0xFA, 0x00, 0x00, 0x02, 0x0E, 0x08, 0xFC, 0xF4, 0x44, 0x01, 0x0A, 0x04,
    0x44, 0x44, 0x0A, 0x04, 0x04, 0x4C, 0x02, 0x08, 0x04, 0x08, 0x01, 0x20,
    0x0A, 0x02, 0x62, 0x02, 0x08, 0x06, 0xF2, 0x0E, 0x08, 0xF6, 0xFA, 0x00,
    0x00, 0x02, 0x0E, 0x08, 0xFC, 0xF4, 0xC4, 0x01, 0x9C, 0x0A, 0x03, 0x42,
    0x20, 0x0A, 0x03, 0x62, 0x05, 0x0A, 0x03, 0x42, 0x06, 0x94, 0x02, 0x08,
    0xF8, 0xF7, 0x01, 0x7C, 0x02, 0x08, 0x0E, 0x04, 0x0E, 0x08, 0xF6, 0xFA,
    0x00, 0x00, 0x02, 0x0E, 0x08, 0xFC, 0xF4, 0x60, 0x01, 0x0A, 0x02, 0xE2,
    0xA4, 0x02, 0x48, 0x01, 0x70, 0x0A, 0x01, 0x62, 0x02, 0x08, 0x07, 0xF3,
    0x0E, 0x08, 0xF6, 0xFA, 0x00, 0x00, 0x02, 0x0E, 0x08, 0xFB, 0xEE, 0x08,
    0x01, 0x09, 0x01, 0x80, 0x02, 0x08, 0xF7, 0xF7, 0x54, 0x01, 0x0A, 0x05,
    0x44, 0x84, 0x0A, 0x05, 0x04, 0x8C, 0x02, 0x5E, 0xB0, 0x0E, 0x08, 0xF5,
    0xFA, 0x00, 0x00, 0x02, 0x0E, 0x08, 0xFA, 0xEE, 0x10, 0x01, 0x40, 0x44,
    0x18, 0x0A, 0x04, 0xE2, 0x64, 0x0A, 0x04, 0xC2, 0x40, 0x0A, 0x04, 0xA2,
    0x6C, 0x0A, 0x04, 0x82, 0x18, 0x4C, 0x40, 0x02, 0x80, 0x0E, 0x08, 0xF3,
    0xFA, 0x00, 0x00, 0x02, 0x0E, 0x08, 0xFC, 0xEE, 0xE4, 0x24, 0x01, 0x0A,
    0x02, 0x44, 0x0A, 0x02, 0xC0, 0x0A, 0x02, 0x04, 0x02, 0x08, 0x0C, 0xF0,
    0x0E, 0x08, 0xF8, 0xFA, 0x00, 0x00, 0x02, 0x0E, 0x08, 0xFA, 0xEE, 0x20,
    0x01, 0x08, 0x08, 0x12, 0x02, 0x2E, 0x7C, 0x01, 0x0A, 0x06, 0x00, 0x02,
    0x6E, 0x3C, 0x0E, 0xC9, 0x00, 0x00, 0x02, 0x0E, 0x08, 0xFC, 0xF2, 0x01,
    0x80, 0x02, 0xA4, 0x01, 0x88, 0x02, 0x42, 0x01, 0x8C, 0x02, 0x6E, 0x40,
    0x0E, 0x08, 0xF6, 0xFA, 0x00, 0x00, 0x02, 0x0E, 0x08, 0xFC, 0xEE, 0x24,
    0x01, 0x08, 0x08, 0x05, 0x08, 0xF8, 0x05, 0x02, 0xCC, 0x01, 0x80, 0x02,
    0x40, 0x0E, 0x08, 0xF8, 0xFA, 0x00, 0x00, 0x02, 0x0E, 0x08, 0xFC, 0xEE,
    0x01, 0x80, 0x02, 0x24, 0x01, 0x08, 0xF8, 0x05, 0x08, 0x08, 0x05, 0x02,
    0xCC, 0x40, 0x0E, 0x08, 0xF8, 0xFA, 0x00, 0x00, 0x02, 0x0E, 0x08, 0xFB,
    0xEE, 0x74, 0x01, 0x0A, 0x02, 0xC4, 0x0A, 0x02, 0x44, 0x02, 0x34, 0x01,
    0x0A, 0x02, 0x84, 0x0A, 0x02, 0x04, 0x02, 0x08, 0x0C, 0xF6, 0x0E, 0x08,
    0xF9, 0xFB, 0x00, 0x00, 0x02, 0x0E, 0x08, 0xFD, 0xEE, 0xF4, 0x01, 0x0A,
    0x03, 0x40, 0x02, 0xFC, 0xC0, 0x0E, 0x08, 0xF7, 0x06, 0x00, 0x00, 0x02,
    0x0E, 0x08, 0xF8, 0xEB, 0xC4, 0x01, 0x09, 0x04, 0xF4, 0x06, 0x15, 0x0B,
    0x00, 0x00, 0x00, 0x02, 0x08, 0xF9, 0xEB, 0x0E, 0x08, 0xFA, 0xFA, 0x00,
    0x00, 0x02, 0x0E, 0x08, 0xFE, 0xEE, 0x08, 0x04, 0x0E, 0x01, 0x48, 0x32,
    0x0C, 0x01, 0x02, 0x19, 0x0A, 0x02, 0x04, 0x02, 0x08, 0x08, 0xED, 0x0E,
    0x08, 0xFA, 0xFD, 0x00, 0x45, 0x4F, 0x46
])

SHX2 = bytes([
    0x41, 0x75, 0x74, 0x6F, 0x43, 0x41, 0x44, 0x2D, 0x38, 0x36, 0x20, 0x75,
    0x6E, 0x69, 0x66, 0x6F, 0x6E, 0x74, 0x20, 0x31, 0x2E, 0x30, 0x0D, 0x0A,
    0x1A, 0xEE, 0x00, 0x00, 0x00, 0x34, 0x00, 0x49, 0x53, 0x4F, 0x43, 0x50,
    0x20, 0x43, 0x6F, 0x70, 0x79, 0x72, 0x69, 0x67, 0x68, 0x74, 0x20, 0xA9,
    0x20, 0x31, 0x39, 0x39, 0x36, 0x2D, 0x32, 0x30, 0x30, 0x33, 0x20, 0x62,
    0x79, 0x20, 0x41, 0x75, 0x74, 0x6F, 0x64, 0x65, 0x73, 0x6B, 0x2C, 0x20,
    0x49, 0x6E, 0x63, 0x2E, 0x00, 0x28, 0x0C, 0x00, 0x00, 0x00, 0x00, 0x0A,
    0x00, 0x06, 0x00, 0x00, 0x02, 0x08, 0x00, 0xB4, 0x00, 0x20, 0x00, 0x06,
    0x00, 0x00, 0x02, 0x08, 0x10, 0x00, 0x00, 0x21, 0x00, 0x13, 0x00, 0x21,
    0x00, 0x02, 0x08, 0x06, 0x02, 0x01, 0x24, 0x02, 0xA4, 0x01, 0x08, 0x00,
    0x18, 0x02, 0x08, 0x06, 0xDA, 0x00, 0x22, 0x00, 0x15, 0x00, 0x22, 0x00,
    0x02, 0x08, 0x06, 0x1E, 0x01, 0x08, 0x02, 0x08, 0x02, 0xA0, 0x01, 0x08,
    0xFE, 0xF8, 0x02, 0x08, 0x08, 0xE2, 0x00, 0x23, 0x00, 0x23, 0x00, 0x23,
    0x00, 0x02, 0x08, 0x0C, 0x02, 0x01, 0x08, 0x00, 0x24, 0x02, 0xC0, 0x01,
    0x08, 0x00, 0xDC, 0x02, 0x08, 0x06, 0x0C, 0x01, 0x08, 0xE8, 0x00, 0x02,
    0xC4, 0x01, 0x08, 0x18, 0x00, 0x02, 0x08, 0x06, 0xE6, 0x00, 0x24, 0x00,
    0x29, 0x00, 0x24, 0x00, 0x02, 0x08, 0x0E, 0x02, 0x01, 0x08, 0x00, 0x24,
    0x02, 0x08, 0x08, 0xFA, 0x01, 0x0C, 0xF8, 0x02, 0x10, 0x28, 0x0C, 0xFD,
    0xF5, 0x62, 0x08, 0x0A, 0xFA, 0x0C, 0xFD, 0xF5, 0x9E, 0x28, 0x0C, 0xF8,
    0x02, 0xF0, 0x02, 0x08, 0x16, 0xF8, 0x00, 0x25, 0x00, 0x1F, 0x00, 0x25,
    0x00, 0x02, 0x08, 0x06, 0x02, 0x01, 0x08, 0x20, 0x24, 0x02, 0x08, 0xEC,
    0xFA, 0x01, 0x0A, 0x06, 0x00, 0x02, 0x08, 0x08, 0xE8, 0x01, 0x0A, 0x06,
    0x40, 0x02, 0x08, 0x12, 0xF8, 0x00, 0x26, 0x00, 0x2B, 0x00, 0x26, 0x00,
    0x02, 0x08, 0x1E, 0x0E, 0x01, 0x03, 0x0A, 0x04, 0x02, 0x08, 0xB8, 0xCA,
    0x03, 0x02, 0x0C, 0xA7, 0x4C, 0x9A, 0x04, 0x04, 0x08, 0x1F, 0x2D, 0x03,
    0x04, 0x04, 0x0A, 0x0B, 0x44, 0xBB, 0x00, 0x06, 0x76, 0x08, 0x11, 0xE5,
    0x02, 0x08, 0x0A, 0xFE, 0x00,  # => *00027
    ])
# fmt:on

if __name__ == "__main__":
    pytest.main([__file__])
