/* BEGIN software license
 *
 * MsXpertSuite - mass spectrometry software suite
 * -----------------------------------------------
 * Copyright (C) 2009--2020 Filippo Rusconi
 *
 * http://www.msxpertsuite.org
 *
 * This file is part of the MsXpertSuite project.
 *
 * The MsXpertSuite project is the successor of the massXpert project. This
 * project now includes various independent modules:
 *
 * - massXpert, model polymer chemistries and simulate mass spectrometric data;
 * - mineXpert, a powerful TIC chromatogram/mass spectrum viewer/miner;
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 *
 * END software license
 */


#pragma once

/////////////////////// stdlib includes


/////////////////////// Qt includes
#include <QDialog>
#include <QRadioButton>
#include <QSpinBox>
#include <QDoubleSpinBox>
#include <QWidget>
#include <QTimer>


/////////////////////// pappsomspp includes


/////////////////////// libXpertMassCore includes
#include "MsXpS/libXpertMassCore/jsclassregistrar.h"
#include "MsXpS/libXpertMassCore/MassPeakShaperConfig.hpp"
#include "MsXpS/libXpertMassCore/globals.hpp"


/////////////////////// libXpertMassGUI includes


/////////////////////// Local includes
#include "MsXpS/export-import-config.h"

namespace Ui
{
class MassPeakShaperConfigWidget;
}

namespace MsXpS
{
namespace libXpertMassGui
{


/*  BEGIN CLASS JS REFERENCE
 *  namespace: MsXpS::libXpertMassGui
 *  class name: MassPeakShaperConfigWidget
 */

class DECLSPEC MassPeakShaperConfigWidget: public QWidget
{
  Q_OBJECT

  Q_PROPERTY(double referencePeakMz MEMBER m_referencePeakMz)
  Q_PROPERTY(libXpertMassCore::MassPeakShaperConfig *config READ getConfig
               MEMBER mp_config)
  Q_PROPERTY(double normalizingIntensity MEMBER m_normalizingIntensity)

  public:
  explicit MassPeakShaperConfigWidget(
    const libXpertMassCore::MassPeakShaperConfig &config, QWidget *parent_p);

  virtual ~MassPeakShaperConfigWidget();


  void writeSettings(const QString &configSettingsFilePath);
  void readSettings(const QString &configSettingsFilePath);

  void setReferencePeakMz(double mz);
  double getReferencePeakMz();

  Q_INVOKABLE libXpertMassCore::MassPeakShaperConfig *getConfig();

  QSpinBox *getResolutionSpinBox();
  QDoubleSpinBox *getFwhmDoubleSpinBox();
  QSpinBox *getFmwhToBinSizeDivisorSpinBox();
  QSpinBox *getPointCountSpinBox();
  QRadioButton *getGaussianRadioButton();
  QRadioButton *getLorentzianRadioButton();


  // This function needs to be public because it is necessary to the user of
  // this widget when checking correctness of the parameters.
  Q_INVOKABLE bool processBinSizeConfig();

  Q_INVOKABLE void
  setParameters(double reference_peak_mz,
                int point_count,
                libXpertMassCore::Enums::MassPeakShapeType mass_peak_shape_type,
                double resolution,
                double fwhm,
                int bin_size_divisor    = 6,
                bool should_create_bins = true);
  Q_INVOKABLE void setBinSizeManually(double bin_size,
                                      bool should_create_bins = false);
  Q_INVOKABLE bool checkParameters();
  Q_INVOKABLE bool checkParameters(libXpertMassCore::ErrorList *error_list);

  Q_INVOKABLE bool isReferencePeakMzKept() const;

  Q_INVOKABLE QPair<double, double> getMzRangeValuePair() const;

  Q_INVOKABLE const libXpertMassCore::MassPeakShaperConfig *getConfig() const;

  static void registerJsConstructor(QJSEngine *engine);

  public slots:

  void resolutionEditingFinished();
  void fwhmEditingFinished();
  void fwhmBinSizeDivisorValueChanged(int value);
  void pointCountEditingFinished();
  void gaussianRadioButtonToggled(bool checked);
  void lorentzianRadioButtonToggled(bool checked);
  void referencePeakMzValueChanged(double value);
  void computeReferencePeakMzValue(double value);

  signals:

  void updatedMassPeakShaperConfigSignal(
    const libXpertMassCore::MassPeakShaperConfig &config);

  protected:
  Ui::MassPeakShaperConfigWidget *mp_ui;
  QWidget *mp_parent = nullptr;

  double m_referencePeakMz   = 0.0;
  bool m_keepReferencePeakMz = false;

  double m_mzRangeStart = qQNaN();
  double m_mzRangeStop  = qQNaN();

  libXpertMassCore::MassPeakShaperConfig *mp_config = nullptr;
  double m_normalizingIntensity = std::numeric_limits<double>::min();

  std::shared_ptr<QTimer> msp_msgTimer = std::make_shared<QTimer>();

  void setupWidget();
  void message(const QString &message, int timeout = 3000);
};

/*  END CLASS JS REFERENCE
 *  namespace: MsXpS::libXpertMassGui
 *  class name: MassPeakShaperConfigWidget
 */

MSXPS_REGISTER_JS_CLASS(MsXpS::libXpertMassGui, MassPeakShaperConfigWidget)

} // namespace libXpertMassGui
} // namespace MsXpS
