/*-
 *   BSD LICENSE
 *
 *   Copyright(c) 2015-2016 Intel Corporation. All rights reserved.
 *
 *   Redistribution and use in source and binary forms, with or without
 *   modification, are permitted provided that the following conditions
 *   are met:
 *
 *	 * Redistributions of source code must retain the above copyright
 *	   notice, this list of conditions and the following disclaimer.
 *	 * Redistributions in binary form must reproduce the above copyright
 *	   notice, this list of conditions and the following disclaimer in
 *	   the documentation and/or other materials provided with the
 *	   distribution.
 *	 * Neither the name of Intel Corporation nor the names of its
 *	   contributors may be used to endorse or promote products derived
 *	   from this software without specific prior written permission.
 *
 *   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *   OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include <rte_common.h>
#include <rte_mbuf.h>
#include <rte_malloc.h>
#include <rte_memcpy.h>

#include <rte_crypto.h>
#include <rte_cryptodev.h>
#include <rte_cycles.h>

#include "test.h"
#include "test_cryptodev.h"


#define PERF_NUM_OPS_INFLIGHT		(128)
#define DEFAULT_NUM_REQS_TO_SUBMIT	(10000000)

struct crypto_testsuite_params {
	struct rte_mempool *mbuf_mp;
	struct rte_mempool *op_mpool;

	uint16_t nb_queue_pairs;

	struct rte_cryptodev_config conf;
	struct rte_cryptodev_qp_conf qp_conf;
	uint8_t dev_id;
};

enum chain_mode {
	CIPHER_HASH,
	HASH_CIPHER,
	CIPHER_ONLY,
	HASH_ONLY
};

struct perf_test_params {

	unsigned total_operations;
	unsigned burst_size;
	unsigned buf_size;

	enum chain_mode chain;

	enum rte_crypto_cipher_algorithm cipher_algo;
	unsigned cipher_key_length;
	enum rte_crypto_auth_algorithm auth_algo;
};

#define MAX_NUM_OF_OPS_PER_UT	(128)

struct crypto_unittest_params {
	struct rte_crypto_sym_xform cipher_xform;
	struct rte_crypto_sym_xform auth_xform;

	struct rte_cryptodev_sym_session *sess;

	struct rte_crypto_op *op;

	struct rte_mbuf *obuf[MAX_NUM_OF_OPS_PER_UT];
	struct rte_mbuf *ibuf[MAX_NUM_OF_OPS_PER_UT];

	uint8_t *digest;
};

static struct rte_cryptodev_sym_session *
test_perf_create_snow3g_session(uint8_t dev_id, enum chain_mode chain,
		enum rte_crypto_cipher_algorithm cipher_algo, unsigned cipher_key_len,
		enum rte_crypto_auth_algorithm auth_algo);
static struct rte_mbuf *
test_perf_create_pktmbuf(struct rte_mempool *mpool, unsigned buf_sz);
static inline struct rte_crypto_op *
test_perf_set_crypto_op_snow3g(struct rte_crypto_op *op, struct rte_mbuf *m,
		struct rte_cryptodev_sym_session *sess, unsigned data_len,
		unsigned digest_len);
static uint32_t get_auth_digest_length(enum rte_crypto_auth_algorithm algo);


static const char *chain_mode_name(enum chain_mode mode)
{
	switch (mode) {
	case CIPHER_HASH: return "cipher_hash"; break;
	case HASH_CIPHER: return "hash_cipher"; break;
	case CIPHER_ONLY: return "cipher_only"; break;
	case HASH_ONLY: return "hash_only"; break;
	default: return ""; break;
	}
}

static const char *pmd_name(enum rte_cryptodev_type pmd)
{
	switch (pmd) {
	case RTE_CRYPTODEV_NULL_PMD: return RTE_STR(CRYPTODEV_NAME_NULL_PMD); break;
	case RTE_CRYPTODEV_AESNI_GCM_PMD:
		return RTE_STR(CRYPTODEV_NAME_AESNI_GCM_PMD);
	case RTE_CRYPTODEV_AESNI_MB_PMD:
		return RTE_STR(CRYPTODEV_NAME_AESNI_MB_PMD);
	case RTE_CRYPTODEV_QAT_SYM_PMD:
		return RTE_STR(CRYPTODEV_NAME_QAT_SYM_PMD);
	case RTE_CRYPTODEV_SNOW3G_PMD:
		return RTE_STR(CRYPTODEV_NAME_SNOW3G_PMD);
	default:
		return "";
	}
}

static const char *cipher_algo_name(enum rte_crypto_cipher_algorithm cipher_algo)
{
	switch (cipher_algo) {
	case RTE_CRYPTO_CIPHER_NULL: return "NULL";
	case RTE_CRYPTO_CIPHER_3DES_CBC: return "3DES_CBC";
	case RTE_CRYPTO_CIPHER_3DES_CTR: return "3DES_CTR";
	case RTE_CRYPTO_CIPHER_3DES_ECB: return "3DES_ECB";
	case RTE_CRYPTO_CIPHER_AES_CBC: return "AES_CBC";
	case RTE_CRYPTO_CIPHER_AES_CCM: return "AES_CCM";
	case RTE_CRYPTO_CIPHER_AES_CTR: return "AES_CTR";
	case RTE_CRYPTO_CIPHER_AES_ECB: return "AES_ECB";
	case RTE_CRYPTO_CIPHER_AES_F8: return "AES_F8";
	case RTE_CRYPTO_CIPHER_AES_GCM: return "AES_GCM";
	case RTE_CRYPTO_CIPHER_AES_XTS: return "AES_XTS";
	case RTE_CRYPTO_CIPHER_ARC4: return "ARC4";
	case RTE_CRYPTO_CIPHER_KASUMI_F8: return "KASUMI_F8";
	case RTE_CRYPTO_CIPHER_SNOW3G_UEA2: return "SNOW3G_UEA2";
	case RTE_CRYPTO_CIPHER_ZUC_EEA3: return "ZUC_EEA3";
	default: return "Another cipher algo";
	}
}

static const char *auth_algo_name(enum rte_crypto_auth_algorithm auth_algo)
{
	switch (auth_algo) {
	case RTE_CRYPTO_AUTH_NULL: return "NULL"; break;
	case RTE_CRYPTO_AUTH_AES_CBC_MAC: return "AES_CBC_MAC"; break;
	case RTE_CRYPTO_AUTH_AES_CCM: return "AES_CCM"; break;
	case RTE_CRYPTO_AUTH_AES_CMAC: return "AES_CMAC,"; break;
	case RTE_CRYPTO_AUTH_AES_GCM: return "AES_GCM"; break;
	case RTE_CRYPTO_AUTH_AES_GMAC: return "AES_GMAC"; break;
	case RTE_CRYPTO_AUTH_AES_XCBC_MAC: return "AES_XCBC_MAC"; break;
	case RTE_CRYPTO_AUTH_KASUMI_F9: return "KASUMI_F9"; break;
	case RTE_CRYPTO_AUTH_MD5: return "MD5"; break;
	case RTE_CRYPTO_AUTH_MD5_HMAC: return "MD5_HMAC,"; break;
	case RTE_CRYPTO_AUTH_SHA1: return "SHA1"; break;
	case RTE_CRYPTO_AUTH_SHA1_HMAC: return "SHA1_HMAC"; break;
	case RTE_CRYPTO_AUTH_SHA224: return "SHA224"; break;
	case RTE_CRYPTO_AUTH_SHA224_HMAC: return "SHA224_HMAC"; break;
	case RTE_CRYPTO_AUTH_SHA256: return "SHA256"; break;
	case RTE_CRYPTO_AUTH_SHA256_HMAC: return "SHA256_HMAC"; break;
	case RTE_CRYPTO_AUTH_SHA384: return "SHA384,"; break;
	case RTE_CRYPTO_AUTH_SHA384_HMAC: return "SHA384_HMAC,"; break;
	case RTE_CRYPTO_AUTH_SHA512: return "SHA512,"; break;
	case RTE_CRYPTO_AUTH_SHA512_HMAC: return "SHA512_HMAC,"; break;
	case RTE_CRYPTO_AUTH_SNOW3G_UIA2: return "SNOW3G_UIA2"; break;
	case RTE_CRYPTO_AUTH_ZUC_EIA3: return "RTE_CRYPTO_AUTH_ZUC_EIA3"; break;
	default: return "Another auth algo"; break;
	};
}

static struct rte_mbuf *
setup_test_string(struct rte_mempool *mpool,
		const uint8_t *data, size_t len, uint8_t blocksize)
{
	struct rte_mbuf *m = rte_pktmbuf_alloc(mpool);
	size_t t_len = len - (blocksize ? (len % blocksize) : 0);

	if (m) {
		char *dst = rte_pktmbuf_append(m, t_len);

		if (!dst) {
			rte_pktmbuf_free(m);
			return NULL;
		}

		rte_memcpy(dst, (const void *)data, t_len);
	}
	return m;
}

static struct crypto_testsuite_params testsuite_params = { NULL };
static struct crypto_unittest_params unittest_params;
static enum rte_cryptodev_type gbl_cryptodev_perftest_devtype;

static int
testsuite_setup(void)
{
	struct crypto_testsuite_params *ts_params = &testsuite_params;
	struct rte_cryptodev_info info;
	unsigned i, nb_devs, valid_dev_id = 0;
	int ret;
	uint16_t qp_id;

	ts_params->mbuf_mp = rte_mempool_lookup("CRYPTO_PERF_MBUFPOOL");
	if (ts_params->mbuf_mp == NULL) {
		/* Not already created so create */
		ts_params->mbuf_mp = rte_pktmbuf_pool_create(
				"CRYPTO_PERF_MBUFPOOL",
				NUM_MBUFS, MBUF_CACHE_SIZE, 0, MBUF_SIZE,
				rte_socket_id());
		if (ts_params->mbuf_mp == NULL) {
			RTE_LOG(ERR, USER1, "Can't create CRYPTO_PERF_MBUFPOOL\n");
			return TEST_FAILED;
		}
	}


	ts_params->op_mpool = rte_crypto_op_pool_create("CRYPTO_OP_POOL",
			RTE_CRYPTO_OP_TYPE_SYMMETRIC,
			NUM_MBUFS, MBUF_CACHE_SIZE,
			DEFAULT_NUM_XFORMS *
			sizeof(struct rte_crypto_sym_xform),
			rte_socket_id());
		if (ts_params->op_mpool == NULL) {
			RTE_LOG(ERR, USER1, "Can't create CRYPTO_OP_POOL\n");
			return TEST_FAILED;
		}

	/* Create 2 AESNI MB devices if required */
	if (gbl_cryptodev_perftest_devtype == RTE_CRYPTODEV_AESNI_MB_PMD) {
		nb_devs = rte_cryptodev_count_devtype(RTE_CRYPTODEV_AESNI_MB_PMD);
		if (nb_devs < 2) {
			for (i = nb_devs; i < 2; i++) {
				ret = rte_eal_vdev_init(
					RTE_STR(CRYPTODEV_NAME_AESNI_MB_PMD), NULL);

				TEST_ASSERT(ret == 0,
					"Failed to create instance %u of pmd : %s",
					i, RTE_STR(CRYPTODEV_NAME_AESNI_MB_PMD));
			}
		}
	}

	/* Create 2 SNOW3G devices if required */
	if (gbl_cryptodev_perftest_devtype == RTE_CRYPTODEV_SNOW3G_PMD) {
		nb_devs = rte_cryptodev_count_devtype(RTE_CRYPTODEV_SNOW3G_PMD);
		if (nb_devs < 2) {
			for (i = nb_devs; i < 2; i++) {
				ret = rte_eal_vdev_init(
					RTE_STR(CRYPTODEV_NAME_SNOW3G_PMD), NULL);

				TEST_ASSERT(ret == 0,
					"Failed to create instance %u of pmd : %s",
					i, RTE_STR(CRYPTODEV_NAME_SNOW3G_PMD));
			}
		}
	}

	nb_devs = rte_cryptodev_count();
	if (nb_devs < 1) {
		RTE_LOG(ERR, USER1, "No crypto devices found?");
		return TEST_FAILED;
	}

	/* Search for the first valid */
	for (i = 0; i < nb_devs; i++) {
		rte_cryptodev_info_get(i, &info);
		if (info.dev_type == gbl_cryptodev_perftest_devtype) {
			ts_params->dev_id = i;
			valid_dev_id = 1;
			break;
		}
	}

	if (!valid_dev_id)
		return TEST_FAILED;

	/*
	 * Using Crypto Device Id 0 by default.
	 * Since we can't free and re-allocate queue memory always set the queues
	 * on this device up to max size first so enough memory is allocated for
	 * any later re-configures needed by other tests
	 */

	rte_cryptodev_info_get(ts_params->dev_id, &info);

	ts_params->conf.nb_queue_pairs = DEFAULT_NUM_QPS_PER_QAT_DEVICE;
	ts_params->conf.socket_id = SOCKET_ID_ANY;
	ts_params->conf.session_mp.nb_objs = info.sym.max_nb_sessions;

	TEST_ASSERT_SUCCESS(rte_cryptodev_configure(ts_params->dev_id,
			&ts_params->conf),
			"Failed to configure cryptodev %u",
			ts_params->dev_id);


	ts_params->qp_conf.nb_descriptors = MAX_NUM_OPS_INFLIGHT;

	for (qp_id = 0; qp_id < ts_params->conf.nb_queue_pairs ; qp_id++) {
		TEST_ASSERT_SUCCESS(rte_cryptodev_queue_pair_setup(
			ts_params->dev_id, qp_id,
			&ts_params->qp_conf,
			rte_cryptodev_socket_id(ts_params->dev_id)),
			"Failed to setup queue pair %u on cryptodev %u",
			qp_id, ts_params->dev_id);
	}

	/*Now reconfigure queues to size we actually want to use in this testsuite.*/
	ts_params->qp_conf.nb_descriptors = PERF_NUM_OPS_INFLIGHT;
	for (qp_id = 0; qp_id < ts_params->conf.nb_queue_pairs ; qp_id++) {

		TEST_ASSERT_SUCCESS(rte_cryptodev_queue_pair_setup(
			ts_params->dev_id, qp_id,
				&ts_params->qp_conf,
				rte_cryptodev_socket_id(ts_params->dev_id)),
				"Failed to setup queue pair %u on cryptodev %u",
				qp_id, ts_params->dev_id);
	}

	return TEST_SUCCESS;
}
static void
testsuite_teardown(void)
{
	struct crypto_testsuite_params *ts_params = &testsuite_params;

	if (ts_params->mbuf_mp != NULL)
		RTE_LOG(DEBUG, USER1, "CRYPTO_PERF_MBUFPOOL count %u\n",
		rte_mempool_avail_count(ts_params->mbuf_mp));
	if (ts_params->op_mpool != NULL)
		RTE_LOG(DEBUG, USER1, "CRYPTO_PERF_OP POOL count %u\n",
		rte_mempool_avail_count(ts_params->op_mpool));
}

static int
ut_setup(void)
{
	struct crypto_testsuite_params *ts_params = &testsuite_params;
	struct crypto_unittest_params *ut_params = &unittest_params;

	/* Clear unit test parameters before running test */
	memset(ut_params, 0, sizeof(*ut_params));

	rte_cryptodev_stats_reset(ts_params->dev_id);

	/* Start the device */
	TEST_ASSERT_SUCCESS(rte_cryptodev_start(ts_params->dev_id),
			"Failed to start cryptodev %u",
			ts_params->dev_id);

	return TEST_SUCCESS;
}

static void
ut_teardown(void)
{
	struct crypto_testsuite_params *ts_params = &testsuite_params;
	struct crypto_unittest_params *ut_params = &unittest_params;
	struct rte_cryptodev_stats stats;

	unsigned i;

	/* free crypto session structure */
	if (ut_params->sess)
		rte_cryptodev_sym_session_free(ts_params->dev_id,
				ut_params->sess);

	/* free crypto operation structure */
	if (ut_params->op)
		rte_crypto_op_free(ut_params->op);

	for (i = 0; i < MAX_NUM_OF_OPS_PER_UT; i++) {
		if (ut_params->obuf[i])
			rte_pktmbuf_free(ut_params->obuf[i]);
		else if (ut_params->ibuf[i])
			rte_pktmbuf_free(ut_params->ibuf[i]);
	}

	if (ts_params->mbuf_mp != NULL)
		RTE_LOG(DEBUG, USER1, "CRYPTO_PERF_MBUFPOOL count %u\n",
			rte_mempool_avail_count(ts_params->mbuf_mp));

	rte_cryptodev_stats_get(ts_params->dev_id, &stats);

	/* Stop the device */
	rte_cryptodev_stop(ts_params->dev_id);
}

const char plaintext_quote[] =
		"THE COUNT OF MONTE CRISTO by Alexandre Dumas, Pere Chapter 1. "
		"Marseilles--The Arrival. On the 24th of February, 1815, the "
		"look-out at Notre-Dame de la Garde signalled the three-master,"
		" the Pharaon from Smyrna, Trieste, and Naples. As usual, a "
		"pilot put off immediately, and rounding the Chateau d'If, got "
		"on board the vessel between Cape Morgion and Rion island. "
		"Immediately, and according to custom, the ramparts of Fort "
		"Saint-Jean were covered with spectators; it is always an event "
		"at Marseilles for a ship to come into port, especially when "
		"this ship, like the Pharaon, has been built, rigged, and laden"
		" at the old Phocee docks, and belongs to an owner of the city."
		" The ship drew on and had safely passed the strait, which some"
		" volcanic shock has made between the Calasareigne and Jaros "
		"islands; had doubled Pomegue, and approached the harbor under"
		" topsails, jib, and spanker, but so slowly and sedately that"
		" the idlers, with that instinct which is the forerunner of "
		"evil, asked one another what misfortune could have happened "
		"on board. However, those experienced in navigation saw plainly"
		" that if any accident had occurred, it was not to the vessel "
		"herself, for she bore down with all the evidence of being "
		"skilfully handled, the anchor a-cockbill, the jib-boom guys "
		"already eased off, and standing by the side of the pilot, who"
		" was steering the Pharaon towards the narrow entrance of the"
		" inner port, was a young man, who, with activity and vigilant"
		" eye, watched every motion of the ship, and repeated each "
		"direction of the pilot. The vague disquietude which prevailed "
		"among the spectators had so much affected one of the crowd "
		"that he did not await the arrival of the vessel in harbor, but"
		" jumping into a small skiff, desired to be pulled alongside "
		"the Pharaon, which he reached as she rounded into La Reserve "
		"basin. When the young man on board saw this person approach, "
		"he left his station by the pilot, and, hat in hand, leaned "
		"over the ship's bulwarks. He was a fine, tall, slim young "
		"fellow of eighteen or twenty, with black eyes, and hair as "
		"dark as a raven's wing; and his whole appearance bespoke that "
		"calmness and resolution peculiar to men accustomed from their "
		"cradle to contend with danger. \"Ah, is it you, Dantes?\" "
		"cried the man in the skiff. \"What's the matter? and why have "
		"you such an air of sadness aboard?\" \"A great misfortune, M. "
		"Morrel,\" replied the young man,--\"a great misfortune, for me"
		" especially! Off Civita Vecchia we lost our brave Captain "
		"Leclere.\" \"And the cargo?\" inquired the owner, eagerly. "
		"\"Is all safe, M. Morrel; and I think you will be satisfied on"
		" that head. But poor Captain Leclere--\" \"What happened to "
		"him?\" asked the owner, with an air of considerable "
		"resignation. \"What happened to the worthy captain?\" \"He "
		"died.\" \"Fell into the sea?\" \"No, sir, he died of "
		"brain-fever in dreadful agony.\" Then turning to the crew, "
		"he said, \"Bear a hand there, to take in sail!\" All hands "
		"obeyed, and at once the eight or ten seamen who composed the "
		"crew, sprang to their respective stations at the spanker "
		"brails and outhaul, topsail sheets and halyards, the jib "
		"downhaul, and the topsail clewlines and buntlines. The young "
		"sailor gave a look to see that his orders were promptly and "
		"accurately obeyed, and then turned again to the owner. \"And "
		"how did this misfortune occur?\" inquired the latter, resuming"
		" the interrupted conversation. \"Alas, sir, in the most "
		"unexpected manner. After a long talk with the harbor-master, "
		"Captain Leclere left Naples greatly disturbed in mind. In "
		"twenty-four hours he was attacked by a fever, and died three "
		"days afterwards. We performed the usual burial service, and he"
		" is at his rest, sewn up in his hammock with a thirty-six "
		"pound shot at his head and his heels, off El Giglio island. "
		"We bring to his widow his sword and cross of honor. It was "
		"worth while, truly,\" added the young man with a melancholy "
		"smile, \"to make war against the English for ten years, and "
		"to die in his bed at last, like everybody else.";

#define QUOTE_LEN_64B		(64)
#define QUOTE_LEN_128B		(128)
#define QUOTE_LEN_256B		(256)
#define QUOTE_LEN_512B		(512)
#define QUOTE_LEN_768B		(768)
#define QUOTE_LEN_1024B		(1024)
#define QUOTE_LEN_1280B		(1280)
#define QUOTE_LEN_1536B		(1536)
#define QUOTE_LEN_1792B		(1792)
#define QUOTE_LEN_2048B		(2048)


/* ***** AES-CBC / HMAC-SHA256 Performance Tests ***** */

#define HMAC_KEY_LENGTH_SHA256	(DIGEST_BYTE_LENGTH_SHA256)

#define CIPHER_KEY_LENGTH_AES_CBC	(16)
#define CIPHER_IV_LENGTH_AES_CBC	(CIPHER_KEY_LENGTH_AES_CBC)

static uint8_t aes_cbc_128_key[] = {
		0xE4, 0x23, 0x33, 0x8A, 0x35, 0x64, 0x61, 0xE2,
		0xF1, 0x35, 0x5C, 0x3B, 0xDD, 0x9A, 0x65, 0xBA };

static uint8_t aes_cbc_128_iv[] = {
		0xf5, 0xd3, 0x89, 0x0f, 0x47, 0x00, 0xcb, 0x52,
		0x42, 0x1a, 0x7d, 0x3d, 0xf5, 0x82, 0x80, 0xf1 };

static uint8_t hmac_sha256_key[] = {
		0xff, 0xcb, 0x37, 0x30, 0x1d, 0x4a, 0xc2, 0x41,
		0x49, 0x03, 0xDD, 0xC6, 0xB8, 0xCA, 0x55, 0x7A,
		0x58, 0x34, 0x85, 0x61, 0x1C, 0x42, 0x10, 0x76,
		0x9a, 0x4f, 0x88, 0x1b, 0xb6, 0x8f, 0xd8, 0x60 };


/* Cipher text output */

static const uint8_t AES_CBC_ciphertext_64B[] = {
		0x05, 0x15, 0x77, 0x32, 0xc9, 0x66, 0x91, 0x50,
		0x93, 0x9f, 0xbb, 0x4e, 0x2e, 0x5a, 0x02, 0xd0,
		0x2d, 0x9d, 0x31, 0x5d, 0xc8, 0x9e, 0x86, 0x36,
		0x54, 0x5c, 0x50, 0xe8, 0x75, 0x54, 0x74, 0x5e,
		0xd5, 0xa2, 0x84, 0x21, 0x2d, 0xc5, 0xf8, 0x1c,
		0x55, 0x1a, 0xba, 0x91, 0xce, 0xb5, 0xa3, 0x1e,
		0x31, 0xbf, 0xe9, 0xa1, 0x97, 0x5c, 0x2b, 0xd6,
		0x57, 0xa5, 0x9f, 0xab, 0xbd, 0xb0, 0x9b, 0x9c
};

static const uint8_t AES_CBC_ciphertext_128B[] = {
		0x79, 0x92, 0x65, 0xc8, 0xfb, 0x0a, 0xc7, 0xc4,
		0x9b, 0x3b, 0xbe, 0x69, 0x7f, 0x7c, 0xf4, 0x4e,
		0xa5, 0x0d, 0xf6, 0x33, 0xc4, 0xdf, 0xf3, 0x0d,
		0xdb, 0xb9, 0x68, 0x34, 0xb0, 0x0d, 0xbd, 0xb9,
		0xa7, 0xf3, 0x86, 0x50, 0x2a, 0xbe, 0x50, 0x5d,
		0xb3, 0xbe, 0x72, 0xf9, 0x02, 0xb1, 0x69, 0x0b,
		0x8c, 0x96, 0x4c, 0x3c, 0x0c, 0x1e, 0x76, 0xe5,
		0x7e, 0x75, 0xdd, 0xd0, 0xa9, 0x75, 0x00, 0x13,
		0x6b, 0x1e, 0xc0, 0xad, 0xfc, 0x03, 0xb5, 0x99,
		0xdc, 0x37, 0x35, 0xfc, 0x16, 0x34, 0xfd, 0xb4,
		0xea, 0x1e, 0xb6, 0x51, 0xdf, 0xab, 0x87, 0xd6,
		0x87, 0x41, 0xfa, 0x1c, 0xc6, 0x78, 0xa6, 0x3c,
		0x1d, 0x76, 0xfe, 0xff, 0x65, 0xfc, 0x63, 0x1e,
		0x1f, 0xe2, 0x7c, 0x9b, 0xa2, 0x72, 0xc3, 0x34,
		0x23, 0xdf, 0x01, 0xf0, 0xfd, 0x02, 0x8b, 0x97,
		0x00, 0x2b, 0x97, 0x4e, 0xab, 0x98, 0x21, 0x3c
};

static const uint8_t AES_CBC_ciphertext_256B[] = {
		0xc7, 0x71, 0x2b, 0xed, 0x2c, 0x97, 0x59, 0xfa,
		0xcf, 0x5a, 0xb9, 0x31, 0x92, 0xe0, 0xc9, 0x92,
		0xc0, 0x2d, 0xd5, 0x9c, 0x84, 0xbf, 0x70, 0x36,
		0x13, 0x48, 0xe0, 0xb1, 0xbf, 0x6c, 0xcd, 0x91,
		0xa0, 0xc3, 0x57, 0x6c, 0x3f, 0x0e, 0x34, 0x41,
		0xe7, 0x9c, 0xc0, 0xec, 0x18, 0x0c, 0x05, 0x52,
		0x78, 0xe2, 0x3c, 0x6e, 0xdf, 0xa5, 0x49, 0xc7,
		0xf2, 0x55, 0x00, 0x8f, 0x65, 0x6d, 0x4b, 0xd0,
		0xcb, 0xd4, 0xd2, 0x0b, 0xea, 0xf4, 0xb0, 0x85,
		0x61, 0x9e, 0x36, 0xc0, 0x71, 0xb7, 0x80, 0xad,
		0x40, 0x78, 0xb4, 0x70, 0x2b, 0xe8, 0x80, 0xc5,
		0x19, 0x35, 0x96, 0x55, 0x3b, 0x40, 0x03, 0xbb,
		0x9f, 0xa6, 0xc2, 0x82, 0x92, 0x04, 0xc3, 0xa6,
		0x96, 0xc4, 0x7f, 0x4c, 0x3e, 0x3c, 0x79, 0x82,
		0x88, 0x8b, 0x3f, 0x8b, 0xc5, 0x9f, 0x44, 0xbe,
		0x71, 0xe7, 0x09, 0xa2, 0x40, 0xa2, 0x23, 0x4e,
		0x9f, 0x31, 0xab, 0x6f, 0xdf, 0x59, 0x40, 0xe1,
		0x12, 0x15, 0x55, 0x4b, 0xea, 0x3f, 0xa1, 0x41,
		0x4f, 0xaf, 0xcd, 0x27, 0x2a, 0x61, 0xa1, 0x9e,
		0x82, 0x30, 0x05, 0x05, 0x55, 0xce, 0x99, 0xd3,
		0x8f, 0x3f, 0x86, 0x79, 0xdc, 0x9f, 0x33, 0x07,
		0x75, 0x26, 0xc8, 0x72, 0x81, 0x0f, 0x9b, 0xf7,
		0xb1, 0xfb, 0xd3, 0x91, 0x36, 0x08, 0xab, 0x26,
		0x70, 0x53, 0x0c, 0x99, 0xfd, 0xa9, 0x07, 0xb4,
		0xe9, 0xce, 0xc1, 0xd6, 0xd2, 0x2c, 0x71, 0x80,
		0xec, 0x59, 0x61, 0x0b, 0x24, 0xf0, 0x6d, 0x33,
		0x73, 0x45, 0x6e, 0x80, 0x03, 0x45, 0xf2, 0x76,
		0xa5, 0x8a, 0xc9, 0xcf, 0xaf, 0x4a, 0xed, 0x35,
		0xc0, 0x97, 0x52, 0xc5, 0x00, 0xdf, 0xef, 0xc7,
		0x9f, 0xf2, 0xe8, 0x15, 0x3e, 0xb3, 0x30, 0xe7,
		0x00, 0xd0, 0x4e, 0xeb, 0x79, 0xf6, 0xf6, 0xcf,
		0xf0, 0xe7, 0x61, 0xd5, 0x3d, 0x6a, 0x73, 0x9d
};

static const uint8_t AES_CBC_ciphertext_512B[] = {
		0xb4, 0xc6, 0xc6, 0x5f, 0x7e, 0xca, 0x05, 0x70,
		0x21, 0x7b, 0x92, 0x9e, 0x23, 0xe7, 0x92, 0xb8,
		0x27, 0x3d, 0x20, 0x29, 0x57, 0xfa, 0x1f, 0x26,
		0x0a, 0x04, 0x34, 0xa6, 0xf2, 0xdc, 0x44, 0xb6,
		0x43, 0x40, 0x62, 0xde, 0x0c, 0xde, 0x1c, 0x30,
		0x43, 0x85, 0x0b, 0xe8, 0x93, 0x1f, 0xa1, 0x2a,
		0x8a, 0x27, 0x35, 0x39, 0x14, 0x9f, 0x37, 0x64,
		0x59, 0xb5, 0x0e, 0x96, 0x82, 0x5d, 0x63, 0x45,
		0xd6, 0x93, 0x89, 0x46, 0xe4, 0x71, 0x31, 0xeb,
		0x0e, 0xd1, 0x7b, 0xda, 0x90, 0xb5, 0x81, 0xac,
		0x76, 0x54, 0x54, 0x85, 0x0b, 0xa9, 0x46, 0x9c,
		0xf0, 0xfd, 0xde, 0x5d, 0xa8, 0xe3, 0xee, 0xe9,
		0xf4, 0x9d, 0x34, 0x76, 0x39, 0xe7, 0xc3, 0x4a,
		0x84, 0x38, 0x92, 0x61, 0xf1, 0x12, 0x9f, 0x05,
		0xda, 0xdb, 0xc1, 0xd4, 0xb0, 0xa0, 0x27, 0x19,
		0xa0, 0x56, 0x5d, 0x9b, 0xcc, 0x47, 0x7c, 0x15,
		0x1d, 0x52, 0x66, 0xd5, 0xff, 0xef, 0x12, 0x23,
		0x86, 0xe2, 0xee, 0x81, 0x2c, 0x3d, 0x7d, 0x28,
		0xd5, 0x42, 0xdf, 0xdb, 0x75, 0x1c, 0xeb, 0xdf,
		0x13, 0x23, 0xd5, 0x17, 0x89, 0xea, 0xd7, 0x01,
		0xff, 0x57, 0x6a, 0x44, 0x61, 0xf4, 0xea, 0xbe,
		0x97, 0x9b, 0xc2, 0xb1, 0x9c, 0x5d, 0xff, 0x4f,
		0x73, 0x2d, 0x3f, 0x57, 0x28, 0x38, 0xbf, 0x3d,
		0x9f, 0xda, 0x49, 0x55, 0x8f, 0xb2, 0x77, 0xec,
		0x0f, 0xbc, 0xce, 0xb8, 0xc6, 0xe1, 0x03, 0xed,
		0x35, 0x9c, 0xf2, 0x4d, 0xa4, 0x29, 0x6c, 0xd6,
		0x6e, 0x05, 0x53, 0x46, 0xc1, 0x41, 0x09, 0x36,
		0x0b, 0x7d, 0xf4, 0x9e, 0x0f, 0xba, 0x86, 0x33,
		0xdd, 0xf1, 0xa7, 0xf7, 0xd5, 0x29, 0xa8, 0xa7,
		0x4d, 0xce, 0x0c, 0xf5, 0xb4, 0x6c, 0xd8, 0x27,
		0xb0, 0x87, 0x2a, 0x6f, 0x7f, 0x3f, 0x8f, 0xc3,
		0xe2, 0x3e, 0x94, 0xcf, 0x61, 0x4a, 0x09, 0x3d,
		0xf9, 0x55, 0x19, 0x31, 0xf2, 0xd2, 0x4a, 0x3e,
		0xc1, 0xf5, 0xed, 0x7c, 0x45, 0xb0, 0x0c, 0x7b,
		0xdd, 0xa6, 0x0a, 0x26, 0x66, 0xec, 0x85, 0x49,
		0x00, 0x38, 0x05, 0x7c, 0x9c, 0x1c, 0x92, 0xf5,
		0xf7, 0xdb, 0x5d, 0xbd, 0x61, 0x0c, 0xc9, 0xaf,
		0xfd, 0x57, 0x3f, 0xee, 0x2b, 0xad, 0x73, 0xef,
		0xa3, 0xc1, 0x66, 0x26, 0x44, 0x5e, 0xf9, 0x12,
		0x86, 0x66, 0xa9, 0x61, 0x75, 0xa1, 0xbc, 0x40,
		0x7f, 0xa8, 0x08, 0x02, 0xc0, 0x76, 0x0e, 0x76,
		0xb3, 0x26, 0x3d, 0x1c, 0x40, 0x65, 0xe4, 0x18,
		0x0f, 0x62, 0x17, 0x8f, 0x1e, 0x61, 0xb8, 0x08,
		0x83, 0x54, 0x42, 0x11, 0x03, 0x30, 0x8e, 0xb7,
		0xc1, 0x9c, 0xec, 0x69, 0x52, 0x95, 0xfb, 0x7b,
		0x1a, 0x0c, 0x20, 0x24, 0xf7, 0xb8, 0x38, 0x0c,
		0xb8, 0x7b, 0xb6, 0x69, 0x70, 0xd0, 0x61, 0xb9,
		0x70, 0x06, 0xc2, 0x5b, 0x20, 0x47, 0xf7, 0xd9,
		0x32, 0xc2, 0xf2, 0x90, 0xb6, 0x4d, 0xcd, 0x3c,
		0x6d, 0x74, 0xea, 0x82, 0x35, 0x1b, 0x08, 0x44,
		0xba, 0xb7, 0x33, 0x82, 0x33, 0x27, 0x54, 0x77,
		0x6e, 0x58, 0xfe, 0x46, 0x5a, 0xb4, 0x88, 0x53,
		0x8d, 0x9b, 0xb1, 0xab, 0xdf, 0x04, 0xe1, 0xfb,
		0xd7, 0x1e, 0xd7, 0x38, 0x64, 0x54, 0xba, 0xb0,
		0x6c, 0x84, 0x7a, 0x0f, 0xa7, 0x80, 0x6b, 0x86,
		0xd9, 0xc9, 0xc6, 0x31, 0x95, 0xfa, 0x8a, 0x2c,
		0x14, 0xe1, 0x85, 0x66, 0x27, 0xfd, 0x63, 0x3e,
		0xf0, 0xfa, 0x81, 0xc9, 0x89, 0x4f, 0xe2, 0x6a,
		0x8c, 0x17, 0xb5, 0xc7, 0x9f, 0x5d, 0x3f, 0x6b,
		0x3f, 0xcd, 0x13, 0x7a, 0x3c, 0xe6, 0x4e, 0xfa,
		0x7a, 0x10, 0xb8, 0x7c, 0x40, 0xec, 0x93, 0x11,
		0x1f, 0xd0, 0x9e, 0xc3, 0x56, 0xb9, 0xf5, 0x21,
		0x18, 0x41, 0x31, 0xea, 0x01, 0x8d, 0xea, 0x1c,
		0x95, 0x5e, 0x56, 0x33, 0xbc, 0x7a, 0x3f, 0x6f
};

static const uint8_t AES_CBC_ciphertext_768B[] = {
		0x3e, 0x7f, 0x9e, 0x4c, 0x88, 0x15, 0x68, 0x69,
		0x10, 0x09, 0xe1, 0xa7, 0x0f, 0x27, 0x88, 0x2d,
		0x90, 0x73, 0x4f, 0x67, 0xd3, 0x8b, 0xaf, 0xa1,
		0x2c, 0x37, 0xa5, 0x6c, 0x7c, 0xbd, 0x95, 0x4c,
		0x82, 0xcf, 0x05, 0x49, 0x16, 0x5c, 0xe7, 0x06,
		0xd4, 0xcb, 0x55, 0x65, 0x9a, 0xd0, 0xe1, 0x46,
		0x3a, 0x37, 0x71, 0xad, 0xb0, 0xb4, 0x99, 0x1e,
		0x23, 0x57, 0x48, 0x96, 0x9c, 0xc5, 0xc4, 0xdb,
		0x64, 0x3e, 0xc9, 0x7f, 0x90, 0x5a, 0xa0, 0x08,
		0x75, 0x4c, 0x09, 0x06, 0x31, 0x6e, 0x59, 0x29,
		0xfc, 0x2f, 0x72, 0xde, 0xf2, 0x40, 0x5a, 0xfe,
		0xd3, 0x66, 0x64, 0xb8, 0x9c, 0xc9, 0xa6, 0x1f,
		0xc3, 0x52, 0xcd, 0xb5, 0xd1, 0x4f, 0x43, 0x3f,
		0xf4, 0x59, 0x25, 0xc4, 0xdd, 0x3e, 0x58, 0x7c,
		0x21, 0xd6, 0x21, 0xce, 0xa4, 0xbe, 0x08, 0x23,
		0x46, 0x68, 0xc0, 0x00, 0x91, 0x47, 0xca, 0x9b,
		0xe0, 0xb4, 0xe3, 0xab, 0xbf, 0xcf, 0x68, 0x26,
		0x97, 0x23, 0x09, 0x93, 0x64, 0x8f, 0x57, 0x59,
		0xe2, 0x41, 0x7c, 0xa2, 0x48, 0x7e, 0xd5, 0x2c,
		0x54, 0x09, 0x1b, 0x07, 0x94, 0xca, 0x39, 0x83,
		0xdd, 0xf4, 0x7a, 0x1d, 0x2d, 0xdd, 0x67, 0xf7,
		0x3c, 0x30, 0x89, 0x3e, 0xc1, 0xdc, 0x1d, 0x8f,
		0xfc, 0xb1, 0xe9, 0x13, 0x31, 0xb0, 0x16, 0xdb,
		0x88, 0xf2, 0x32, 0x7e, 0x73, 0xa3, 0xdf, 0x08,
		0x6b, 0x53, 0x92, 0x08, 0xc9, 0x9d, 0x98, 0xb2,
		0xf4, 0x8c, 0xb1, 0x95, 0xdc, 0xb6, 0xfc, 0xec,
		0xf1, 0xc9, 0x0d, 0x6d, 0x42, 0x2c, 0xf5, 0x38,
		0x29, 0xf4, 0xd8, 0x98, 0x0f, 0xb0, 0x81, 0xa5,
		0xaa, 0xe6, 0x1f, 0x6e, 0x87, 0x32, 0x1b, 0x02,
		0x07, 0x57, 0x38, 0x83, 0xf3, 0xe4, 0x54, 0x7c,
		0xa8, 0x43, 0xdf, 0x3f, 0x42, 0xfd, 0x67, 0x28,
		0x06, 0x4d, 0xea, 0xce, 0x1f, 0x84, 0x4a, 0xcd,
		0x8c, 0x61, 0x5e, 0x8f, 0x61, 0xed, 0x84, 0x03,
		0x53, 0x6a, 0x9e, 0xbf, 0x68, 0x83, 0xa7, 0x42,
		0x56, 0x57, 0xcd, 0x45, 0x29, 0xfc, 0x7b, 0x07,
		0xfc, 0xe9, 0xb9, 0x42, 0xfd, 0x29, 0xd5, 0xfd,
		0x98, 0x11, 0xd1, 0x8d, 0x67, 0x29, 0x47, 0x61,
		0xd8, 0x27, 0x37, 0x79, 0x29, 0xd1, 0x94, 0x6f,
		0x8d, 0xf3, 0x1b, 0x3d, 0x6a, 0xb1, 0x59, 0xef,
		0x1b, 0xd4, 0x70, 0x0e, 0xac, 0xab, 0xa0, 0x2b,
		0x1f, 0x5e, 0x04, 0xf0, 0x0e, 0x35, 0x72, 0x90,
		0xfc, 0xcf, 0x86, 0x43, 0xea, 0x45, 0x6d, 0x22,
		0x63, 0x06, 0x1a, 0x58, 0xd7, 0x2d, 0xc5, 0xb0,
		0x60, 0x69, 0xe8, 0x53, 0xc2, 0xa2, 0x57, 0x83,
		0xc4, 0x31, 0xb4, 0xc6, 0xb3, 0xa1, 0x77, 0xb3,
		0x1c, 0xca, 0x89, 0x3f, 0xf5, 0x10, 0x3b, 0x36,
		0x31, 0x7d, 0x00, 0x46, 0x00, 0x92, 0xa0, 0xa0,
		0x34, 0xd8, 0x5e, 0x62, 0xa9, 0xe0, 0x23, 0x37,
		0x50, 0x85, 0xc7, 0x3a, 0x20, 0xa3, 0x98, 0xc0,
		0xac, 0x20, 0x06, 0x0f, 0x17, 0x3c, 0xfc, 0x43,
		0x8c, 0x9d, 0xec, 0xf5, 0x9a, 0x35, 0x96, 0xf7,
		0xb7, 0x4c, 0xf9, 0x69, 0xf8, 0xd4, 0x1e, 0x9e,
		0xf9, 0x7c, 0xc4, 0xd2, 0x11, 0x14, 0x41, 0xb9,
		0x89, 0xd6, 0x07, 0xd2, 0x37, 0x07, 0x5e, 0x5e,
		0xae, 0x60, 0xdc, 0xe4, 0xeb, 0x38, 0x48, 0x6d,
		0x95, 0x8d, 0x71, 0xf2, 0xba, 0xda, 0x5f, 0x08,
		0x9d, 0x4a, 0x0f, 0x56, 0x90, 0x64, 0xab, 0xb6,
		0x88, 0x22, 0xa8, 0x90, 0x1f, 0x76, 0x2c, 0x83,
		0x43, 0xce, 0x32, 0x55, 0x45, 0x84, 0x57, 0x43,
		0xf9, 0xa8, 0xd1, 0x4f, 0xe3, 0xc1, 0x72, 0x9c,
		0xeb, 0x64, 0xf7, 0xe4, 0x61, 0x2b, 0x93, 0xd1,
		0x1f, 0xbb, 0x5c, 0xff, 0xa1, 0x59, 0x69, 0xcf,
		0xf7, 0xaf, 0x58, 0x45, 0xd5, 0x3e, 0x98, 0x7d,
		0x26, 0x39, 0x5c, 0x75, 0x3c, 0x4a, 0xbf, 0x5e,
		0x12, 0x10, 0xb0, 0x93, 0x0f, 0x86, 0x82, 0xcf,
		0xb2, 0xec, 0x70, 0x5c, 0x0b, 0xad, 0x5d, 0x63,
		0x65, 0x32, 0xa6, 0x04, 0x58, 0x03, 0x91, 0x2b,
		0xdb, 0x8f, 0xd3, 0xa3, 0x2b, 0x3a, 0xf5, 0xa1,
		0x62, 0x6c, 0xb6, 0xf0, 0x13, 0x3b, 0x8c, 0x07,
		0x10, 0x82, 0xc9, 0x56, 0x24, 0x87, 0xfc, 0x56,
		0xe8, 0xef, 0x90, 0x8b, 0xd6, 0x48, 0xda, 0x53,
		0x04, 0x49, 0x41, 0xa4, 0x67, 0xe0, 0x33, 0x24,
		0x6b, 0x9c, 0x07, 0x55, 0x4c, 0x5d, 0xe9, 0x35,
		0xfa, 0xbd, 0xea, 0xa8, 0x3f, 0xe9, 0xf5, 0x20,
		0x5c, 0x60, 0x0f, 0x0d, 0x24, 0xcb, 0x1a, 0xd6,
		0xe8, 0x5c, 0xa8, 0x42, 0xae, 0xd0, 0xd2, 0xf2,
		0xa8, 0xbe, 0xea, 0x0f, 0x8d, 0xfb, 0x81, 0xa3,
		0xa4, 0xef, 0xb7, 0x3e, 0x91, 0xbd, 0x26, 0x0f,
		0x8e, 0xf1, 0xb2, 0xa5, 0x47, 0x06, 0xfa, 0x40,
		0x8b, 0x31, 0x7a, 0x5a, 0x74, 0x2a, 0x0a, 0x7c,
		0x62, 0x5d, 0x39, 0xa4, 0xae, 0x14, 0x85, 0x08,
		0x5b, 0x20, 0x85, 0xf1, 0x57, 0x6e, 0x71, 0x13,
		0x4e, 0x2b, 0x49, 0x87, 0x01, 0xdf, 0x37, 0xed,
		0x28, 0xee, 0x4d, 0xa1, 0xf4, 0xb3, 0x3b, 0xba,
		0x2d, 0xb3, 0x46, 0x17, 0x84, 0x80, 0x9d, 0xd7,
		0x93, 0x1f, 0x28, 0x7c, 0xf5, 0xf9, 0xd6, 0x85,
		0x8c, 0xa5, 0x44, 0xe9, 0x2c, 0x65, 0x51, 0x5f,
		0x53, 0x7a, 0x09, 0xd9, 0x30, 0x16, 0x95, 0x89,
		0x9c, 0x0b, 0xef, 0x90, 0x6d, 0x23, 0xd3, 0x48,
		0x57, 0x3b, 0x55, 0x69, 0x96, 0xfc, 0xf7, 0x52,
		0x92, 0x38, 0x36, 0xbf, 0xa9, 0x0a, 0xbb, 0x68,
		0x45, 0x08, 0x25, 0xee, 0x59, 0xfe, 0xee, 0xf2,
		0x2c, 0xd4, 0x5f, 0x78, 0x59, 0x0d, 0x90, 0xf1,
		0xd7, 0xe4, 0x39, 0x0e, 0x46, 0x36, 0xf5, 0x75,
		0x03, 0x3c, 0x28, 0xfb, 0xfa, 0x8f, 0xef, 0xc9,
		0x61, 0x00, 0x94, 0xc3, 0xd2, 0x0f, 0xd9, 0xda
};

static const uint8_t AES_CBC_ciphertext_1024B[] = {
		0x7d, 0x01, 0x7e, 0x2f, 0x92, 0xb3, 0xea, 0x72,
		0x4a, 0x3f, 0x10, 0xf9, 0x2b, 0xb0, 0xd5, 0xb9,
		0x19, 0x68, 0x94, 0xe9, 0x93, 0xe9, 0xd5, 0x26,
		0x20, 0x44, 0xe2, 0x47, 0x15, 0x8d, 0x75, 0x48,
		0x8e, 0xe4, 0x40, 0x81, 0xb5, 0x06, 0xa8, 0xb8,
		0x0e, 0x0f, 0x3b, 0xbc, 0x5b, 0xbe, 0x3b, 0xa2,
		0x2a, 0x0c, 0x48, 0x98, 0x19, 0xdf, 0xe9, 0x25,
		0x75, 0xab, 0x93, 0x44, 0xb1, 0x72, 0x70, 0xbb,
		0x20, 0xcf, 0x78, 0xe9, 0x4d, 0xc6, 0xa9, 0xa9,
		0x84, 0x78, 0xc5, 0xc0, 0xc4, 0xc9, 0x79, 0x1a,
		0xbc, 0x61, 0x25, 0x5f, 0xac, 0x01, 0x03, 0xb7,
		0xef, 0x07, 0xf2, 0x62, 0x98, 0xee, 0xe3, 0xad,
		0x94, 0x75, 0x30, 0x67, 0xb9, 0x15, 0x00, 0xe7,
		0x11, 0x32, 0x2e, 0x6b, 0x55, 0x9f, 0xac, 0x68,
		0xde, 0x61, 0x05, 0x80, 0x01, 0xf3, 0xad, 0xab,
		0xaf, 0x45, 0xe0, 0xf4, 0x68, 0x5c, 0xc0, 0x52,
		0x92, 0xc8, 0x21, 0xb6, 0xf5, 0x8a, 0x1d, 0xbb,
		0xfc, 0x4a, 0x11, 0x62, 0xa2, 0xc4, 0xf1, 0x2d,
		0x0e, 0xb2, 0xc7, 0x17, 0x34, 0xb4, 0x2a, 0x54,
		0x81, 0xc2, 0x1e, 0xcf, 0x51, 0x0a, 0x76, 0x54,
		0xf1, 0x48, 0x0d, 0x5c, 0xcd, 0x38, 0x3e, 0x38,
		0x3e, 0xf8, 0x46, 0x1d, 0x00, 0xf5, 0x62, 0xe1,
		0x5c, 0xb7, 0x8d, 0xce, 0xd0, 0x3f, 0xbb, 0x22,
		0xf1, 0xe5, 0xb1, 0xa0, 0x58, 0x5e, 0x3c, 0x0f,
		0x15, 0xd1, 0xac, 0x3e, 0xc7, 0x72, 0xc4, 0xde,
		0x8b, 0x95, 0x3e, 0x91, 0xf7, 0x1d, 0x04, 0x9a,
		0xc8, 0xe4, 0xbf, 0xd3, 0x22, 0xca, 0x4a, 0xdc,
		0xb6, 0x16, 0x79, 0x81, 0x75, 0x2f, 0x6b, 0xa7,
		0x04, 0x98, 0xa7, 0x4e, 0xc1, 0x19, 0x90, 0x33,
		0x33, 0x3c, 0x7f, 0xdd, 0xac, 0x09, 0x0c, 0xc3,
		0x91, 0x34, 0x74, 0xab, 0xa5, 0x35, 0x0a, 0x13,
		0xc3, 0x56, 0x67, 0x6d, 0x1a, 0x3e, 0xbf, 0x56,
		0x06, 0x67, 0x15, 0x5f, 0xfc, 0x8b, 0xa2, 0x3c,
		0x5e, 0xaf, 0x56, 0x1f, 0xe3, 0x2e, 0x9d, 0x0a,
		0xf9, 0x9b, 0xc7, 0xb5, 0x03, 0x1c, 0x68, 0x99,
		0xfa, 0x3c, 0x37, 0x59, 0xc1, 0xf7, 0x6a, 0x83,
		0x22, 0xee, 0xca, 0x7f, 0x7d, 0x49, 0xe6, 0x48,
		0x84, 0x54, 0x7a, 0xff, 0xb3, 0x72, 0x21, 0xd8,
		0x7a, 0x5d, 0xb1, 0x4b, 0xcc, 0x01, 0x6f, 0x90,
		0xc6, 0x68, 0x1c, 0x2c, 0xa1, 0xe2, 0x74, 0x40,
		0x26, 0x9b, 0x57, 0x53, 0xa3, 0x7c, 0x0b, 0x0d,
		0xcf, 0x05, 0x5d, 0x62, 0x4f, 0x75, 0x06, 0x62,
		0x1f, 0x26, 0x32, 0xaa, 0x25, 0xcc, 0x26, 0x8d,
		0xae, 0x01, 0x47, 0xa3, 0x00, 0x42, 0xe2, 0x4c,
		0xee, 0x29, 0xa2, 0x81, 0xa0, 0xfd, 0xeb, 0xff,
		0x9a, 0x66, 0x6e, 0x47, 0x5b, 0xab, 0x93, 0x5a,
		0x02, 0x6d, 0x6f, 0xf2, 0x6e, 0x02, 0x9d, 0xb1,
		0xab, 0x56, 0xdc, 0x8b, 0x9b, 0x17, 0xa8, 0xfb,
		0x87, 0x42, 0x7c, 0x91, 0x1e, 0x14, 0xc6, 0x6f,
		0xdc, 0xf0, 0x27, 0x30, 0xfa, 0x3f, 0xc4, 0xad,
		0x57, 0x85, 0xd2, 0xc9, 0x32, 0x2c, 0x13, 0xa6,
		0x04, 0x04, 0x50, 0x05, 0x2f, 0x72, 0xd9, 0x44,
		0x55, 0x6e, 0x93, 0x40, 0xed, 0x7e, 0xd4, 0x40,
		0x3e, 0x88, 0x3b, 0x8b, 0xb6, 0xeb, 0xc6, 0x5d,
		0x9c, 0x99, 0xa1, 0xcf, 0x30, 0xb2, 0xdc, 0x48,
		0x8a, 0x01, 0xa7, 0x61, 0x77, 0x50, 0x14, 0xf3,
		0x0c, 0x49, 0x53, 0xb3, 0xb4, 0xb4, 0x28, 0x41,
		0x4a, 0x2d, 0xd2, 0x4d, 0x2a, 0x30, 0x31, 0x83,
		0x03, 0x5e, 0xaa, 0xd3, 0xa3, 0xd1, 0xa1, 0xca,
		0x62, 0xf0, 0xe1, 0xf2, 0xff, 0xf0, 0x19, 0xa6,
		0xde, 0x22, 0x47, 0xb5, 0x28, 0x7d, 0xf7, 0x07,
		0x16, 0x0d, 0xb1, 0x55, 0x81, 0x95, 0xe5, 0x1d,
		0x4d, 0x78, 0xa9, 0x3e, 0xce, 0xe3, 0x1c, 0xf9,
		0x47, 0xc8, 0xec, 0xc5, 0xc5, 0x93, 0x4c, 0x34,
		0x20, 0x6b, 0xee, 0x9a, 0xe6, 0x86, 0x57, 0x58,
		0xd5, 0x58, 0xf1, 0x33, 0x10, 0x29, 0x9e, 0x93,
		0x2f, 0xf5, 0x90, 0x00, 0x17, 0x67, 0x4f, 0x39,
		0x18, 0xe1, 0xcf, 0x55, 0x78, 0xbb, 0xe6, 0x29,
		0x3e, 0x77, 0xd5, 0x48, 0xb7, 0x42, 0x72, 0x53,
		0x27, 0xfa, 0x5b, 0xe0, 0x36, 0x14, 0x97, 0xb8,
		0x9b, 0x3c, 0x09, 0x77, 0xc1, 0x0a, 0xe4, 0xa2,
		0x63, 0xfc, 0xbe, 0x5c, 0x17, 0xcf, 0x01, 0xf5,
		0x03, 0x0f, 0x17, 0xbc, 0x93, 0xdd, 0x5f, 0xe2,
		0xf3, 0x08, 0xa8, 0xb1, 0x85, 0xb6, 0x34, 0x3f,
		0x87, 0x42, 0xa5, 0x42, 0x3b, 0x0e, 0xd6, 0x83,
		0x6a, 0xfd, 0x5d, 0xc9, 0x67, 0xd5, 0x51, 0xc9,
		0x2a, 0x4e, 0x91, 0xb0, 0x59, 0xb2, 0x0f, 0xa2,
		0xe6, 0x47, 0x73, 0xc2, 0xa2, 0xae, 0xbb, 0xc8,
		0x42, 0xa3, 0x2a, 0x27, 0x29, 0x48, 0x8c, 0x54,
		0x6c, 0xec, 0x00, 0x2a, 0x42, 0xa3, 0x7a, 0x0f,
		0x12, 0x66, 0x6b, 0x96, 0xf6, 0xd0, 0x56, 0x4f,
		0x49, 0x5c, 0x47, 0xec, 0x05, 0x62, 0x54, 0xb2,
		0x64, 0x5a, 0x69, 0x1f, 0x19, 0xb4, 0x84, 0x5c,
		0xbe, 0x48, 0x8e, 0xfc, 0x58, 0x21, 0xce, 0xfa,
		0xaa, 0x84, 0xd2, 0xc1, 0x08, 0xb3, 0x87, 0x0f,
		0x4f, 0xa3, 0x3a, 0xb6, 0x44, 0xbe, 0x2e, 0x9a,
		0xdd, 0xb5, 0x44, 0x80, 0xca, 0xf4, 0xc3, 0x6e,
		0xba, 0x93, 0x77, 0xe0, 0x53, 0xfb, 0x37, 0xfb,
		0x88, 0xc3, 0x1f, 0x25, 0xde, 0x3e, 0x11, 0xf4,
		0x89, 0xe7, 0xd1, 0x3b, 0xb4, 0x23, 0xcb, 0x70,
		0xba, 0x35, 0x97, 0x7c, 0xbe, 0x84, 0x13, 0xcf,
		0xe0, 0x4d, 0x33, 0x91, 0x71, 0x85, 0xbb, 0x4b,
		0x97, 0x32, 0x5d, 0xa0, 0xb9, 0x8f, 0xdc, 0x27,
		0x5a, 0xeb, 0x71, 0xf1, 0xd5, 0x0d, 0x65, 0xb4,
		0x22, 0x81, 0xde, 0xa7, 0x58, 0x20, 0x0b, 0x18,
		0x11, 0x76, 0x5c, 0xe6, 0x6a, 0x2c, 0x99, 0x69,
		0xdc, 0xed, 0x67, 0x08, 0x5d, 0x5e, 0xe9, 0x1e,
		0x55, 0x70, 0xc1, 0x5a, 0x76, 0x1b, 0x8d, 0x2e,
		0x0d, 0xf9, 0xcc, 0x30, 0x8c, 0x44, 0x0f, 0x63,
		0x8c, 0x42, 0x8a, 0x9f, 0x4c, 0xd1, 0x48, 0x28,
		0x8a, 0xf5, 0x56, 0x2e, 0x23, 0x12, 0xfe, 0x67,
		0x9a, 0x13, 0x65, 0x75, 0x83, 0xf1, 0x3c, 0x98,
		0x07, 0x6b, 0xb7, 0x27, 0x5b, 0xf0, 0x70, 0xda,
		0x30, 0xf8, 0x74, 0x4e, 0x7a, 0x32, 0x84, 0xcc,
		0x0e, 0xcd, 0x80, 0x8b, 0x82, 0x31, 0x9a, 0x48,
		0xcf, 0x75, 0x00, 0x1f, 0x4f, 0xe0, 0x8e, 0xa3,
		0x6a, 0x2c, 0xd4, 0x73, 0x4c, 0x63, 0x7c, 0xa6,
		0x4d, 0x5e, 0xfd, 0x43, 0x3b, 0x27, 0xe1, 0x5e,
		0xa3, 0xa9, 0x5c, 0x3b, 0x60, 0xdd, 0xc6, 0x8d,
		0x5a, 0xf1, 0x3e, 0x89, 0x4b, 0x24, 0xcf, 0x01,
		0x3a, 0x2d, 0x44, 0xe7, 0xda, 0xe7, 0xa1, 0xac,
		0x11, 0x05, 0x0c, 0xa9, 0x7a, 0x82, 0x8c, 0x5c,
		0x29, 0x68, 0x9c, 0x73, 0x13, 0xcc, 0x67, 0x32,
		0x11, 0x5e, 0xe5, 0xcc, 0x8c, 0xf5, 0xa7, 0x52,
		0x83, 0x9a, 0x70, 0xef, 0xde, 0x55, 0x9c, 0xc7,
		0x8a, 0xed, 0xad, 0x28, 0x4a, 0xc5, 0x92, 0x6d,
		0x8e, 0x47, 0xca, 0xe3, 0xf8, 0x77, 0xb5, 0x26,
		0x64, 0x84, 0xc2, 0xf1, 0xd7, 0xae, 0x0c, 0xb9,
		0x39, 0x0f, 0x43, 0x6b, 0xe9, 0xe0, 0x09, 0x4b,
		0xe5, 0xe3, 0x17, 0xa6, 0x68, 0x69, 0x46, 0xf4,
		0xf0, 0x68, 0x7f, 0x2f, 0x1c, 0x7e, 0x4c, 0xd2,
		0xb5, 0xc6, 0x16, 0x85, 0xcf, 0x02, 0x4c, 0x89,
		0x0b, 0x25, 0xb0, 0xeb, 0xf3, 0x77, 0x08, 0x6a,
		0x46, 0x5c, 0xf6, 0x2f, 0xf1, 0x24, 0xc3, 0x4d,
		0x80, 0x60, 0x4d, 0x69, 0x98, 0xde, 0xc7, 0xa1,
		0xf6, 0x4e, 0x18, 0x0c, 0x2a, 0xb0, 0xb2, 0xe0,
		0x46, 0xe7, 0x49, 0x37, 0xc8, 0x5a, 0x23, 0x24,
		0xe3, 0x0f, 0xcc, 0x92, 0xb4, 0x8d, 0xdc, 0x9e
};

static const uint8_t AES_CBC_ciphertext_1280B[] = {
		0x91, 0x99, 0x5e, 0x9e, 0x84, 0xff, 0x59, 0x45,
		0xc1, 0xf4, 0xbc, 0x9c, 0xb9, 0x30, 0x6c, 0x51,
		0x73, 0x52, 0xb4, 0x44, 0x09, 0x79, 0xe2, 0x89,
		0x75, 0xeb, 0x54, 0x26, 0xce, 0xd8, 0x24, 0x98,
		0xaa, 0xf8, 0x13, 0x16, 0x68, 0x58, 0xc4, 0x82,
		0x0e, 0x31, 0xd3, 0x6a, 0x13, 0x58, 0x31, 0xe9,
		0x3a, 0xc1, 0x8b, 0xc5, 0x3f, 0x50, 0x42, 0xd1,
		0x93, 0xe4, 0x9b, 0x65, 0x2b, 0xf4, 0x1d, 0x9e,
		0x2d, 0xdb, 0x48, 0xef, 0x9a, 0x01, 0x68, 0xb6,
		0xea, 0x7a, 0x2b, 0xad, 0xfe, 0x77, 0x44, 0x7e,
		0x5a, 0xc5, 0x64, 0xb4, 0xfe, 0x5c, 0x80, 0xf3,
		0x20, 0x7e, 0xaf, 0x5b, 0xf8, 0xd1, 0x38, 0xa0,
		0x8d, 0x09, 0x77, 0x06, 0xfe, 0xf5, 0xf4, 0xe4,
		0xee, 0xb8, 0x95, 0x27, 0xed, 0x07, 0xb8, 0xaa,
		0x25, 0xb4, 0xe1, 0x4c, 0xeb, 0x3f, 0xdb, 0x39,
		0x66, 0x28, 0x1b, 0x60, 0x42, 0x8b, 0x99, 0xd9,
		0x49, 0xd6, 0x8c, 0xa4, 0x9d, 0xd8, 0x93, 0x58,
		0x8f, 0xfa, 0xd3, 0xf7, 0x37, 0x9c, 0x88, 0xab,
		0x16, 0x50, 0xfe, 0x01, 0x1f, 0x88, 0x48, 0xbe,
		0x21, 0xa9, 0x90, 0x9e, 0x73, 0xe9, 0x82, 0xf7,
		0xbf, 0x4b, 0x43, 0xf4, 0xbf, 0x22, 0x3c, 0x45,
		0x47, 0x95, 0x5b, 0x49, 0x71, 0x07, 0x1c, 0x8b,
		0x49, 0xa4, 0xa3, 0x49, 0xc4, 0x5f, 0xb1, 0xf5,
		0xe3, 0x6b, 0xf1, 0xdc, 0xea, 0x92, 0x7b, 0x29,
		0x40, 0xc9, 0x39, 0x5f, 0xdb, 0xbd, 0xf3, 0x6a,
		0x09, 0x9b, 0x2a, 0x5e, 0xc7, 0x0b, 0x25, 0x94,
		0x55, 0x71, 0x9c, 0x7e, 0x0e, 0xb4, 0x08, 0x12,
		0x8c, 0x6e, 0x77, 0xb8, 0x29, 0xf1, 0xc6, 0x71,
		0x04, 0x40, 0x77, 0x18, 0x3f, 0x01, 0x09, 0x9c,
		0x23, 0x2b, 0x5d, 0x2a, 0x88, 0x20, 0x23, 0x59,
		0x74, 0x2a, 0x67, 0x8f, 0xb7, 0xba, 0x38, 0x9f,
		0x0f, 0xcf, 0x94, 0xdf, 0xe1, 0x8f, 0x35, 0x5e,
		0x34, 0x0c, 0x32, 0x92, 0x2b, 0x23, 0x81, 0xf4,
		0x73, 0xa0, 0x5a, 0x2a, 0xbd, 0xa6, 0x6b, 0xae,
		0x43, 0xe2, 0xdc, 0x01, 0xc1, 0xc6, 0xc3, 0x04,
		0x06, 0xbb, 0xb0, 0x89, 0xb3, 0x4e, 0xbd, 0x81,
		0x1b, 0x03, 0x63, 0x93, 0xed, 0x4e, 0xf6, 0xe5,
		0x94, 0x6f, 0xd6, 0xf3, 0x20, 0xf3, 0xbc, 0x30,
		0xc5, 0xd6, 0xbe, 0x1c, 0x05, 0x34, 0x26, 0x4d,
		0x46, 0x5e, 0x56, 0x63, 0xfb, 0xdb, 0xcd, 0xed,
		0xb0, 0x7f, 0x83, 0x94, 0x55, 0x54, 0x2f, 0xab,
		0xc9, 0xb7, 0x16, 0x4f, 0x9e, 0x93, 0x25, 0xd7,
		0x9f, 0x39, 0x2b, 0x63, 0xcf, 0x1e, 0xa3, 0x0e,
		0x28, 0x47, 0x8a, 0x5f, 0x40, 0x02, 0x89, 0x1f,
		0x83, 0xe7, 0x87, 0xd1, 0x90, 0x17, 0xb8, 0x27,
		0x64, 0xe1, 0xe1, 0x48, 0x5a, 0x55, 0x74, 0x99,
		0x27, 0x9d, 0x05, 0x67, 0xda, 0x70, 0x12, 0x8f,
		0x94, 0x96, 0xfd, 0x36, 0xa4, 0x1d, 0x22, 0xe5,
		0x0b, 0xe5, 0x2f, 0x38, 0x55, 0xa3, 0x5d, 0x0b,
		0xcf, 0xd4, 0xa9, 0xb8, 0xd6, 0x9a, 0x16, 0x2e,
		0x6c, 0x4a, 0x25, 0x51, 0x7a, 0x09, 0x48, 0xdd,
		0xf0, 0xa3, 0x5b, 0x08, 0x1e, 0x2f, 0x03, 0x91,
		0x80, 0xe8, 0x0f, 0xe9, 0x5a, 0x2f, 0x90, 0xd3,
		0x64, 0xed, 0xd7, 0x51, 0x17, 0x66, 0x53, 0x40,
		0x43, 0x74, 0xef, 0x0a, 0x0d, 0x49, 0x41, 0xf2,
		0x67, 0x6e, 0xea, 0x14, 0xc8, 0x74, 0xd6, 0xa9,
		0xb9, 0x6a, 0xe3, 0xec, 0x7d, 0xe8, 0x6a, 0x21,
		0x3a, 0x52, 0x42, 0xfe, 0x9a, 0x15, 0x6d, 0x60,
		0x64, 0x88, 0xc5, 0xb2, 0x8b, 0x15, 0x2c, 0xff,
		0xe2, 0x35, 0xc3, 0xee, 0x9f, 0xcd, 0x82, 0xd9,
		0x14, 0x35, 0x2a, 0xb7, 0xf5, 0x2f, 0x7b, 0xbc,
		0x01, 0xfd, 0xa8, 0xe0, 0x21, 0x4e, 0x73, 0xf9,
		0xf2, 0xb0, 0x79, 0xc9, 0x10, 0x52, 0x8f, 0xa8,
		0x3e, 0x3b, 0xbe, 0xc5, 0xde, 0xf6, 0x53, 0xe3,
		0x1c, 0x25, 0x3a, 0x1f, 0x13, 0xbf, 0x13, 0xbb,
		0x94, 0xc2, 0x97, 0x43, 0x64, 0x47, 0x8f, 0x76,
		0xd7, 0xaa, 0xeb, 0xa4, 0x03, 0x50, 0x0c, 0x10,
		0x50, 0xd8, 0xf7, 0x75, 0x52, 0x42, 0xe2, 0x94,
		0x67, 0xf4, 0x60, 0xfb, 0x21, 0x9b, 0x7a, 0x05,
		0x50, 0x7c, 0x1b, 0x4a, 0x8b, 0x29, 0xe1, 0xac,
		0xd7, 0x99, 0xfd, 0x0d, 0x65, 0x92, 0xcd, 0x23,
		0xa7, 0x35, 0x8e, 0x13, 0xf2, 0xe4, 0x10, 0x74,
		0xc6, 0x4f, 0x19, 0xf7, 0x01, 0x0b, 0x46, 0xab,
		0xef, 0x8d, 0x4a, 0x4a, 0xfa, 0xda, 0xf3, 0xfb,
		0x40, 0x28, 0x88, 0xa2, 0x65, 0x98, 0x4d, 0x88,
		0xc7, 0xbf, 0x00, 0xc8, 0xd0, 0x91, 0xcb, 0x89,
		0x2f, 0xb0, 0x85, 0xfc, 0xa1, 0xc1, 0x9e, 0x83,
		0x88, 0xad, 0x95, 0xc0, 0x31, 0xa0, 0xad, 0xa2,
		0x42, 0xb5, 0xe7, 0x55, 0xd4, 0x93, 0x5a, 0x74,
		0x4e, 0x41, 0xc3, 0xcf, 0x96, 0x83, 0x46, 0xa1,
		0xb7, 0x5b, 0xb1, 0x34, 0x67, 0x4e, 0xb1, 0xd7,
		0x40, 0x20, 0x72, 0xe9, 0xc8, 0x74, 0xb7, 0xde,
		0x72, 0x29, 0x77, 0x4c, 0x74, 0x7e, 0xcc, 0x18,
		0xa5, 0x8d, 0x79, 0x8c, 0xd6, 0x6e, 0xcb, 0xd9,
		0xe1, 0x61, 0xe7, 0x36, 0xbc, 0x37, 0xea, 0xee,
		0xd8, 0x3c, 0x5e, 0x7c, 0x47, 0x50, 0xd5, 0xec,
		0x37, 0xc5, 0x63, 0xc3, 0xc9, 0x99, 0x23, 0x9f,
		0x64, 0x39, 0xdf, 0x13, 0x96, 0x6d, 0xea, 0x08,
		0x0c, 0x27, 0x2d, 0xfe, 0x0f, 0xc2, 0xa3, 0x97,
		0x04, 0x12, 0x66, 0x0d, 0x94, 0xbf, 0xbe, 0x3e,
		0xb9, 0xcf, 0x8e, 0xc1, 0x9d, 0xb1, 0x64, 0x17,
		0x54, 0x92, 0x3f, 0x0a, 0x51, 0xc8, 0xf5, 0x82,
		0x98, 0x73, 0x03, 0xc0, 0x5a, 0x51, 0x01, 0x67,
		0xb4, 0x01, 0x04, 0x06, 0xbc, 0x37, 0xde, 0x96,
		0x23, 0x3c, 0xce, 0x98, 0x3f, 0xd6, 0x51, 0x1b,
		0x01, 0x83, 0x0a, 0x1c, 0xf9, 0xeb, 0x7e, 0x72,
		0xa9, 0x51, 0x23, 0xc8, 0xd7, 0x2f, 0x12, 0xbc,
		0x08, 0xac, 0x07, 0xe7, 0xa7, 0xe6, 0x46, 0xae,
		0x54, 0xa3, 0xc2, 0xf2, 0x05, 0x2d, 0x06, 0x5e,
		0xfc, 0xe2, 0xa2, 0x23, 0xac, 0x86, 0xf2, 0x54,
		0x83, 0x4a, 0xb6, 0x48, 0x93, 0xa1, 0x78, 0xc2,
		0x07, 0xec, 0x82, 0xf0, 0x74, 0xa9, 0x18, 0xe9,
		0x53, 0x44, 0x49, 0xc2, 0x94, 0xf8, 0x94, 0x92,
		0x08, 0x3f, 0xbf, 0xa6, 0xe5, 0xc6, 0x03, 0x8a,
		0xc6, 0x90, 0x48, 0x6c, 0xee, 0xbd, 0x44, 0x92,
		0x1f, 0x2a, 0xce, 0x1d, 0xb8, 0x31, 0xa2, 0x9d,
		0x24, 0x93, 0xa8, 0x9f, 0x36, 0x00, 0x04, 0x7b,
		0xcb, 0x93, 0x59, 0xa1, 0x53, 0xdb, 0x13, 0x7a,
		0x54, 0xb1, 0x04, 0xdb, 0xce, 0x48, 0x4f, 0xe5,
		0x2f, 0xcb, 0xdf, 0x8f, 0x50, 0x7c, 0xfc, 0x76,
		0x80, 0xb4, 0xdc, 0x3b, 0xc8, 0x98, 0x95, 0xf5,
		0x50, 0xba, 0x70, 0x5a, 0x97, 0xd5, 0xfc, 0x98,
		0x4d, 0xf3, 0x61, 0x0f, 0xcf, 0xac, 0x49, 0x0a,
		0xdb, 0xc1, 0x42, 0x8f, 0xb6, 0x29, 0xd5, 0x65,
		0xef, 0x83, 0xf1, 0x30, 0x4b, 0x84, 0xd0, 0x69,
		0xde, 0xd2, 0x99, 0xe5, 0xec, 0xd3, 0x90, 0x86,
		0x39, 0x2a, 0x6e, 0xd5, 0x32, 0xe3, 0x0d, 0x2d,
		0x01, 0x8b, 0x17, 0x55, 0x1d, 0x65, 0x57, 0xbf,
		0xd8, 0x75, 0xa4, 0x85, 0xb6, 0x4e, 0x35, 0x14,
		0x58, 0xe4, 0x89, 0xb8, 0x7a, 0x58, 0x86, 0x0c,
		0xbd, 0x8b, 0x05, 0x7b, 0x63, 0xc0, 0x86, 0x80,
		0x33, 0x46, 0xd4, 0x9b, 0xb6, 0x0a, 0xeb, 0x6c,
		0xae, 0xd6, 0x57, 0x7a, 0xc7, 0x59, 0x33, 0xa0,
		0xda, 0xa4, 0x12, 0xbf, 0x52, 0x22, 0x05, 0x8d,
		0xeb, 0xee, 0xd5, 0xec, 0xea, 0x29, 0x9b, 0x76,
		0x95, 0x50, 0x6d, 0x99, 0xe1, 0x45, 0x63, 0x09,
		0x16, 0x5f, 0xb0, 0xf2, 0x5b, 0x08, 0x33, 0xdd,
		0x8f, 0xb7, 0x60, 0x7a, 0x8e, 0xc6, 0xfc, 0xac,
		0xa9, 0x56, 0x2c, 0xa9, 0x8b, 0x74, 0x33, 0xad,
		0x2a, 0x7e, 0x96, 0xb6, 0xba, 0x22, 0x28, 0xcf,
		0x4d, 0x96, 0xb7, 0xd1, 0xfa, 0x99, 0x4a, 0x61,
		0xe6, 0x84, 0xd1, 0x94, 0xca, 0xf5, 0x86, 0xb0,
		0xba, 0x34, 0x7a, 0x04, 0xcc, 0xd4, 0x81, 0xcd,
		0xd9, 0x86, 0xb6, 0xe0, 0x5a, 0x6f, 0x9b, 0x99,
		0xf0, 0xdf, 0x49, 0xae, 0x6d, 0xc2, 0x54, 0x67,
		0xe0, 0xb4, 0x34, 0x2d, 0x1c, 0x46, 0xdf, 0x73,
		0x3b, 0x45, 0x43, 0xe7, 0x1f, 0xa3, 0x36, 0x35,
		0x25, 0x33, 0xd9, 0xc0, 0x54, 0x38, 0x6e, 0x6b,
		0x80, 0xcf, 0x50, 0xa4, 0xb6, 0x21, 0x17, 0xfd,
		0x9b, 0x5c, 0x36, 0xca, 0xcc, 0x73, 0x73, 0xad,
		0xe0, 0x57, 0x77, 0x90, 0x0e, 0x7f, 0x0f, 0x87,
		0x7f, 0xdb, 0x73, 0xbf, 0xda, 0xc2, 0xb3, 0x05,
		0x22, 0x06, 0xf5, 0xa3, 0xfc, 0x1e, 0x8f, 0xda,
		0xcf, 0x49, 0xd6, 0xb3, 0x66, 0x2c, 0xb5, 0x00,
		0xaf, 0x85, 0x6e, 0xb8, 0x5b, 0x8c, 0xa1, 0xa4,
		0x21, 0xce, 0x40, 0xf3, 0x98, 0xac, 0xec, 0x88,
		0x62, 0x43, 0x2a, 0xac, 0xca, 0xcf, 0xb9, 0x30,
		0xeb, 0xfc, 0xef, 0xf0, 0x6e, 0x64, 0x6d, 0xe7,
		0x54, 0x88, 0x6b, 0x22, 0x29, 0xbe, 0xa5, 0x8c,
		0x31, 0x23, 0x3b, 0x4a, 0x80, 0x37, 0xe6, 0xd0,
		0x05, 0xfc, 0x10, 0x0e, 0xdd, 0xbb, 0x00, 0xc5,
		0x07, 0x20, 0x59, 0xd3, 0x41, 0x17, 0x86, 0x46,
		0xab, 0x68, 0xf6, 0x48, 0x3c, 0xea, 0x5a, 0x06,
		0x30, 0x21, 0x19, 0xed, 0x74, 0xbe, 0x0b, 0x97,
		0xee, 0x91, 0x35, 0x94, 0x1f, 0xcb, 0x68, 0x7f,
		0xe4, 0x48, 0xb0, 0x16, 0xfb, 0xf0, 0x74, 0xdb,
		0x06, 0x59, 0x2e, 0x5a, 0x9c, 0xce, 0x8f, 0x7d,
		0xba, 0x48, 0xd5, 0x3f, 0x5c, 0xb0, 0xc2, 0x33,
		0x48, 0x60, 0x17, 0x08, 0x85, 0xba, 0xff, 0xb9,
		0x34, 0x0a, 0x3d, 0x8f, 0x21, 0x13, 0x12, 0x1b
};

static const uint8_t AES_CBC_ciphertext_1536B[] = {
		0x89, 0x93, 0x05, 0x99, 0xa9, 0xed, 0xea, 0x62,
		0xc9, 0xda, 0x51, 0x15, 0xce, 0x42, 0x91, 0xc3,
		0x80, 0xc8, 0x03, 0x88, 0xc2, 0x63, 0xda, 0x53,
		0x1a, 0xf3, 0xeb, 0xd5, 0xba, 0x6f, 0x23, 0xb2,
		0xed, 0x8f, 0x89, 0xb1, 0xb3, 0xca, 0x90, 0x7a,
		0xdd, 0x3f, 0xf6, 0xca, 0x86, 0x58, 0x54, 0xbc,
		0xab, 0x0f, 0xf4, 0xab, 0x6d, 0x5d, 0x42, 0xd0,
		0x17, 0x49, 0x17, 0xd1, 0x93, 0xea, 0xe8, 0x22,
		0xc1, 0x34, 0x9f, 0x3a, 0x3b, 0xaa, 0xe9, 0x1b,
		0x93, 0xff, 0x6b, 0x68, 0xba, 0xe6, 0xd2, 0x39,
		0x3d, 0x55, 0x34, 0x8f, 0x98, 0x86, 0xb4, 0xd8,
		0x7c, 0x0d, 0x3e, 0x01, 0x63, 0x04, 0x01, 0xff,
		0x16, 0x0f, 0x51, 0x5f, 0x73, 0x53, 0xf0, 0x3a,
		0x38, 0xb4, 0x4d, 0x8d, 0xaf, 0xa3, 0xca, 0x2f,
		0x6f, 0xdf, 0xc0, 0x41, 0x6c, 0x48, 0x60, 0x1a,
		0xe4, 0xe7, 0x8a, 0x65, 0x6f, 0x8d, 0xd7, 0xe1,
		0x10, 0xab, 0x78, 0x5b, 0xb9, 0x69, 0x1f, 0xe0,
		0x5c, 0xf1, 0x19, 0x12, 0x21, 0xc7, 0x51, 0xbc,
		0x61, 0x5f, 0xc0, 0x36, 0x17, 0xc0, 0x28, 0xd9,
		0x51, 0xcb, 0x43, 0xd9, 0xfa, 0xd1, 0xad, 0x79,
		0x69, 0x86, 0x49, 0xc5, 0xe5, 0x69, 0x27, 0xce,
		0x22, 0xd0, 0xe1, 0x6a, 0xf9, 0x02, 0xca, 0x6c,
		0x34, 0xc7, 0xb8, 0x02, 0xc1, 0x38, 0x7f, 0xd5,
		0x15, 0xf5, 0xd6, 0xeb, 0xf9, 0x30, 0x40, 0x43,
		0xea, 0x87, 0xde, 0x35, 0xf6, 0x83, 0x59, 0x09,
		0x68, 0x62, 0x00, 0x87, 0xb8, 0xe7, 0xca, 0x05,
		0x0f, 0xac, 0x42, 0x58, 0x45, 0xaa, 0xc9, 0x9b,
		0xfd, 0x2a, 0xda, 0x65, 0x33, 0x93, 0x9d, 0xc6,
		0x93, 0x8d, 0xe2, 0xc5, 0x71, 0xc1, 0x5c, 0x13,
		0xde, 0x7b, 0xd4, 0xb9, 0x4c, 0x35, 0x61, 0x85,
		0x90, 0x78, 0xf7, 0x81, 0x98, 0x45, 0x99, 0x24,
		0x58, 0x73, 0x28, 0xf8, 0x31, 0xab, 0x54, 0x2e,
		0xc0, 0x38, 0x77, 0x25, 0x5c, 0x06, 0x9c, 0xc3,
		0x69, 0x21, 0x92, 0x76, 0xe1, 0x16, 0xdc, 0xa9,
		0xee, 0xb6, 0x80, 0x66, 0x43, 0x11, 0x24, 0xb3,
		0x07, 0x17, 0x89, 0x0f, 0xcb, 0xe0, 0x60, 0xa8,
		0x9d, 0x06, 0x4b, 0x6e, 0x72, 0xb7, 0xbc, 0x4f,
		0xb8, 0xc0, 0x80, 0xa2, 0xfb, 0x46, 0x5b, 0x8f,
		0x11, 0x01, 0x92, 0x9d, 0x37, 0x09, 0x98, 0xc8,
		0x0a, 0x46, 0xae, 0x12, 0xac, 0x61, 0x3f, 0xe7,
		0x41, 0x1a, 0xaa, 0x2e, 0xdc, 0xd7, 0x2a, 0x47,
		0xee, 0xdf, 0x08, 0xd1, 0xff, 0xea, 0x13, 0xc6,
		0x05, 0xdb, 0x29, 0xcc, 0x03, 0xba, 0x7b, 0x6d,
		0x40, 0xc1, 0xc9, 0x76, 0x75, 0x03, 0x7a, 0x71,
		0xc9, 0x5f, 0xd9, 0xe0, 0x61, 0x69, 0x36, 0x8f,
		0xb2, 0xbc, 0x28, 0xf3, 0x90, 0x71, 0xda, 0x5f,
		0x08, 0xd5, 0x0d, 0xc1, 0xe6, 0xbd, 0x2b, 0xc6,
		0x6c, 0x42, 0xfd, 0xbf, 0x10, 0xe8, 0x5f, 0x87,
		0x3d, 0x21, 0x42, 0x85, 0x01, 0x0a, 0xbf, 0x8e,
		0x49, 0xd3, 0x9c, 0x89, 0x3b, 0xea, 0xe1, 0xbf,
		0xe9, 0x9b, 0x5e, 0x0e, 0xb8, 0xeb, 0xcd, 0x3a,
		0xf6, 0x29, 0x41, 0x35, 0xdd, 0x9b, 0x13, 0x24,
		0xe0, 0x1d, 0x8a, 0xcb, 0x20, 0xf8, 0x41, 0x51,
		0x3e, 0x23, 0x8c, 0x67, 0x98, 0x39, 0x53, 0x77,
		0x2a, 0x68, 0xf4, 0x3c, 0x7e, 0xd6, 0xc4, 0x6e,
		0xf1, 0x53, 0xe9, 0xd8, 0x5c, 0xc1, 0xa9, 0x38,
		0x6f, 0x5e, 0xe4, 0xd4, 0x29, 0x1c, 0x6c, 0xee,
		0x2f, 0xea, 0xde, 0x61, 0x71, 0x5a, 0xea, 0xce,
		0x23, 0x6e, 0x1b, 0x16, 0x43, 0xb7, 0xc0, 0xe3,
		0x87, 0xa1, 0x95, 0x1e, 0x97, 0x4d, 0xea, 0xa6,
		0xf7, 0x25, 0xac, 0x82, 0x2a, 0xd3, 0xa6, 0x99,
		0x75, 0xdd, 0xc1, 0x55, 0x32, 0x6b, 0xea, 0x33,
		0x88, 0xce, 0x06, 0xac, 0x15, 0x39, 0x19, 0xa3,
		0x59, 0xaf, 0x7a, 0x1f, 0xd9, 0x72, 0x5e, 0xf7,
		0x4c, 0xf3, 0x5d, 0x6b, 0xf2, 0x16, 0x92, 0xa8,
		0x9e, 0x3d, 0xd4, 0x4c, 0x72, 0x55, 0x4e, 0x4a,
		0xf7, 0x8b, 0x2f, 0x67, 0x5a, 0x90, 0xb7, 0xcf,
		0x16, 0xd3, 0x7b, 0x5a, 0x9a, 0xc8, 0x9f, 0xbf,
		0x01, 0x76, 0x3b, 0x86, 0x2c, 0x2a, 0x78, 0x10,
		0x70, 0x05, 0x38, 0xf9, 0xdd, 0x2a, 0x1d, 0x00,
		0x25, 0xb7, 0x10, 0xac, 0x3b, 0x3c, 0x4d, 0x3c,
		0x01, 0x68, 0x3c, 0x5a, 0x29, 0xc2, 0xa0, 0x1b,
		0x95, 0x67, 0xf9, 0x0a, 0x60, 0xb7, 0x11, 0x9c,
		0x40, 0x45, 0xd7, 0xb0, 0xda, 0x49, 0x87, 0xcd,
		0xb0, 0x9b, 0x61, 0x8c, 0xf4, 0x0d, 0x94, 0x1d,
		0x79, 0x66, 0x13, 0x0b, 0xc6, 0x6b, 0x19, 0xee,
		0xa0, 0x6b, 0x64, 0x7d, 0xc4, 0xff, 0x98, 0x72,
		0x60, 0xab, 0x7f, 0x0f, 0x4d, 0x5d, 0x6b, 0xc3,
		0xba, 0x5e, 0x0d, 0x04, 0xd9, 0x59, 0x17, 0xd0,
		0x64, 0xbe, 0xfb, 0x58, 0xfc, 0xed, 0x18, 0xf6,
		0xac, 0x19, 0xa4, 0xfd, 0x16, 0x59, 0x80, 0x58,
		0xb8, 0x0f, 0x79, 0x24, 0x60, 0x18, 0x62, 0xa9,
		0xa3, 0xa0, 0xe8, 0x81, 0xd6, 0xec, 0x5b, 0xfe,
		0x5b, 0xb8, 0xa4, 0x00, 0xa9, 0xd0, 0x90, 0x17,
		0xe5, 0x50, 0x3d, 0x2b, 0x12, 0x6e, 0x2a, 0x13,
		0x65, 0x7c, 0xdf, 0xdf, 0xa7, 0xdd, 0x9f, 0x78,
		0x5f, 0x8f, 0x4e, 0x90, 0xa6, 0x10, 0xe4, 0x7b,
		0x68, 0x6b, 0xfd, 0xa9, 0x6d, 0x47, 0xfa, 0xec,
		0x42, 0x35, 0x07, 0x12, 0x3e, 0x78, 0x23, 0x15,
		0xff, 0xe2, 0x65, 0xc7, 0x47, 0x89, 0x2f, 0x97,
		0x7c, 0xd7, 0x6b, 0x69, 0x35, 0x79, 0x6f, 0x85,
		0xb4, 0xa9, 0x75, 0x04, 0x32, 0x9a, 0xfe, 0xf0,
		0xce, 0xe3, 0xf1, 0xab, 0x15, 0x47, 0xe4, 0x9c,
		0xc1, 0x48, 0x32, 0x3c, 0xbe, 0x44, 0x72, 0xc9,
		0xaa, 0x50, 0x37, 0xa6, 0xbe, 0x41, 0xcf, 0xe8,
		0x17, 0x4e, 0x37, 0xbe, 0xf1, 0x34, 0x2c, 0xd9,
		0x60, 0x48, 0x09, 0xa5, 0x26, 0x00, 0x31, 0x77,
		0x4e, 0xac, 0x7c, 0x89, 0x75, 0xe3, 0xde, 0x26,
		0x4c, 0x32, 0x54, 0x27, 0x8e, 0x92, 0x26, 0x42,
		0x85, 0x76, 0x01, 0x76, 0x62, 0x4c, 0x29, 0xe9,
		0x38, 0x05, 0x51, 0x54, 0x97, 0xa3, 0x03, 0x59,
		0x5e, 0xec, 0x0c, 0xe4, 0x96, 0xb7, 0x15, 0xa8,
		0x41, 0x06, 0x2b, 0x78, 0x95, 0x24, 0xf6, 0x32,
		0xc5, 0xec, 0xd7, 0x89, 0x28, 0x1e, 0xec, 0xb1,
		0xc7, 0x21, 0x0c, 0xd3, 0x80, 0x7c, 0x5a, 0xe6,
		0xb1, 0x3a, 0x52, 0x33, 0x84, 0x4e, 0x32, 0x6e,
		0x7a, 0xf6, 0x43, 0x15, 0x5b, 0xa6, 0xba, 0xeb,
		0xa8, 0xe4, 0xff, 0x4f, 0xbd, 0xbd, 0xa8, 0x5e,
		0xbe, 0x27, 0xaf, 0xc5, 0xf7, 0x9e, 0xdf, 0x48,
		0x22, 0xca, 0x6a, 0x0b, 0x3c, 0xd7, 0xe0, 0xdc,
		0xf3, 0x71, 0x08, 0xdc, 0x28, 0x13, 0x08, 0xf2,
		0x08, 0x1d, 0x9d, 0x7b, 0xd9, 0xde, 0x6f, 0xe6,
		0xe8, 0x88, 0x18, 0xc2, 0xcd, 0x93, 0xc5, 0x38,
		0x21, 0x68, 0x4c, 0x9a, 0xfb, 0xb6, 0x18, 0x16,
		0x73, 0x2c, 0x1d, 0x6f, 0x95, 0xfb, 0x65, 0x4f,
		0x7c, 0xec, 0x8d, 0x6c, 0xa8, 0xc0, 0x55, 0x28,
		0xc6, 0xc3, 0xea, 0xeb, 0x05, 0xf5, 0x65, 0xeb,
		0x53, 0xe1, 0x54, 0xef, 0xb8, 0x64, 0x98, 0x2d,
		0x98, 0x9e, 0xc8, 0xfe, 0xa2, 0x07, 0x30, 0xf7,
		0xf7, 0xae, 0xdb, 0x32, 0xf8, 0x71, 0x9d, 0x06,
		0xdf, 0x9b, 0xda, 0x61, 0x7d, 0xdb, 0xae, 0x06,
		0x24, 0x63, 0x74, 0xb6, 0xf3, 0x1b, 0x66, 0x09,
		0x60, 0xff, 0x2b, 0x29, 0xf5, 0xa9, 0x9d, 0x61,
		0x5d, 0x55, 0x10, 0x82, 0x21, 0xbb, 0x64, 0x0d,
		0xef, 0x5c, 0xe3, 0x30, 0x1b, 0x60, 0x1e, 0x5b,
		0xfe, 0x6c, 0xf5, 0x15, 0xa3, 0x86, 0x27, 0x58,
		0x46, 0x00, 0x20, 0xcb, 0x86, 0x9a, 0x52, 0x29,
		0x20, 0x68, 0x4d, 0x67, 0x88, 0x70, 0xc2, 0x31,
		0xd8, 0xbb, 0xa5, 0xa7, 0x88, 0x7f, 0x66, 0xbc,
		0xaa, 0x0f, 0xe1, 0x78, 0x7b, 0x97, 0x3c, 0xb7,
		0xd7, 0xd8, 0x04, 0xe0, 0x09, 0x60, 0xc8, 0xd0,
		0x9e, 0xe5, 0x6b, 0x31, 0x7f, 0x88, 0xfe, 0xc3,
		0xfd, 0x89, 0xec, 0x76, 0x4b, 0xb3, 0xa7, 0x37,
		0x03, 0xb7, 0xc6, 0x10, 0x7c, 0x9d, 0x0c, 0x75,
		0xd3, 0x08, 0x14, 0x94, 0x03, 0x42, 0x25, 0x26,
		0x85, 0xf7, 0xf0, 0x90, 0x06, 0x3e, 0x6f, 0x60,
		0x52, 0x55, 0xd5, 0x0f, 0x79, 0x64, 0x69, 0x69,
		0x46, 0xf9, 0x7f, 0x7f, 0x03, 0xf1, 0x1f, 0xdb,
		0x39, 0x05, 0xba, 0x4a, 0x8f, 0x17, 0xe7, 0xba,
		0xe2, 0x07, 0x7c, 0x1d, 0x9e, 0xbc, 0x94, 0xc0,
		0x61, 0x59, 0x8e, 0x72, 0xaf, 0xfc, 0x99, 0xe4,
		0xd5, 0xa8, 0xee, 0x0a, 0x48, 0x2d, 0x82, 0x8b,
		0x34, 0x54, 0x8a, 0xce, 0xc7, 0xfa, 0xdd, 0xba,
		0x54, 0xdf, 0xb3, 0x30, 0x33, 0x73, 0x2e, 0xd5,
		0x52, 0xab, 0x49, 0x91, 0x4e, 0x0a, 0xd6, 0x2f,
		0x67, 0xe4, 0xdd, 0x64, 0x48, 0x16, 0xd9, 0x85,
		0xaa, 0x52, 0xa5, 0x0b, 0xd3, 0xb4, 0x2d, 0x77,
		0x5e, 0x52, 0x77, 0x17, 0xcf, 0xbe, 0x88, 0x04,
		0x01, 0x52, 0xe2, 0xf1, 0x46, 0xe2, 0x91, 0x30,
		0x65, 0xcf, 0xc0, 0x65, 0x45, 0xc3, 0x7e, 0xf4,
		0x2e, 0xb5, 0xaf, 0x6f, 0xab, 0x1a, 0xfa, 0x70,
		0x35, 0xb8, 0x4f, 0x2d, 0x78, 0x90, 0x33, 0xb5,
		0x9a, 0x67, 0xdb, 0x2f, 0x28, 0x32, 0xb6, 0x54,
		0xab, 0x4c, 0x6b, 0x85, 0xed, 0x6c, 0x3e, 0x05,
		0x2a, 0xc7, 0x32, 0xe8, 0xf5, 0xa3, 0x7b, 0x4e,
		0x7b, 0x58, 0x24, 0x73, 0xf7, 0xfd, 0xc7, 0xc8,
		0x6c, 0x71, 0x68, 0xb1, 0xf6, 0xc5, 0x9e, 0x1e,
		0xe3, 0x5c, 0x25, 0xc0, 0x5b, 0x3e, 0x59, 0xa1,
		0x18, 0x5a, 0xe8, 0xb5, 0xd1, 0x44, 0x13, 0xa3,
		0xe6, 0x05, 0x76, 0xd2, 0x8d, 0x6e, 0x54, 0x68,
		0x0c, 0xa4, 0x7b, 0x8b, 0xd3, 0x8c, 0x42, 0x13,
		0x87, 0xda, 0xdf, 0x8f, 0xa5, 0x83, 0x7a, 0x42,
		0x99, 0xb7, 0xeb, 0xe2, 0x79, 0xe0, 0xdb, 0xda,
		0x33, 0xa8, 0x50, 0x3a, 0xd7, 0xe7, 0xd3, 0x61,
		0x18, 0xb8, 0xaa, 0x2d, 0xc8, 0xd8, 0x2c, 0x28,
		0xe5, 0x97, 0x0a, 0x7c, 0x6c, 0x7f, 0x09, 0xd7,
		0x88, 0x80, 0xac, 0x12, 0xed, 0xf8, 0xc6, 0xb5,
		0x2d, 0xd6, 0x63, 0x9b, 0x98, 0x35, 0x26, 0xde,
		0xf6, 0x31, 0xee, 0x7e, 0xa0, 0xfb, 0x16, 0x98,
		0xb1, 0x96, 0x1d, 0xee, 0xe3, 0x2f, 0xfb, 0x41,
		0xdd, 0xea, 0x10, 0x1e, 0x03, 0x89, 0x18, 0xd2,
		0x47, 0x0c, 0xa0, 0x57, 0xda, 0x76, 0x3a, 0x37,
		0x2c, 0xe4, 0xf9, 0x77, 0xc8, 0x43, 0x5f, 0xcb,
		0xd6, 0x85, 0xf7, 0x22, 0xe4, 0x32, 0x25, 0xa8,
		0xdc, 0x21, 0xc0, 0xf5, 0x95, 0xb2, 0xf8, 0x83,
		0xf0, 0x65, 0x61, 0x15, 0x48, 0x94, 0xb7, 0x03,
		0x7f, 0x66, 0xa1, 0x39, 0x1f, 0xdd, 0xce, 0x96,
		0xfe, 0x58, 0x81, 0x3d, 0x41, 0x11, 0x87, 0x13,
		0x26, 0x1b, 0x6d, 0xf3, 0xca, 0x2e, 0x2c, 0x76,
		0xd3, 0x2f, 0x6d, 0x49, 0x70, 0x53, 0x05, 0x96,
		0xcc, 0x30, 0x2b, 0x83, 0xf2, 0xc6, 0xb2, 0x4b,
		0x22, 0x13, 0x95, 0x42, 0xeb, 0x56, 0x4d, 0x22,
		0xe6, 0x43, 0x6f, 0xba, 0xe7, 0x3b, 0xe5, 0x59,
		0xce, 0x57, 0x88, 0x85, 0xb6, 0xbf, 0x15, 0x37,
		0xb3, 0x7a, 0x7e, 0xc4, 0xbc, 0x99, 0xfc, 0xe4,
		0x89, 0x00, 0x68, 0x39, 0xbc, 0x5a, 0xba, 0xab,
		0x52, 0xab, 0xe6, 0x81, 0xfd, 0x93, 0x62, 0xe9,
		0xb7, 0x12, 0xd1, 0x18, 0x1a, 0xb9, 0x55, 0x4a,
		0x0f, 0xae, 0x35, 0x11, 0x04, 0x27, 0xf3, 0x42,
		0x4e, 0xca, 0xdf, 0x9f, 0x12, 0x62, 0xea, 0x03,
		0xc0, 0xa9, 0x22, 0x7b, 0x6c, 0x6c, 0xe3, 0xdf,
		0x16, 0xad, 0x03, 0xc9, 0xfe, 0xa4, 0xdd, 0x4f
};

static const uint8_t AES_CBC_ciphertext_1792B[] = {
		0x59, 0xcc, 0xfe, 0x8f, 0xb4, 0x9d, 0x0e, 0xd1,
		0x85, 0xfc, 0x9b, 0x43, 0xc1, 0xb7, 0x54, 0x67,
		0x01, 0xef, 0xb8, 0x71, 0x36, 0xdb, 0x50, 0x48,
		0x7a, 0xea, 0xcf, 0xce, 0xba, 0x30, 0x10, 0x2e,
		0x96, 0x2b, 0xfd, 0xcf, 0x00, 0xe3, 0x1f, 0xac,
		0x66, 0x14, 0x30, 0x86, 0x49, 0xdb, 0x01, 0x8b,
		0x07, 0xdd, 0x00, 0x9d, 0x0d, 0x5c, 0x19, 0x11,
		0xe8, 0x44, 0x2b, 0x25, 0x70, 0xed, 0x7c, 0x33,
		0x0d, 0xe3, 0x34, 0x93, 0x63, 0xad, 0x26, 0xb1,
		0x11, 0x91, 0x34, 0x2e, 0x1d, 0x50, 0xaa, 0xd4,
		0xef, 0x3a, 0x6d, 0xd7, 0x33, 0x20, 0x0d, 0x3f,
		0x9b, 0xdd, 0xc3, 0xa5, 0xc5, 0xf1, 0x99, 0xdc,
		0xea, 0x52, 0xda, 0x55, 0xea, 0xa2, 0x7a, 0xc5,
		0x78, 0x44, 0x4a, 0x02, 0x33, 0x19, 0x62, 0x37,
		0xf8, 0x8b, 0xd1, 0x0c, 0x21, 0xdf, 0x40, 0x19,
		0x81, 0xea, 0xfb, 0x1c, 0xa7, 0xcc, 0x60, 0xfe,
		0x63, 0x25, 0x8f, 0xf3, 0x73, 0x0f, 0x45, 0xe6,
		0x6a, 0x18, 0xbf, 0xbe, 0xad, 0x92, 0x2a, 0x1e,
		0x15, 0x65, 0x6f, 0xef, 0x92, 0xcd, 0x0e, 0x19,
		0x3d, 0x42, 0xa8, 0xfc, 0x0d, 0x32, 0x58, 0xe0,
		0x56, 0x9f, 0xd6, 0x9b, 0x8b, 0xec, 0xe0, 0x45,
		0x4d, 0x7e, 0x73, 0x87, 0xff, 0x74, 0x92, 0x59,
		0x60, 0x13, 0x93, 0xda, 0xec, 0xbf, 0xfa, 0x20,
		0xb6, 0xe7, 0xdf, 0xc7, 0x10, 0xf5, 0x79, 0xb4,
		0xd7, 0xac, 0xaf, 0x2b, 0x37, 0x52, 0x30, 0x1d,
		0xbe, 0x0f, 0x60, 0x77, 0x3d, 0x03, 0x63, 0xa9,
		0xae, 0xb1, 0xf3, 0xca, 0xca, 0xb4, 0x21, 0xd7,
		0x6f, 0x2e, 0x5e, 0x9b, 0x68, 0x53, 0x80, 0xab,
		0x30, 0x23, 0x0a, 0x72, 0x6b, 0xb1, 0xd8, 0x25,
		0x5d, 0x3a, 0x62, 0x9b, 0x4f, 0x59, 0x3b, 0x79,
		0xa8, 0x9e, 0x08, 0x6d, 0x37, 0xb0, 0xfc, 0x42,
		0x51, 0x25, 0x86, 0xbd, 0x54, 0x5a, 0x95, 0x20,
		0x6c, 0xac, 0xb9, 0x30, 0x1c, 0x03, 0xc9, 0x49,
		0x38, 0x55, 0x31, 0x49, 0xed, 0xa9, 0x0e, 0xc3,
		0x65, 0xb4, 0x68, 0x6b, 0x07, 0x4c, 0x0a, 0xf9,
		0x21, 0x69, 0x7c, 0x9f, 0x28, 0x80, 0xe9, 0x49,
		0x22, 0x7c, 0xec, 0x97, 0xf7, 0x70, 0xb4, 0xb8,
		0x25, 0xe7, 0x80, 0x2c, 0x43, 0x24, 0x8a, 0x2e,
		0xac, 0xa2, 0x84, 0x20, 0xe7, 0xf4, 0x6b, 0x86,
		0x37, 0x05, 0xc7, 0x59, 0x04, 0x49, 0x2a, 0x99,
		0x80, 0x46, 0x32, 0x19, 0xe6, 0x30, 0xce, 0xc0,
		0xef, 0x6e, 0xec, 0xe5, 0x2f, 0x24, 0xc1, 0x78,
		0x45, 0x02, 0xd3, 0x64, 0x99, 0xf5, 0xc7, 0xbc,
		0x8f, 0x8c, 0x75, 0xb1, 0x0a, 0xc8, 0xc3, 0xbd,
		0x5e, 0x7e, 0xbd, 0x0e, 0xdf, 0x4b, 0x96, 0x6a,
		0xfd, 0x03, 0xdb, 0xd1, 0x31, 0x1e, 0x27, 0xf9,
		0xe5, 0x83, 0x9a, 0xfc, 0x13, 0x4c, 0xd3, 0x04,
		0xdb, 0xdb, 0x3f, 0x35, 0x93, 0x4e, 0x14, 0x6b,
		0x00, 0x5c, 0xb6, 0x11, 0x50, 0xee, 0x61, 0x5c,
		0x10, 0x5c, 0xd0, 0x90, 0x02, 0x2e, 0x12, 0xe0,
		0x50, 0x44, 0xad, 0x75, 0xcd, 0x94, 0xcf, 0x92,
		0xcb, 0xe3, 0xe8, 0x77, 0x4b, 0xd7, 0x1a, 0x7c,
		0xdd, 0x6b, 0x49, 0x21, 0x7c, 0xe8, 0x2c, 0x25,
		0x49, 0x86, 0x1e, 0x54, 0xae, 0xfc, 0x0e, 0x80,
		0xb1, 0xd5, 0xa5, 0x23, 0xcf, 0xcc, 0x0e, 0x11,
		0xe2, 0x7c, 0x3c, 0x25, 0x78, 0x64, 0x03, 0xa1,
		0xdd, 0x9f, 0x74, 0x12, 0x7b, 0x21, 0xb5, 0x73,
		0x15, 0x3c, 0xed, 0xad, 0x07, 0x62, 0x21, 0x79,
		0xd4, 0x2f, 0x0d, 0x72, 0xe9, 0x7c, 0x6b, 0x96,
		0x6e, 0xe5, 0x36, 0x4a, 0xd2, 0x38, 0xe1, 0xff,
		0x6e, 0x26, 0xa4, 0xac, 0x83, 0x07, 0xe6, 0x67,
		0x74, 0x6c, 0xec, 0x8b, 0x4b, 0x79, 0x33, 0x50,
		0x2f, 0x8f, 0xa0, 0x8f, 0xfa, 0x38, 0x6a, 0xa2,
		0x3a, 0x42, 0x85, 0x15, 0x90, 0xd0, 0xb3, 0x0d,
		0x8a, 0xe4, 0x60, 0x03, 0xef, 0xf9, 0x65, 0x8a,
		0x4e, 0x50, 0x8c, 0x65, 0xba, 0x61, 0x16, 0xc3,
		0x93, 0xb7, 0x75, 0x21, 0x98, 0x25, 0x60, 0x6e,
		0x3d, 0x68, 0xba, 0x7c, 0xe4, 0xf3, 0xd9, 0x9b,
		0xfb, 0x7a, 0xed, 0x1f, 0xb3, 0x4b, 0x88, 0x74,
		0x2c, 0xb8, 0x8c, 0x22, 0x95, 0xce, 0x90, 0xf1,
		0xdb, 0x80, 0xa6, 0x39, 0xae, 0x82, 0xa1, 0xef,
		0x75, 0xec, 0xfe, 0xf1, 0xe8, 0x04, 0xfd, 0x99,
		0x1b, 0x5f, 0x45, 0x87, 0x4f, 0xfa, 0xa2, 0x3e,
		0x3e, 0xb5, 0x01, 0x4b, 0x46, 0xeb, 0x13, 0x9a,
		0xe4, 0x7d, 0x03, 0x87, 0xb1, 0x59, 0x91, 0x8e,
		0x37, 0xd3, 0x16, 0xce, 0xef, 0x4b, 0xe9, 0x46,
		0x8d, 0x2a, 0x50, 0x2f, 0x41, 0xd3, 0x7b, 0xcf,
		0xf0, 0xb7, 0x8b, 0x65, 0x0f, 0xa3, 0x27, 0x10,
		0xe9, 0xa9, 0xe9, 0x2c, 0xbe, 0xbb, 0x82, 0xe3,
		0x7b, 0x0b, 0x81, 0x3e, 0xa4, 0x6a, 0x4f, 0x3b,
		0xd5, 0x61, 0xf8, 0x47, 0x04, 0x99, 0x5b, 0xff,
		0xf3, 0x14, 0x6e, 0x57, 0x5b, 0xbf, 0x1b, 0xb4,
		0x3f, 0xf9, 0x31, 0xf6, 0x95, 0xd5, 0x10, 0xa9,
		0x72, 0x28, 0x23, 0xa9, 0x6a, 0xa2, 0xcf, 0x7d,
		0xe3, 0x18, 0x95, 0xda, 0xbc, 0x6f, 0xe9, 0xd8,
		0xef, 0x49, 0x3f, 0xd3, 0xef, 0x1f, 0xe1, 0x50,
		0xe8, 0x8a, 0xc0, 0xce, 0xcc, 0xb7, 0x5e, 0x0e,
		0x8b, 0x95, 0x80, 0xfd, 0x58, 0x2a, 0x9b, 0xc8,
		0xb4, 0x17, 0x04, 0x46, 0x74, 0xd4, 0x68, 0x91,
		0x33, 0xc8, 0x31, 0x15, 0x84, 0x16, 0x35, 0x03,
		0x64, 0x6d, 0xa9, 0x4e, 0x20, 0xeb, 0xa9, 0x3f,
		0x21, 0x5e, 0x9b, 0x09, 0xc3, 0x45, 0xf8, 0x7c,
		0x59, 0x62, 0x29, 0x9a, 0x5c, 0xcf, 0xb4, 0x27,
		0x5e, 0x13, 0xea, 0xb3, 0xef, 0xd9, 0x01, 0x2a,
		0x65, 0x5f, 0x14, 0xf4, 0xbf, 0x28, 0x89, 0x3d,
		0xdd, 0x9d, 0x52, 0xbd, 0x9e, 0x5b, 0x3b, 0xd2,
		0xc2, 0x81, 0x35, 0xb6, 0xac, 0xdd, 0x27, 0xc3,
		0x7b, 0x01, 0x5a, 0x6d, 0x4c, 0x5e, 0x2c, 0x30,
		0xcb, 0x3a, 0xfa, 0xc1, 0xd7, 0x31, 0x67, 0x3e,
		0x08, 0x6a, 0xe8, 0x8c, 0x75, 0xac, 0x1a, 0x6a,
		0x52, 0xf7, 0x51, 0xcd, 0x85, 0x3f, 0x3c, 0xa7,
		0xea, 0xbc, 0xd7, 0x18, 0x9e, 0x27, 0x73, 0xe6,
		0x2b, 0x58, 0xb6, 0xd2, 0x29, 0x68, 0xd5, 0x8f,
		0x00, 0x4d, 0x55, 0xf6, 0x61, 0x5a, 0xcc, 0x51,
		0xa6, 0x5e, 0x85, 0xcb, 0x0b, 0xfd, 0x06, 0xca,
		0xf5, 0xbf, 0x0d, 0x13, 0x74, 0x78, 0x6d, 0x9e,
		0x20, 0x11, 0x84, 0x3e, 0x78, 0x17, 0x04, 0x4f,
		0x64, 0x2c, 0x3b, 0x3e, 0x93, 0x7b, 0x58, 0x33,
		0x07, 0x52, 0xf7, 0x60, 0x6a, 0xa8, 0x3b, 0x19,
		0x27, 0x7a, 0x93, 0xc5, 0x53, 0xad, 0xec, 0xf6,
		0xc8, 0x94, 0xee, 0x92, 0xea, 0xee, 0x7e, 0xea,
		0xb9, 0x5f, 0xac, 0x59, 0x5d, 0x2e, 0x78, 0x53,
		0x72, 0x81, 0x92, 0xdd, 0x1c, 0x63, 0xbe, 0x02,
		0xeb, 0xa8, 0x1b, 0x2a, 0x6e, 0x72, 0xe3, 0x2d,
		0x84, 0x0d, 0x8a, 0x22, 0xf6, 0xba, 0xab, 0x04,
		0x8e, 0x04, 0x24, 0xdb, 0xcc, 0xe2, 0x69, 0xeb,
		0x4e, 0xfa, 0x6b, 0x5b, 0xc8, 0xc0, 0xd9, 0x25,
		0xcb, 0x40, 0x8d, 0x4b, 0x8e, 0xa0, 0xd4, 0x72,
		0x98, 0x36, 0x46, 0x3b, 0x4f, 0x5f, 0x96, 0x84,
		0x03, 0x28, 0x86, 0x4d, 0xa1, 0x8a, 0xd7, 0xb2,
		0x5b, 0x27, 0x01, 0x80, 0x62, 0x49, 0x56, 0xb9,
		0xa0, 0xa1, 0xe3, 0x6e, 0x22, 0x2a, 0x5d, 0x03,
		0x86, 0x40, 0x36, 0x22, 0x5e, 0xd2, 0xe5, 0xc0,
		0x6b, 0xfa, 0xac, 0x80, 0x4e, 0x09, 0x99, 0xbc,
		0x2f, 0x9b, 0xcc, 0xf3, 0x4e, 0xf7, 0x99, 0x98,
		0x11, 0x6e, 0x6f, 0x62, 0x22, 0x6b, 0x92, 0x95,
		0x3b, 0xc3, 0xd2, 0x8e, 0x0f, 0x07, 0xc2, 0x51,
		0x5c, 0x4d, 0xb2, 0x6e, 0xc0, 0x27, 0x73, 0xcd,
		0x57, 0xb7, 0xf0, 0xe9, 0x2e, 0xc8, 0xe2, 0x0c,
		0xd1, 0xb5, 0x0f, 0xff, 0xf9, 0xec, 0x38, 0xba,
		0x97, 0xd6, 0x94, 0x9b, 0xd1, 0x79, 0xb6, 0x6a,
		0x01, 0x17, 0xe4, 0x7e, 0xa6, 0xd5, 0x86, 0x19,
		0xae, 0xf3, 0xf0, 0x62, 0x73, 0xc0, 0xf0, 0x0a,
		0x7a, 0x96, 0x93, 0x72, 0x89, 0x7e, 0x25, 0x57,
		0xf8, 0xf7, 0xd5, 0x1e, 0xe5, 0xac, 0xd6, 0x38,
		0x4f, 0xe8, 0x81, 0xd1, 0x53, 0x41, 0x07, 0x2d,
		0x58, 0x34, 0x1c, 0xef, 0x74, 0x2e, 0x61, 0xca,
		0xd3, 0xeb, 0xd6, 0x93, 0x0a, 0xf2, 0xf2, 0x86,
		0x9c, 0xe3, 0x7a, 0x52, 0xf5, 0x42, 0xf1, 0x8b,
		0x10, 0xf2, 0x25, 0x68, 0x7e, 0x61, 0xb1, 0x19,
		0xcf, 0x8f, 0x5a, 0x53, 0xb7, 0x68, 0x4f, 0x1a,
		0x71, 0xe9, 0x83, 0x91, 0x3a, 0x78, 0x0f, 0xf7,
		0xd4, 0x74, 0xf5, 0x06, 0xd2, 0x88, 0xb0, 0x06,
		0xe5, 0xc0, 0xfb, 0xb3, 0x91, 0xad, 0xc0, 0x84,
		0x31, 0xf2, 0x3a, 0xcf, 0x63, 0xe6, 0x4a, 0xd3,
		0x78, 0xbe, 0xde, 0x73, 0x3e, 0x02, 0x8e, 0xb8,
		0x3a, 0xf6, 0x55, 0xa7, 0xf8, 0x5a, 0xb5, 0x0e,
		0x0c, 0xc5, 0xe5, 0x66, 0xd5, 0xd2, 0x18, 0xf3,
		0xef, 0xa5, 0xc9, 0x68, 0x69, 0xe0, 0xcd, 0x00,
		0x33, 0x99, 0x6e, 0xea, 0xcb, 0x06, 0x7a, 0xe1,
		0xe1, 0x19, 0x0b, 0xe7, 0x08, 0xcd, 0x09, 0x1b,
		0x85, 0xec, 0xc4, 0xd4, 0x75, 0xf0, 0xd6, 0xfb,
		0x84, 0x95, 0x07, 0x44, 0xca, 0xa5, 0x2a, 0x6c,
		0xc2, 0x00, 0x58, 0x08, 0x87, 0x9e, 0x0a, 0xd4,
		0x06, 0xe2, 0x91, 0x5f, 0xb7, 0x1b, 0x11, 0xfa,
		0x85, 0xfc, 0x7c, 0xf2, 0x0f, 0x6e, 0x3c, 0x8a,
		0xe1, 0x0f, 0xa0, 0x33, 0x84, 0xce, 0x81, 0x4d,
		0x32, 0x4d, 0xeb, 0x41, 0xcf, 0x5a, 0x05, 0x60,
		0x47, 0x6c, 0x2a, 0xc4, 0x17, 0xd5, 0x16, 0x3a,
		0xe4, 0xe7, 0xab, 0x84, 0x94, 0x22, 0xff, 0x56,
		0xb0, 0x0c, 0x92, 0x6c, 0x19, 0x11, 0x4c, 0xb3,
		0xed, 0x58, 0x48, 0x84, 0x2a, 0xe2, 0x19, 0x2a,
		0xe1, 0xc0, 0x56, 0x82, 0x3c, 0x83, 0xb4, 0x58,
		0x2d, 0xf0, 0xb5, 0x1e, 0x76, 0x85, 0x51, 0xc2,
		0xe4, 0x95, 0x27, 0x96, 0xd1, 0x90, 0xc3, 0x17,
		0x75, 0xa1, 0xbb, 0x46, 0x5f, 0xa6, 0xf2, 0xef,
		0x71, 0x56, 0x92, 0xc5, 0x8a, 0x85, 0x52, 0xe4,
		0x63, 0x21, 0x6f, 0x55, 0x85, 0x2b, 0x6b, 0x0d,
		0xc9, 0x92, 0x77, 0x67, 0xe3, 0xff, 0x2a, 0x2b,
		0x90, 0x01, 0x3d, 0x74, 0x63, 0x04, 0x61, 0x3c,
		0x8e, 0xf8, 0xfc, 0x04, 0xdd, 0x21, 0x85, 0x92,
		0x1e, 0x4d, 0x51, 0x8d, 0xb5, 0x6b, 0xf1, 0xda,
		0x96, 0xf5, 0x8e, 0x3c, 0x38, 0x5a, 0xac, 0x9b,
		0xba, 0x0c, 0x84, 0x5d, 0x50, 0x12, 0xc7, 0xc5,
		0x7a, 0xcb, 0xb1, 0xfa, 0x16, 0x93, 0xdf, 0x98,
		0xda, 0x3f, 0x49, 0xa3, 0x94, 0x78, 0x70, 0xc7,
		0x0b, 0xb6, 0x91, 0xa6, 0x16, 0x2e, 0xcf, 0xfd,
		0x51, 0x6a, 0x5b, 0xad, 0x7a, 0xdd, 0xa9, 0x48,
		0x48, 0xac, 0xd6, 0x45, 0xbc, 0x23, 0x31, 0x1d,
		0x86, 0x54, 0x8a, 0x7f, 0x04, 0x97, 0x71, 0x9e,
		0xbc, 0x2e, 0x6b, 0xd9, 0x33, 0xc8, 0x20, 0xc9,
		0xe0, 0x25, 0x86, 0x59, 0x15, 0xcf, 0x63, 0xe5,
		0x99, 0xf1, 0x24, 0xf1, 0xba, 0xc4, 0x15, 0x02,
		0xe2, 0xdb, 0xfe, 0x4a, 0xf8, 0x3b, 0x91, 0x13,
		0x8d, 0x03, 0x81, 0x9f, 0xb3, 0x3f, 0x04, 0x03,
		0x58, 0xc0, 0xef, 0x27, 0x82, 0x14, 0xd2, 0x7f,
		0x93, 0x70, 0xb7, 0xb2, 0x02, 0x21, 0xb3, 0x07,
		0x7f, 0x1c, 0xef, 0x88, 0xee, 0x29, 0x7a, 0x0b,
		0x3d, 0x75, 0x5a, 0x93, 0xfe, 0x7f, 0x14, 0xf7,
		0x4e, 0x4b, 0x7f, 0x21, 0x02, 0xad, 0xf9, 0x43,
		0x29, 0x1a, 0xe8, 0x1b, 0xf5, 0x32, 0xb2, 0x96,
		0xe6, 0xe8, 0x96, 0x20, 0x9b, 0x96, 0x8e, 0x7b,
		0xfe, 0xd8, 0xc9, 0x9c, 0x65, 0x16, 0xd6, 0x68,
		0x95, 0xf8, 0x22, 0xe2, 0xae, 0x84, 0x03, 0xfd,
		0x87, 0xa2, 0x72, 0x79, 0x74, 0x95, 0xfa, 0xe1,
		0xfe, 0xd0, 0x4e, 0x3d, 0x39, 0x2e, 0x67, 0x55,
		0x71, 0x6c, 0x89, 0x33, 0x49, 0x0c, 0x1b, 0x46,
		0x92, 0x31, 0x6f, 0xa6, 0xf0, 0x09, 0xbd, 0x2d,
		0xe2, 0xca, 0xda, 0x18, 0x33, 0xce, 0x67, 0x37,
		0xfd, 0x6f, 0xcb, 0x9d, 0xbd, 0x42, 0xbc, 0xb2,
		0x9c, 0x28, 0xcd, 0x65, 0x3c, 0x61, 0xbc, 0xde,
		0x9d, 0xe1, 0x2a, 0x3e, 0xbf, 0xee, 0x3c, 0xcb,
		0xb1, 0x50, 0xa9, 0x2c, 0xbe, 0xb5, 0x43, 0xd0,
		0xec, 0x29, 0xf9, 0x16, 0x6f, 0x31, 0xd9, 0x9b,
		0x92, 0xb1, 0x32, 0xae, 0x0f, 0xb6, 0x9d, 0x0e,
		0x25, 0x7f, 0x89, 0x1f, 0x1d, 0x01, 0x68, 0xab,
		0x3d, 0xd1, 0x74, 0x5b, 0x4c, 0x38, 0x7f, 0x3d,
		0x33, 0xa5, 0xa2, 0x9f, 0xda, 0x84, 0xa5, 0x82,
		0x2d, 0x16, 0x66, 0x46, 0x08, 0x30, 0x14, 0x48,
		0x5e, 0xca, 0xe3, 0xf4, 0x8c, 0xcb, 0x32, 0xc6,
		0xf1, 0x43, 0x62, 0xc6, 0xef, 0x16, 0xfa, 0x43,
		0xae, 0x9c, 0x53, 0xe3, 0x49, 0x45, 0x80, 0xfd,
		0x1d, 0x8c, 0xa9, 0x6d, 0x77, 0x76, 0xaa, 0x40,
		0xc4, 0x4e, 0x7b, 0x78, 0x6b, 0xe0, 0x1d, 0xce,
		0x56, 0x3d, 0xf0, 0x11, 0xfe, 0x4f, 0x6a, 0x6d,
		0x0f, 0x4f, 0x90, 0x38, 0x92, 0x17, 0xfa, 0x56,
		0x12, 0xa6, 0xa1, 0x0a, 0xea, 0x2f, 0x50, 0xf9,
		0x60, 0x66, 0x6c, 0x7d, 0x5a, 0x08, 0x8e, 0x3c,
		0xf3, 0xf0, 0x33, 0x02, 0x11, 0x02, 0xfe, 0x4c,
		0x56, 0x2b, 0x9f, 0x0c, 0xbd, 0x65, 0x8a, 0x83,
		0xde, 0x7c, 0x05, 0x26, 0x93, 0x19, 0xcc, 0xf3,
		0x71, 0x0e, 0xad, 0x2f, 0xb3, 0xc9, 0x38, 0x50,
		0x64, 0xd5, 0x4c, 0x60, 0x5f, 0x02, 0x13, 0x34,
		0xc9, 0x75, 0xc4, 0x60, 0xab, 0x2e, 0x17, 0x7d
};

static const uint8_t AES_CBC_ciphertext_2048B[] = {
		0x8b, 0x55, 0xbd, 0xfd, 0x2b, 0x35, 0x76, 0x5c,
		0xd1, 0x90, 0xd7, 0x6a, 0x63, 0x1e, 0x39, 0x71,
		0x0d, 0x5c, 0xd8, 0x03, 0x00, 0x75, 0xf1, 0x07,
		0x03, 0x8d, 0x76, 0xeb, 0x3b, 0x00, 0x1e, 0x33,
		0x88, 0xfc, 0x8f, 0x08, 0x4d, 0x33, 0xf1, 0x3c,
		0xee, 0xd0, 0x5d, 0x19, 0x8b, 0x3c, 0x50, 0x86,
		0xfd, 0x8d, 0x58, 0x21, 0xb4, 0xae, 0x0f, 0x81,
		0xe9, 0x9f, 0xc9, 0xc0, 0x90, 0xf7, 0x04, 0x6f,
		0x39, 0x1d, 0x8a, 0x3f, 0x8d, 0x32, 0x23, 0xb5,
		0x1f, 0xcc, 0x8a, 0x12, 0x2d, 0x46, 0x82, 0x5e,
		0x6a, 0x34, 0x8c, 0xb1, 0x93, 0x70, 0x3b, 0xde,
		0x55, 0xaf, 0x16, 0x35, 0x99, 0x84, 0xd5, 0x88,
		0xc9, 0x54, 0xb1, 0xb2, 0xd3, 0xeb, 0x9e, 0x55,
		0x9a, 0xa9, 0xa7, 0xf5, 0xda, 0x29, 0xcf, 0xe1,
		0x98, 0x64, 0x45, 0x77, 0xf2, 0x12, 0x69, 0x8f,
		0x78, 0xd8, 0x82, 0x41, 0xb2, 0x9f, 0xe2, 0x1c,
		0x63, 0x9b, 0x24, 0x81, 0x67, 0x95, 0xa2, 0xff,
		0x26, 0x9d, 0x65, 0x48, 0x61, 0x30, 0x66, 0x41,
		0x68, 0x84, 0xbb, 0x59, 0x14, 0x8e, 0x9a, 0x62,
		0xb6, 0xca, 0xda, 0xbe, 0x7c, 0x41, 0x52, 0x6e,
		0x1b, 0x86, 0xbf, 0x08, 0xeb, 0x37, 0x84, 0x60,
		0xe4, 0xc4, 0x1e, 0xa8, 0x4c, 0x84, 0x60, 0x2f,
		0x70, 0x90, 0xf2, 0x26, 0xe7, 0x65, 0x0c, 0xc4,
		0x58, 0x36, 0x8e, 0x4d, 0xdf, 0xff, 0x9a, 0x39,
		0x93, 0x01, 0xcf, 0x6f, 0x6d, 0xde, 0xef, 0x79,
		0xb0, 0xce, 0xe2, 0x98, 0xdb, 0x85, 0x8d, 0x62,
		0x9d, 0xb9, 0x63, 0xfd, 0xf0, 0x35, 0xb5, 0xa9,
		0x1b, 0xf9, 0xe5, 0xd4, 0x2e, 0x22, 0x2d, 0xcc,
		0x42, 0xbf, 0x0e, 0x51, 0xf7, 0x15, 0x07, 0x32,
		0x75, 0x5b, 0x74, 0xbb, 0x00, 0xef, 0xd4, 0x66,
		0x8b, 0xad, 0x71, 0x53, 0x94, 0xd7, 0x7d, 0x2c,
		0x40, 0x3e, 0x69, 0xa0, 0x4c, 0x86, 0x5e, 0x06,
		0xed, 0xdf, 0x22, 0xe2, 0x24, 0x25, 0x4e, 0x9b,
		0x5f, 0x49, 0x74, 0xba, 0xed, 0xb1, 0xa6, 0xeb,
		0xae, 0x3f, 0xc6, 0x9e, 0x0b, 0x29, 0x28, 0x9a,
		0xb6, 0xb2, 0x74, 0x58, 0xec, 0xa6, 0x4a, 0xed,
		0xe5, 0x10, 0x00, 0x85, 0xe1, 0x63, 0x41, 0x61,
		0x30, 0x7c, 0x97, 0xcf, 0x75, 0xcf, 0xb6, 0xf3,
		0xf7, 0xda, 0x35, 0x3f, 0x85, 0x8c, 0x64, 0xca,
		0xb7, 0xea, 0x7f, 0xe4, 0xa3, 0x4d, 0x30, 0x84,
		0x8c, 0x9c, 0x80, 0x5a, 0x50, 0xa5, 0x64, 0xae,
		0x26, 0xd3, 0xb5, 0x01, 0x73, 0x36, 0x8a, 0x92,
		0x49, 0xc4, 0x1a, 0x94, 0x81, 0x9d, 0xf5, 0x6c,
		0x50, 0xe1, 0x58, 0x0b, 0x75, 0xdd, 0x6b, 0x6a,
		0xca, 0x69, 0xea, 0xc3, 0x33, 0x90, 0x9f, 0x3b,
		0x65, 0x5d, 0x5e, 0xee, 0x31, 0xb7, 0x32, 0xfd,
		0x56, 0x83, 0xb6, 0xfb, 0xa8, 0x04, 0xfc, 0x1e,
		0x11, 0xfb, 0x02, 0x23, 0x53, 0x49, 0x45, 0xb1,
		0x07, 0xfc, 0xba, 0xe7, 0x5f, 0x5d, 0x2d, 0x7f,
		0x9e, 0x46, 0xba, 0xe9, 0xb0, 0xdb, 0x32, 0x04,
		0xa4, 0xa7, 0x98, 0xab, 0x91, 0xcd, 0x02, 0x05,
		0xf5, 0x74, 0x31, 0x98, 0x83, 0x3d, 0x33, 0x11,
		0x0e, 0xe3, 0x8d, 0xa8, 0xc9, 0x0e, 0xf3, 0xb9,
		0x47, 0x67, 0xe9, 0x79, 0x2b, 0x34, 0xcd, 0x9b,
		0x45, 0x75, 0x29, 0xf0, 0xbf, 0xcc, 0xda, 0x3a,
		0x91, 0xb2, 0x15, 0x27, 0x7a, 0xe5, 0xf5, 0x6a,
		0x5e, 0xbe, 0x2c, 0x98, 0xe8, 0x40, 0x96, 0x4f,
		0x8a, 0x09, 0xfd, 0xf6, 0xb2, 0xe7, 0x45, 0xb6,
		0x08, 0xc1, 0x69, 0xe1, 0xb3, 0xc4, 0x24, 0x34,
		0x07, 0x85, 0xd5, 0xa9, 0x78, 0xca, 0xfa, 0x4b,
		0x01, 0x19, 0x4d, 0x95, 0xdc, 0xa5, 0xc1, 0x9c,
		0xec, 0x27, 0x5b, 0xa6, 0x54, 0x25, 0xbd, 0xc8,
		0x0a, 0xb7, 0x11, 0xfb, 0x4e, 0xeb, 0x65, 0x2e,
		0xe1, 0x08, 0x9c, 0x3a, 0x45, 0x44, 0x33, 0xef,
		0x0d, 0xb9, 0xff, 0x3e, 0x68, 0x9c, 0x61, 0x2b,
		0x11, 0xb8, 0x5c, 0x47, 0x0f, 0x94, 0xf2, 0xf8,
		0x0b, 0xbb, 0x99, 0x18, 0x85, 0xa3, 0xba, 0x44,
		0xf3, 0x79, 0xb3, 0x63, 0x2c, 0x1f, 0x2a, 0x35,
		0x3b, 0x23, 0x98, 0xab, 0xf4, 0x16, 0x36, 0xf8,
		0xde, 0x86, 0xa4, 0xd4, 0x75, 0xff, 0x51, 0xf9,
		0xeb, 0x42, 0x5f, 0x55, 0xe2, 0xbe, 0xd1, 0x5b,
		0xb5, 0x38, 0xeb, 0xb4, 0x4d, 0xec, 0xec, 0x99,
		0xe1, 0x39, 0x43, 0xaa, 0x64, 0xf7, 0xc9, 0xd8,
		0xf2, 0x9a, 0x71, 0x43, 0x39, 0x17, 0xe8, 0xa8,
		0xa2, 0xe2, 0xa4, 0x2c, 0x18, 0x11, 0x49, 0xdf,
		0x18, 0xdd, 0x85, 0x6e, 0x65, 0x96, 0xe2, 0xba,
		0xa1, 0x0a, 0x2c, 0xca, 0xdc, 0x5f, 0xe4, 0xf4,
		0x35, 0x03, 0xb2, 0xa9, 0xda, 0xcf, 0xb7, 0x6d,
		0x65, 0x82, 0x82, 0x67, 0x9d, 0x0e, 0xf3, 0xe8,
		0x85, 0x6c, 0x69, 0xb8, 0x4c, 0xa6, 0xc6, 0x2e,
		0x40, 0xb5, 0x54, 0x28, 0x95, 0xe4, 0x57, 0xe0,
		0x5b, 0xf8, 0xde, 0x59, 0xe0, 0xfd, 0x89, 0x48,
		0xac, 0x56, 0x13, 0x54, 0xb9, 0x1b, 0xf5, 0x59,
		0x97, 0xb6, 0xb3, 0xe8, 0xac, 0x2d, 0xfc, 0xd2,
		0xea, 0x57, 0x96, 0x57, 0xa8, 0x26, 0x97, 0x2c,
		0x01, 0x89, 0x56, 0xea, 0xec, 0x8c, 0x53, 0xd5,
		0xd7, 0x9e, 0xc9, 0x98, 0x0b, 0xad, 0x03, 0x75,
		0xa0, 0x6e, 0x98, 0x8b, 0x97, 0x8d, 0x8d, 0x85,
		0x7d, 0x74, 0xa7, 0x2d, 0xde, 0x67, 0x0c, 0xcd,
		0x54, 0xb8, 0x15, 0x7b, 0xeb, 0xf5, 0x84, 0xb9,
		0x78, 0xab, 0xd8, 0x68, 0x91, 0x1f, 0x6a, 0xa6,
		0x28, 0x22, 0xf7, 0x00, 0x49, 0x00, 0xbe, 0x41,
		0x71, 0x0a, 0xf5, 0xe7, 0x9f, 0xb4, 0x11, 0x41,
		0x3f, 0xcd, 0xa9, 0xa9, 0x01, 0x8b, 0x6a, 0xeb,
		0x54, 0x4c, 0x58, 0x92, 0x68, 0x02, 0x0e, 0xe9,
		0xed, 0x65, 0x4c, 0xfb, 0x95, 0x48, 0x58, 0xa2,
		0xaa, 0x57, 0x69, 0x13, 0x82, 0x0c, 0x2c, 0x4b,
		0x5d, 0x4e, 0x18, 0x30, 0xef, 0x1c, 0xb1, 0x9d,
		0x05, 0x05, 0x02, 0x1c, 0x97, 0xc9, 0x48, 0xfe,
		0x5e, 0x7b, 0x77, 0xa3, 0x1f, 0x2a, 0x81, 0x42,
		0xf0, 0x4b, 0x85, 0x12, 0x9c, 0x1f, 0x44, 0xb1,
		0x14, 0x91, 0x92, 0x65, 0x77, 0xb1, 0x87, 0xa2,
		0xfc, 0xa4, 0xe7, 0xd2, 0x9b, 0xf2, 0x17, 0xf0,
		0x30, 0x1c, 0x8d, 0x33, 0xbc, 0x25, 0x28, 0x48,
		0xfd, 0x30, 0x79, 0x0a, 0x99, 0x3e, 0xb4, 0x0f,
		0x1e, 0xa6, 0x68, 0x76, 0x19, 0x76, 0x29, 0xac,
		0x5d, 0xb8, 0x1e, 0x42, 0xd6, 0x85, 0x04, 0xbf,
		0x64, 0x1c, 0x2d, 0x53, 0xe9, 0x92, 0x78, 0xf8,
		0xc3, 0xda, 0x96, 0x92, 0x10, 0x6f, 0x45, 0x85,
		0xaf, 0x5e, 0xcc, 0xa8, 0xc0, 0xc6, 0x2e, 0x73,
		0x51, 0x3f, 0x5e, 0xd7, 0x52, 0x33, 0x71, 0x12,
		0x6d, 0x85, 0xee, 0xea, 0x85, 0xa8, 0x48, 0x2b,
		0x40, 0x64, 0x6d, 0x28, 0x73, 0x16, 0xd7, 0x82,
		0xd9, 0x90, 0xed, 0x1f, 0xa7, 0x5c, 0xb1, 0x5c,
		0x27, 0xb9, 0x67, 0x8b, 0xb4, 0x17, 0x13, 0x83,
		0x5f, 0x09, 0x72, 0x0a, 0xd7, 0xa0, 0xec, 0x81,
		0x59, 0x19, 0xb9, 0xa6, 0x5a, 0x37, 0x34, 0x14,
		0x47, 0xf6, 0xe7, 0x6c, 0xd2, 0x09, 0x10, 0xe7,
		0xdd, 0xbb, 0x02, 0xd1, 0x28, 0xfa, 0x01, 0x2c,
		0x93, 0x64, 0x2e, 0x1b, 0x4c, 0x02, 0x52, 0xcb,
		0x07, 0xa1, 0xb6, 0x46, 0x02, 0x80, 0xd9, 0x8f,
		0x5c, 0x62, 0xbe, 0x78, 0x9e, 0x75, 0xc4, 0x97,
		0x91, 0x39, 0x12, 0x65, 0xb9, 0x3b, 0xc2, 0xd1,
		0xaf, 0xf2, 0x1f, 0x4e, 0x4d, 0xd1, 0xf0, 0x9f,
		0xb7, 0x12, 0xfd, 0xe8, 0x75, 0x18, 0xc0, 0x9d,
		0x8c, 0x70, 0xff, 0x77, 0x05, 0xb6, 0x1a, 0x1f,
		0x96, 0x48, 0xf6, 0xfe, 0xd5, 0x5d, 0x98, 0xa5,
		0x72, 0x1c, 0x84, 0x76, 0x3e, 0xb8, 0x87, 0x37,
		0xdd, 0xd4, 0x3a, 0x45, 0xdd, 0x09, 0xd8, 0xe7,
		0x09, 0x2f, 0x3e, 0x33, 0x9e, 0x7b, 0x8c, 0xe4,
		0x85, 0x12, 0x4e, 0xf8, 0x06, 0xb7, 0xb1, 0x85,
		0x24, 0x96, 0xd8, 0xfe, 0x87, 0x92, 0x81, 0xb1,
		0xa3, 0x38, 0xb9, 0x56, 0xe1, 0xf6, 0x36, 0x41,
		0xbb, 0xd6, 0x56, 0x69, 0x94, 0x57, 0xb3, 0xa4,
		0xca, 0xa4, 0xe1, 0x02, 0x3b, 0x96, 0x71, 0xe0,
		0xb2, 0x2f, 0x85, 0x48, 0x1b, 0x4a, 0x41, 0x80,
		0x4b, 0x9c, 0xe0, 0xc9, 0x39, 0xb8, 0xb1, 0xca,
		0x64, 0x77, 0x46, 0x58, 0xe6, 0x84, 0xd5, 0x2b,
		0x65, 0xce, 0xe9, 0x09, 0xa3, 0xaa, 0xfb, 0x83,
		0xa9, 0x28, 0x68, 0xfd, 0xcd, 0xfd, 0x76, 0x83,
		0xe1, 0x20, 0x22, 0x77, 0x3a, 0xa3, 0xb2, 0x93,
		0x14, 0x91, 0xfc, 0xe2, 0x17, 0x63, 0x2b, 0xa6,
		0x29, 0x38, 0x7b, 0x9b, 0x8b, 0x15, 0x77, 0xd6,
		0xaa, 0x92, 0x51, 0x53, 0x50, 0xff, 0xa0, 0x35,
		0xa0, 0x59, 0x7d, 0xf0, 0x11, 0x23, 0x49, 0xdf,
		0x5a, 0x21, 0xc2, 0xfe, 0x35, 0xa0, 0x1d, 0xe2,
		0xae, 0xa2, 0x8a, 0x61, 0x5b, 0xf7, 0xf1, 0x1c,
		0x1c, 0xec, 0xc4, 0xf6, 0xdc, 0xaa, 0xc8, 0xc2,
		0xe5, 0xa1, 0x2e, 0x14, 0xe5, 0xc6, 0xc9, 0x73,
		0x03, 0x78, 0xeb, 0xed, 0xe0, 0x3e, 0xc5, 0xf4,
		0xf1, 0x50, 0xb2, 0x01, 0x91, 0x96, 0xf5, 0xbb,
		0xe1, 0x32, 0xcd, 0xa8, 0x66, 0xbf, 0x73, 0x85,
		0x94, 0xd6, 0x7e, 0x68, 0xc5, 0xe4, 0xed, 0xd5,
		0xe3, 0x67, 0x4c, 0xa5, 0xb3, 0x1f, 0xdf, 0xf8,
		0xb3, 0x73, 0x5a, 0xac, 0xeb, 0x46, 0x16, 0x24,
		0xab, 0xca, 0xa4, 0xdd, 0x87, 0x0e, 0x24, 0x83,
		0x32, 0x04, 0x4c, 0xd8, 0xda, 0x7d, 0xdc, 0xe3,
		0x01, 0x93, 0xf3, 0xc1, 0x5b, 0xbd, 0xc3, 0x1d,
		0x40, 0x62, 0xde, 0x94, 0x03, 0x85, 0x91, 0x2a,
		0xa0, 0x25, 0x10, 0xd3, 0x32, 0x9f, 0x93, 0x00,
		0xa7, 0x8a, 0xfa, 0x77, 0x7c, 0xaf, 0x4d, 0xc8,
		0x7a, 0xf3, 0x16, 0x2b, 0xba, 0xeb, 0x74, 0x51,
		0xb8, 0xdd, 0x32, 0xad, 0x68, 0x7d, 0xdd, 0xca,
		0x60, 0x98, 0xc9, 0x9b, 0xb6, 0x5d, 0x4d, 0x3a,
		0x66, 0x8a, 0xbe, 0x05, 0xf9, 0x0c, 0xc5, 0xba,
		0x52, 0x82, 0x09, 0x1f, 0x5a, 0x66, 0x89, 0x69,
		0xa3, 0x5d, 0x93, 0x50, 0x7d, 0x44, 0xc3, 0x2a,
		0xb8, 0xab, 0xec, 0xa6, 0x5a, 0xae, 0x4a, 0x6a,
		0xcd, 0xfd, 0xb6, 0xff, 0x3d, 0x98, 0x05, 0xd9,
		0x5b, 0x29, 0xc4, 0x6f, 0xe0, 0x76, 0xe2, 0x3f,
		0xec, 0xd7, 0xa4, 0x91, 0x63, 0xf5, 0x4e, 0x4b,
		0xab, 0x20, 0x8c, 0x3a, 0x41, 0xed, 0x8b, 0x4b,
		0xb9, 0x01, 0x21, 0xc0, 0x6d, 0xfd, 0x70, 0x5b,
		0x20, 0x92, 0x41, 0x89, 0x74, 0xb7, 0xe9, 0x8b,
		0xfc, 0x6d, 0x17, 0x3f, 0x7f, 0x89, 0x3d, 0x6b,
		0x8f, 0xbc, 0xd2, 0x57, 0xe9, 0xc9, 0x6e, 0xa7,
		0x19, 0x26, 0x18, 0xad, 0xef, 0xb5, 0x87, 0xbf,
		0xb8, 0xa8, 0xd6, 0x7d, 0xdd, 0x5f, 0x94, 0x54,
		0x09, 0x92, 0x2b, 0xf5, 0x04, 0xf7, 0x36, 0x69,
		0x8e, 0xf4, 0xdc, 0x1d, 0x6e, 0x55, 0xbb, 0xe9,
		0x13, 0x05, 0x83, 0x35, 0x9c, 0xed, 0xcf, 0x8c,
		0x26, 0x8c, 0x7b, 0xc7, 0x0b, 0xba, 0xfd, 0xe2,
		0x84, 0x5c, 0x2a, 0x79, 0x43, 0x99, 0xb2, 0xc3,
		0x82, 0x87, 0xc8, 0xcd, 0x37, 0x6d, 0xa1, 0x2b,
		0x39, 0xb2, 0x38, 0x99, 0xd9, 0xfc, 0x02, 0x15,
		0x55, 0x21, 0x62, 0x59, 0xeb, 0x00, 0x86, 0x08,
		0x20, 0xbe, 0x1a, 0x62, 0x4d, 0x7e, 0xdf, 0x68,
		0x73, 0x5b, 0x5f, 0xaf, 0x84, 0x96, 0x2e, 0x1f,
		0x6b, 0x03, 0xc9, 0xa6, 0x75, 0x18, 0xe9, 0xd4,
		0xbd, 0xc8, 0xec, 0x9a, 0x5a, 0xb3, 0x99, 0xab,
		0x5f, 0x7c, 0x08, 0x7f, 0x69, 0x4d, 0x52, 0xa2,
		0x30, 0x17, 0x3b, 0x16, 0x15, 0x1b, 0x11, 0x62,
		0x3e, 0x80, 0x4b, 0x85, 0x7c, 0x9c, 0xd1, 0x3a,
		0x13, 0x01, 0x5e, 0x45, 0xf1, 0xc8, 0x5f, 0xcd,
		0x0e, 0x21, 0xf5, 0x82, 0xd4, 0x7b, 0x5c, 0x45,
		0x27, 0x6b, 0xef, 0xfe, 0xb8, 0xc0, 0x6f, 0xdc,
		0x60, 0x7b, 0xe4, 0xd5, 0x75, 0x71, 0xe6, 0xe8,
		0x7d, 0x6b, 0x6d, 0x80, 0xaf, 0x76, 0x41, 0x58,
		0xb7, 0xac, 0xb7, 0x13, 0x2f, 0x81, 0xcc, 0xf9,
		0x19, 0x97, 0xe8, 0xee, 0x40, 0x91, 0xfc, 0x89,
		0x13, 0x1e, 0x67, 0x9a, 0xdb, 0x8f, 0x8f, 0xc7,
		0x4a, 0xc9, 0xaf, 0x2f, 0x67, 0x01, 0x3c, 0xb8,
		0xa8, 0x3e, 0x78, 0x93, 0x1b, 0xdf, 0xbb, 0x34,
		0x0b, 0x1a, 0xfa, 0xc2, 0x2d, 0xc5, 0x1c, 0xec,
		0x97, 0x4f, 0x48, 0x41, 0x15, 0x0e, 0x75, 0xed,
		0x66, 0x8c, 0x17, 0x7f, 0xb1, 0x48, 0x13, 0xc1,
		0xfb, 0x60, 0x06, 0xf9, 0x72, 0x41, 0x3e, 0xcf,
		0x6e, 0xb6, 0xc8, 0xeb, 0x4b, 0x5a, 0xd2, 0x0c,
		0x28, 0xda, 0x02, 0x7a, 0x46, 0x21, 0x42, 0xb5,
		0x34, 0xda, 0xcb, 0x5e, 0xbd, 0x66, 0x5c, 0xca,
		0xff, 0x52, 0x43, 0x89, 0xf9, 0x10, 0x9a, 0x9e,
		0x9b, 0xe3, 0xb0, 0x51, 0xe9, 0xf3, 0x0a, 0x35,
		0x77, 0x54, 0xcc, 0xac, 0xa6, 0xf1, 0x2e, 0x36,
		0x89, 0xac, 0xc5, 0xc6, 0x62, 0x5a, 0xc0, 0x6d,
		0xc4, 0xe1, 0xf7, 0x64, 0x30, 0xff, 0x11, 0x40,
		0x13, 0x89, 0xd8, 0xd7, 0x73, 0x3f, 0x93, 0x08,
		0x68, 0xab, 0x66, 0x09, 0x1a, 0xea, 0x78, 0xc9,
		0x52, 0xf2, 0xfd, 0x93, 0x1b, 0x94, 0xbe, 0x5c,
		0xe5, 0x00, 0x6e, 0x00, 0xb9, 0xea, 0x27, 0xaa,
		0xb3, 0xee, 0xe3, 0xc8, 0x6a, 0xb0, 0xc1, 0x8e,
		0x9b, 0x54, 0x40, 0x10, 0x96, 0x06, 0xe8, 0xb3,
		0xf5, 0x55, 0x77, 0xd7, 0x5c, 0x94, 0xc1, 0x74,
		0xf3, 0x07, 0x64, 0xac, 0x1c, 0xde, 0xc7, 0x22,
		0xb0, 0xbf, 0x2a, 0x5a, 0xc0, 0x8f, 0x8a, 0x83,
		0x50, 0xc2, 0x5e, 0x97, 0xa0, 0xbe, 0x49, 0x7e,
		0x47, 0xaf, 0xa7, 0x20, 0x02, 0x35, 0xa4, 0x57,
		0xd9, 0x26, 0x63, 0xdb, 0xf1, 0x34, 0x42, 0x89,
		0x36, 0xd1, 0x77, 0x6f, 0xb1, 0xea, 0x79, 0x7e,
		0x95, 0x10, 0x5a, 0xee, 0xa3, 0xae, 0x6f, 0xba,
		0xa9, 0xef, 0x5a, 0x7e, 0x34, 0x03, 0x04, 0x07,
		0x92, 0xd6, 0x07, 0x79, 0xaa, 0x14, 0x90, 0x97,
		0x05, 0x4d, 0xa6, 0x27, 0x10, 0x5c, 0x25, 0x24,
		0xcb, 0xcc, 0xf6, 0x77, 0x9e, 0x43, 0x23, 0xd4,
		0x98, 0xef, 0x22, 0xa8, 0xad, 0xf2, 0x26, 0x08,
		0x59, 0x69, 0xa4, 0xc3, 0x97, 0xe0, 0x5c, 0x6f,
		0xeb, 0x3d, 0xd4, 0x62, 0x6e, 0x80, 0x61, 0x02,
		0xf4, 0xfc, 0x94, 0x79, 0xbb, 0x4e, 0x6d, 0xd7,
		0x30, 0x5b, 0x10, 0x11, 0x5a, 0x3d, 0xa7, 0x50,
		0x1d, 0x9a, 0x13, 0x5f, 0x4f, 0xa8, 0xa7, 0xb6,
		0x39, 0xc7, 0xea, 0xe6, 0x19, 0x61, 0x69, 0xc7,
		0x9a, 0x3a, 0xeb, 0x9d, 0xdc, 0xf7, 0x06, 0x37,
		0xbd, 0xac, 0xe3, 0x18, 0xff, 0xfe, 0x11, 0xdb,
		0x67, 0x42, 0xb4, 0xea, 0xa8, 0xbd, 0xb0, 0x76,
		0xd2, 0x74, 0x32, 0xc2, 0xa4, 0x9c, 0xe7, 0x60,
		0xc5, 0x30, 0x9a, 0x57, 0x66, 0xcd, 0x0f, 0x02,
		0x4c, 0xea, 0xe9, 0xd3, 0x2a, 0x5c, 0x09, 0xc2,
		0xff, 0x6a, 0xde, 0x5d, 0xb7, 0xe9, 0x75, 0x6b,
		0x29, 0x94, 0xd6, 0xf7, 0xc3, 0xdf, 0xfb, 0x70,
		0xec, 0xb5, 0x8c, 0xb0, 0x78, 0x7a, 0xee, 0x52,
		0x5f, 0x8c, 0xae, 0x85, 0xe5, 0x98, 0xa2, 0xb7,
		0x7c, 0x02, 0x2a, 0xcc, 0x9e, 0xde, 0x99, 0x5f,
		0x84, 0x20, 0xbb, 0xdc, 0xf2, 0xd2, 0x13, 0x46,
		0x3c, 0xd6, 0x4d, 0xe7, 0x50, 0xef, 0x55, 0xc3,
		0x96, 0x9f, 0xec, 0x6c, 0xd8, 0xe2, 0xea, 0xed,
		0xc7, 0x33, 0xc9, 0xb3, 0x1c, 0x4f, 0x1d, 0x83,
		0x1d, 0xe4, 0xdd, 0xb2, 0x24, 0x8f, 0xf9, 0xf5
};


static const uint8_t HMAC_SHA256_ciphertext_64B_digest[] = {
		0xc5, 0x6d, 0x4f, 0x29, 0xf4, 0xd2, 0xcc, 0x87,
		0x3c, 0x81, 0x02, 0x6d, 0x38, 0x7a, 0x67, 0x3e,
		0x95, 0x9c, 0x5c, 0x8f, 0xda, 0x5c, 0x06, 0xe0,
		0x65, 0xf1, 0x6c, 0x51, 0x52, 0x49, 0x3e, 0x5f
};

static const uint8_t HMAC_SHA256_ciphertext_128B_digest[] = {
		0x76, 0x64, 0x2d, 0x69, 0x71, 0x5d, 0x6a, 0xd8,
		0x9f, 0x74, 0x11, 0x2f, 0x58, 0xe0, 0x4a, 0x2f,
		0x6c, 0x88, 0x5e, 0x4d, 0x9c, 0x79, 0x83, 0x1c,
		0x8a, 0x14, 0xd0, 0x07, 0xfb, 0xbf, 0x6c, 0x8f
};

static const uint8_t HMAC_SHA256_ciphertext_256B_digest[] = {
		0x05, 0xa7, 0x44, 0xcd, 0x91, 0x8c, 0x95, 0xcf,
		0x7b, 0x8f, 0xd3, 0x90, 0x86, 0x7e, 0x7b, 0xb9,
		0x05, 0xd6, 0x6e, 0x7a, 0xc1, 0x7b, 0x26, 0xff,
		0xd3, 0x4b, 0xe0, 0x22, 0x8b, 0xa8, 0x47, 0x52
};

static const uint8_t HMAC_SHA256_ciphertext_512B_digest[] = {
		0x08, 0xb7, 0x29, 0x54, 0x18, 0x7e, 0x97, 0x49,
		0xc6, 0x7c, 0x9f, 0x94, 0xa5, 0x4f, 0xa2, 0x25,
		0xd0, 0xe2, 0x30, 0x7b, 0xad, 0x93, 0xc9, 0x12,
		0x0f, 0xf0, 0xf0, 0x71, 0xc2, 0xf6, 0x53, 0x8f
};

static const uint8_t HMAC_SHA256_ciphertext_768B_digest[] = {
		0xe4, 0x3e, 0x73, 0x93, 0x03, 0xaf, 0x6f, 0x9c,
		0xca, 0x57, 0x3b, 0x4a, 0x6e, 0x83, 0x58, 0xf5,
		0x66, 0xc2, 0xb4, 0xa7, 0xe0, 0xee, 0x63, 0x6b,
		0x48, 0xb7, 0x50, 0x45, 0x69, 0xdf, 0x5c, 0x5b
};

static const uint8_t HMAC_SHA256_ciphertext_1024B_digest[] = {
		0x03, 0xb9, 0x96, 0x26, 0xdc, 0x1c, 0xab, 0xe2,
		0xf5, 0x70, 0x55, 0x15, 0x67, 0x6e, 0x48, 0x11,
		0xe7, 0x67, 0xea, 0xfa, 0x5c, 0x6b, 0x28, 0x22,
		0xc9, 0x0e, 0x67, 0x04, 0xb3, 0x71, 0x7f, 0x88
};

static const uint8_t HMAC_SHA256_ciphertext_1280B_digest[] = {
		0x01, 0x91, 0xb8, 0x78, 0xd3, 0x21, 0x74, 0xa5,
		0x1c, 0x8b, 0xd4, 0xd2, 0xc0, 0x49, 0xd7, 0xd2,
		0x16, 0x46, 0x66, 0x85, 0x50, 0x6d, 0x08, 0xcc,
		0xc7, 0x0a, 0xa3, 0x71, 0xcc, 0xde, 0xee, 0xdc
};

static const uint8_t HMAC_SHA256_ciphertext_1536B_digest[] = {
		0xf2, 0xe5, 0xe9, 0x57, 0x53, 0xd7, 0x69, 0x28,
		0x7b, 0x69, 0xb5, 0x49, 0xa3, 0x31, 0x56, 0x5f,
		0xa4, 0xe9, 0x87, 0x26, 0x2f, 0xe0, 0x2d, 0xd6,
		0x08, 0x44, 0x01, 0x71, 0x0c, 0x93, 0x85, 0x84
};

static const uint8_t HMAC_SHA256_ciphertext_1792B_digest[] = {
		0xf6, 0x57, 0x62, 0x01, 0xbf, 0x2d, 0xea, 0x4a,
		0xef, 0x43, 0x85, 0x60, 0x18, 0xdf, 0x8b, 0xb4,
		0x60, 0xc0, 0xfd, 0x2f, 0x90, 0x15, 0xe6, 0x91,
		0x56, 0x61, 0x68, 0x7f, 0x5e, 0x92, 0xa8, 0xdd
};

static const uint8_t HMAC_SHA256_ciphertext_2048B_digest[] = {
		0x81, 0x1a, 0x29, 0xbc, 0x6b, 0x9f, 0xbb, 0xb8,
		0xef, 0x71, 0x7b, 0x1f, 0x6f, 0xd4, 0x7e, 0x68,
		0x3a, 0x9c, 0xb9, 0x98, 0x22, 0x81, 0xfa, 0x95,
		0xee, 0xbc, 0x7f, 0x23, 0x29, 0x88, 0x76, 0xb8
};

struct crypto_data_params {
	const char *name;
	uint16_t length;
	const char *plaintext;
	struct crypto_expected_output {
		const uint8_t *ciphertext;
		const uint8_t *digest;
	} expected;
};

#define MAX_PACKET_SIZE_INDEX	10

struct crypto_data_params aes_cbc_hmac_sha256_output[MAX_PACKET_SIZE_INDEX] = {
	{ "64B", 64, &plaintext_quote[sizeof(plaintext_quote) - 1 - 64],
		{ AES_CBC_ciphertext_64B, HMAC_SHA256_ciphertext_64B_digest } },
	{ "128B", 128, &plaintext_quote[sizeof(plaintext_quote) - 1 - 128],
		{ AES_CBC_ciphertext_128B, HMAC_SHA256_ciphertext_128B_digest } },
	{ "256B", 256, &plaintext_quote[sizeof(plaintext_quote) - 1 - 256],
		{ AES_CBC_ciphertext_256B, HMAC_SHA256_ciphertext_256B_digest } },
	{ "512B", 512, &plaintext_quote[sizeof(plaintext_quote) - 1 - 512],
		{ AES_CBC_ciphertext_512B, HMAC_SHA256_ciphertext_512B_digest } },
	{ "768B", 768, &plaintext_quote[sizeof(plaintext_quote) - 1 - 768],
		{ AES_CBC_ciphertext_768B, HMAC_SHA256_ciphertext_768B_digest } },
	{ "1024B", 1024, &plaintext_quote[sizeof(plaintext_quote) - 1 - 1024],
		{ AES_CBC_ciphertext_1024B, HMAC_SHA256_ciphertext_1024B_digest } },
	{ "1280B", 1280, &plaintext_quote[sizeof(plaintext_quote) - 1 - 1280],
		{ AES_CBC_ciphertext_1280B, HMAC_SHA256_ciphertext_1280B_digest } },
	{ "1536B", 1536, &plaintext_quote[sizeof(plaintext_quote) - 1 - 1536],
		{ AES_CBC_ciphertext_1536B, HMAC_SHA256_ciphertext_1536B_digest } },
	{ "1792B", 1792, &plaintext_quote[sizeof(plaintext_quote) - 1 - 1792],
		{ AES_CBC_ciphertext_1792B, HMAC_SHA256_ciphertext_1792B_digest } },
	{ "2048B", 2048, &plaintext_quote[sizeof(plaintext_quote) - 1 - 2048],
		{ AES_CBC_ciphertext_2048B, HMAC_SHA256_ciphertext_2048B_digest } }
};

static int
test_perf_crypto_qp_vary_burst_size(uint16_t dev_num)
{
	uint32_t num_to_submit = 4096;
	struct rte_crypto_op *c_ops[num_to_submit];
	struct rte_crypto_op *proc_ops[num_to_submit];
	uint64_t failed_polls, retries, start_cycles, end_cycles, total_cycles = 0;
	uint32_t burst_sent, burst_received;
	uint32_t i, burst_size, num_sent, num_received;
	struct crypto_testsuite_params *ts_params = &testsuite_params;
	struct crypto_unittest_params *ut_params = &unittest_params;
	struct crypto_data_params *data_params = aes_cbc_hmac_sha256_output;

	if (rte_cryptodev_count() == 0) {
		printf("\nNo crypto devices available. Is kernel driver loaded?\n");
		return TEST_FAILED;
	}

	/* Setup Cipher Parameters */
	ut_params->cipher_xform.type = RTE_CRYPTO_SYM_XFORM_CIPHER;
	ut_params->cipher_xform.next = &ut_params->auth_xform;

	ut_params->cipher_xform.cipher.algo = RTE_CRYPTO_CIPHER_AES_CBC;
	ut_params->cipher_xform.cipher.op = RTE_CRYPTO_CIPHER_OP_DECRYPT;
	ut_params->cipher_xform.cipher.key.data = aes_cbc_128_key;
	ut_params->cipher_xform.cipher.key.length = CIPHER_IV_LENGTH_AES_CBC;


	/* Setup HMAC Parameters */
	ut_params->auth_xform.type = RTE_CRYPTO_SYM_XFORM_AUTH;
	ut_params->auth_xform.next = NULL;

	ut_params->auth_xform.auth.op = RTE_CRYPTO_AUTH_OP_VERIFY;
	ut_params->auth_xform.auth.algo = RTE_CRYPTO_AUTH_SHA256_HMAC;
	ut_params->auth_xform.auth.key.data = hmac_sha256_key;
	ut_params->auth_xform.auth.key.length = HMAC_KEY_LENGTH_SHA256;
	ut_params->auth_xform.auth.digest_length = DIGEST_BYTE_LENGTH_SHA256;

	/* Create Crypto session*/
	ut_params->sess = rte_cryptodev_sym_session_create(ts_params->dev_id,
		&ut_params->cipher_xform);

	TEST_ASSERT_NOT_NULL(ut_params->sess, "Session creation failed");

	/* Generate Crypto op data structure(s) */
	for (i = 0; i < num_to_submit ; i++) {
		struct rte_mbuf *m = setup_test_string(ts_params->mbuf_mp,
				data_params[0].expected.ciphertext,
				data_params[0].length, 0);
		TEST_ASSERT_NOT_NULL(m, "Failed to allocate tx_buf");

		ut_params->digest = (uint8_t *)rte_pktmbuf_append(m,
				DIGEST_BYTE_LENGTH_SHA256);
		TEST_ASSERT_NOT_NULL(ut_params->digest,
				"no room to append digest");

		rte_memcpy(ut_params->digest, data_params[0].expected.digest,
			DIGEST_BYTE_LENGTH_SHA256);


		struct rte_crypto_op *op =
				rte_crypto_op_alloc(ts_params->op_mpool,
						RTE_CRYPTO_OP_TYPE_SYMMETRIC);

		rte_crypto_op_attach_sym_session(op, ut_params->sess);

		op->sym->auth.digest.data = ut_params->digest;
		op->sym->auth.digest.phys_addr = rte_pktmbuf_mtophys_offset(m,
				data_params[0].length);
		op->sym->auth.digest.length = DIGEST_BYTE_LENGTH_SHA256;

		op->sym->auth.data.offset = CIPHER_IV_LENGTH_AES_CBC;
		op->sym->auth.data.length = data_params[0].length;


		op->sym->cipher.iv.data = (uint8_t *)rte_pktmbuf_prepend(m,
				CIPHER_IV_LENGTH_AES_CBC);
		op->sym->cipher.iv.phys_addr = rte_pktmbuf_mtophys(m);
		op->sym->cipher.iv.length = CIPHER_IV_LENGTH_AES_CBC;

		rte_memcpy(op->sym->cipher.iv.data, aes_cbc_128_iv,
				CIPHER_IV_LENGTH_AES_CBC);

		op->sym->cipher.data.offset = CIPHER_IV_LENGTH_AES_CBC;
		op->sym->cipher.data.length = data_params[0].length;

		op->sym->m_src = m;

		c_ops[i] = op;
	}

	printf("\nTest to measure the IA cycle cost using AES128_CBC_SHA256_HMAC "
			"algorithm with a constant request size of %u.",
			data_params[0].length);
	printf("\nThis test will keep retries at 0 and only measure IA cycle "
			"cost for each request.");
	printf("\nDev No\tQP No\tNum Sent\tNum Received\tTx/Rx burst");
	printf("\tRetries (Device Busy)\tAverage IA cycle cost "
			"(assuming 0 retries)");
	for (i = 2; i <= 128 ; i *= 2) {
		num_sent = 0;
		num_received = 0;
		retries = 0;
		failed_polls = 0;
		burst_size = i;
		total_cycles = 0;
		while (num_sent < num_to_submit) {
			start_cycles = rte_rdtsc_precise();
			burst_sent = rte_cryptodev_enqueue_burst(dev_num,
					0, &c_ops[num_sent],
					((num_to_submit-num_sent) < burst_size) ?
					num_to_submit-num_sent : burst_size);
			if (burst_sent == 0)
				retries++;
			else
				num_sent += burst_sent;
			end_cycles = rte_rdtsc_precise();
			total_cycles += (end_cycles - start_cycles);
			/*
			 * Wait until requests have been sent.
			 */
			rte_delay_ms(1);

			start_cycles = rte_rdtsc_precise();
			burst_received = rte_cryptodev_dequeue_burst(
					dev_num, 0, proc_ops, burst_size);
			if (burst_received == 0)
				failed_polls++;
			else
				num_received += burst_received;
			end_cycles = rte_rdtsc_precise();
			total_cycles += end_cycles - start_cycles;
		}

		while (num_received != num_to_submit) {
			if (gbl_cryptodev_perftest_devtype ==
					RTE_CRYPTODEV_AESNI_MB_PMD)
				rte_cryptodev_enqueue_burst(dev_num, 0,
						NULL, 0);

			burst_received = rte_cryptodev_dequeue_burst(
					dev_num, 0, proc_ops, burst_size);
			if (burst_received == 0)
				failed_polls++;
			else
				num_received += burst_received;
		}

		printf("\n%u\t%u\t%u\t\t%u\t\t%u", dev_num, 0,
					num_sent, num_received, burst_size);
		printf("\t\t%"PRIu64, retries);
		printf("\t\t\t%"PRIu64, total_cycles/num_received);
	}
	printf("\n");

	for (i = 0; i < num_to_submit ; i++) {
		rte_pktmbuf_free(c_ops[i]->sym->m_src);
		rte_crypto_op_free(c_ops[i]);
	}
	return TEST_SUCCESS;
}

static int
test_perf_snow3G_optimise_cyclecount(struct perf_test_params *pparams)
{
	uint32_t num_to_submit = pparams->total_operations;
	struct rte_crypto_op *c_ops[num_to_submit];
	struct rte_crypto_op *proc_ops[num_to_submit];
	uint64_t failed_polls, retries, start_cycles, end_cycles, total_cycles = 0;
	uint32_t burst_sent = 0, burst_received = 0;
	uint32_t i, burst_size, num_sent, num_ops_received;
	struct crypto_testsuite_params *ts_params = &testsuite_params;
	static struct rte_cryptodev_sym_session *sess;

	if (rte_cryptodev_count() == 0) {
		printf("\nNo crypto devices found. Is PMD build configured?\n");
		printf("\nAnd is kernel driver loaded for HW PMDs?\n");
		return TEST_FAILED;
	}

	/* Create Crypto session*/
	sess = test_perf_create_snow3g_session(ts_params->dev_id,
			pparams->chain, pparams->cipher_algo,
			pparams->cipher_key_length, pparams->auth_algo);
	TEST_ASSERT_NOT_NULL(sess, "Session creation failed");

	/* Generate Crypto op data structure(s)*/
	for (i = 0; i < num_to_submit ; i++) {
		struct rte_mbuf *m = test_perf_create_pktmbuf(
						ts_params->mbuf_mp,
						pparams->buf_size);
		TEST_ASSERT_NOT_NULL(m, "Failed to allocate tx_buf");

		struct rte_crypto_op *op =
				rte_crypto_op_alloc(ts_params->op_mpool,
						RTE_CRYPTO_OP_TYPE_SYMMETRIC);
		TEST_ASSERT_NOT_NULL(op, "Failed to allocate op");

		op = test_perf_set_crypto_op_snow3g(op, m, sess, pparams->buf_size,
					get_auth_digest_length(pparams->auth_algo));
		TEST_ASSERT_NOT_NULL(op, "Failed to attach op to session");

		c_ops[i] = op;
	}

	printf("\nOn %s dev%u qp%u, %s, cipher algo:%s, auth_algo:%s, "
			"Packet Size %u bytes",
			pmd_name(gbl_cryptodev_perftest_devtype),
			ts_params->dev_id, 0,
			chain_mode_name(pparams->chain),
			cipher_algo_name(pparams->cipher_algo),
			auth_algo_name(pparams->auth_algo),
			pparams->buf_size);
	printf("\nOps Tx\tOps Rx\tOps/burst  ");
	printf("Retries  EmptyPolls\tIACycles/CyOp\tIACycles/Burst\tIACycles/Byte");

	for (i = 2; i <= 128 ; i *= 2) {
		num_sent = 0;
		num_ops_received = 0;
		retries = 0;
		failed_polls = 0;
		burst_size = i;
		total_cycles = 0;
		while (num_sent < num_to_submit) {
			start_cycles = rte_rdtsc_precise();
			burst_sent = rte_cryptodev_enqueue_burst(ts_params->dev_id,
					0, &c_ops[num_sent],
					((num_to_submit-num_sent) < burst_size) ?
					num_to_submit-num_sent : burst_size);
			end_cycles = rte_rdtsc_precise();
			if (burst_sent == 0)
				retries++;
			num_sent += burst_sent;
			total_cycles += (end_cycles - start_cycles);

			/* Wait until requests have been sent. */

			rte_delay_ms(1);

			start_cycles = rte_rdtsc_precise();
			burst_received = rte_cryptodev_dequeue_burst(
					ts_params->dev_id, 0, proc_ops, burst_size);
			end_cycles = rte_rdtsc_precise();
			if (burst_received < burst_sent)
				failed_polls++;
			num_ops_received += burst_received;

			total_cycles += end_cycles - start_cycles;
		}

		while (num_ops_received != num_to_submit) {
			if (gbl_cryptodev_perftest_devtype ==
					RTE_CRYPTODEV_AESNI_MB_PMD)
				rte_cryptodev_enqueue_burst(ts_params->dev_id, 0,
						NULL, 0);
			start_cycles = rte_rdtsc_precise();
			burst_received = rte_cryptodev_dequeue_burst(
					ts_params->dev_id, 0, proc_ops, burst_size);
			end_cycles = rte_rdtsc_precise();
			total_cycles += end_cycles - start_cycles;
			if (burst_received == 0)
				failed_polls++;
			num_ops_received += burst_received;
		}

		printf("\n%u\t%u\t%u", num_sent, num_ops_received, burst_size);
		printf("\t\t%"PRIu64, retries);
		printf("\t%"PRIu64, failed_polls);
		printf("\t\t%"PRIu64, total_cycles/num_ops_received);
		printf("\t\t%"PRIu64, (total_cycles/num_ops_received)*burst_size);
		printf("\t\t%"PRIu64, total_cycles/(num_ops_received*pparams->buf_size));
	}
	printf("\n");

	for (i = 0; i < num_to_submit ; i++) {
		rte_pktmbuf_free(c_ops[i]->sym->m_src);
		rte_crypto_op_free(c_ops[i]);
	}

	return TEST_SUCCESS;
}

static int
test_perf_snow3G_vary_burst_size(void)
{
	unsigned total_operations = 4096;
	/*no need to vary pkt size for QAT, should have no effect on IA cycles */
	uint16_t buf_lengths[] = {40};
	uint8_t i, j;

	struct perf_test_params params_set[] = {
			{
					.chain = CIPHER_ONLY,
					.cipher_algo  = RTE_CRYPTO_CIPHER_SNOW3G_UEA2,
					.cipher_key_length = 16,
					.auth_algo  = RTE_CRYPTO_AUTH_NULL,
			},
			{
					.chain = HASH_ONLY,
					.cipher_algo = RTE_CRYPTO_CIPHER_NULL,
					.auth_algo  = RTE_CRYPTO_AUTH_SNOW3G_UIA2,
					.cipher_key_length = 16
			},
	};

	printf("\n\nStart %s.", __func__);
	printf("\nThis Test measures the average IA cycle cost using a "
			"constant request(packet) size. ");
	printf("Cycle cost is only valid when indicators show device is not busy,"
			" i.e. Retries and EmptyPolls = 0");

	for (i = 0; i < RTE_DIM(params_set); i++) {
		printf("\n");
		params_set[i].total_operations = total_operations;

		for (j = 0;
			j < RTE_DIM(buf_lengths);
			j++) {

			params_set[i].buf_size = buf_lengths[j];

			test_perf_snow3G_optimise_cyclecount(&params_set[i]);
		}

	}

	return 0;
}

static uint32_t get_auth_key_max_length(enum rte_crypto_auth_algorithm algo)
{
	switch (algo) {
	case RTE_CRYPTO_AUTH_SNOW3G_UIA2:
		return 16;
	case RTE_CRYPTO_AUTH_SHA1_HMAC:
		return 64;
	case RTE_CRYPTO_AUTH_SHA224_HMAC:
		return 64;
	case RTE_CRYPTO_AUTH_SHA256_HMAC:
		return 64;
	case RTE_CRYPTO_AUTH_SHA384_HMAC:
		return 128;
	case RTE_CRYPTO_AUTH_SHA512_HMAC:
		return 128;
	default:
		return 0;
	}
}

static uint32_t get_auth_digest_length(enum rte_crypto_auth_algorithm algo)
{
	switch (algo) {
	case RTE_CRYPTO_AUTH_SNOW3G_UIA2:
		return 4;
	case RTE_CRYPTO_AUTH_SHA1_HMAC:
		return TRUNCATED_DIGEST_BYTE_LENGTH_SHA1;
	case RTE_CRYPTO_AUTH_SHA224_HMAC:
		return TRUNCATED_DIGEST_BYTE_LENGTH_SHA224;
	case RTE_CRYPTO_AUTH_SHA256_HMAC:
		return TRUNCATED_DIGEST_BYTE_LENGTH_SHA256;
	case RTE_CRYPTO_AUTH_SHA384_HMAC:
		return TRUNCATED_DIGEST_BYTE_LENGTH_SHA384;
	case RTE_CRYPTO_AUTH_SHA512_HMAC:
		return TRUNCATED_DIGEST_BYTE_LENGTH_SHA512;
	default:
		return 0;
	}
}

static uint8_t aes_cbc_key[] = {
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00
};

static uint8_t aes_cbc_iv[] = {
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00
};

static uint8_t hmac_sha_key[] = {
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00
};

static uint8_t snow3g_cipher_key[] = {
		0x2B, 0xD6, 0x45, 0x9F, 0x82, 0xC5, 0xB3, 0x00,
		0x95, 0x2C, 0x49, 0x10, 0x48, 0x81, 0xFF, 0x48
};

static uint8_t snow3g_iv[] = {
		0x72, 0xA4, 0xF2, 0x0F, 0x64, 0x00, 0x00, 0x00,
		0x72, 0xA4, 0xF2, 0x0F, 0x64, 0x00, 0x00, 0x00
};

static uint8_t snow3g_hash_key[] = {
		0xC7, 0x36, 0xC6, 0xAA, 0xB2, 0x2B, 0xFF, 0xF9,
		0x1E, 0x26, 0x98, 0xD2, 0xE2, 0x2A, 0xD5, 0x7E
};

static struct rte_cryptodev_sym_session *
test_perf_create_aes_sha_session(uint8_t dev_id, enum chain_mode chain,
		enum rte_crypto_cipher_algorithm cipher_algo,
		unsigned cipher_key_len,
		enum rte_crypto_auth_algorithm auth_algo)
{
	struct rte_crypto_sym_xform cipher_xform = { 0 };
	struct rte_crypto_sym_xform auth_xform = { 0 };


	/* Setup Cipher Parameters */
	cipher_xform.type = RTE_CRYPTO_SYM_XFORM_CIPHER;
	cipher_xform.cipher.algo = cipher_algo;
	cipher_xform.cipher.op = RTE_CRYPTO_CIPHER_OP_ENCRYPT;

	cipher_xform.cipher.key.data = aes_cbc_key;
	cipher_xform.cipher.key.length = cipher_key_len;

	/* Setup HMAC Parameters */
	auth_xform.type = RTE_CRYPTO_SYM_XFORM_AUTH;
	auth_xform.auth.op = RTE_CRYPTO_AUTH_OP_GENERATE;
	auth_xform.auth.algo = auth_algo;

	auth_xform.auth.key.data = hmac_sha_key;
	auth_xform.auth.key.length =  get_auth_key_max_length(auth_algo);
	auth_xform.auth.digest_length = get_auth_digest_length(auth_algo);

	switch (chain) {
	case CIPHER_HASH:
		cipher_xform.next = &auth_xform;
		auth_xform.next = NULL;
		/* Create Crypto session*/
		return rte_cryptodev_sym_session_create(dev_id,	&cipher_xform);
	case HASH_CIPHER:
		auth_xform.next = &cipher_xform;
		cipher_xform.next = NULL;
		/* Create Crypto session*/
		return rte_cryptodev_sym_session_create(dev_id,	&auth_xform);
	default:
		return NULL;
	}
}

static struct rte_cryptodev_sym_session *
test_perf_create_snow3g_session(uint8_t dev_id, enum chain_mode chain,
		enum rte_crypto_cipher_algorithm cipher_algo, unsigned cipher_key_len,
		enum rte_crypto_auth_algorithm auth_algo)
{
	struct rte_crypto_sym_xform cipher_xform = {0};
	struct rte_crypto_sym_xform auth_xform = {0};


	/* Setup Cipher Parameters */
	cipher_xform.type = RTE_CRYPTO_SYM_XFORM_CIPHER;
	cipher_xform.cipher.algo = cipher_algo;
	cipher_xform.cipher.op = RTE_CRYPTO_CIPHER_OP_ENCRYPT;

	cipher_xform.cipher.key.data = snow3g_cipher_key;
	cipher_xform.cipher.key.length = cipher_key_len;

	/* Setup HMAC Parameters */
	auth_xform.type = RTE_CRYPTO_SYM_XFORM_AUTH;
	auth_xform.auth.op = RTE_CRYPTO_AUTH_OP_GENERATE;
	auth_xform.auth.algo = auth_algo;

	auth_xform.auth.key.data = snow3g_hash_key;
	auth_xform.auth.key.length =  get_auth_key_max_length(auth_algo);
	auth_xform.auth.digest_length = get_auth_digest_length(auth_algo);

	switch (chain) {
	case CIPHER_HASH:
		cipher_xform.next = &auth_xform;
		auth_xform.next = NULL;
		/* Create Crypto session*/
		return rte_cryptodev_sym_session_create(dev_id,	&cipher_xform);
	case HASH_CIPHER:
		auth_xform.next = &cipher_xform;
		cipher_xform.next = NULL;
		/* Create Crypto session*/
		return rte_cryptodev_sym_session_create(dev_id,	&auth_xform);
	case CIPHER_ONLY:
		cipher_xform.next = NULL;
		/* Create Crypto session*/
		return rte_cryptodev_sym_session_create(dev_id, &cipher_xform);
	case HASH_ONLY:
		auth_xform.next = NULL;
		/* Create Crypto session */
		return rte_cryptodev_sym_session_create(dev_id,	&auth_xform);
	default:
		return NULL;
	}
}

#define AES_CBC_BLOCK_SIZE 16
#define AES_CBC_CIPHER_IV_LENGTH 16
#define SNOW3G_CIPHER_IV_LENGTH 16

static struct rte_mbuf *
test_perf_create_pktmbuf(struct rte_mempool *mpool, unsigned buf_sz)
{
	struct rte_mbuf *m = rte_pktmbuf_alloc(mpool);

	if (rte_pktmbuf_append(m, buf_sz) == NULL) {
		rte_pktmbuf_free(m);
		return NULL;
	}

	memset(rte_pktmbuf_mtod(m, uint8_t *), 0, buf_sz);

	return m;
}

static inline struct rte_crypto_op *
test_perf_set_crypto_op(struct rte_crypto_op *op, struct rte_mbuf *m,
		struct rte_cryptodev_sym_session *sess, unsigned data_len,
		unsigned digest_len)
{
	if (rte_crypto_op_attach_sym_session(op, sess) != 0) {
		rte_crypto_op_free(op);
		return NULL;
	}

	/* Authentication Parameters */
	op->sym->auth.digest.data = (uint8_t *)m->buf_addr +
					(m->data_off + data_len);
	op->sym->auth.digest.phys_addr = rte_pktmbuf_mtophys_offset(m, data_len);
	op->sym->auth.digest.length = digest_len;
	op->sym->auth.aad.data = aes_cbc_iv;
	op->sym->auth.aad.length = AES_CBC_CIPHER_IV_LENGTH;

	/* Cipher Parameters */
	op->sym->cipher.iv.data = aes_cbc_iv;
	op->sym->cipher.iv.length = AES_CBC_CIPHER_IV_LENGTH;

	/* Data lengths/offsets Parameters */
	op->sym->auth.data.offset = 0;
	op->sym->auth.data.length = data_len;

	op->sym->cipher.data.offset = AES_CBC_BLOCK_SIZE;
	op->sym->cipher.data.length = data_len - AES_CBC_BLOCK_SIZE;

	op->sym->m_src = m;

	return op;
}

static inline struct rte_crypto_op *
test_perf_set_crypto_op_snow3g(struct rte_crypto_op *op, struct rte_mbuf *m,
		struct rte_cryptodev_sym_session *sess, unsigned data_len,
		unsigned digest_len)
{
	if (rte_crypto_op_attach_sym_session(op, sess) != 0) {
		rte_crypto_op_free(op);
		return NULL;
	}

	/* Authentication Parameters */
	op->sym->auth.digest.data = (uint8_t *)m->buf_addr +
						(m->data_off + data_len);
	op->sym->auth.digest.phys_addr = rte_pktmbuf_mtophys_offset(m, data_len);
	op->sym->auth.digest.length = digest_len;
	op->sym->auth.aad.data = snow3g_iv;
	op->sym->auth.aad.length = SNOW3G_CIPHER_IV_LENGTH;

	/* Cipher Parameters */
	op->sym->cipher.iv.data = snow3g_iv;
	op->sym->cipher.iv.length = SNOW3G_CIPHER_IV_LENGTH;

	/* Data lengths/offsets Parameters */
	op->sym->auth.data.offset = 0;
	op->sym->auth.data.length = data_len << 3;

	op->sym->cipher.data.offset = 0;
	op->sym->cipher.data.length = data_len << 3;

	op->sym->m_src = m;

	return op;
}



/* An mbuf set is used in each burst. An mbuf can be used by multiple bursts at
 * same time, i.e. as they're not dereferenced there's no need to wait until
 * finished with to re-use */
#define NUM_MBUF_SETS 8

static int
test_perf_aes_sha(uint8_t dev_id, uint16_t queue_id,
		struct perf_test_params *pparams)
{
	uint16_t i, k, l, m;
	uint16_t j = 0;
	uint16_t ops_unused = 0;

	uint64_t burst_enqueued = 0, total_enqueued = 0, burst_dequeued = 0;
	uint64_t processed = 0, failed_polls = 0, retries = 0;
	uint64_t tsc_start = 0, tsc_end = 0;

	uint16_t digest_length = get_auth_digest_length(pparams->auth_algo);

	struct rte_crypto_op *ops[pparams->burst_size];
	struct rte_crypto_op *proc_ops[pparams->burst_size];

	struct rte_mbuf *mbufs[pparams->burst_size * 8];

	struct crypto_testsuite_params *ts_params = &testsuite_params;

	static struct rte_cryptodev_sym_session *sess;

	if (rte_cryptodev_count() == 0) {
		printf("\nNo crypto devices available. Is kernel driver loaded?\n");
		return TEST_FAILED;
	}

	/* Create Crypto session*/
	sess = test_perf_create_aes_sha_session(ts_params->dev_id,
			pparams->chain, pparams->cipher_algo,
			pparams->cipher_key_length, pparams->auth_algo);
	TEST_ASSERT_NOT_NULL(sess, "Session creation failed");

	/* Generate a burst of crypto operations */
	for (i = 0; i < (pparams->burst_size * NUM_MBUF_SETS); i++) {
		mbufs[i] = test_perf_create_pktmbuf(
				ts_params->mbuf_mp,
				pparams->buf_size);

		if (mbufs[i] == NULL) {
			printf("\nFailed to get mbuf - freeing the rest.\n");
			for (k = 0; k < i; k++)
				rte_pktmbuf_free(mbufs[k]);
			return -1;
		}

	}


	tsc_start = rte_rdtsc_precise();

	while (total_enqueued < pparams->total_operations) {
		uint16_t burst_size =
		total_enqueued+pparams->burst_size <= pparams->total_operations ?
		pparams->burst_size : pparams->total_operations-total_enqueued;
		uint16_t ops_needed = burst_size-ops_unused;

		if (ops_needed != rte_crypto_op_bulk_alloc(ts_params->op_mpool,
				RTE_CRYPTO_OP_TYPE_SYMMETRIC, ops, ops_needed)){
			printf("\nFailed to alloc enough ops, finish dequeuing "
				"and free ops below.");
		} else {
			for (i = 0; i < ops_needed; i++)
				ops[i] = test_perf_set_crypto_op(ops[i],
					mbufs[i + (pparams->burst_size *
						(j % NUM_MBUF_SETS))],
					sess, pparams->buf_size, digest_length);

			/* enqueue burst */
			burst_enqueued = rte_cryptodev_enqueue_burst(dev_id,
					queue_id, ops, burst_size);

			if (burst_enqueued < burst_size)
				retries++;

			ops_unused = burst_size-burst_enqueued;
			total_enqueued += burst_enqueued;
		}

		/* dequeue burst */
		burst_dequeued = rte_cryptodev_dequeue_burst(dev_id, queue_id,
				proc_ops, pparams->burst_size);
		if (burst_dequeued == 0)
			failed_polls++;
		else {
			processed += burst_dequeued;

			for (l = 0; l < burst_dequeued; l++)
				rte_crypto_op_free(proc_ops[l]);
		}
		j++;
	}

	/* Dequeue any operations still in the crypto device */
	while (processed < pparams->total_operations) {
		/* Sending 0 length burst to flush sw crypto device */
		rte_cryptodev_enqueue_burst(dev_id, queue_id, NULL, 0);

		/* dequeue burst */
		burst_dequeued = rte_cryptodev_dequeue_burst(dev_id, queue_id,
				proc_ops, pparams->burst_size);
		if (burst_dequeued == 0)
			failed_polls++;
		else {
			processed += burst_dequeued;

			for (m = 0; m < burst_dequeued; m++)
				rte_crypto_op_free(proc_ops[m]);
		}
	}

	tsc_end = rte_rdtsc_precise();

	double ops_s = ((double)processed / (tsc_end - tsc_start)) * rte_get_tsc_hz();
	double throughput = (ops_s * pparams->buf_size * 8) / 1000000000;

	printf("\t%u\t%6.2f\t%10.2f\t%8"PRIu64"\t%8"PRIu64, pparams->buf_size, ops_s/1000000,
			throughput, retries, failed_polls);

	for (i = 0; i < pparams->burst_size * NUM_MBUF_SETS; i++)
		rte_pktmbuf_free(mbufs[i]);

	printf("\n");
	return TEST_SUCCESS;
}


static int
test_perf_snow3g(uint8_t dev_id, uint16_t queue_id,
		struct perf_test_params *pparams)
{
	uint16_t i, k, l, m;
	uint16_t j = 0;
	uint16_t ops_unused = 0;
	uint64_t burst_enqueued = 0, total_enqueued = 0, burst_dequeued = 0;
	uint64_t processed = 0, failed_polls = 0, retries = 0;
	uint64_t tsc_start = 0, tsc_end = 0;

	uint16_t digest_length = get_auth_digest_length(pparams->auth_algo);

	struct rte_crypto_op *ops[pparams->burst_size];
	struct rte_crypto_op *proc_ops[pparams->burst_size];

	struct rte_mbuf *mbufs[pparams->burst_size * NUM_MBUF_SETS];

	struct crypto_testsuite_params *ts_params = &testsuite_params;

	static struct rte_cryptodev_sym_session *sess;

	if (rte_cryptodev_count() == 0) {
		printf("\nNo crypto devices found. Is PMD build configured?\n");
		printf("\nAnd is kernel driver loaded for HW PMDs?\n");
		return TEST_FAILED;
	}

	/* Create Crypto session*/
	sess = test_perf_create_snow3g_session(ts_params->dev_id,
			pparams->chain, pparams->cipher_algo,
			pparams->cipher_key_length, pparams->auth_algo);
	TEST_ASSERT_NOT_NULL(sess, "Session creation failed");

	/* Generate a burst of crypto operations */
	for (i = 0; i < (pparams->burst_size * NUM_MBUF_SETS); i++) {
		mbufs[i] = test_perf_create_pktmbuf(
				ts_params->mbuf_mp,
				pparams->buf_size);

		if (mbufs[i] == NULL) {
			printf("\nFailed to get mbuf - freeing the rest.\n");
			for (k = 0; k < i; k++)
				rte_pktmbuf_free(mbufs[k]);
			return -1;
		}

	}

	tsc_start = rte_rdtsc_precise();

	while (total_enqueued < pparams->total_operations) {
		uint16_t burst_size =
				(total_enqueued+pparams->burst_size)
						<= pparams->total_operations ?
		pparams->burst_size : pparams->total_operations-total_enqueued;
		uint16_t ops_needed = burst_size-ops_unused;
		/* Handle the last burst correctly */
		uint16_t op_offset = pparams->burst_size - burst_size;

		if (ops_needed !=
			rte_crypto_op_bulk_alloc(ts_params->op_mpool,
						RTE_CRYPTO_OP_TYPE_SYMMETRIC,
						ops+op_offset, ops_needed)) {
			printf("\nFailed to alloc enough ops.");
			/*Don't exit, dequeue, more ops should become available*/
		} else {
			for (i = 0; i < ops_needed; i++) {
				ops[i+op_offset] =
				test_perf_set_crypto_op_snow3g(ops[i+op_offset],
				mbufs[i +
				  (pparams->burst_size * (j % NUM_MBUF_SETS))],
				sess,
				pparams->buf_size, digest_length);
			}

			/* enqueue burst */
			burst_enqueued =
				rte_cryptodev_enqueue_burst(dev_id, queue_id,
						ops+op_offset, burst_size);

			if (burst_enqueued < burst_size)
				retries++;

			ops_unused = burst_size-burst_enqueued;
			total_enqueued += burst_enqueued;
		}

		/* dequeue burst */
		burst_dequeued = rte_cryptodev_dequeue_burst(dev_id, queue_id,
					proc_ops, pparams->burst_size);
		if (burst_dequeued == 0) {
			failed_polls++;
		} else {
			processed += burst_dequeued;
			for (l = 0; l < burst_dequeued; l++)
				rte_crypto_op_free(proc_ops[l]);
		}
		j++;
	}

	/* Dequeue any operations still in the crypto device */
	while (processed < pparams->total_operations) {
		/* Sending 0 length burst to flush sw crypto device */
		rte_cryptodev_enqueue_burst(dev_id, queue_id, NULL, 0);

		/* dequeue burst */
		burst_dequeued = rte_cryptodev_dequeue_burst(dev_id, queue_id,
				proc_ops, pparams->burst_size);
		if (burst_dequeued == 0)
			failed_polls++;
		else {
			processed += burst_dequeued;
			for (m = 0; m < burst_dequeued; m++)
				rte_crypto_op_free(proc_ops[m]);
		}
	}

	tsc_end = rte_rdtsc_precise();

	double ops_s = ((double)processed / (tsc_end - tsc_start)) * rte_get_tsc_hz();
	double cycles_burst = (double) (tsc_end - tsc_start) /
					(double) processed * pparams->burst_size;
	double cycles_buff = (double) (tsc_end - tsc_start) / (double) processed;
	double cycles_B = cycles_buff / pparams->buf_size;
	double throughput = (ops_s * pparams->buf_size * 8) / 1000000;

	if (gbl_cryptodev_perftest_devtype == RTE_CRYPTODEV_QAT_SYM_PMD) {
		/* Cycle count misleading on HW devices for this test, so don't print */
		printf("%4u\t%6.2f\t%10.2f\t n/a \t\t n/a "
			"\t\t n/a \t\t%8"PRIu64"\t%8"PRIu64,
			pparams->buf_size, ops_s/1000000,
			throughput, retries, failed_polls);
	} else {
		printf("%4u\t%6.2f\t%10.2f\t%10.2f\t%8.2f"
			"\t%8.2f\t%8"PRIu64"\t%8"PRIu64,
			pparams->buf_size, ops_s/1000000, throughput, cycles_burst,
			cycles_buff, cycles_B, retries, failed_polls);
	}

	for (i = 0; i < pparams->burst_size * NUM_MBUF_SETS; i++)
		rte_pktmbuf_free(mbufs[i]);

	printf("\n");
	return TEST_SUCCESS;
}

/*

    perf_test_aes_sha("avx2", HASH_CIPHER, 16, CBC, SHA1);
    perf_test_aes_sha("avx2", HASH_CIPHER, 16, CBC, SHA_256);
    perf_test_aes_sha("avx2", HASH_CIPHER, 16, CBC, SHA_512);

    perf_test_aes_sha("avx2", CIPHER_HASH, 32, CBC, SHA1);
    perf_test_aes_sha("avx2", CIPHER_HASH, 32, CBC, SHA_256);
    perf_test_aes_sha("avx2", CIPHER_HASH, 32, CBC, SHA_512);

    perf_test_aes_sha("avx2", HASH_CIPHER, 32, CBC, SHA1);
    perf_test_aes_sha("avx2", HASH_CIPHER, 32, CBC, SHA_256);
    perf_test_aes_sha("avx2", HASH_CIPHER, 32, CBC, SHA_512);
 */
static int
test_perf_aes_cbc_encrypt_digest_vary_pkt_size(void)
{
	unsigned total_operations = 1000000;
	unsigned burst_size = 32;
	unsigned buf_lengths[] = { 64, 128, 256, 512, 768, 1024, 1280, 1536, 1792, 2048 };
	uint8_t i, j;

	struct perf_test_params params_set[] = {
		{
			.chain = CIPHER_HASH,

			.cipher_algo  = RTE_CRYPTO_CIPHER_AES_CBC,
			.cipher_key_length = 16,
			.auth_algo = RTE_CRYPTO_AUTH_SHA1_HMAC
		},
		{
			.chain = CIPHER_HASH,

			.cipher_algo  = RTE_CRYPTO_CIPHER_AES_CBC,
			.cipher_key_length = 16,
			.auth_algo = RTE_CRYPTO_AUTH_SHA256_HMAC
		},
		{
			.chain = CIPHER_HASH,

			.cipher_algo  = RTE_CRYPTO_CIPHER_AES_CBC,
			.cipher_key_length = 16,
			.auth_algo = RTE_CRYPTO_AUTH_SHA512_HMAC
		},
		{
			.chain = CIPHER_HASH,

			.cipher_algo  = RTE_CRYPTO_CIPHER_AES_CBC,
			.cipher_key_length = 32,
			.auth_algo = RTE_CRYPTO_AUTH_SHA1_HMAC
		},
		{
			.chain = CIPHER_HASH,

			.cipher_algo  = RTE_CRYPTO_CIPHER_AES_CBC,
			.cipher_key_length = 32,
			.auth_algo = RTE_CRYPTO_AUTH_SHA256_HMAC
		},
		{
			.chain = CIPHER_HASH,

			.cipher_algo  = RTE_CRYPTO_CIPHER_AES_CBC,
			.cipher_key_length = 32,
			.auth_algo = RTE_CRYPTO_AUTH_SHA512_HMAC
		},
	};

	for (i = 0; i < RTE_DIM(params_set); i++) {

		params_set[i].total_operations = total_operations;
		params_set[i].burst_size = burst_size;
		printf("\n%s. cipher algo: %s auth algo: %s cipher key size=%u."
				" burst_size: %d ops\n",
				chain_mode_name(params_set[i].chain),
				cipher_algo_name(params_set[i].cipher_algo),
				auth_algo_name(params_set[i].auth_algo),
				params_set[i].cipher_key_length,
				burst_size);
		printf("\nBuffer Size(B)\tOPS(M)\tThroughput(Gbps)\t"
			"Retries\tEmptyPolls\n");
		for (j = 0; j < RTE_DIM(buf_lengths); j++) {
			params_set[i].buf_size = buf_lengths[j];
			test_perf_aes_sha(testsuite_params.dev_id, 0,
					&params_set[i]);
		}
	}
	return 0;
}

static int
test_perf_snow3G_vary_pkt_size(void)
{
	unsigned total_operations = 1000000;
	uint8_t i, j;
	unsigned k;
	uint16_t burst_sizes[] = {64};
	uint16_t buf_lengths[] = {40, 64, 80, 120, 240, 256, 400, 512, 600, 1024, 2048};

	struct perf_test_params params_set[] = {
		{
			.chain = CIPHER_ONLY,
			.cipher_algo  = RTE_CRYPTO_CIPHER_SNOW3G_UEA2,
			.cipher_key_length = 16,
			.auth_algo  = RTE_CRYPTO_AUTH_NULL,
		},
		{
			.chain = HASH_ONLY,
			.cipher_algo = RTE_CRYPTO_CIPHER_NULL,
			.auth_algo  = RTE_CRYPTO_AUTH_SNOW3G_UIA2,
			.cipher_key_length = 16
		},
	};

	printf("\n\nStart %s.", __func__);
	printf("\nTest to measure max throughput at various pkt sizes.");
	printf("\nOn HW devices t'put maximised when high Retries and EmptyPolls"
			" so cycle cost not relevant (n/a displayed).");

	for (i = 0; i < RTE_DIM(params_set); i++) {
		printf("\n\n");
		params_set[i].total_operations = total_operations;
		for (k = 0; k < RTE_DIM(burst_sizes); k++) {
			printf("\nOn %s dev%u qp%u, %s, "
				"cipher algo:%s, auth algo:%s, burst_size: %d ops",
				pmd_name(gbl_cryptodev_perftest_devtype),
				testsuite_params.dev_id, 0,
				chain_mode_name(params_set[i].chain),
				cipher_algo_name(params_set[i].cipher_algo),
				auth_algo_name(params_set[i].auth_algo),
				burst_sizes[k]);

			params_set[i].burst_size = burst_sizes[k];
			printf("\nPktSzB\tOp/s(M)\tThruput(Mbps)\tCycles/Burst\t"
				"Cycles/buf\tCycles/B\tRetries\t\tEmptyPolls\n");
			for (j = 0; j < RTE_DIM(buf_lengths); j++) {

				params_set[i].buf_size = buf_lengths[j];

				test_perf_snow3g(testsuite_params.dev_id, 0, &params_set[i]);
			}
		}
	}

	return 0;
}

static int
test_perf_aes_cbc_vary_burst_size(void)
{
	return test_perf_crypto_qp_vary_burst_size(testsuite_params.dev_id);
}

#if 1
static struct unit_test_suite cryptodev_testsuite  = {
	.suite_name = "Crypto Device Unit Test Suite",
	.setup = testsuite_setup,
	.teardown = testsuite_teardown,
	.unit_test_cases = {
		TEST_CASE_ST(ut_setup, ut_teardown,
				test_perf_aes_cbc_encrypt_digest_vary_pkt_size),
		TEST_CASE_ST(ut_setup, ut_teardown,
				test_perf_aes_cbc_vary_burst_size),
		TEST_CASES_END() /**< NULL terminate unit test array */
	}
};
#endif
static struct unit_test_suite cryptodev_aes_testsuite  = {
	.suite_name = "Crypto Device AESNI MB Unit Test Suite",
	.setup = testsuite_setup,
	.teardown = testsuite_teardown,
	.unit_test_cases = {
		TEST_CASE_ST(ut_setup, ut_teardown,
				test_perf_aes_cbc_encrypt_digest_vary_pkt_size),
		TEST_CASES_END() /**< NULL terminate unit test array */
	}
};

static struct unit_test_suite cryptodev_snow3g_testsuite  = {
	.suite_name = "Crypto Device Snow3G Unit Test Suite",
	.setup = testsuite_setup,
	.teardown = testsuite_teardown,
	.unit_test_cases = {
		TEST_CASE_ST(ut_setup, ut_teardown,
				test_perf_snow3G_vary_pkt_size),
		TEST_CASE_ST(ut_setup, ut_teardown,
				test_perf_snow3G_vary_burst_size),
		TEST_CASES_END() /**< NULL terminate unit test array */
	}
};

static int
perftest_aesni_mb_cryptodev(void /*argv __rte_unused, int argc __rte_unused*/)
{
	gbl_cryptodev_perftest_devtype = RTE_CRYPTODEV_AESNI_MB_PMD;

	return unit_test_suite_runner(&cryptodev_aes_testsuite);
}

static int
perftest_qat_cryptodev(void /*argv __rte_unused, int argc __rte_unused*/)
{
	gbl_cryptodev_perftest_devtype = RTE_CRYPTODEV_QAT_SYM_PMD;

	return unit_test_suite_runner(&cryptodev_testsuite);
}

static int
perftest_sw_snow3g_cryptodev(void /*argv __rte_unused, int argc __rte_unused*/)
{
	gbl_cryptodev_perftest_devtype = RTE_CRYPTODEV_SNOW3G_PMD;

	return unit_test_suite_runner(&cryptodev_snow3g_testsuite);
}

static int
perftest_qat_snow3g_cryptodev(void /*argv __rte_unused, int argc __rte_unused*/)
{
	gbl_cryptodev_perftest_devtype = RTE_CRYPTODEV_QAT_SYM_PMD;

	return unit_test_suite_runner(&cryptodev_snow3g_testsuite);
}

REGISTER_TEST_COMMAND(cryptodev_aesni_mb_perftest, perftest_aesni_mb_cryptodev);
REGISTER_TEST_COMMAND(cryptodev_qat_perftest, perftest_qat_cryptodev);
REGISTER_TEST_COMMAND(cryptodev_sw_snow3g_perftest, perftest_sw_snow3g_cryptodev);
REGISTER_TEST_COMMAND(cryptodev_qat_snow3g_perftest, perftest_qat_snow3g_cryptodev);
