/*
 * Copyright © 2012 Intel Corporation
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice (including the next
 * paragraph) shall be included in all copies or substantial portions of the
 * Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
 * DEALINGS IN THE SOFTWARE.
 */

/**
 * \file indirect_api.cpp
 * Validate the generated code in indirect_init.c
 *
 * Tests various apsects of the dispatch table generated by
 * \c __glXNewIndirectAPI.
 *
 *    * No entry in the table should be \c NULL.
 *
 *    * Entries in the table that correspond to "known" functions with GLX
 *      protocol should point to the correct function.
 *
 *    * Entries beyond the end of the "known" part of the table (i.e., entries
 *      that can be allocated by drivers for extensions) should point to a
 *      no-op function.
 *
 *    * Entries in the table that correspond to "known" functions that lack
 *      GLX protocol should point to a no-op function.
 *
 * Very few entries in the last catogory are tests.  See \c OpenGL_20_is_nop.
 */

#include <gtest/gtest.h>
#include "main/glheader.h"
extern "C" {
#include "../indirect_init.h"
#include "glapi/glapi.h"
#include "../../mesa/main/dispatch.h"
}

static const void *nil = 0;

#define EXTRA_DISPATCH 111

static bool dispatch_table_size_was_queried = false;

extern "C" GLuint
_glapi_get_dispatch_table_size(void)
{
   dispatch_table_size_was_queried = true;
   return _gloffset_COUNT + EXTRA_DISPATCH;
}

/**
 * \name Indirect-rendering function stubs
 *
 * These are necessary so that indirect_init.c doesn't try to link with
 * indirect.c.  Linking with indirect.c would require linking with various X
 * libraries and a bunch of other stuff.  While this is ugly, it does simplify
 * the build quite a bit.
 */
/*@{*/
extern "C" {
void __indirect_glAccum(void) { }
void __indirect_glAlphaFunc(void) { }
void __indirect_glBegin(void) { }
void __indirect_glBitmap(void) { }
void __indirect_glBlendFunc(void) { }
void __indirect_glCallList(void) { }
void __indirect_glCallLists(void) { }
void __indirect_glClear(void) { }
void __indirect_glClearAccum(void) { }
void __indirect_glClearColor(void) { }
void __indirect_glClearDepth(void) { }
void __indirect_glClearIndex(void) { }
void __indirect_glClearStencil(void) { }
void __indirect_glClipPlane(void) { }
void __indirect_glColor3b(void) { }
void __indirect_glColor3bv(void) { }
void __indirect_glColor3d(void) { }
void __indirect_glColor3dv(void) { }
void __indirect_glColor3f(void) { }
void __indirect_glColor3fv(void) { }
void __indirect_glColor3i(void) { }
void __indirect_glColor3iv(void) { }
void __indirect_glColor3s(void) { }
void __indirect_glColor3sv(void) { }
void __indirect_glColor3ub(void) { }
void __indirect_glColor3ubv(void) { }
void __indirect_glColor3ui(void) { }
void __indirect_glColor3uiv(void) { }
void __indirect_glColor3us(void) { }
void __indirect_glColor3usv(void) { }
void __indirect_glColor4b(void) { }
void __indirect_glColor4bv(void) { }
void __indirect_glColor4d(void) { }
void __indirect_glColor4dv(void) { }
void __indirect_glColor4f(void) { }
void __indirect_glColor4fv(void) { }
void __indirect_glColor4i(void) { }
void __indirect_glColor4iv(void) { }
void __indirect_glColor4s(void) { }
void __indirect_glColor4sv(void) { }
void __indirect_glColor4ub(void) { }
void __indirect_glColor4ubv(void) { }
void __indirect_glColor4ui(void) { }
void __indirect_glColor4uiv(void) { }
void __indirect_glColor4us(void) { }
void __indirect_glColor4usv(void) { }
void __indirect_glColorMask(void) { }
void __indirect_glColorMaterial(void) { }
void __indirect_glCopyPixels(void) { }
void __indirect_glCullFace(void) { }
void __indirect_glDeleteLists(void) { }
void __indirect_glDepthFunc(void) { }
void __indirect_glDepthMask(void) { }
void __indirect_glDepthRange(void) { }
void __indirect_glDisable(void) { }
void __indirect_glDrawBuffer(void) { }
void __indirect_glDrawPixels(void) { }
void __indirect_glEdgeFlag(void) { }
void __indirect_glEdgeFlagv(void) { }
void __indirect_glEnable(void) { }
void __indirect_glEnd(void) { }
void __indirect_glEndList(void) { }
void __indirect_glEvalCoord1d(void) { }
void __indirect_glEvalCoord1dv(void) { }
void __indirect_glEvalCoord1f(void) { }
void __indirect_glEvalCoord1fv(void) { }
void __indirect_glEvalCoord2d(void) { }
void __indirect_glEvalCoord2dv(void) { }
void __indirect_glEvalCoord2f(void) { }
void __indirect_glEvalCoord2fv(void) { }
void __indirect_glEvalMesh1(void) { }
void __indirect_glEvalMesh2(void) { }
void __indirect_glEvalPoint1(void) { }
void __indirect_glEvalPoint2(void) { }
void __indirect_glFeedbackBuffer(void) { }
void __indirect_glFinish(void) { }
void __indirect_glFlush(void) { }
void __indirect_glFogf(void) { }
void __indirect_glFogfv(void) { }
void __indirect_glFogi(void) { }
void __indirect_glFogiv(void) { }
void __indirect_glFrontFace(void) { }
void __indirect_glFrustum(void) { }
void __indirect_glGenLists(void) { }
void __indirect_glGetBooleanv(void) { }
void __indirect_glGetClipPlane(void) { }
void __indirect_glGetDoublev(void) { }
void __indirect_glGetError(void) { }
void __indirect_glGetFloatv(void) { }
void __indirect_glGetIntegerv(void) { }
void __indirect_glGetLightfv(void) { }
void __indirect_glGetLightiv(void) { }
void __indirect_glGetMapdv(void) { }
void __indirect_glGetMapfv(void) { }
void __indirect_glGetMapiv(void) { }
void __indirect_glGetMaterialfv(void) { }
void __indirect_glGetMaterialiv(void) { }
void __indirect_glGetPixelMapfv(void) { }
void __indirect_glGetPixelMapuiv(void) { }
void __indirect_glGetPixelMapusv(void) { }
void __indirect_glGetPolygonStipple(void) { }
void __indirect_glGetString(void) { }
void __indirect_glGetTexEnvfv(void) { }
void __indirect_glGetTexEnviv(void) { }
void __indirect_glGetTexGendv(void) { }
void __indirect_glGetTexGenfv(void) { }
void __indirect_glGetTexGeniv(void) { }
void __indirect_glGetTexImage(void) { }
void __indirect_glGetTexLevelParameterfv(void) { }
void __indirect_glGetTexLevelParameteriv(void) { }
void __indirect_glGetTexParameterfv(void) { }
void __indirect_glGetTexParameteriv(void) { }
void __indirect_glHint(void) { }
void __indirect_glIndexMask(void) { }
void __indirect_glIndexd(void) { }
void __indirect_glIndexdv(void) { }
void __indirect_glIndexf(void) { }
void __indirect_glIndexfv(void) { }
void __indirect_glIndexi(void) { }
void __indirect_glIndexiv(void) { }
void __indirect_glIndexs(void) { }
void __indirect_glIndexsv(void) { }
void __indirect_glInitNames(void) { }
void __indirect_glIsEnabled(void) { }
void __indirect_glIsList(void) { }
void __indirect_glLightModelf(void) { }
void __indirect_glLightModelfv(void) { }
void __indirect_glLightModeli(void) { }
void __indirect_glLightModeliv(void) { }
void __indirect_glLightf(void) { }
void __indirect_glLightfv(void) { }
void __indirect_glLighti(void) { }
void __indirect_glLightiv(void) { }
void __indirect_glLineStipple(void) { }
void __indirect_glLineWidth(void) { }
void __indirect_glListBase(void) { }
void __indirect_glLoadIdentity(void) { }
void __indirect_glLoadMatrixd(void) { }
void __indirect_glLoadMatrixf(void) { }
void __indirect_glLoadName(void) { }
void __indirect_glLogicOp(void) { }
void __indirect_glMap1d(void) { }
void __indirect_glMap1f(void) { }
void __indirect_glMap2d(void) { }
void __indirect_glMap2f(void) { }
void __indirect_glMapGrid1d(void) { }
void __indirect_glMapGrid1f(void) { }
void __indirect_glMapGrid2d(void) { }
void __indirect_glMapGrid2f(void) { }
void __indirect_glMaterialf(void) { }
void __indirect_glMaterialfv(void) { }
void __indirect_glMateriali(void) { }
void __indirect_glMaterialiv(void) { }
void __indirect_glMatrixMode(void) { }
void __indirect_glMultMatrixd(void) { }
void __indirect_glMultMatrixf(void) { }
void __indirect_glNewList(void) { }
void __indirect_glNormal3b(void) { }
void __indirect_glNormal3bv(void) { }
void __indirect_glNormal3d(void) { }
void __indirect_glNormal3dv(void) { }
void __indirect_glNormal3f(void) { }
void __indirect_glNormal3fv(void) { }
void __indirect_glNormal3i(void) { }
void __indirect_glNormal3iv(void) { }
void __indirect_glNormal3s(void) { }
void __indirect_glNormal3sv(void) { }
void __indirect_glOrtho(void) { }
void __indirect_glPassThrough(void) { }
void __indirect_glPixelMapfv(void) { }
void __indirect_glPixelMapuiv(void) { }
void __indirect_glPixelMapusv(void) { }
void __indirect_glPixelStoref(void) { }
void __indirect_glPixelStorei(void) { }
void __indirect_glPixelTransferf(void) { }
void __indirect_glPixelTransferi(void) { }
void __indirect_glPixelZoom(void) { }
void __indirect_glPointSize(void) { }
void __indirect_glPolygonMode(void) { }
void __indirect_glPolygonStipple(void) { }
void __indirect_glPopAttrib(void) { }
void __indirect_glPopMatrix(void) { }
void __indirect_glPopName(void) { }
void __indirect_glPushAttrib(void) { }
void __indirect_glPushMatrix(void) { }
void __indirect_glPushName(void) { }
void __indirect_glRasterPos2d(void) { }
void __indirect_glRasterPos2dv(void) { }
void __indirect_glRasterPos2f(void) { }
void __indirect_glRasterPos2fv(void) { }
void __indirect_glRasterPos2i(void) { }
void __indirect_glRasterPos2iv(void) { }
void __indirect_glRasterPos2s(void) { }
void __indirect_glRasterPos2sv(void) { }
void __indirect_glRasterPos3d(void) { }
void __indirect_glRasterPos3dv(void) { }
void __indirect_glRasterPos3f(void) { }
void __indirect_glRasterPos3fv(void) { }
void __indirect_glRasterPos3i(void) { }
void __indirect_glRasterPos3iv(void) { }
void __indirect_glRasterPos3s(void) { }
void __indirect_glRasterPos3sv(void) { }
void __indirect_glRasterPos4d(void) { }
void __indirect_glRasterPos4dv(void) { }
void __indirect_glRasterPos4f(void) { }
void __indirect_glRasterPos4fv(void) { }
void __indirect_glRasterPos4i(void) { }
void __indirect_glRasterPos4iv(void) { }
void __indirect_glRasterPos4s(void) { }
void __indirect_glRasterPos4sv(void) { }
void __indirect_glReadBuffer(void) { }
void __indirect_glReadPixels(void) { }
void __indirect_glRectd(void) { }
void __indirect_glRectdv(void) { }
void __indirect_glRectf(void) { }
void __indirect_glRectfv(void) { }
void __indirect_glRecti(void) { }
void __indirect_glRectiv(void) { }
void __indirect_glRects(void) { }
void __indirect_glRectsv(void) { }
void __indirect_glRenderMode(void) { }
void __indirect_glRotated(void) { }
void __indirect_glRotatef(void) { }
void __indirect_glScaled(void) { }
void __indirect_glScalef(void) { }
void __indirect_glScissor(void) { }
void __indirect_glSelectBuffer(void) { }
void __indirect_glShadeModel(void) { }
void __indirect_glStencilFunc(void) { }
void __indirect_glStencilMask(void) { }
void __indirect_glStencilOp(void) { }
void __indirect_glTexCoord1d(void) { }
void __indirect_glTexCoord1dv(void) { }
void __indirect_glTexCoord1f(void) { }
void __indirect_glTexCoord1fv(void) { }
void __indirect_glTexCoord1i(void) { }
void __indirect_glTexCoord1iv(void) { }
void __indirect_glTexCoord1s(void) { }
void __indirect_glTexCoord1sv(void) { }
void __indirect_glTexCoord2d(void) { }
void __indirect_glTexCoord2dv(void) { }
void __indirect_glTexCoord2f(void) { }
void __indirect_glTexCoord2fv(void) { }
void __indirect_glTexCoord2i(void) { }
void __indirect_glTexCoord2iv(void) { }
void __indirect_glTexCoord2s(void) { }
void __indirect_glTexCoord2sv(void) { }
void __indirect_glTexCoord3d(void) { }
void __indirect_glTexCoord3dv(void) { }
void __indirect_glTexCoord3f(void) { }
void __indirect_glTexCoord3fv(void) { }
void __indirect_glTexCoord3i(void) { }
void __indirect_glTexCoord3iv(void) { }
void __indirect_glTexCoord3s(void) { }
void __indirect_glTexCoord3sv(void) { }
void __indirect_glTexCoord4d(void) { }
void __indirect_glTexCoord4dv(void) { }
void __indirect_glTexCoord4f(void) { }
void __indirect_glTexCoord4fv(void) { }
void __indirect_glTexCoord4i(void) { }
void __indirect_glTexCoord4iv(void) { }
void __indirect_glTexCoord4s(void) { }
void __indirect_glTexCoord4sv(void) { }
void __indirect_glTexEnvf(void) { }
void __indirect_glTexEnvfv(void) { }
void __indirect_glTexEnvi(void) { }
void __indirect_glTexEnviv(void) { }
void __indirect_glTexGend(void) { }
void __indirect_glTexGendv(void) { }
void __indirect_glTexGenf(void) { }
void __indirect_glTexGenfv(void) { }
void __indirect_glTexGeni(void) { }
void __indirect_glTexGeniv(void) { }
void __indirect_glTexImage1D(void) { }
void __indirect_glTexImage2D(void) { }
void __indirect_glTexParameterf(void) { }
void __indirect_glTexParameterfv(void) { }
void __indirect_glTexParameteri(void) { }
void __indirect_glTexParameteriv(void) { }
void __indirect_glTranslated(void) { }
void __indirect_glTranslatef(void) { }
void __indirect_glVertex2d(void) { }
void __indirect_glVertex2dv(void) { }
void __indirect_glVertex2f(void) { }
void __indirect_glVertex2fv(void) { }
void __indirect_glVertex2i(void) { }
void __indirect_glVertex2iv(void) { }
void __indirect_glVertex2s(void) { }
void __indirect_glVertex2sv(void) { }
void __indirect_glVertex3d(void) { }
void __indirect_glVertex3dv(void) { }
void __indirect_glVertex3f(void) { }
void __indirect_glVertex3fv(void) { }
void __indirect_glVertex3i(void) { }
void __indirect_glVertex3iv(void) { }
void __indirect_glVertex3s(void) { }
void __indirect_glVertex3sv(void) { }
void __indirect_glVertex4d(void) { }
void __indirect_glVertex4dv(void) { }
void __indirect_glVertex4f(void) { }
void __indirect_glVertex4fv(void) { }
void __indirect_glVertex4i(void) { }
void __indirect_glVertex4iv(void) { }
void __indirect_glVertex4s(void) { }
void __indirect_glVertex4sv(void) { }
void __indirect_glViewport(void) { }
void __indirect_glAreTexturesResident(void) { }
void __indirect_glArrayElement(void) { }
void __indirect_glBindTexture(void) { }
void __indirect_glColorPointer(void) { }
void __indirect_glCopyTexImage1D(void) { }
void __indirect_glCopyTexImage2D(void) { }
void __indirect_glCopyTexSubImage1D(void) { }
void __indirect_glCopyTexSubImage2D(void) { }
void __indirect_glDeleteTextures(void) { }
void __indirect_glDisableClientState(void) { }
void __indirect_glDrawArrays(void) { }
void __indirect_glDrawElements(void) { }
void __indirect_glEdgeFlagPointer(void) { }
void __indirect_glEnableClientState(void) { }
void __indirect_glGenTextures(void) { }
void __indirect_glGetPointerv(void) { }
void __indirect_glIndexPointer(void) { }
void __indirect_glIndexub(void) { }
void __indirect_glIndexubv(void) { }
void __indirect_glInterleavedArrays(void) { }
void __indirect_glIsTexture(void) { }
void __indirect_glNormalPointer(void) { }
void __indirect_glPolygonOffset(void) { }
void __indirect_glPopClientAttrib(void) { }
void __indirect_glPrioritizeTextures(void) { }
void __indirect_glPushClientAttrib(void) { }
void __indirect_glTexCoordPointer(void) { }
void __indirect_glTexSubImage1D(void) { }
void __indirect_glTexSubImage2D(void) { }
void __indirect_glVertexPointer(void) { }
void __indirect_glBlendColor(void) { }
void __indirect_glBlendEquation(void) { }
void __indirect_glColorSubTable(void) { }
void __indirect_glColorTable(void) { }
void __indirect_glColorTableParameterfv(void) { }
void __indirect_glColorTableParameteriv(void) { }
void __indirect_glConvolutionFilter1D(void) { }
void __indirect_glConvolutionFilter2D(void) { }
void __indirect_glConvolutionParameterf(void) { }
void __indirect_glConvolutionParameterfv(void) { }
void __indirect_glConvolutionParameteri(void) { }
void __indirect_glConvolutionParameteriv(void) { }
void __indirect_glCopyColorSubTable(void) { }
void __indirect_glCopyColorTable(void) { }
void __indirect_glCopyConvolutionFilter1D(void) { }
void __indirect_glCopyConvolutionFilter2D(void) { }
void __indirect_glCopyTexSubImage3D(void) { }
void __indirect_glDrawRangeElements(void) { }
void __indirect_glGetColorTable(void) { }
void __indirect_glGetColorTableParameterfv(void) { }
void __indirect_glGetColorTableParameteriv(void) { }
void __indirect_glGetConvolutionFilter(void) { }
void __indirect_glGetConvolutionParameterfv(void) { }
void __indirect_glGetConvolutionParameteriv(void) { }
void __indirect_glGetHistogram(void) { }
void __indirect_glGetHistogramParameterfv(void) { }
void __indirect_glGetHistogramParameteriv(void) { }
void __indirect_glGetMinmax(void) { }
void __indirect_glGetMinmaxParameterfv(void) { }
void __indirect_glGetMinmaxParameteriv(void) { }
void __indirect_glGetSeparableFilter(void) { }
void __indirect_glHistogram(void) { }
void __indirect_glMinmax(void) { }
void __indirect_glResetHistogram(void) { }
void __indirect_glResetMinmax(void) { }
void __indirect_glSeparableFilter2D(void) { }
void __indirect_glTexImage3D(void) { }
void __indirect_glTexSubImage3D(void) { }
void __indirect_glActiveTexture(void) { }
void __indirect_glClientActiveTexture(void) { }
void __indirect_glMultiTexCoord1d(void) { }
void __indirect_glMultiTexCoord1dv(void) { }
void __indirect_glMultiTexCoord1fARB(void) { }
void __indirect_glMultiTexCoord1fvARB(void) { }
void __indirect_glMultiTexCoord1i(void) { }
void __indirect_glMultiTexCoord1iv(void) { }
void __indirect_glMultiTexCoord1s(void) { }
void __indirect_glMultiTexCoord1sv(void) { }
void __indirect_glMultiTexCoord2d(void) { }
void __indirect_glMultiTexCoord2dv(void) { }
void __indirect_glMultiTexCoord2fARB(void) { }
void __indirect_glMultiTexCoord2fvARB(void) { }
void __indirect_glMultiTexCoord2i(void) { }
void __indirect_glMultiTexCoord2iv(void) { }
void __indirect_glMultiTexCoord2s(void) { }
void __indirect_glMultiTexCoord2sv(void) { }
void __indirect_glMultiTexCoord3d(void) { }
void __indirect_glMultiTexCoord3dv(void) { }
void __indirect_glMultiTexCoord3fARB(void) { }
void __indirect_glMultiTexCoord3fvARB(void) { }
void __indirect_glMultiTexCoord3i(void) { }
void __indirect_glMultiTexCoord3iv(void) { }
void __indirect_glMultiTexCoord3s(void) { }
void __indirect_glMultiTexCoord3sv(void) { }
void __indirect_glMultiTexCoord4d(void) { }
void __indirect_glMultiTexCoord4dv(void) { }
void __indirect_glMultiTexCoord4fARB(void) { }
void __indirect_glMultiTexCoord4fvARB(void) { }
void __indirect_glMultiTexCoord4i(void) { }
void __indirect_glMultiTexCoord4iv(void) { }
void __indirect_glMultiTexCoord4s(void) { }
void __indirect_glMultiTexCoord4sv(void) { }
void __indirect_glLoadTransposeMatrixd(void) { }
void __indirect_glLoadTransposeMatrixf(void) { }
void __indirect_glMultTransposeMatrixd(void) { }
void __indirect_glMultTransposeMatrixf(void) { }
void __indirect_glSampleCoverage(void) { }
void __indirect_glCompressedTexImage1D(void) { }
void __indirect_glCompressedTexImage2D(void) { }
void __indirect_glCompressedTexImage3D(void) { }
void __indirect_glCompressedTexSubImage1D(void) { }
void __indirect_glCompressedTexSubImage2D(void) { }
void __indirect_glCompressedTexSubImage3D(void) { }
void __indirect_glGetCompressedTexImage(void) { }
void __indirect_glDisableVertexAttribArray(void) { }
void __indirect_glEnableVertexAttribArray(void) { }
void __indirect_glGetProgramEnvParameterdvARB(void) { }
void __indirect_glGetProgramEnvParameterfvARB(void) { }
void __indirect_glGetProgramLocalParameterdvARB(void) { }
void __indirect_glGetProgramLocalParameterfvARB(void) { }
void __indirect_glGetProgramStringARB(void) { }
void __indirect_glGetProgramivARB(void) { }
void __indirect_glGetVertexAttribdv(void) { }
void __indirect_glGetVertexAttribfv(void) { }
void __indirect_glGetVertexAttribiv(void) { }
void __indirect_glProgramEnvParameter4dARB(void) { }
void __indirect_glProgramEnvParameter4dvARB(void) { }
void __indirect_glProgramEnvParameter4fARB(void) { }
void __indirect_glProgramEnvParameter4fvARB(void) { }
void __indirect_glProgramLocalParameter4dARB(void) { }
void __indirect_glProgramLocalParameter4dvARB(void) { }
void __indirect_glProgramLocalParameter4fARB(void) { }
void __indirect_glProgramLocalParameter4fvARB(void) { }
void __indirect_glProgramStringARB(void) { }
void __indirect_glVertexAttrib1d(void) { }
void __indirect_glVertexAttrib1dv(void) { }
void __indirect_glVertexAttrib1fARB(void) { }
void __indirect_glVertexAttrib1fvARB(void) { }
void __indirect_glVertexAttrib1s(void) { }
void __indirect_glVertexAttrib1sv(void) { }
void __indirect_glVertexAttrib2d(void) { }
void __indirect_glVertexAttrib2dv(void) { }
void __indirect_glVertexAttrib2fARB(void) { }
void __indirect_glVertexAttrib2fvARB(void) { }
void __indirect_glVertexAttrib2s(void) { }
void __indirect_glVertexAttrib2sv(void) { }
void __indirect_glVertexAttrib3d(void) { }
void __indirect_glVertexAttrib3dv(void) { }
void __indirect_glVertexAttrib3fARB(void) { }
void __indirect_glVertexAttrib3fvARB(void) { }
void __indirect_glVertexAttrib3s(void) { }
void __indirect_glVertexAttrib3sv(void) { }
void __indirect_glVertexAttrib4Nbv(void) { }
void __indirect_glVertexAttrib4Niv(void) { }
void __indirect_glVertexAttrib4Nsv(void) { }
void __indirect_glVertexAttrib4Nub(void) { }
void __indirect_glVertexAttrib4Nubv(void) { }
void __indirect_glVertexAttrib4Nuiv(void) { }
void __indirect_glVertexAttrib4Nusv(void) { }
void __indirect_glVertexAttrib4bv(void) { }
void __indirect_glVertexAttrib4d(void) { }
void __indirect_glVertexAttrib4dv(void) { }
void __indirect_glVertexAttrib4fARB(void) { }
void __indirect_glVertexAttrib4fvARB(void) { }
void __indirect_glVertexAttrib4iv(void) { }
void __indirect_glVertexAttrib4s(void) { }
void __indirect_glVertexAttrib4sv(void) { }
void __indirect_glVertexAttrib4ubv(void) { }
void __indirect_glVertexAttrib4uiv(void) { }
void __indirect_glVertexAttrib4usv(void) { }
void __indirect_glVertexAttribPointer(void) { }
void __indirect_glBeginQuery(void) { }
void __indirect_glDeleteQueries(void) { }
void __indirect_glEndQuery(void) { }
void __indirect_glGenQueries(void) { }
void __indirect_glGetQueryObjectiv(void) { }
void __indirect_glGetQueryObjectuiv(void) { }
void __indirect_glGetQueryiv(void) { }
void __indirect_glIsQuery(void) { }
void __indirect_glDrawBuffers(void) { }
void __indirect_glClampColor(void) { }
void __indirect_glRenderbufferStorageMultisample(void) { }
void __indirect_glSampleMaskSGIS(void) { }
void __indirect_glSamplePatternSGIS(void) { }
void __indirect_glColorPointerEXT(void) { }
void __indirect_glEdgeFlagPointerEXT(void) { }
void __indirect_glIndexPointerEXT(void) { }
void __indirect_glNormalPointerEXT(void) { }
void __indirect_glTexCoordPointerEXT(void) { }
void __indirect_glVertexPointerEXT(void) { }
void __indirect_glPointParameterf(void) { }
void __indirect_glPointParameterfv(void) { }
void __indirect_glSecondaryColor3b(void) { }
void __indirect_glSecondaryColor3bv(void) { }
void __indirect_glSecondaryColor3d(void) { }
void __indirect_glSecondaryColor3dv(void) { }
void __indirect_glSecondaryColor3fEXT(void) { }
void __indirect_glSecondaryColor3fvEXT(void) { }
void __indirect_glSecondaryColor3i(void) { }
void __indirect_glSecondaryColor3iv(void) { }
void __indirect_glSecondaryColor3s(void) { }
void __indirect_glSecondaryColor3sv(void) { }
void __indirect_glSecondaryColor3ub(void) { }
void __indirect_glSecondaryColor3ubv(void) { }
void __indirect_glSecondaryColor3ui(void) { }
void __indirect_glSecondaryColor3uiv(void) { }
void __indirect_glSecondaryColor3us(void) { }
void __indirect_glSecondaryColor3usv(void) { }
void __indirect_glSecondaryColorPointer(void) { }
void __indirect_glMultiDrawArrays(void) { }
void __indirect_glMultiDrawElementsEXT(void) { }
void __indirect_glFogCoordPointer(void) { }
void __indirect_glFogCoordd(void) { }
void __indirect_glFogCoorddv(void) { }
void __indirect_glFogCoordfEXT(void) { }
void __indirect_glFogCoordfvEXT(void) { }
void __indirect_glBlendFuncSeparate(void) { }
void __indirect_glWindowPos2d(void) { }
void __indirect_glWindowPos2dv(void) { }
void __indirect_glWindowPos2f(void) { }
void __indirect_glWindowPos2fv(void) { }
void __indirect_glWindowPos2i(void) { }
void __indirect_glWindowPos2iv(void) { }
void __indirect_glWindowPos2s(void) { }
void __indirect_glWindowPos2sv(void) { }
void __indirect_glWindowPos3d(void) { }
void __indirect_glWindowPos3dv(void) { }
void __indirect_glWindowPos3f(void) { }
void __indirect_glWindowPos3fv(void) { }
void __indirect_glWindowPos3i(void) { }
void __indirect_glWindowPos3iv(void) { }
void __indirect_glWindowPos3s(void) { }
void __indirect_glWindowPos3sv(void) { }
void __indirect_glAreProgramsResidentNV(void) { }
void __indirect_glBindProgramARB(void) { }
void __indirect_glDeleteProgramsARB(void) { }
void __indirect_glExecuteProgramNV(void) { }
void __indirect_glGenProgramsARB(void) { }
void __indirect_glGetProgramParameterdvNV(void) { }
void __indirect_glGetProgramParameterfvNV(void) { }
void __indirect_glGetProgramStringNV(void) { }
void __indirect_glGetProgramivNV(void) { }
void __indirect_glGetTrackMatrixivNV(void) { }
void __indirect_glGetVertexAttribPointerv(void) { }
void __indirect_glGetVertexAttribdvNV(void) { }
void __indirect_glGetVertexAttribfvNV(void) { }
void __indirect_glGetVertexAttribivNV(void) { }
void __indirect_glIsProgramARB(void) { }
void __indirect_glLoadProgramNV(void) { }
void __indirect_glProgramParameters4dvNV(void) { }
void __indirect_glProgramParameters4fvNV(void) { }
void __indirect_glRequestResidentProgramsNV(void) { }
void __indirect_glTrackMatrixNV(void) { }
void __indirect_glVertexAttrib1dNV(void) { }
void __indirect_glVertexAttrib1dvNV(void) { }
void __indirect_glVertexAttrib1fNV(void) { }
void __indirect_glVertexAttrib1fvNV(void) { }
void __indirect_glVertexAttrib1sNV(void) { }
void __indirect_glVertexAttrib1svNV(void) { }
void __indirect_glVertexAttrib2dNV(void) { }
void __indirect_glVertexAttrib2dvNV(void) { }
void __indirect_glVertexAttrib2fNV(void) { }
void __indirect_glVertexAttrib2fvNV(void) { }
void __indirect_glVertexAttrib2sNV(void) { }
void __indirect_glVertexAttrib2svNV(void) { }
void __indirect_glVertexAttrib3dNV(void) { }
void __indirect_glVertexAttrib3dvNV(void) { }
void __indirect_glVertexAttrib3fNV(void) { }
void __indirect_glVertexAttrib3fvNV(void) { }
void __indirect_glVertexAttrib3sNV(void) { }
void __indirect_glVertexAttrib3svNV(void) { }
void __indirect_glVertexAttrib4dNV(void) { }
void __indirect_glVertexAttrib4dvNV(void) { }
void __indirect_glVertexAttrib4fNV(void) { }
void __indirect_glVertexAttrib4fvNV(void) { }
void __indirect_glVertexAttrib4sNV(void) { }
void __indirect_glVertexAttrib4svNV(void) { }
void __indirect_glVertexAttrib4ubNV(void) { }
void __indirect_glVertexAttrib4ubvNV(void) { }
void __indirect_glVertexAttribPointerNV(void) { }
void __indirect_glVertexAttribs1dvNV(void) { }
void __indirect_glVertexAttribs1fvNV(void) { }
void __indirect_glVertexAttribs1svNV(void) { }
void __indirect_glVertexAttribs2dvNV(void) { }
void __indirect_glVertexAttribs2fvNV(void) { }
void __indirect_glVertexAttribs2svNV(void) { }
void __indirect_glVertexAttribs3dvNV(void) { }
void __indirect_glVertexAttribs3fvNV(void) { }
void __indirect_glVertexAttribs3svNV(void) { }
void __indirect_glVertexAttribs4dvNV(void) { }
void __indirect_glVertexAttribs4fvNV(void) { }
void __indirect_glVertexAttribs4svNV(void) { }
void __indirect_glVertexAttribs4ubvNV(void) { }
void __indirect_glPointParameteri(void) { }
void __indirect_glPointParameteriv(void) { }
void __indirect_glActiveStencilFaceEXT(void) { }
void __indirect_glGetProgramNamedParameterdvNV(void) { }
void __indirect_glGetProgramNamedParameterfvNV(void) { }
void __indirect_glProgramNamedParameter4dNV(void) { }
void __indirect_glProgramNamedParameter4dvNV(void) { }
void __indirect_glProgramNamedParameter4fNV(void) { }
void __indirect_glProgramNamedParameter4fvNV(void) { }
void __indirect_glBlendEquationSeparate(void) { }
void __indirect_glBindFramebufferEXT(void) { }
void __indirect_glBindRenderbufferEXT(void) { }
void __indirect_glBindFramebuffer(void) { }
void __indirect_glBindRenderbuffer(void) { }
void __indirect_glCheckFramebufferStatus(void) { }
void __indirect_glDeleteFramebuffers(void) { }
void __indirect_glDeleteRenderbuffers(void) { }
void __indirect_glFramebufferRenderbuffer(void) { }
void __indirect_glFramebufferTexture1D(void) { }
void __indirect_glFramebufferTexture2D(void) { }
void __indirect_glFramebufferTexture3D(void) { }
void __indirect_glGenFramebuffers(void) { }
void __indirect_glGenRenderbuffers(void) { }
void __indirect_glGenerateMipmap(void) { }
void __indirect_glGetFramebufferAttachmentParameteriv(void) { }
void __indirect_glGetRenderbufferParameteriv(void) { }
void __indirect_glIsFramebuffer(void) { }
void __indirect_glIsRenderbuffer(void) { }
void __indirect_glRenderbufferStorage(void) { }
void __indirect_glBlitFramebuffer(void) { }
void __indirect_glFramebufferTextureLayer(void) { }
}
/*@}*/

class IndirectAPI : public ::testing::Test {
public:
   virtual void SetUp();
   virtual void TearDown();

   _glapi_proc *table;
};

void
IndirectAPI::SetUp()
{
   this->table = (_glapi_proc *) __glXNewIndirectAPI();
}

void
IndirectAPI::TearDown()
{
   free(this->table);
   this->table = NULL;
}

TEST_F(IndirectAPI, DispatchTableSizeWasQueried)
{
   dispatch_table_size_was_queried = false;

   /* __glXNewIndirectAPI determines the size of the dispatch table by calling
    * _glapi_get_dispatch_table_size.  Our stub version of this function sets
    * a flag so that we can detect that __glXNewIndirectAPI called it.
    */
   EXPECT_FALSE(dispatch_table_size_was_queried);
   _glapi_table *table = __glXNewIndirectAPI();
   EXPECT_TRUE(dispatch_table_size_was_queried);

   /* Since a second dispatch table was created, it must be freed by hand.
    */
   free(table);
}

TEST_F(IndirectAPI, NoNullEntries)
{
   const unsigned entries = _glapi_get_dispatch_table_size();

   /* There should not be any NULL entries in the dispatch table.  All entires
    * should either point to a real implementation or to a no-op function.
    */
   for (unsigned i = 0; i < entries; i++) {
      EXPECT_NE(nil, table[i]);
   }
}

TEST_F(IndirectAPI, OpenGL_10)
{
   EXPECT_EQ((_glapi_proc) __indirect_glAccum, table[_gloffset_Accum]);
   EXPECT_EQ((_glapi_proc) __indirect_glAlphaFunc, table[_gloffset_AlphaFunc]);
   EXPECT_EQ((_glapi_proc) __indirect_glBegin, table[_gloffset_Begin]);
   EXPECT_EQ((_glapi_proc) __indirect_glBitmap, table[_gloffset_Bitmap]);
   EXPECT_EQ((_glapi_proc) __indirect_glBlendFunc, table[_gloffset_BlendFunc]);
   EXPECT_EQ((_glapi_proc) __indirect_glCallList, table[_gloffset_CallList]);
   EXPECT_EQ((_glapi_proc) __indirect_glCallLists, table[_gloffset_CallLists]);
   EXPECT_EQ((_glapi_proc) __indirect_glClear, table[_gloffset_Clear]);
   EXPECT_EQ((_glapi_proc) __indirect_glClearAccum, table[_gloffset_ClearAccum]);
   EXPECT_EQ((_glapi_proc) __indirect_glClearColor, table[_gloffset_ClearColor]);
   EXPECT_EQ((_glapi_proc) __indirect_glClearDepth, table[_gloffset_ClearDepth]);
   EXPECT_EQ((_glapi_proc) __indirect_glClearIndex, table[_gloffset_ClearIndex]);
   EXPECT_EQ((_glapi_proc) __indirect_glClearStencil, table[_gloffset_ClearStencil]);
   EXPECT_EQ((_glapi_proc) __indirect_glClipPlane, table[_gloffset_ClipPlane]);
   EXPECT_EQ((_glapi_proc) __indirect_glColor3b, table[_gloffset_Color3b]);
   EXPECT_EQ((_glapi_proc) __indirect_glColor3bv, table[_gloffset_Color3bv]);
   EXPECT_EQ((_glapi_proc) __indirect_glColor3d, table[_gloffset_Color3d]);
   EXPECT_EQ((_glapi_proc) __indirect_glColor3dv, table[_gloffset_Color3dv]);
   EXPECT_EQ((_glapi_proc) __indirect_glColor3f, table[_gloffset_Color3f]);
   EXPECT_EQ((_glapi_proc) __indirect_glColor3fv, table[_gloffset_Color3fv]);
   EXPECT_EQ((_glapi_proc) __indirect_glColor3i, table[_gloffset_Color3i]);
   EXPECT_EQ((_glapi_proc) __indirect_glColor3iv, table[_gloffset_Color3iv]);
   EXPECT_EQ((_glapi_proc) __indirect_glColor3s, table[_gloffset_Color3s]);
   EXPECT_EQ((_glapi_proc) __indirect_glColor3sv, table[_gloffset_Color3sv]);
   EXPECT_EQ((_glapi_proc) __indirect_glColor3ub, table[_gloffset_Color3ub]);
   EXPECT_EQ((_glapi_proc) __indirect_glColor3ubv, table[_gloffset_Color3ubv]);
   EXPECT_EQ((_glapi_proc) __indirect_glColor3ui, table[_gloffset_Color3ui]);
   EXPECT_EQ((_glapi_proc) __indirect_glColor3uiv, table[_gloffset_Color3uiv]);
   EXPECT_EQ((_glapi_proc) __indirect_glColor3us, table[_gloffset_Color3us]);
   EXPECT_EQ((_glapi_proc) __indirect_glColor3usv, table[_gloffset_Color3usv]);
   EXPECT_EQ((_glapi_proc) __indirect_glColor4b, table[_gloffset_Color4b]);
   EXPECT_EQ((_glapi_proc) __indirect_glColor4bv, table[_gloffset_Color4bv]);
   EXPECT_EQ((_glapi_proc) __indirect_glColor4d, table[_gloffset_Color4d]);
   EXPECT_EQ((_glapi_proc) __indirect_glColor4dv, table[_gloffset_Color4dv]);
   EXPECT_EQ((_glapi_proc) __indirect_glColor4f, table[_gloffset_Color4f]);
   EXPECT_EQ((_glapi_proc) __indirect_glColor4fv, table[_gloffset_Color4fv]);
   EXPECT_EQ((_glapi_proc) __indirect_glColor4i, table[_gloffset_Color4i]);
   EXPECT_EQ((_glapi_proc) __indirect_glColor4iv, table[_gloffset_Color4iv]);
   EXPECT_EQ((_glapi_proc) __indirect_glColor4s, table[_gloffset_Color4s]);
   EXPECT_EQ((_glapi_proc) __indirect_glColor4sv, table[_gloffset_Color4sv]);
   EXPECT_EQ((_glapi_proc) __indirect_glColor4ub, table[_gloffset_Color4ub]);
   EXPECT_EQ((_glapi_proc) __indirect_glColor4ubv, table[_gloffset_Color4ubv]);
   EXPECT_EQ((_glapi_proc) __indirect_glColor4ui, table[_gloffset_Color4ui]);
   EXPECT_EQ((_glapi_proc) __indirect_glColor4uiv, table[_gloffset_Color4uiv]);
   EXPECT_EQ((_glapi_proc) __indirect_glColor4us, table[_gloffset_Color4us]);
   EXPECT_EQ((_glapi_proc) __indirect_glColor4usv, table[_gloffset_Color4usv]);
   EXPECT_EQ((_glapi_proc) __indirect_glColorMask, table[_gloffset_ColorMask]);
   EXPECT_EQ((_glapi_proc) __indirect_glColorMaterial, table[_gloffset_ColorMaterial]);
   EXPECT_EQ((_glapi_proc) __indirect_glCopyPixels, table[_gloffset_CopyPixels]);
   EXPECT_EQ((_glapi_proc) __indirect_glCullFace, table[_gloffset_CullFace]);
   EXPECT_EQ((_glapi_proc) __indirect_glDeleteLists, table[_gloffset_DeleteLists]);
   EXPECT_EQ((_glapi_proc) __indirect_glDepthFunc, table[_gloffset_DepthFunc]);
   EXPECT_EQ((_glapi_proc) __indirect_glDepthMask, table[_gloffset_DepthMask]);
   EXPECT_EQ((_glapi_proc) __indirect_glDepthRange, table[_gloffset_DepthRange]);
   EXPECT_EQ((_glapi_proc) __indirect_glDisable, table[_gloffset_Disable]);
   EXPECT_EQ((_glapi_proc) __indirect_glDrawBuffer, table[_gloffset_DrawBuffer]);
   EXPECT_EQ((_glapi_proc) __indirect_glDrawPixels, table[_gloffset_DrawPixels]);
   EXPECT_EQ((_glapi_proc) __indirect_glEdgeFlag, table[_gloffset_EdgeFlag]);
   EXPECT_EQ((_glapi_proc) __indirect_glEdgeFlagv, table[_gloffset_EdgeFlagv]);
   EXPECT_EQ((_glapi_proc) __indirect_glEnable, table[_gloffset_Enable]);
   EXPECT_EQ((_glapi_proc) __indirect_glEnd, table[_gloffset_End]);
   EXPECT_EQ((_glapi_proc) __indirect_glEndList, table[_gloffset_EndList]);
   EXPECT_EQ((_glapi_proc) __indirect_glEvalCoord1d, table[_gloffset_EvalCoord1d]);
   EXPECT_EQ((_glapi_proc) __indirect_glEvalCoord1dv, table[_gloffset_EvalCoord1dv]);
   EXPECT_EQ((_glapi_proc) __indirect_glEvalCoord1f, table[_gloffset_EvalCoord1f]);
   EXPECT_EQ((_glapi_proc) __indirect_glEvalCoord1fv, table[_gloffset_EvalCoord1fv]);
   EXPECT_EQ((_glapi_proc) __indirect_glEvalCoord2d, table[_gloffset_EvalCoord2d]);
   EXPECT_EQ((_glapi_proc) __indirect_glEvalCoord2dv, table[_gloffset_EvalCoord2dv]);
   EXPECT_EQ((_glapi_proc) __indirect_glEvalCoord2f, table[_gloffset_EvalCoord2f]);
   EXPECT_EQ((_glapi_proc) __indirect_glEvalCoord2fv, table[_gloffset_EvalCoord2fv]);
   EXPECT_EQ((_glapi_proc) __indirect_glEvalMesh1, table[_gloffset_EvalMesh1]);
   EXPECT_EQ((_glapi_proc) __indirect_glEvalMesh2, table[_gloffset_EvalMesh2]);
   EXPECT_EQ((_glapi_proc) __indirect_glEvalPoint1, table[_gloffset_EvalPoint1]);
   EXPECT_EQ((_glapi_proc) __indirect_glEvalPoint2, table[_gloffset_EvalPoint2]);
   EXPECT_EQ((_glapi_proc) __indirect_glFeedbackBuffer, table[_gloffset_FeedbackBuffer]);
   EXPECT_EQ((_glapi_proc) __indirect_glFinish, table[_gloffset_Finish]);
   EXPECT_EQ((_glapi_proc) __indirect_glFlush, table[_gloffset_Flush]);
   EXPECT_EQ((_glapi_proc) __indirect_glFogf, table[_gloffset_Fogf]);
   EXPECT_EQ((_glapi_proc) __indirect_glFogfv, table[_gloffset_Fogfv]);
   EXPECT_EQ((_glapi_proc) __indirect_glFogi, table[_gloffset_Fogi]);
   EXPECT_EQ((_glapi_proc) __indirect_glFogiv, table[_gloffset_Fogiv]);
   EXPECT_EQ((_glapi_proc) __indirect_glFrontFace, table[_gloffset_FrontFace]);
   EXPECT_EQ((_glapi_proc) __indirect_glFrustum, table[_gloffset_Frustum]);
   EXPECT_EQ((_glapi_proc) __indirect_glGenLists, table[_gloffset_GenLists]);
   EXPECT_EQ((_glapi_proc) __indirect_glGetBooleanv, table[_gloffset_GetBooleanv]);
   EXPECT_EQ((_glapi_proc) __indirect_glGetClipPlane, table[_gloffset_GetClipPlane]);
   EXPECT_EQ((_glapi_proc) __indirect_glGetDoublev, table[_gloffset_GetDoublev]);
   EXPECT_EQ((_glapi_proc) __indirect_glGetError, table[_gloffset_GetError]);
   EXPECT_EQ((_glapi_proc) __indirect_glGetFloatv, table[_gloffset_GetFloatv]);
   EXPECT_EQ((_glapi_proc) __indirect_glGetIntegerv, table[_gloffset_GetIntegerv]);
   EXPECT_EQ((_glapi_proc) __indirect_glGetLightfv, table[_gloffset_GetLightfv]);
   EXPECT_EQ((_glapi_proc) __indirect_glGetLightiv, table[_gloffset_GetLightiv]);
   EXPECT_EQ((_glapi_proc) __indirect_glGetMapdv, table[_gloffset_GetMapdv]);
   EXPECT_EQ((_glapi_proc) __indirect_glGetMapfv, table[_gloffset_GetMapfv]);
   EXPECT_EQ((_glapi_proc) __indirect_glGetMapiv, table[_gloffset_GetMapiv]);
   EXPECT_EQ((_glapi_proc) __indirect_glGetMaterialfv, table[_gloffset_GetMaterialfv]);
   EXPECT_EQ((_glapi_proc) __indirect_glGetMaterialiv, table[_gloffset_GetMaterialiv]);
   EXPECT_EQ((_glapi_proc) __indirect_glGetPixelMapfv, table[_gloffset_GetPixelMapfv]);
   EXPECT_EQ((_glapi_proc) __indirect_glGetPixelMapuiv, table[_gloffset_GetPixelMapuiv]);
   EXPECT_EQ((_glapi_proc) __indirect_glGetPixelMapusv, table[_gloffset_GetPixelMapusv]);
   EXPECT_EQ((_glapi_proc) __indirect_glGetPolygonStipple, table[_gloffset_GetPolygonStipple]);
   EXPECT_EQ((_glapi_proc) __indirect_glGetString, table[_gloffset_GetString]);
   EXPECT_EQ((_glapi_proc) __indirect_glGetTexEnvfv, table[_gloffset_GetTexEnvfv]);
   EXPECT_EQ((_glapi_proc) __indirect_glGetTexEnviv, table[_gloffset_GetTexEnviv]);
   EXPECT_EQ((_glapi_proc) __indirect_glGetTexGendv, table[_gloffset_GetTexGendv]);
   EXPECT_EQ((_glapi_proc) __indirect_glGetTexGenfv, table[_gloffset_GetTexGenfv]);
   EXPECT_EQ((_glapi_proc) __indirect_glGetTexGeniv, table[_gloffset_GetTexGeniv]);
   EXPECT_EQ((_glapi_proc) __indirect_glGetTexImage, table[_gloffset_GetTexImage]);
   EXPECT_EQ((_glapi_proc) __indirect_glGetTexLevelParameterfv, table[_gloffset_GetTexLevelParameterfv]);
   EXPECT_EQ((_glapi_proc) __indirect_glGetTexLevelParameteriv, table[_gloffset_GetTexLevelParameteriv]);
   EXPECT_EQ((_glapi_proc) __indirect_glGetTexParameterfv, table[_gloffset_GetTexParameterfv]);
   EXPECT_EQ((_glapi_proc) __indirect_glGetTexParameteriv, table[_gloffset_GetTexParameteriv]);
   EXPECT_EQ((_glapi_proc) __indirect_glHint, table[_gloffset_Hint]);
   EXPECT_EQ((_glapi_proc) __indirect_glIndexMask, table[_gloffset_IndexMask]);
   EXPECT_EQ((_glapi_proc) __indirect_glIndexd, table[_gloffset_Indexd]);
   EXPECT_EQ((_glapi_proc) __indirect_glIndexdv, table[_gloffset_Indexdv]);
   EXPECT_EQ((_glapi_proc) __indirect_glIndexf, table[_gloffset_Indexf]);
   EXPECT_EQ((_glapi_proc) __indirect_glIndexfv, table[_gloffset_Indexfv]);
   EXPECT_EQ((_glapi_proc) __indirect_glIndexi, table[_gloffset_Indexi]);
   EXPECT_EQ((_glapi_proc) __indirect_glIndexiv, table[_gloffset_Indexiv]);
   EXPECT_EQ((_glapi_proc) __indirect_glIndexs, table[_gloffset_Indexs]);
   EXPECT_EQ((_glapi_proc) __indirect_glIndexsv, table[_gloffset_Indexsv]);
   EXPECT_EQ((_glapi_proc) __indirect_glInitNames, table[_gloffset_InitNames]);
   EXPECT_EQ((_glapi_proc) __indirect_glIsEnabled, table[_gloffset_IsEnabled]);
   EXPECT_EQ((_glapi_proc) __indirect_glIsList, table[_gloffset_IsList]);
   EXPECT_EQ((_glapi_proc) __indirect_glLightModelf, table[_gloffset_LightModelf]);
   EXPECT_EQ((_glapi_proc) __indirect_glLightModelfv, table[_gloffset_LightModelfv]);
   EXPECT_EQ((_glapi_proc) __indirect_glLightModeli, table[_gloffset_LightModeli]);
   EXPECT_EQ((_glapi_proc) __indirect_glLightModeliv, table[_gloffset_LightModeliv]);
   EXPECT_EQ((_glapi_proc) __indirect_glLightf, table[_gloffset_Lightf]);
   EXPECT_EQ((_glapi_proc) __indirect_glLightfv, table[_gloffset_Lightfv]);
   EXPECT_EQ((_glapi_proc) __indirect_glLighti, table[_gloffset_Lighti]);
   EXPECT_EQ((_glapi_proc) __indirect_glLightiv, table[_gloffset_Lightiv]);
   EXPECT_EQ((_glapi_proc) __indirect_glLineStipple, table[_gloffset_LineStipple]);
   EXPECT_EQ((_glapi_proc) __indirect_glLineWidth, table[_gloffset_LineWidth]);
   EXPECT_EQ((_glapi_proc) __indirect_glListBase, table[_gloffset_ListBase]);
   EXPECT_EQ((_glapi_proc) __indirect_glLoadIdentity, table[_gloffset_LoadIdentity]);
   EXPECT_EQ((_glapi_proc) __indirect_glLoadMatrixd, table[_gloffset_LoadMatrixd]);
   EXPECT_EQ((_glapi_proc) __indirect_glLoadMatrixf, table[_gloffset_LoadMatrixf]);
   EXPECT_EQ((_glapi_proc) __indirect_glLoadName, table[_gloffset_LoadName]);
   EXPECT_EQ((_glapi_proc) __indirect_glLogicOp, table[_gloffset_LogicOp]);
   EXPECT_EQ((_glapi_proc) __indirect_glMap1d, table[_gloffset_Map1d]);
   EXPECT_EQ((_glapi_proc) __indirect_glMap1f, table[_gloffset_Map1f]);
   EXPECT_EQ((_glapi_proc) __indirect_glMap2d, table[_gloffset_Map2d]);
   EXPECT_EQ((_glapi_proc) __indirect_glMap2f, table[_gloffset_Map2f]);
   EXPECT_EQ((_glapi_proc) __indirect_glMapGrid1d, table[_gloffset_MapGrid1d]);
   EXPECT_EQ((_glapi_proc) __indirect_glMapGrid1f, table[_gloffset_MapGrid1f]);
   EXPECT_EQ((_glapi_proc) __indirect_glMapGrid2d, table[_gloffset_MapGrid2d]);
   EXPECT_EQ((_glapi_proc) __indirect_glMapGrid2f, table[_gloffset_MapGrid2f]);
   EXPECT_EQ((_glapi_proc) __indirect_glMaterialf, table[_gloffset_Materialf]);
   EXPECT_EQ((_glapi_proc) __indirect_glMaterialfv, table[_gloffset_Materialfv]);
   EXPECT_EQ((_glapi_proc) __indirect_glMateriali, table[_gloffset_Materiali]);
   EXPECT_EQ((_glapi_proc) __indirect_glMaterialiv, table[_gloffset_Materialiv]);
   EXPECT_EQ((_glapi_proc) __indirect_glMatrixMode, table[_gloffset_MatrixMode]);
   EXPECT_EQ((_glapi_proc) __indirect_glMultMatrixd, table[_gloffset_MultMatrixd]);
   EXPECT_EQ((_glapi_proc) __indirect_glMultMatrixf, table[_gloffset_MultMatrixf]);
   EXPECT_EQ((_glapi_proc) __indirect_glNewList, table[_gloffset_NewList]);
   EXPECT_EQ((_glapi_proc) __indirect_glNormal3b, table[_gloffset_Normal3b]);
   EXPECT_EQ((_glapi_proc) __indirect_glNormal3bv, table[_gloffset_Normal3bv]);
   EXPECT_EQ((_glapi_proc) __indirect_glNormal3d, table[_gloffset_Normal3d]);
   EXPECT_EQ((_glapi_proc) __indirect_glNormal3dv, table[_gloffset_Normal3dv]);
   EXPECT_EQ((_glapi_proc) __indirect_glNormal3f, table[_gloffset_Normal3f]);
   EXPECT_EQ((_glapi_proc) __indirect_glNormal3fv, table[_gloffset_Normal3fv]);
   EXPECT_EQ((_glapi_proc) __indirect_glNormal3i, table[_gloffset_Normal3i]);
   EXPECT_EQ((_glapi_proc) __indirect_glNormal3iv, table[_gloffset_Normal3iv]);
   EXPECT_EQ((_glapi_proc) __indirect_glNormal3s, table[_gloffset_Normal3s]);
   EXPECT_EQ((_glapi_proc) __indirect_glNormal3sv, table[_gloffset_Normal3sv]);
   EXPECT_EQ((_glapi_proc) __indirect_glOrtho, table[_gloffset_Ortho]);
   EXPECT_EQ((_glapi_proc) __indirect_glPassThrough, table[_gloffset_PassThrough]);
   EXPECT_EQ((_glapi_proc) __indirect_glPixelMapfv, table[_gloffset_PixelMapfv]);
   EXPECT_EQ((_glapi_proc) __indirect_glPixelMapuiv, table[_gloffset_PixelMapuiv]);
   EXPECT_EQ((_glapi_proc) __indirect_glPixelMapusv, table[_gloffset_PixelMapusv]);
   EXPECT_EQ((_glapi_proc) __indirect_glPixelStoref, table[_gloffset_PixelStoref]);
   EXPECT_EQ((_glapi_proc) __indirect_glPixelStorei, table[_gloffset_PixelStorei]);
   EXPECT_EQ((_glapi_proc) __indirect_glPixelTransferf, table[_gloffset_PixelTransferf]);
   EXPECT_EQ((_glapi_proc) __indirect_glPixelTransferi, table[_gloffset_PixelTransferi]);
   EXPECT_EQ((_glapi_proc) __indirect_glPixelZoom, table[_gloffset_PixelZoom]);
   EXPECT_EQ((_glapi_proc) __indirect_glPointSize, table[_gloffset_PointSize]);
   EXPECT_EQ((_glapi_proc) __indirect_glPolygonMode, table[_gloffset_PolygonMode]);
   EXPECT_EQ((_glapi_proc) __indirect_glPolygonStipple, table[_gloffset_PolygonStipple]);
   EXPECT_EQ((_glapi_proc) __indirect_glPopAttrib, table[_gloffset_PopAttrib]);
   EXPECT_EQ((_glapi_proc) __indirect_glPopMatrix, table[_gloffset_PopMatrix]);
   EXPECT_EQ((_glapi_proc) __indirect_glPopName, table[_gloffset_PopName]);
   EXPECT_EQ((_glapi_proc) __indirect_glPushAttrib, table[_gloffset_PushAttrib]);
   EXPECT_EQ((_glapi_proc) __indirect_glPushMatrix, table[_gloffset_PushMatrix]);
   EXPECT_EQ((_glapi_proc) __indirect_glPushName, table[_gloffset_PushName]);
   EXPECT_EQ((_glapi_proc) __indirect_glRasterPos2d, table[_gloffset_RasterPos2d]);
   EXPECT_EQ((_glapi_proc) __indirect_glRasterPos2dv, table[_gloffset_RasterPos2dv]);
   EXPECT_EQ((_glapi_proc) __indirect_glRasterPos2f, table[_gloffset_RasterPos2f]);
   EXPECT_EQ((_glapi_proc) __indirect_glRasterPos2fv, table[_gloffset_RasterPos2fv]);
   EXPECT_EQ((_glapi_proc) __indirect_glRasterPos2i, table[_gloffset_RasterPos2i]);
   EXPECT_EQ((_glapi_proc) __indirect_glRasterPos2iv, table[_gloffset_RasterPos2iv]);
   EXPECT_EQ((_glapi_proc) __indirect_glRasterPos2s, table[_gloffset_RasterPos2s]);
   EXPECT_EQ((_glapi_proc) __indirect_glRasterPos2sv, table[_gloffset_RasterPos2sv]);
   EXPECT_EQ((_glapi_proc) __indirect_glRasterPos3d, table[_gloffset_RasterPos3d]);
   EXPECT_EQ((_glapi_proc) __indirect_glRasterPos3dv, table[_gloffset_RasterPos3dv]);
   EXPECT_EQ((_glapi_proc) __indirect_glRasterPos3f, table[_gloffset_RasterPos3f]);
   EXPECT_EQ((_glapi_proc) __indirect_glRasterPos3fv, table[_gloffset_RasterPos3fv]);
   EXPECT_EQ((_glapi_proc) __indirect_glRasterPos3i, table[_gloffset_RasterPos3i]);
   EXPECT_EQ((_glapi_proc) __indirect_glRasterPos3iv, table[_gloffset_RasterPos3iv]);
   EXPECT_EQ((_glapi_proc) __indirect_glRasterPos3s, table[_gloffset_RasterPos3s]);
   EXPECT_EQ((_glapi_proc) __indirect_glRasterPos3sv, table[_gloffset_RasterPos3sv]);
   EXPECT_EQ((_glapi_proc) __indirect_glRasterPos4d, table[_gloffset_RasterPos4d]);
   EXPECT_EQ((_glapi_proc) __indirect_glRasterPos4dv, table[_gloffset_RasterPos4dv]);
   EXPECT_EQ((_glapi_proc) __indirect_glRasterPos4f, table[_gloffset_RasterPos4f]);
   EXPECT_EQ((_glapi_proc) __indirect_glRasterPos4fv, table[_gloffset_RasterPos4fv]);
   EXPECT_EQ((_glapi_proc) __indirect_glRasterPos4i, table[_gloffset_RasterPos4i]);
   EXPECT_EQ((_glapi_proc) __indirect_glRasterPos4iv, table[_gloffset_RasterPos4iv]);
   EXPECT_EQ((_glapi_proc) __indirect_glRasterPos4s, table[_gloffset_RasterPos4s]);
   EXPECT_EQ((_glapi_proc) __indirect_glRasterPos4sv, table[_gloffset_RasterPos4sv]);
   EXPECT_EQ((_glapi_proc) __indirect_glReadBuffer, table[_gloffset_ReadBuffer]);
   EXPECT_EQ((_glapi_proc) __indirect_glReadPixels, table[_gloffset_ReadPixels]);
   EXPECT_EQ((_glapi_proc) __indirect_glRectd, table[_gloffset_Rectd]);
   EXPECT_EQ((_glapi_proc) __indirect_glRectdv, table[_gloffset_Rectdv]);
   EXPECT_EQ((_glapi_proc) __indirect_glRectf, table[_gloffset_Rectf]);
   EXPECT_EQ((_glapi_proc) __indirect_glRectfv, table[_gloffset_Rectfv]);
   EXPECT_EQ((_glapi_proc) __indirect_glRecti, table[_gloffset_Recti]);
   EXPECT_EQ((_glapi_proc) __indirect_glRectiv, table[_gloffset_Rectiv]);
   EXPECT_EQ((_glapi_proc) __indirect_glRects, table[_gloffset_Rects]);
   EXPECT_EQ((_glapi_proc) __indirect_glRectsv, table[_gloffset_Rectsv]);
   EXPECT_EQ((_glapi_proc) __indirect_glRenderMode, table[_gloffset_RenderMode]);
   EXPECT_EQ((_glapi_proc) __indirect_glRotated, table[_gloffset_Rotated]);
   EXPECT_EQ((_glapi_proc) __indirect_glRotatef, table[_gloffset_Rotatef]);
   EXPECT_EQ((_glapi_proc) __indirect_glScaled, table[_gloffset_Scaled]);
   EXPECT_EQ((_glapi_proc) __indirect_glScalef, table[_gloffset_Scalef]);
   EXPECT_EQ((_glapi_proc) __indirect_glScissor, table[_gloffset_Scissor]);
   EXPECT_EQ((_glapi_proc) __indirect_glSelectBuffer, table[_gloffset_SelectBuffer]);
   EXPECT_EQ((_glapi_proc) __indirect_glShadeModel, table[_gloffset_ShadeModel]);
   EXPECT_EQ((_glapi_proc) __indirect_glStencilFunc, table[_gloffset_StencilFunc]);
   EXPECT_EQ((_glapi_proc) __indirect_glStencilMask, table[_gloffset_StencilMask]);
   EXPECT_EQ((_glapi_proc) __indirect_glStencilOp, table[_gloffset_StencilOp]);
   EXPECT_EQ((_glapi_proc) __indirect_glTexCoord1d, table[_gloffset_TexCoord1d]);
   EXPECT_EQ((_glapi_proc) __indirect_glTexCoord1dv, table[_gloffset_TexCoord1dv]);
   EXPECT_EQ((_glapi_proc) __indirect_glTexCoord1f, table[_gloffset_TexCoord1f]);
   EXPECT_EQ((_glapi_proc) __indirect_glTexCoord1fv, table[_gloffset_TexCoord1fv]);
   EXPECT_EQ((_glapi_proc) __indirect_glTexCoord1i, table[_gloffset_TexCoord1i]);
   EXPECT_EQ((_glapi_proc) __indirect_glTexCoord1iv, table[_gloffset_TexCoord1iv]);
   EXPECT_EQ((_glapi_proc) __indirect_glTexCoord1s, table[_gloffset_TexCoord1s]);
   EXPECT_EQ((_glapi_proc) __indirect_glTexCoord1sv, table[_gloffset_TexCoord1sv]);
   EXPECT_EQ((_glapi_proc) __indirect_glTexCoord2d, table[_gloffset_TexCoord2d]);
   EXPECT_EQ((_glapi_proc) __indirect_glTexCoord2dv, table[_gloffset_TexCoord2dv]);
   EXPECT_EQ((_glapi_proc) __indirect_glTexCoord2f, table[_gloffset_TexCoord2f]);
   EXPECT_EQ((_glapi_proc) __indirect_glTexCoord2fv, table[_gloffset_TexCoord2fv]);
   EXPECT_EQ((_glapi_proc) __indirect_glTexCoord2i, table[_gloffset_TexCoord2i]);
   EXPECT_EQ((_glapi_proc) __indirect_glTexCoord2iv, table[_gloffset_TexCoord2iv]);
   EXPECT_EQ((_glapi_proc) __indirect_glTexCoord2s, table[_gloffset_TexCoord2s]);
   EXPECT_EQ((_glapi_proc) __indirect_glTexCoord2sv, table[_gloffset_TexCoord2sv]);
   EXPECT_EQ((_glapi_proc) __indirect_glTexCoord3d, table[_gloffset_TexCoord3d]);
   EXPECT_EQ((_glapi_proc) __indirect_glTexCoord3dv, table[_gloffset_TexCoord3dv]);
   EXPECT_EQ((_glapi_proc) __indirect_glTexCoord3f, table[_gloffset_TexCoord3f]);
   EXPECT_EQ((_glapi_proc) __indirect_glTexCoord3fv, table[_gloffset_TexCoord3fv]);
   EXPECT_EQ((_glapi_proc) __indirect_glTexCoord3i, table[_gloffset_TexCoord3i]);
   EXPECT_EQ((_glapi_proc) __indirect_glTexCoord3iv, table[_gloffset_TexCoord3iv]);
   EXPECT_EQ((_glapi_proc) __indirect_glTexCoord3s, table[_gloffset_TexCoord3s]);
   EXPECT_EQ((_glapi_proc) __indirect_glTexCoord3sv, table[_gloffset_TexCoord3sv]);
   EXPECT_EQ((_glapi_proc) __indirect_glTexCoord4d, table[_gloffset_TexCoord4d]);
   EXPECT_EQ((_glapi_proc) __indirect_glTexCoord4dv, table[_gloffset_TexCoord4dv]);
   EXPECT_EQ((_glapi_proc) __indirect_glTexCoord4f, table[_gloffset_TexCoord4f]);
   EXPECT_EQ((_glapi_proc) __indirect_glTexCoord4fv, table[_gloffset_TexCoord4fv]);
   EXPECT_EQ((_glapi_proc) __indirect_glTexCoord4i, table[_gloffset_TexCoord4i]);
   EXPECT_EQ((_glapi_proc) __indirect_glTexCoord4iv, table[_gloffset_TexCoord4iv]);
   EXPECT_EQ((_glapi_proc) __indirect_glTexCoord4s, table[_gloffset_TexCoord4s]);
   EXPECT_EQ((_glapi_proc) __indirect_glTexCoord4sv, table[_gloffset_TexCoord4sv]);
   EXPECT_EQ((_glapi_proc) __indirect_glTexEnvf, table[_gloffset_TexEnvf]);
   EXPECT_EQ((_glapi_proc) __indirect_glTexEnvfv, table[_gloffset_TexEnvfv]);
   EXPECT_EQ((_glapi_proc) __indirect_glTexEnvi, table[_gloffset_TexEnvi]);
   EXPECT_EQ((_glapi_proc) __indirect_glTexEnviv, table[_gloffset_TexEnviv]);
   EXPECT_EQ((_glapi_proc) __indirect_glTexGend, table[_gloffset_TexGend]);
   EXPECT_EQ((_glapi_proc) __indirect_glTexGendv, table[_gloffset_TexGendv]);
   EXPECT_EQ((_glapi_proc) __indirect_glTexGenf, table[_gloffset_TexGenf]);
   EXPECT_EQ((_glapi_proc) __indirect_glTexGenfv, table[_gloffset_TexGenfv]);
   EXPECT_EQ((_glapi_proc) __indirect_glTexGeni, table[_gloffset_TexGeni]);
   EXPECT_EQ((_glapi_proc) __indirect_glTexGeniv, table[_gloffset_TexGeniv]);
   EXPECT_EQ((_glapi_proc) __indirect_glTexImage1D, table[_gloffset_TexImage1D]);
   EXPECT_EQ((_glapi_proc) __indirect_glTexImage2D, table[_gloffset_TexImage2D]);
   EXPECT_EQ((_glapi_proc) __indirect_glTexParameterf, table[_gloffset_TexParameterf]);
   EXPECT_EQ((_glapi_proc) __indirect_glTexParameterfv, table[_gloffset_TexParameterfv]);
   EXPECT_EQ((_glapi_proc) __indirect_glTexParameteri, table[_gloffset_TexParameteri]);
   EXPECT_EQ((_glapi_proc) __indirect_glTexParameteriv, table[_gloffset_TexParameteriv]);
   EXPECT_EQ((_glapi_proc) __indirect_glTranslated, table[_gloffset_Translated]);
   EXPECT_EQ((_glapi_proc) __indirect_glTranslatef, table[_gloffset_Translatef]);
   EXPECT_EQ((_glapi_proc) __indirect_glVertex2d, table[_gloffset_Vertex2d]);
   EXPECT_EQ((_glapi_proc) __indirect_glVertex2dv, table[_gloffset_Vertex2dv]);
   EXPECT_EQ((_glapi_proc) __indirect_glVertex2f, table[_gloffset_Vertex2f]);
   EXPECT_EQ((_glapi_proc) __indirect_glVertex2fv, table[_gloffset_Vertex2fv]);
   EXPECT_EQ((_glapi_proc) __indirect_glVertex2i, table[_gloffset_Vertex2i]);
   EXPECT_EQ((_glapi_proc) __indirect_glVertex2iv, table[_gloffset_Vertex2iv]);
   EXPECT_EQ((_glapi_proc) __indirect_glVertex2s, table[_gloffset_Vertex2s]);
   EXPECT_EQ((_glapi_proc) __indirect_glVertex2sv, table[_gloffset_Vertex2sv]);
   EXPECT_EQ((_glapi_proc) __indirect_glVertex3d, table[_gloffset_Vertex3d]);
   EXPECT_EQ((_glapi_proc) __indirect_glVertex3dv, table[_gloffset_Vertex3dv]);
   EXPECT_EQ((_glapi_proc) __indirect_glVertex3f, table[_gloffset_Vertex3f]);
   EXPECT_EQ((_glapi_proc) __indirect_glVertex3fv, table[_gloffset_Vertex3fv]);
   EXPECT_EQ((_glapi_proc) __indirect_glVertex3i, table[_gloffset_Vertex3i]);
   EXPECT_EQ((_glapi_proc) __indirect_glVertex3iv, table[_gloffset_Vertex3iv]);
   EXPECT_EQ((_glapi_proc) __indirect_glVertex3s, table[_gloffset_Vertex3s]);
   EXPECT_EQ((_glapi_proc) __indirect_glVertex3sv, table[_gloffset_Vertex3sv]);
   EXPECT_EQ((_glapi_proc) __indirect_glVertex4d, table[_gloffset_Vertex4d]);
   EXPECT_EQ((_glapi_proc) __indirect_glVertex4dv, table[_gloffset_Vertex4dv]);
   EXPECT_EQ((_glapi_proc) __indirect_glVertex4f, table[_gloffset_Vertex4f]);
   EXPECT_EQ((_glapi_proc) __indirect_glVertex4fv, table[_gloffset_Vertex4fv]);
   EXPECT_EQ((_glapi_proc) __indirect_glVertex4i, table[_gloffset_Vertex4i]);
   EXPECT_EQ((_glapi_proc) __indirect_glVertex4iv, table[_gloffset_Vertex4iv]);
   EXPECT_EQ((_glapi_proc) __indirect_glVertex4s, table[_gloffset_Vertex4s]);
   EXPECT_EQ((_glapi_proc) __indirect_glVertex4sv, table[_gloffset_Vertex4sv]);
   EXPECT_EQ((_glapi_proc) __indirect_glViewport, table[_gloffset_Viewport]);
}

TEST_F(IndirectAPI, OpenGL_11)
{
   EXPECT_EQ((_glapi_proc) __indirect_glAreTexturesResident, table[_gloffset_AreTexturesResident]);
   EXPECT_EQ((_glapi_proc) __indirect_glArrayElement, table[_gloffset_ArrayElement]);
   EXPECT_EQ((_glapi_proc) __indirect_glBindTexture, table[_gloffset_BindTexture]);
   EXPECT_EQ((_glapi_proc) __indirect_glColorPointer, table[_gloffset_ColorPointer]);
   EXPECT_EQ((_glapi_proc) __indirect_glCopyTexImage1D, table[_gloffset_CopyTexImage1D]);
   EXPECT_EQ((_glapi_proc) __indirect_glCopyTexImage2D, table[_gloffset_CopyTexImage2D]);
   EXPECT_EQ((_glapi_proc) __indirect_glCopyTexSubImage1D, table[_gloffset_CopyTexSubImage1D]);
   EXPECT_EQ((_glapi_proc) __indirect_glCopyTexSubImage2D, table[_gloffset_CopyTexSubImage2D]);
   EXPECT_EQ((_glapi_proc) __indirect_glDeleteTextures, table[_gloffset_DeleteTextures]);
   EXPECT_EQ((_glapi_proc) __indirect_glDisableClientState, table[_gloffset_DisableClientState]);
   EXPECT_EQ((_glapi_proc) __indirect_glDrawArrays, table[_gloffset_DrawArrays]);
   EXPECT_EQ((_glapi_proc) __indirect_glDrawElements, table[_gloffset_DrawElements]);
   EXPECT_EQ((_glapi_proc) __indirect_glEdgeFlagPointer, table[_gloffset_EdgeFlagPointer]);
   EXPECT_EQ((_glapi_proc) __indirect_glEnableClientState, table[_gloffset_EnableClientState]);
   EXPECT_EQ((_glapi_proc) __indirect_glGenTextures, table[_gloffset_GenTextures]);
   EXPECT_EQ((_glapi_proc) __indirect_glGetPointerv, table[_gloffset_GetPointerv]);
   EXPECT_EQ((_glapi_proc) __indirect_glIndexPointer, table[_gloffset_IndexPointer]);
   EXPECT_EQ((_glapi_proc) __indirect_glIndexub, table[_gloffset_Indexub]);
   EXPECT_EQ((_glapi_proc) __indirect_glIndexubv, table[_gloffset_Indexubv]);
   EXPECT_EQ((_glapi_proc) __indirect_glInterleavedArrays, table[_gloffset_InterleavedArrays]);
   EXPECT_EQ((_glapi_proc) __indirect_glIsTexture, table[_gloffset_IsTexture]);
   EXPECT_EQ((_glapi_proc) __indirect_glNormalPointer, table[_gloffset_NormalPointer]);
   EXPECT_EQ((_glapi_proc) __indirect_glPolygonOffset, table[_gloffset_PolygonOffset]);
   EXPECT_EQ((_glapi_proc) __indirect_glPopClientAttrib, table[_gloffset_PopClientAttrib]);
   EXPECT_EQ((_glapi_proc) __indirect_glPrioritizeTextures, table[_gloffset_PrioritizeTextures]);
   EXPECT_EQ((_glapi_proc) __indirect_glPushClientAttrib, table[_gloffset_PushClientAttrib]);
   EXPECT_EQ((_glapi_proc) __indirect_glTexCoordPointer, table[_gloffset_TexCoordPointer]);
   EXPECT_EQ((_glapi_proc) __indirect_glTexSubImage1D, table[_gloffset_TexSubImage1D]);
   EXPECT_EQ((_glapi_proc) __indirect_glTexSubImage2D, table[_gloffset_TexSubImage2D]);
   EXPECT_EQ((_glapi_proc) __indirect_glVertexPointer, table[_gloffset_VertexPointer]);
}

TEST_F(IndirectAPI, OpenGL_12)
{
   EXPECT_EQ((_glapi_proc) __indirect_glBlendColor, table[_gloffset_BlendColor]);
   EXPECT_EQ((_glapi_proc) __indirect_glBlendEquation, table[_gloffset_BlendEquation]);
   EXPECT_EQ((_glapi_proc) __indirect_glColorSubTable, table[_gloffset_ColorSubTable]);
   EXPECT_EQ((_glapi_proc) __indirect_glColorTable, table[_gloffset_ColorTable]);
   EXPECT_EQ((_glapi_proc) __indirect_glColorTableParameterfv, table[_gloffset_ColorTableParameterfv]);
   EXPECT_EQ((_glapi_proc) __indirect_glColorTableParameteriv, table[_gloffset_ColorTableParameteriv]);
   EXPECT_EQ((_glapi_proc) __indirect_glConvolutionFilter1D, table[_gloffset_ConvolutionFilter1D]);
   EXPECT_EQ((_glapi_proc) __indirect_glConvolutionFilter2D, table[_gloffset_ConvolutionFilter2D]);
   EXPECT_EQ((_glapi_proc) __indirect_glConvolutionParameterf, table[_gloffset_ConvolutionParameterf]);
   EXPECT_EQ((_glapi_proc) __indirect_glConvolutionParameterfv, table[_gloffset_ConvolutionParameterfv]);
   EXPECT_EQ((_glapi_proc) __indirect_glConvolutionParameteri, table[_gloffset_ConvolutionParameteri]);
   EXPECT_EQ((_glapi_proc) __indirect_glConvolutionParameteriv, table[_gloffset_ConvolutionParameteriv]);
   EXPECT_EQ((_glapi_proc) __indirect_glCopyColorSubTable, table[_gloffset_CopyColorSubTable]);
   EXPECT_EQ((_glapi_proc) __indirect_glCopyColorTable, table[_gloffset_CopyColorTable]);
   EXPECT_EQ((_glapi_proc) __indirect_glCopyConvolutionFilter1D, table[_gloffset_CopyConvolutionFilter1D]);
   EXPECT_EQ((_glapi_proc) __indirect_glCopyConvolutionFilter2D, table[_gloffset_CopyConvolutionFilter2D]);
   EXPECT_EQ((_glapi_proc) __indirect_glCopyTexSubImage3D, table[_gloffset_CopyTexSubImage3D]);
   EXPECT_EQ((_glapi_proc) __indirect_glDrawRangeElements, table[_gloffset_DrawRangeElements]);
   EXPECT_EQ((_glapi_proc) __indirect_glGetColorTable, table[_gloffset_GetColorTable]);
   EXPECT_EQ((_glapi_proc) __indirect_glGetColorTableParameterfv, table[_gloffset_GetColorTableParameterfv]);
   EXPECT_EQ((_glapi_proc) __indirect_glGetColorTableParameteriv, table[_gloffset_GetColorTableParameteriv]);
   EXPECT_EQ((_glapi_proc) __indirect_glGetConvolutionFilter, table[_gloffset_GetConvolutionFilter]);
   EXPECT_EQ((_glapi_proc) __indirect_glGetConvolutionParameterfv, table[_gloffset_GetConvolutionParameterfv]);
   EXPECT_EQ((_glapi_proc) __indirect_glGetConvolutionParameteriv, table[_gloffset_GetConvolutionParameteriv]);
   EXPECT_EQ((_glapi_proc) __indirect_glGetHistogram, table[_gloffset_GetHistogram]);
   EXPECT_EQ((_glapi_proc) __indirect_glGetHistogramParameterfv, table[_gloffset_GetHistogramParameterfv]);
   EXPECT_EQ((_glapi_proc) __indirect_glGetHistogramParameteriv, table[_gloffset_GetHistogramParameteriv]);
   EXPECT_EQ((_glapi_proc) __indirect_glGetMinmax, table[_gloffset_GetMinmax]);
   EXPECT_EQ((_glapi_proc) __indirect_glGetMinmaxParameterfv, table[_gloffset_GetMinmaxParameterfv]);
   EXPECT_EQ((_glapi_proc) __indirect_glGetMinmaxParameteriv, table[_gloffset_GetMinmaxParameteriv]);
   EXPECT_EQ((_glapi_proc) __indirect_glGetSeparableFilter, table[_gloffset_GetSeparableFilter]);
   EXPECT_EQ((_glapi_proc) __indirect_glHistogram, table[_gloffset_Histogram]);
   EXPECT_EQ((_glapi_proc) __indirect_glMinmax, table[_gloffset_Minmax]);
   EXPECT_EQ((_glapi_proc) __indirect_glResetHistogram, table[_gloffset_ResetHistogram]);
   EXPECT_EQ((_glapi_proc) __indirect_glResetMinmax, table[_gloffset_ResetMinmax]);
   EXPECT_EQ((_glapi_proc) __indirect_glSeparableFilter2D, table[_gloffset_SeparableFilter2D]);
   EXPECT_EQ((_glapi_proc) __indirect_glTexImage3D, table[_gloffset_TexImage3D]);
   EXPECT_EQ((_glapi_proc) __indirect_glTexSubImage3D, table[_gloffset_TexSubImage3D]);
}

TEST_F(IndirectAPI, OpenGL_20_is_nop)
{
   _glapi_proc const nop = table[_gloffset_COUNT];

   EXPECT_EQ(nop, table[_glapi_get_proc_offset("glAttachShader")]);
   EXPECT_EQ(nop, table[_glapi_get_proc_offset("glCreateProgram")]);
   EXPECT_EQ(nop, table[_glapi_get_proc_offset("glCreateShader")]);
   EXPECT_EQ(nop, table[_glapi_get_proc_offset("glDeleteProgram")]);
   EXPECT_EQ(nop, table[_glapi_get_proc_offset("glDeleteShader")]);
   EXPECT_EQ(nop, table[_glapi_get_proc_offset("glDetachShader")]);
   EXPECT_EQ(nop, table[_glapi_get_proc_offset("glGetAttachedShaders")]);
   EXPECT_EQ(nop, table[_glapi_get_proc_offset("glGetProgramInfoLog")]);
   EXPECT_EQ(nop, table[_glapi_get_proc_offset("glGetProgramiv")]);
   EXPECT_EQ(nop, table[_glapi_get_proc_offset("glGetShaderInfoLog")]);
   EXPECT_EQ(nop, table[_glapi_get_proc_offset("glGetShaderiv")]);
   EXPECT_EQ(nop, table[_glapi_get_proc_offset("glIsProgram")]);
   EXPECT_EQ(nop, table[_glapi_get_proc_offset("glIsShader")]);
   EXPECT_EQ(nop, table[_glapi_get_proc_offset("glStencilFuncSeparate")]);
   EXPECT_EQ(nop, table[_glapi_get_proc_offset("glStencilMaskSeparate")]);
   EXPECT_EQ(nop, table[_glapi_get_proc_offset("glStencilOpSeparate")]);
}

TEST_F(IndirectAPI, ARB_multitexture)
{
   EXPECT_EQ((_glapi_proc) __indirect_glActiveTexture, table[_gloffset_ActiveTexture]);
   EXPECT_EQ((_glapi_proc) __indirect_glClientActiveTexture, table[_gloffset_ClientActiveTexture]);
   EXPECT_EQ((_glapi_proc) __indirect_glMultiTexCoord1d, table[_gloffset_MultiTexCoord1d]);
   EXPECT_EQ((_glapi_proc) __indirect_glMultiTexCoord1dv, table[_gloffset_MultiTexCoord1dv]);
   EXPECT_EQ((_glapi_proc) __indirect_glMultiTexCoord1fARB, table[_gloffset_MultiTexCoord1fARB]);
   EXPECT_EQ((_glapi_proc) __indirect_glMultiTexCoord1fvARB, table[_gloffset_MultiTexCoord1fvARB]);
   EXPECT_EQ((_glapi_proc) __indirect_glMultiTexCoord1i, table[_gloffset_MultiTexCoord1i]);
   EXPECT_EQ((_glapi_proc) __indirect_glMultiTexCoord1iv, table[_gloffset_MultiTexCoord1iv]);
   EXPECT_EQ((_glapi_proc) __indirect_glMultiTexCoord1s, table[_gloffset_MultiTexCoord1s]);
   EXPECT_EQ((_glapi_proc) __indirect_glMultiTexCoord1sv, table[_gloffset_MultiTexCoord1sv]);
   EXPECT_EQ((_glapi_proc) __indirect_glMultiTexCoord2d, table[_gloffset_MultiTexCoord2d]);
   EXPECT_EQ((_glapi_proc) __indirect_glMultiTexCoord2dv, table[_gloffset_MultiTexCoord2dv]);
   EXPECT_EQ((_glapi_proc) __indirect_glMultiTexCoord2fARB, table[_gloffset_MultiTexCoord2fARB]);
   EXPECT_EQ((_glapi_proc) __indirect_glMultiTexCoord2fvARB, table[_gloffset_MultiTexCoord2fvARB]);
   EXPECT_EQ((_glapi_proc) __indirect_glMultiTexCoord2i, table[_gloffset_MultiTexCoord2i]);
   EXPECT_EQ((_glapi_proc) __indirect_glMultiTexCoord2iv, table[_gloffset_MultiTexCoord2iv]);
   EXPECT_EQ((_glapi_proc) __indirect_glMultiTexCoord2s, table[_gloffset_MultiTexCoord2s]);
   EXPECT_EQ((_glapi_proc) __indirect_glMultiTexCoord2sv, table[_gloffset_MultiTexCoord2sv]);
   EXPECT_EQ((_glapi_proc) __indirect_glMultiTexCoord3d, table[_gloffset_MultiTexCoord3d]);
   EXPECT_EQ((_glapi_proc) __indirect_glMultiTexCoord3dv, table[_gloffset_MultiTexCoord3dv]);
   EXPECT_EQ((_glapi_proc) __indirect_glMultiTexCoord3fARB, table[_gloffset_MultiTexCoord3fARB]);
   EXPECT_EQ((_glapi_proc) __indirect_glMultiTexCoord3fvARB, table[_gloffset_MultiTexCoord3fvARB]);
   EXPECT_EQ((_glapi_proc) __indirect_glMultiTexCoord3i, table[_gloffset_MultiTexCoord3i]);
   EXPECT_EQ((_glapi_proc) __indirect_glMultiTexCoord3iv, table[_gloffset_MultiTexCoord3iv]);
   EXPECT_EQ((_glapi_proc) __indirect_glMultiTexCoord3s, table[_gloffset_MultiTexCoord3s]);
   EXPECT_EQ((_glapi_proc) __indirect_glMultiTexCoord3sv, table[_gloffset_MultiTexCoord3sv]);
   EXPECT_EQ((_glapi_proc) __indirect_glMultiTexCoord4d, table[_gloffset_MultiTexCoord4d]);
   EXPECT_EQ((_glapi_proc) __indirect_glMultiTexCoord4dv, table[_gloffset_MultiTexCoord4dv]);
   EXPECT_EQ((_glapi_proc) __indirect_glMultiTexCoord4fARB, table[_gloffset_MultiTexCoord4fARB]);
   EXPECT_EQ((_glapi_proc) __indirect_glMultiTexCoord4fvARB, table[_gloffset_MultiTexCoord4fvARB]);
   EXPECT_EQ((_glapi_proc) __indirect_glMultiTexCoord4i, table[_gloffset_MultiTexCoord4i]);
   EXPECT_EQ((_glapi_proc) __indirect_glMultiTexCoord4iv, table[_gloffset_MultiTexCoord4iv]);
   EXPECT_EQ((_glapi_proc) __indirect_glMultiTexCoord4s, table[_gloffset_MultiTexCoord4s]);
   EXPECT_EQ((_glapi_proc) __indirect_glMultiTexCoord4sv, table[_gloffset_MultiTexCoord4sv]);
}

TEST_F(IndirectAPI, ARB_transpose_matrix)
{
   EXPECT_EQ((_glapi_proc) __indirect_glLoadTransposeMatrixd, table[_glapi_get_proc_offset("glLoadTransposeMatrixdARB")]);
   EXPECT_EQ((_glapi_proc) __indirect_glLoadTransposeMatrixf, table[_glapi_get_proc_offset("glLoadTransposeMatrixfARB")]);
   EXPECT_EQ((_glapi_proc) __indirect_glMultTransposeMatrixd, table[_glapi_get_proc_offset("glMultTransposeMatrixdARB")]);
   EXPECT_EQ((_glapi_proc) __indirect_glMultTransposeMatrixf, table[_glapi_get_proc_offset("glMultTransposeMatrixfARB")]);
}

TEST_F(IndirectAPI, ARB_multisample)
{
   EXPECT_EQ((_glapi_proc) __indirect_glSampleCoverage, table[_glapi_get_proc_offset("glSampleCoverageARB")]);
}

TEST_F(IndirectAPI, ARB_texture_compression)
{
   EXPECT_EQ((_glapi_proc) __indirect_glCompressedTexImage1D, table[_glapi_get_proc_offset("glCompressedTexImage1DARB")]);
   EXPECT_EQ((_glapi_proc) __indirect_glCompressedTexImage2D, table[_glapi_get_proc_offset("glCompressedTexImage2DARB")]);
   EXPECT_EQ((_glapi_proc) __indirect_glCompressedTexImage3D, table[_glapi_get_proc_offset("glCompressedTexImage3DARB")]);
   EXPECT_EQ((_glapi_proc) __indirect_glCompressedTexSubImage1D, table[_glapi_get_proc_offset("glCompressedTexSubImage1DARB")]);
   EXPECT_EQ((_glapi_proc) __indirect_glCompressedTexSubImage2D, table[_glapi_get_proc_offset("glCompressedTexSubImage2DARB")]);
   EXPECT_EQ((_glapi_proc) __indirect_glCompressedTexSubImage3D, table[_glapi_get_proc_offset("glCompressedTexSubImage3DARB")]);
   EXPECT_EQ((_glapi_proc) __indirect_glGetCompressedTexImage, table[_glapi_get_proc_offset("glGetCompressedTexImageARB")]);
}

TEST_F(IndirectAPI, ARB_vertex_program)
{
   EXPECT_EQ((_glapi_proc) __indirect_glDisableVertexAttribArray, table[_glapi_get_proc_offset("glDisableVertexAttribArrayARB")]);
   EXPECT_EQ((_glapi_proc) __indirect_glEnableVertexAttribArray, table[_glapi_get_proc_offset("glEnableVertexAttribArrayARB")]);
   EXPECT_EQ((_glapi_proc) __indirect_glGetProgramEnvParameterdvARB, table[_glapi_get_proc_offset("glGetProgramEnvParameterdvARB")]);
   EXPECT_EQ((_glapi_proc) __indirect_glGetProgramEnvParameterfvARB, table[_glapi_get_proc_offset("glGetProgramEnvParameterfvARB")]);
   EXPECT_EQ((_glapi_proc) __indirect_glGetProgramLocalParameterdvARB, table[_glapi_get_proc_offset("glGetProgramLocalParameterdvARB")]);
   EXPECT_EQ((_glapi_proc) __indirect_glGetProgramLocalParameterfvARB, table[_glapi_get_proc_offset("glGetProgramLocalParameterfvARB")]);
   EXPECT_EQ((_glapi_proc) __indirect_glGetProgramStringARB, table[_glapi_get_proc_offset("glGetProgramStringARB")]);
   EXPECT_EQ((_glapi_proc) __indirect_glGetProgramivARB, table[_glapi_get_proc_offset("glGetProgramivARB")]);
   EXPECT_EQ((_glapi_proc) __indirect_glGetVertexAttribdv, table[_glapi_get_proc_offset("glGetVertexAttribdvARB")]);
   EXPECT_EQ((_glapi_proc) __indirect_glGetVertexAttribfv, table[_glapi_get_proc_offset("glGetVertexAttribfvARB")]);
   EXPECT_EQ((_glapi_proc) __indirect_glGetVertexAttribiv, table[_glapi_get_proc_offset("glGetVertexAttribivARB")]);
   EXPECT_EQ((_glapi_proc) __indirect_glProgramEnvParameter4dARB, table[_glapi_get_proc_offset("glProgramEnvParameter4dARB")]);
   EXPECT_EQ((_glapi_proc) __indirect_glProgramEnvParameter4dvARB, table[_glapi_get_proc_offset("glProgramEnvParameter4dvARB")]);
   EXPECT_EQ((_glapi_proc) __indirect_glProgramEnvParameter4fARB, table[_glapi_get_proc_offset("glProgramEnvParameter4fARB")]);
   EXPECT_EQ((_glapi_proc) __indirect_glProgramEnvParameter4fvARB, table[_glapi_get_proc_offset("glProgramEnvParameter4fvARB")]);
   EXPECT_EQ((_glapi_proc) __indirect_glProgramLocalParameter4dARB, table[_glapi_get_proc_offset("glProgramLocalParameter4dARB")]);
   EXPECT_EQ((_glapi_proc) __indirect_glProgramLocalParameter4dvARB, table[_glapi_get_proc_offset("glProgramLocalParameter4dvARB")]);
   EXPECT_EQ((_glapi_proc) __indirect_glProgramLocalParameter4fARB, table[_glapi_get_proc_offset("glProgramLocalParameter4fARB")]);
   EXPECT_EQ((_glapi_proc) __indirect_glProgramLocalParameter4fvARB, table[_glapi_get_proc_offset("glProgramLocalParameter4fvARB")]);
   EXPECT_EQ((_glapi_proc) __indirect_glProgramStringARB, table[_glapi_get_proc_offset("glProgramStringARB")]);
   EXPECT_EQ((_glapi_proc) __indirect_glVertexAttrib1d, table[_glapi_get_proc_offset("glVertexAttrib1dARB")]);
   EXPECT_EQ((_glapi_proc) __indirect_glVertexAttrib1dv, table[_glapi_get_proc_offset("glVertexAttrib1dvARB")]);
   EXPECT_EQ((_glapi_proc) __indirect_glVertexAttrib1fARB, table[_glapi_get_proc_offset("glVertexAttrib1fARB")]);
   EXPECT_EQ((_glapi_proc) __indirect_glVertexAttrib1fvARB, table[_glapi_get_proc_offset("glVertexAttrib1fvARB")]);
   EXPECT_EQ((_glapi_proc) __indirect_glVertexAttrib1s, table[_glapi_get_proc_offset("glVertexAttrib1sARB")]);
   EXPECT_EQ((_glapi_proc) __indirect_glVertexAttrib1sv, table[_glapi_get_proc_offset("glVertexAttrib1svARB")]);
   EXPECT_EQ((_glapi_proc) __indirect_glVertexAttrib2d, table[_glapi_get_proc_offset("glVertexAttrib2dARB")]);
   EXPECT_EQ((_glapi_proc) __indirect_glVertexAttrib2dv, table[_glapi_get_proc_offset("glVertexAttrib2dvARB")]);
   EXPECT_EQ((_glapi_proc) __indirect_glVertexAttrib2fARB, table[_glapi_get_proc_offset("glVertexAttrib2fARB")]);
   EXPECT_EQ((_glapi_proc) __indirect_glVertexAttrib2fvARB, table[_glapi_get_proc_offset("glVertexAttrib2fvARB")]);
   EXPECT_EQ((_glapi_proc) __indirect_glVertexAttrib2s, table[_glapi_get_proc_offset("glVertexAttrib2sARB")]);
   EXPECT_EQ((_glapi_proc) __indirect_glVertexAttrib2sv, table[_glapi_get_proc_offset("glVertexAttrib2svARB")]);
   EXPECT_EQ((_glapi_proc) __indirect_glVertexAttrib3d, table[_glapi_get_proc_offset("glVertexAttrib3dARB")]);
   EXPECT_EQ((_glapi_proc) __indirect_glVertexAttrib3dv, table[_glapi_get_proc_offset("glVertexAttrib3dvARB")]);
   EXPECT_EQ((_glapi_proc) __indirect_glVertexAttrib3fARB, table[_glapi_get_proc_offset("glVertexAttrib3fARB")]);
   EXPECT_EQ((_glapi_proc) __indirect_glVertexAttrib3fvARB, table[_glapi_get_proc_offset("glVertexAttrib3fvARB")]);
   EXPECT_EQ((_glapi_proc) __indirect_glVertexAttrib3s, table[_glapi_get_proc_offset("glVertexAttrib3sARB")]);
   EXPECT_EQ((_glapi_proc) __indirect_glVertexAttrib3sv, table[_glapi_get_proc_offset("glVertexAttrib3svARB")]);
   EXPECT_EQ((_glapi_proc) __indirect_glVertexAttrib4Nbv, table[_glapi_get_proc_offset("glVertexAttrib4NbvARB")]);
   EXPECT_EQ((_glapi_proc) __indirect_glVertexAttrib4Niv, table[_glapi_get_proc_offset("glVertexAttrib4NivARB")]);
   EXPECT_EQ((_glapi_proc) __indirect_glVertexAttrib4Nsv, table[_glapi_get_proc_offset("glVertexAttrib4NsvARB")]);
   EXPECT_EQ((_glapi_proc) __indirect_glVertexAttrib4Nub, table[_glapi_get_proc_offset("glVertexAttrib4NubARB")]);
   EXPECT_EQ((_glapi_proc) __indirect_glVertexAttrib4Nubv, table[_glapi_get_proc_offset("glVertexAttrib4NubvARB")]);
   EXPECT_EQ((_glapi_proc) __indirect_glVertexAttrib4Nuiv, table[_glapi_get_proc_offset("glVertexAttrib4NuivARB")]);
   EXPECT_EQ((_glapi_proc) __indirect_glVertexAttrib4Nusv, table[_glapi_get_proc_offset("glVertexAttrib4NusvARB")]);
   EXPECT_EQ((_glapi_proc) __indirect_glVertexAttrib4bv, table[_glapi_get_proc_offset("glVertexAttrib4bvARB")]);
   EXPECT_EQ((_glapi_proc) __indirect_glVertexAttrib4d, table[_glapi_get_proc_offset("glVertexAttrib4dARB")]);
   EXPECT_EQ((_glapi_proc) __indirect_glVertexAttrib4dv, table[_glapi_get_proc_offset("glVertexAttrib4dvARB")]);
   EXPECT_EQ((_glapi_proc) __indirect_glVertexAttrib4fARB, table[_glapi_get_proc_offset("glVertexAttrib4fARB")]);
   EXPECT_EQ((_glapi_proc) __indirect_glVertexAttrib4fvARB, table[_glapi_get_proc_offset("glVertexAttrib4fvARB")]);
   EXPECT_EQ((_glapi_proc) __indirect_glVertexAttrib4iv, table[_glapi_get_proc_offset("glVertexAttrib4ivARB")]);
   EXPECT_EQ((_glapi_proc) __indirect_glVertexAttrib4s, table[_glapi_get_proc_offset("glVertexAttrib4sARB")]);
   EXPECT_EQ((_glapi_proc) __indirect_glVertexAttrib4sv, table[_glapi_get_proc_offset("glVertexAttrib4svARB")]);
   EXPECT_EQ((_glapi_proc) __indirect_glVertexAttrib4ubv, table[_glapi_get_proc_offset("glVertexAttrib4ubvARB")]);
   EXPECT_EQ((_glapi_proc) __indirect_glVertexAttrib4uiv, table[_glapi_get_proc_offset("glVertexAttrib4uivARB")]);
   EXPECT_EQ((_glapi_proc) __indirect_glVertexAttrib4usv, table[_glapi_get_proc_offset("glVertexAttrib4usvARB")]);
   EXPECT_EQ((_glapi_proc) __indirect_glVertexAttribPointer, table[_glapi_get_proc_offset("glVertexAttribPointerARB")]);
}

TEST_F(IndirectAPI, ARB_occlusion_query)
{
   EXPECT_EQ((_glapi_proc) __indirect_glBeginQuery, table[_glapi_get_proc_offset("glBeginQueryARB")]);
   EXPECT_EQ((_glapi_proc) __indirect_glDeleteQueries, table[_glapi_get_proc_offset("glDeleteQueriesARB")]);
   EXPECT_EQ((_glapi_proc) __indirect_glEndQuery, table[_glapi_get_proc_offset("glEndQueryARB")]);
   EXPECT_EQ((_glapi_proc) __indirect_glGenQueries, table[_glapi_get_proc_offset("glGenQueriesARB")]);
   EXPECT_EQ((_glapi_proc) __indirect_glGetQueryObjectiv, table[_glapi_get_proc_offset("glGetQueryObjectivARB")]);
   EXPECT_EQ((_glapi_proc) __indirect_glGetQueryObjectuiv, table[_glapi_get_proc_offset("glGetQueryObjectuivARB")]);
   EXPECT_EQ((_glapi_proc) __indirect_glGetQueryiv, table[_glapi_get_proc_offset("glGetQueryivARB")]);
   EXPECT_EQ((_glapi_proc) __indirect_glIsQuery, table[_glapi_get_proc_offset("glIsQueryARB")]);
}

TEST_F(IndirectAPI, ARB_draw_buffers)
{
   EXPECT_EQ((_glapi_proc) __indirect_glDrawBuffers, table[_glapi_get_proc_offset("glDrawBuffersARB")]);
}

TEST_F(IndirectAPI, ARB_color_buffer_float)
{
   EXPECT_EQ((_glapi_proc) __indirect_glClampColor, table[_glapi_get_proc_offset("glClampColorARB")]);
}

TEST_F(IndirectAPI, ARB_framebuffer_object)
{
   EXPECT_EQ((_glapi_proc) __indirect_glRenderbufferStorageMultisample, table[_glapi_get_proc_offset("glRenderbufferStorageMultisample")]);
}

TEST_F(IndirectAPI, SGIS_multisample)
{
   EXPECT_EQ((_glapi_proc) __indirect_glSampleMaskSGIS, table[_glapi_get_proc_offset("glSampleMaskSGIS")]);
   EXPECT_EQ((_glapi_proc) __indirect_glSamplePatternSGIS, table[_glapi_get_proc_offset("glSamplePatternSGIS")]);
}

TEST_F(IndirectAPI, EXT_vertex_array)
{
   EXPECT_EQ((_glapi_proc) __indirect_glColorPointerEXT, table[_glapi_get_proc_offset("glColorPointerEXT")]);
   EXPECT_EQ((_glapi_proc) __indirect_glEdgeFlagPointerEXT, table[_glapi_get_proc_offset("glEdgeFlagPointerEXT")]);
   EXPECT_EQ((_glapi_proc) __indirect_glIndexPointerEXT, table[_glapi_get_proc_offset("glIndexPointerEXT")]);
   EXPECT_EQ((_glapi_proc) __indirect_glNormalPointerEXT, table[_glapi_get_proc_offset("glNormalPointerEXT")]);
   EXPECT_EQ((_glapi_proc) __indirect_glTexCoordPointerEXT, table[_glapi_get_proc_offset("glTexCoordPointerEXT")]);
   EXPECT_EQ((_glapi_proc) __indirect_glVertexPointerEXT, table[_glapi_get_proc_offset("glVertexPointerEXT")]);
}

TEST_F(IndirectAPI, EXT_point_parameters)
{
   EXPECT_EQ((_glapi_proc) __indirect_glPointParameterf, table[_glapi_get_proc_offset("glPointParameterfEXT")]);
   EXPECT_EQ((_glapi_proc) __indirect_glPointParameterfv, table[_glapi_get_proc_offset("glPointParameterfvEXT")]);
}

TEST_F(IndirectAPI, EXT_secondary_color)
{
   EXPECT_EQ((_glapi_proc) __indirect_glSecondaryColor3b, table[_glapi_get_proc_offset("glSecondaryColor3bEXT")]);
   EXPECT_EQ((_glapi_proc) __indirect_glSecondaryColor3bv, table[_glapi_get_proc_offset("glSecondaryColor3bvEXT")]);
   EXPECT_EQ((_glapi_proc) __indirect_glSecondaryColor3d, table[_glapi_get_proc_offset("glSecondaryColor3dEXT")]);
   EXPECT_EQ((_glapi_proc) __indirect_glSecondaryColor3dv, table[_glapi_get_proc_offset("glSecondaryColor3dvEXT")]);
   EXPECT_EQ((_glapi_proc) __indirect_glSecondaryColor3fEXT, table[_glapi_get_proc_offset("glSecondaryColor3fEXT")]);
   EXPECT_EQ((_glapi_proc) __indirect_glSecondaryColor3fvEXT, table[_glapi_get_proc_offset("glSecondaryColor3fvEXT")]);
   EXPECT_EQ((_glapi_proc) __indirect_glSecondaryColor3i, table[_glapi_get_proc_offset("glSecondaryColor3iEXT")]);
   EXPECT_EQ((_glapi_proc) __indirect_glSecondaryColor3iv, table[_glapi_get_proc_offset("glSecondaryColor3ivEXT")]);
   EXPECT_EQ((_glapi_proc) __indirect_glSecondaryColor3s, table[_glapi_get_proc_offset("glSecondaryColor3sEXT")]);
   EXPECT_EQ((_glapi_proc) __indirect_glSecondaryColor3sv, table[_glapi_get_proc_offset("glSecondaryColor3svEXT")]);
   EXPECT_EQ((_glapi_proc) __indirect_glSecondaryColor3ub, table[_glapi_get_proc_offset("glSecondaryColor3ubEXT")]);
   EXPECT_EQ((_glapi_proc) __indirect_glSecondaryColor3ubv, table[_glapi_get_proc_offset("glSecondaryColor3ubvEXT")]);
   EXPECT_EQ((_glapi_proc) __indirect_glSecondaryColor3ui, table[_glapi_get_proc_offset("glSecondaryColor3uiEXT")]);
   EXPECT_EQ((_glapi_proc) __indirect_glSecondaryColor3uiv, table[_glapi_get_proc_offset("glSecondaryColor3uivEXT")]);
   EXPECT_EQ((_glapi_proc) __indirect_glSecondaryColor3us, table[_glapi_get_proc_offset("glSecondaryColor3usEXT")]);
   EXPECT_EQ((_glapi_proc) __indirect_glSecondaryColor3usv, table[_glapi_get_proc_offset("glSecondaryColor3usvEXT")]);
   EXPECT_EQ((_glapi_proc) __indirect_glSecondaryColorPointer, table[_glapi_get_proc_offset("glSecondaryColorPointerEXT")]);
}

TEST_F(IndirectAPI, EXT_multi_draw_arrays)
{
   EXPECT_EQ((_glapi_proc) __indirect_glMultiDrawArrays, table[_glapi_get_proc_offset("glMultiDrawArraysEXT")]);
   EXPECT_EQ((_glapi_proc) __indirect_glMultiDrawElementsEXT, table[_glapi_get_proc_offset("glMultiDrawElementsEXT")]);
}

TEST_F(IndirectAPI, EXT_fog_coord)
{
   EXPECT_EQ((_glapi_proc) __indirect_glFogCoordPointer, table[_glapi_get_proc_offset("glFogCoordPointerEXT")]);
   EXPECT_EQ((_glapi_proc) __indirect_glFogCoordd, table[_glapi_get_proc_offset("glFogCoorddEXT")]);
   EXPECT_EQ((_glapi_proc) __indirect_glFogCoorddv, table[_glapi_get_proc_offset("glFogCoorddvEXT")]);
   EXPECT_EQ((_glapi_proc) __indirect_glFogCoordfEXT, table[_glapi_get_proc_offset("glFogCoordfEXT")]);
   EXPECT_EQ((_glapi_proc) __indirect_glFogCoordfvEXT, table[_glapi_get_proc_offset("glFogCoordfvEXT")]);
}

TEST_F(IndirectAPI, EXT_blend_func_separate)
{
   EXPECT_EQ((_glapi_proc) __indirect_glBlendFuncSeparate, table[_glapi_get_proc_offset("glBlendFuncSeparateEXT")]);
}

TEST_F(IndirectAPI, MESA_window_pos)
{
   EXPECT_EQ((_glapi_proc) __indirect_glWindowPos2d, table[_glapi_get_proc_offset("glWindowPos2dMESA")]);
   EXPECT_EQ((_glapi_proc) __indirect_glWindowPos2dv, table[_glapi_get_proc_offset("glWindowPos2dvMESA")]);
   EXPECT_EQ((_glapi_proc) __indirect_glWindowPos2f, table[_glapi_get_proc_offset("glWindowPos2fMESA")]);
   EXPECT_EQ((_glapi_proc) __indirect_glWindowPos2fv, table[_glapi_get_proc_offset("glWindowPos2fvMESA")]);
   EXPECT_EQ((_glapi_proc) __indirect_glWindowPos2i, table[_glapi_get_proc_offset("glWindowPos2iMESA")]);
   EXPECT_EQ((_glapi_proc) __indirect_glWindowPos2iv, table[_glapi_get_proc_offset("glWindowPos2ivMESA")]);
   EXPECT_EQ((_glapi_proc) __indirect_glWindowPos2s, table[_glapi_get_proc_offset("glWindowPos2sMESA")]);
   EXPECT_EQ((_glapi_proc) __indirect_glWindowPos2sv, table[_glapi_get_proc_offset("glWindowPos2svMESA")]);
   EXPECT_EQ((_glapi_proc) __indirect_glWindowPos3d, table[_glapi_get_proc_offset("glWindowPos3dMESA")]);
   EXPECT_EQ((_glapi_proc) __indirect_glWindowPos3dv, table[_glapi_get_proc_offset("glWindowPos3dvMESA")]);
   EXPECT_EQ((_glapi_proc) __indirect_glWindowPos3f, table[_glapi_get_proc_offset("glWindowPos3fMESA")]);
   EXPECT_EQ((_glapi_proc) __indirect_glWindowPos3fv, table[_glapi_get_proc_offset("glWindowPos3fvMESA")]);
   EXPECT_EQ((_glapi_proc) __indirect_glWindowPos3i, table[_glapi_get_proc_offset("glWindowPos3iMESA")]);
   EXPECT_EQ((_glapi_proc) __indirect_glWindowPos3iv, table[_glapi_get_proc_offset("glWindowPos3ivMESA")]);
   EXPECT_EQ((_glapi_proc) __indirect_glWindowPos3s, table[_glapi_get_proc_offset("glWindowPos3sMESA")]);
   EXPECT_EQ((_glapi_proc) __indirect_glWindowPos3sv, table[_glapi_get_proc_offset("glWindowPos3svMESA")]);
}

TEST_F(IndirectAPI, NV_vertex_program)
{
   EXPECT_EQ((_glapi_proc) __indirect_glAreProgramsResidentNV, table[_glapi_get_proc_offset("glAreProgramsResidentNV")]);
   EXPECT_EQ((_glapi_proc) __indirect_glBindProgramARB, table[_glapi_get_proc_offset("glBindProgramNV")]);
   EXPECT_EQ((_glapi_proc) __indirect_glDeleteProgramsARB, table[_glapi_get_proc_offset("glDeleteProgramsNV")]);
   EXPECT_EQ((_glapi_proc) __indirect_glExecuteProgramNV, table[_glapi_get_proc_offset("glExecuteProgramNV")]);
   EXPECT_EQ((_glapi_proc) __indirect_glGenProgramsARB, table[_glapi_get_proc_offset("glGenProgramsNV")]);
   EXPECT_EQ((_glapi_proc) __indirect_glGetProgramParameterdvNV, table[_glapi_get_proc_offset("glGetProgramParameterdvNV")]);
   EXPECT_EQ((_glapi_proc) __indirect_glGetProgramParameterfvNV, table[_glapi_get_proc_offset("glGetProgramParameterfvNV")]);
   EXPECT_EQ((_glapi_proc) __indirect_glGetProgramStringNV, table[_glapi_get_proc_offset("glGetProgramStringNV")]);
   EXPECT_EQ((_glapi_proc) __indirect_glGetProgramivNV, table[_glapi_get_proc_offset("glGetProgramivNV")]);
   EXPECT_EQ((_glapi_proc) __indirect_glGetTrackMatrixivNV, table[_glapi_get_proc_offset("glGetTrackMatrixivNV")]);
   EXPECT_EQ((_glapi_proc) __indirect_glGetVertexAttribPointerv, table[_glapi_get_proc_offset("glGetVertexAttribPointervNV")]);
   EXPECT_EQ((_glapi_proc) __indirect_glGetVertexAttribdvNV, table[_glapi_get_proc_offset("glGetVertexAttribdvNV")]);
   EXPECT_EQ((_glapi_proc) __indirect_glGetVertexAttribfvNV, table[_glapi_get_proc_offset("glGetVertexAttribfvNV")]);
   EXPECT_EQ((_glapi_proc) __indirect_glGetVertexAttribivNV, table[_glapi_get_proc_offset("glGetVertexAttribivNV")]);
   EXPECT_EQ((_glapi_proc) __indirect_glIsProgramARB, table[_glapi_get_proc_offset("glIsProgramNV")]);
   EXPECT_EQ((_glapi_proc) __indirect_glLoadProgramNV, table[_glapi_get_proc_offset("glLoadProgramNV")]);
   EXPECT_EQ((_glapi_proc) __indirect_glProgramParameters4dvNV, table[_glapi_get_proc_offset("glProgramParameters4dvNV")]);
   EXPECT_EQ((_glapi_proc) __indirect_glProgramParameters4fvNV, table[_glapi_get_proc_offset("glProgramParameters4fvNV")]);
   EXPECT_EQ((_glapi_proc) __indirect_glRequestResidentProgramsNV, table[_glapi_get_proc_offset("glRequestResidentProgramsNV")]);
   EXPECT_EQ((_glapi_proc) __indirect_glTrackMatrixNV, table[_glapi_get_proc_offset("glTrackMatrixNV")]);
   EXPECT_EQ((_glapi_proc) __indirect_glVertexAttrib1dNV, table[_glapi_get_proc_offset("glVertexAttrib1dNV")]);
   EXPECT_EQ((_glapi_proc) __indirect_glVertexAttrib1dvNV, table[_glapi_get_proc_offset("glVertexAttrib1dvNV")]);
   EXPECT_EQ((_glapi_proc) __indirect_glVertexAttrib1fNV, table[_glapi_get_proc_offset("glVertexAttrib1fNV")]);
   EXPECT_EQ((_glapi_proc) __indirect_glVertexAttrib1fvNV, table[_glapi_get_proc_offset("glVertexAttrib1fvNV")]);
   EXPECT_EQ((_glapi_proc) __indirect_glVertexAttrib1sNV, table[_glapi_get_proc_offset("glVertexAttrib1sNV")]);
   EXPECT_EQ((_glapi_proc) __indirect_glVertexAttrib1svNV, table[_glapi_get_proc_offset("glVertexAttrib1svNV")]);
   EXPECT_EQ((_glapi_proc) __indirect_glVertexAttrib2dNV, table[_glapi_get_proc_offset("glVertexAttrib2dNV")]);
   EXPECT_EQ((_glapi_proc) __indirect_glVertexAttrib2dvNV, table[_glapi_get_proc_offset("glVertexAttrib2dvNV")]);
   EXPECT_EQ((_glapi_proc) __indirect_glVertexAttrib2fNV, table[_glapi_get_proc_offset("glVertexAttrib2fNV")]);
   EXPECT_EQ((_glapi_proc) __indirect_glVertexAttrib2fvNV, table[_glapi_get_proc_offset("glVertexAttrib2fvNV")]);
   EXPECT_EQ((_glapi_proc) __indirect_glVertexAttrib2sNV, table[_glapi_get_proc_offset("glVertexAttrib2sNV")]);
   EXPECT_EQ((_glapi_proc) __indirect_glVertexAttrib2svNV, table[_glapi_get_proc_offset("glVertexAttrib2svNV")]);
   EXPECT_EQ((_glapi_proc) __indirect_glVertexAttrib3dNV, table[_glapi_get_proc_offset("glVertexAttrib3dNV")]);
   EXPECT_EQ((_glapi_proc) __indirect_glVertexAttrib3dvNV, table[_glapi_get_proc_offset("glVertexAttrib3dvNV")]);
   EXPECT_EQ((_glapi_proc) __indirect_glVertexAttrib3fNV, table[_glapi_get_proc_offset("glVertexAttrib3fNV")]);
   EXPECT_EQ((_glapi_proc) __indirect_glVertexAttrib3fvNV, table[_glapi_get_proc_offset("glVertexAttrib3fvNV")]);
   EXPECT_EQ((_glapi_proc) __indirect_glVertexAttrib3sNV, table[_glapi_get_proc_offset("glVertexAttrib3sNV")]);
   EXPECT_EQ((_glapi_proc) __indirect_glVertexAttrib3svNV, table[_glapi_get_proc_offset("glVertexAttrib3svNV")]);
   EXPECT_EQ((_glapi_proc) __indirect_glVertexAttrib4dNV, table[_glapi_get_proc_offset("glVertexAttrib4dNV")]);
   EXPECT_EQ((_glapi_proc) __indirect_glVertexAttrib4dvNV, table[_glapi_get_proc_offset("glVertexAttrib4dvNV")]);
   EXPECT_EQ((_glapi_proc) __indirect_glVertexAttrib4fNV, table[_glapi_get_proc_offset("glVertexAttrib4fNV")]);
   EXPECT_EQ((_glapi_proc) __indirect_glVertexAttrib4fvNV, table[_glapi_get_proc_offset("glVertexAttrib4fvNV")]);
   EXPECT_EQ((_glapi_proc) __indirect_glVertexAttrib4sNV, table[_glapi_get_proc_offset("glVertexAttrib4sNV")]);
   EXPECT_EQ((_glapi_proc) __indirect_glVertexAttrib4svNV, table[_glapi_get_proc_offset("glVertexAttrib4svNV")]);
   EXPECT_EQ((_glapi_proc) __indirect_glVertexAttrib4ubNV, table[_glapi_get_proc_offset("glVertexAttrib4ubNV")]);
   EXPECT_EQ((_glapi_proc) __indirect_glVertexAttrib4ubvNV, table[_glapi_get_proc_offset("glVertexAttrib4ubvNV")]);
   EXPECT_EQ((_glapi_proc) __indirect_glVertexAttribPointerNV, table[_glapi_get_proc_offset("glVertexAttribPointerNV")]);
   EXPECT_EQ((_glapi_proc) __indirect_glVertexAttribs1dvNV, table[_glapi_get_proc_offset("glVertexAttribs1dvNV")]);
   EXPECT_EQ((_glapi_proc) __indirect_glVertexAttribs1fvNV, table[_glapi_get_proc_offset("glVertexAttribs1fvNV")]);
   EXPECT_EQ((_glapi_proc) __indirect_glVertexAttribs1svNV, table[_glapi_get_proc_offset("glVertexAttribs1svNV")]);
   EXPECT_EQ((_glapi_proc) __indirect_glVertexAttribs2dvNV, table[_glapi_get_proc_offset("glVertexAttribs2dvNV")]);
   EXPECT_EQ((_glapi_proc) __indirect_glVertexAttribs2fvNV, table[_glapi_get_proc_offset("glVertexAttribs2fvNV")]);
   EXPECT_EQ((_glapi_proc) __indirect_glVertexAttribs2svNV, table[_glapi_get_proc_offset("glVertexAttribs2svNV")]);
   EXPECT_EQ((_glapi_proc) __indirect_glVertexAttribs3dvNV, table[_glapi_get_proc_offset("glVertexAttribs3dvNV")]);
   EXPECT_EQ((_glapi_proc) __indirect_glVertexAttribs3fvNV, table[_glapi_get_proc_offset("glVertexAttribs3fvNV")]);
   EXPECT_EQ((_glapi_proc) __indirect_glVertexAttribs3svNV, table[_glapi_get_proc_offset("glVertexAttribs3svNV")]);
   EXPECT_EQ((_glapi_proc) __indirect_glVertexAttribs4dvNV, table[_glapi_get_proc_offset("glVertexAttribs4dvNV")]);
   EXPECT_EQ((_glapi_proc) __indirect_glVertexAttribs4fvNV, table[_glapi_get_proc_offset("glVertexAttribs4fvNV")]);
   EXPECT_EQ((_glapi_proc) __indirect_glVertexAttribs4svNV, table[_glapi_get_proc_offset("glVertexAttribs4svNV")]);
   EXPECT_EQ((_glapi_proc) __indirect_glVertexAttribs4ubvNV, table[_glapi_get_proc_offset("glVertexAttribs4ubvNV")]);
}

TEST_F(IndirectAPI, NV_point_sprite)
{
   EXPECT_EQ((_glapi_proc) __indirect_glPointParameteri, table[_glapi_get_proc_offset("glPointParameteriNV")]);
   EXPECT_EQ((_glapi_proc) __indirect_glPointParameteriv, table[_glapi_get_proc_offset("glPointParameterivNV")]);
}

TEST_F(IndirectAPI, EXT_stencil_two_side)
{
   EXPECT_EQ((_glapi_proc) __indirect_glActiveStencilFaceEXT, table[_glapi_get_proc_offset("glActiveStencilFaceEXT")]);
}

TEST_F(IndirectAPI, NV_fragment_program)
{
   EXPECT_EQ((_glapi_proc) __indirect_glGetProgramNamedParameterdvNV, table[_glapi_get_proc_offset("glGetProgramNamedParameterdvNV")]);
   EXPECT_EQ((_glapi_proc) __indirect_glGetProgramNamedParameterfvNV, table[_glapi_get_proc_offset("glGetProgramNamedParameterfvNV")]);
   EXPECT_EQ((_glapi_proc) __indirect_glProgramNamedParameter4dNV, table[_glapi_get_proc_offset("glProgramNamedParameter4dNV")]);
   EXPECT_EQ((_glapi_proc) __indirect_glProgramNamedParameter4dvNV, table[_glapi_get_proc_offset("glProgramNamedParameter4dvNV")]);
   EXPECT_EQ((_glapi_proc) __indirect_glProgramNamedParameter4fNV, table[_glapi_get_proc_offset("glProgramNamedParameter4fNV")]);
   EXPECT_EQ((_glapi_proc) __indirect_glProgramNamedParameter4fvNV, table[_glapi_get_proc_offset("glProgramNamedParameter4fvNV")]);
}

TEST_F(IndirectAPI, EXT_blend_equation_separate)
{
   EXPECT_EQ((_glapi_proc) __indirect_glBlendEquationSeparate, table[_glapi_get_proc_offset("glBlendEquationSeparateEXT")]);
}

TEST_F(IndirectAPI, EXT_framebuffer_object)
{
   EXPECT_EQ((_glapi_proc) __indirect_glBindFramebufferEXT, table[_glapi_get_proc_offset("glBindFramebufferEXT")]);
   EXPECT_EQ((_glapi_proc) __indirect_glBindRenderbufferEXT, table[_glapi_get_proc_offset("glBindRenderbufferEXT")]);
   EXPECT_EQ((_glapi_proc) __indirect_glBindFramebuffer, table[_glapi_get_proc_offset("glBindFramebuffer")]);
   EXPECT_EQ((_glapi_proc) __indirect_glBindRenderbuffer, table[_glapi_get_proc_offset("glBindRenderbuffer")]);
   EXPECT_EQ((_glapi_proc) __indirect_glCheckFramebufferStatus, table[_glapi_get_proc_offset("glCheckFramebufferStatusEXT")]);
   EXPECT_EQ((_glapi_proc) __indirect_glDeleteFramebuffers, table[_glapi_get_proc_offset("glDeleteFramebuffersEXT")]);
   EXPECT_EQ((_glapi_proc) __indirect_glDeleteRenderbuffers, table[_glapi_get_proc_offset("glDeleteRenderbuffersEXT")]);
   EXPECT_EQ((_glapi_proc) __indirect_glFramebufferRenderbuffer, table[_glapi_get_proc_offset("glFramebufferRenderbufferEXT")]);
   EXPECT_EQ((_glapi_proc) __indirect_glFramebufferTexture1D, table[_glapi_get_proc_offset("glFramebufferTexture1DEXT")]);
   EXPECT_EQ((_glapi_proc) __indirect_glFramebufferTexture2D, table[_glapi_get_proc_offset("glFramebufferTexture2DEXT")]);
   EXPECT_EQ((_glapi_proc) __indirect_glFramebufferTexture3D, table[_glapi_get_proc_offset("glFramebufferTexture3DEXT")]);
   EXPECT_EQ((_glapi_proc) __indirect_glGenFramebuffers, table[_glapi_get_proc_offset("glGenFramebuffersEXT")]);
   EXPECT_EQ((_glapi_proc) __indirect_glGenRenderbuffers, table[_glapi_get_proc_offset("glGenRenderbuffersEXT")]);
   EXPECT_EQ((_glapi_proc) __indirect_glGenerateMipmap, table[_glapi_get_proc_offset("glGenerateMipmapEXT")]);
   EXPECT_EQ((_glapi_proc) __indirect_glGetFramebufferAttachmentParameteriv, table[_glapi_get_proc_offset("glGetFramebufferAttachmentParameterivEXT")]);
   EXPECT_EQ((_glapi_proc) __indirect_glGetRenderbufferParameteriv, table[_glapi_get_proc_offset("glGetRenderbufferParameterivEXT")]);
   EXPECT_EQ((_glapi_proc) __indirect_glIsFramebuffer, table[_glapi_get_proc_offset("glIsFramebufferEXT")]);
   EXPECT_EQ((_glapi_proc) __indirect_glIsRenderbuffer, table[_glapi_get_proc_offset("glIsRenderbufferEXT")]);
   EXPECT_EQ((_glapi_proc) __indirect_glRenderbufferStorage, table[_glapi_get_proc_offset("glRenderbufferStorageEXT")]);
}

TEST_F(IndirectAPI, EXT_framebuffer_blit)
{
   EXPECT_EQ((_glapi_proc) __indirect_glBlitFramebuffer, table[_glapi_get_proc_offset("glBlitFramebufferEXT")]);
}

TEST_F(IndirectAPI, EXT_texture_array)
{
   EXPECT_EQ((_glapi_proc) __indirect_glFramebufferTextureLayer, table[_glapi_get_proc_offset("glFramebufferTextureLayerEXT")]);
}
