/*
 * Argus Client Software.  Tools to read, analyze and manage Argus data.
 * Copyright (c) 2000-2003 QoSient, LLC
 * All rights reserved.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

#ifndef ArgusClient_h
#define ArgusClient_h


#include <stdio.h>
#include <string.h>
#include <sys/time.h>


#if defined(HAVE_SOLARIS)
#include <sys/types.h>
#include <sys/socket.h>
#endif

#include <netinet/in.h>

#include <compat.h>
#include <ctype.h>

#include <pcap.h>

#include <argus_def.h>
#include <argus_out.h>
#include <argus_parse.h>
#include <argus_filter.h>

#define RA_MODELNAMETAGSTR	"RAGATOR_MODEL_NAME="
#define RA_PRESERVETAGSTR	"RAGATOR_PRESERVE_FIELDS="
#define RA_REPORTTAGSTR		"RAGATOR_REPORT_AGGREGATION="
#define RA_AUTOCORRECTSTR	"RAGATOR_AUTO_CORRECTION="
#define RA_MODELTAGSTR		"Model"
#define RA_FLOWTAGSTR		"Flow"

#define RA_MODELIST		1
#define RA_FLOWLIST		2

#define RA_FLOWPOLICYFIELDNUM	11
#define RA_MODELPOLICYFIELDNUM	8
  
#define RA_LABELSTRING		0
#define RA_POLICYID		1
#define RA_POLICYTYPE		2
#define RA_POLICYSRCADDR	3
#define RA_POLICYDSTADDR	4
#define RA_POLICYPROTO		5
#define RA_POLICYSRCPORT	6
#define RA_POLICYDSTPORT	7
#define RA_POLICYMODELST	8
#define RA_POLICYTIMEOUT	9
#define RA_POLICYIDLETIMEOUT	10

#define RA_MODIFIED		1

#define RA_CON			1
#define RA_DONE			2

#define RA_CIDR_TYPE		1
#define RA_NETCLASS_TYPE	2


#define RA_HASHTABLESIZE	0x1000
#define RA_SVCPASSED		0x0100
#define RA_SVCFAILED		0x0200
#define RA_SVCINCOMPLETE        0x0400
#define RA_SVCTEST		(RA_SVCFAILED|RA_SVCPASSED|RA_SVCINCOMPLETE)
#define RA_SVCDISCOVERY		0x0800
#define RA_SVCMULTICAST		0x1000


#define ARGUS_FAR_SRCADDR_MODIFIED      0x0100
#define ARGUS_FAR_DSTADDR_MODIFIED      0x0200
#define ARGUS_FAR_PROTO_MODIFIED        0x0400
#define ARGUS_FAR_SRCPORT_MODIFIED      0x0800
#define ARGUS_FAR_DSTPORT_MODIFIED      0x1000
#define ARGUS_FAR_TPVAL_MODIFIED        0x2000

#define ARGUS_MAX_S_OPTIONS	34
#define ARGUS_MAX_SORT_ALG	34
#define MAX_SORT_ALG_TYPES	27

#define RASORTTIME		0
#define RASORTSTARTTIME		1
#define RASORTLASTTIME		2
#define RASORTTRANS   		3
#define RASORTDURATION		4
#define RASORTAVGDURATION	5
#define RASORTSRCADDR		6
#define RASORTDSTADDR		7
#define RASORTPROTOCOL		8
#define RASORTSRCPORT		9
#define RASORTDSTPORT		10
#define RASORTSRCTOS		11
#define RASORTDSTTOS		12
#define RASORTSRCTTL		13
#define RASORTDSTTTL		14
#define RASORTBYTECOUNT		15
#define RASORTSRCBYTECOUNT	16
#define RASORTDSTBYTECOUNT	17
#define RASORTPKTSCOUNT		18
#define RASORTSRCPKTSCOUNT	19
#define RASORTDSTPKTSCOUNT      20
#define RASORTLOAD              21
#define RASORTRATE              22
#define RASORTLOSS              23
#define RASORTTRANREF           24
#define RASORTSEQ               25
#define RASORTSRCID             26


#define ARGUS_MAX_PRINT_ALG	47
#define MAX_PRINT_ALG_TYPES	47

struct ArgusServiceRecord {
   u_int status;
   struct ArgusRecord *argus;
   struct RaSrvSignature *sig;
};

struct ArgusHoldingBuffer {
   int size, count;
   struct ArgusListStruct **array;
};

struct ArgusQueueHeader {
   struct ArgusQueueHeader *prv, *nxt;
   struct RaQueueStruct *queue;
   int status;
   struct timeval lasttime, logtime;
};

struct RaHashTableStruct {
   int size, count;
   struct RaHashTableHeader **array;
};

struct RaQueueStruct {
   int count;
   struct RaHashTableStruct *htable;
   struct ArgusQueueHeader **array;
   struct ArgusQueueHeader *start;
};

struct ArgusTimeStatsStruct {
   int n;
   double sumtime;
   double sumsqrd;
};

struct ArgusRecordData {
   int status, farhdrstatus;
   struct ArgusFarHeaderStruct *farhdrs[32];
   unsigned char srcdatamask[16];
   unsigned char dstdatamask[16];
   struct ArgusFlow flow;
   struct ArgusRecord *argus;
   struct ArgusAGRStruct agr;
   int pad[1];
   struct ArgusTimeStatsStruct act, idle;
};

#define ARGUS_TIMESERIES                16

#define RACAST_PROBEOBJ			1
#define RACAST_NETWORKOBJ		2

struct ArgusStoreObjectStruct {
   unsigned char type, length;
   unsigned short status;
};


struct ArgusRecordStore {
   struct ArgusQueueHeader qhdr;
   struct RaHashTableStruct *htable;
   struct RaHashTableHeader *rahtblhdr;
   struct RaQueueStruct *queue;
   struct timeval startime, lasttime;
   int  ArgusTimeout, ArgusIdleTimeout;
   int  status, sortag;
   char *displaystr;
   struct ArgusRecordData **data;
   struct ArgusRecord *cor[8];
   struct RaSrvSignature *sig;
   struct ArgusStoreObjectStruct *object;
};

struct RaCIDRAddr {
   u_int type, addr, mask;
};

struct RaFlowModelStruct {
   char *desc; 
   int preserve, report, autocorrect;
   int pindex, mindex;
   struct RaPolicyStruct **policy;
   struct RaPolicyStruct **model;
};

struct RaPolicyStruct {
   u_int RaEntryType, RaPolicyId;
   struct RaCIDRAddr src, dst;
   u_short type;
   u_char proto, pad;
   u_short sport, dport;
   u_int RaModelId, ArgusTimeout, ArgusIdleTimeout;
   char *str;
};


#define RA_HASHSIZE		64

struct RaHashStruct {
   int len;
   unsigned short hash;
   unsigned int buf[RA_HASHSIZE];
};

struct RaHashTableHeader {
   struct RaHashTableHeader *nxt, *prv;
   struct RaHashTableStruct *htable;
   struct RaHashStruct hstruct;
   struct ArgusRecordStore *storeobj;
   void *object;
};


struct ArgusListObjectStruct {
   struct ArgusListObjectStruct *nxt, *prv;
   void *obj;
};


struct ArgusListStruct {
   struct ArgusListObjectStruct *start;
   struct timeval outputTime, reportTime;
   unsigned int count;
};
 
void RaProcessRecord (struct ArgusRecord *);
void RaProcessSrvRecord (struct ArgusServiceRecord *);
 
extern int *RaHistoTimeValues;
extern struct ArgusCanonicalRecord *RaThisCanon;


#ifdef ArgusClient

char *appOptstring = NULL;

#if defined(ARGUS_SASL)
int ArgusMaxSsf = 128;
int ArgusMinSsf = 40;
#endif

char **RaFlowModelFile = NULL;

int RaPrintDate = 1;
int RaPrintDistance = 0;
int RaPrintSrcTime = 1;
int RaPrintDstTime = 1;
int RaPrintActTime = 1;
int RaPrintIdleTime = 1;

int RaCumulativeMerge = 1;
int RaPrintActualValues = 1;
int RaReportAggregation = 1;
int RaAutoCorrection = 0;

int RaAllocHashTableHeaders = 0;
int RaAllocArgusRecordStore = 0;
int RaAllocArgusRecord      = 0;

int ArgusMinuteUpdate = 1;
int ArgusHourlyUpdate = 1;
int RaHistoTimeSeries = 1;

struct ArgusFlow RaArgusFlowBuf, *RaArgusFlow = &RaArgusFlowBuf;
long long RaThisActiveDuration = 0;
int RaThisActiveIndex = 0;

struct RaCIDRAddr RaCIDRBuffer, *RaCIDRPtr = &RaCIDRBuffer;

int RaThisFlowNum = 0;
int RaThisModelNum = 0;
int RaParseError = 0;

struct RaHashTableStruct RaHashTable;

struct RaPolicyStruct *RaFlowModelPolicy = NULL;

struct RaQueueStruct *RaModelerQueue = NULL;
struct RaQueueStruct *RaAvailableQueue = NULL;
struct RaQueueStruct *RaFailedQueue = NULL;

struct RaQueueStruct *RaConnectedQueue = NULL;
struct RaQueueStruct *RaRejectedQueue = NULL;
struct RaQueueStruct *RaMulticastQueue = NULL;
struct RaQueueStruct *RaBeaconQueue = NULL;


int RaSOptionIndex = 0;

void RaSortQueue (struct RaQueueStruct *);
char *RaSOptionStrings[ARGUS_MAX_S_OPTIONS];

int (*RaSortAlgorithm)(struct ArgusRecord *, struct ArgusRecord *) = NULL;
int (*RaSortAlgorithms[ARGUS_MAX_SORT_ALG])(struct ArgusRecord *, struct ArgusRecord *);

char *RaPrintKeyWords[MAX_PRINT_ALG_TYPES] = {
   "time",
   "startime",
   "lasttime",
   "trans",
   "dur",
   "avgdur",
   "snet",
   "saddr",
   "dnet",
   "daddr",
   "proto",
   "sport",
   "dport",
   "stos",
   "dtos",
   "sttl",
   "dttl",
   "bytes",
   "sbytes",
   "dbytes",
   "pkts",
   "spkts",
   "dpkts",
   "sload",
   "dload",
   "load",
   "loss",
   "srate",
   "drate",
   "rate",
   "srcid",
   "ind",
   "mac",
   "dir",
   "jitter",
   "sjitter",
   "djitter",
   "status",
   "user",
   "tcpext",
   "win",
   "seq",
   "bins",
   "binnum",
   "mpls",
   "vlan",
   "ipid",
};

char *RaSortKeyWords[MAX_SORT_ALG_TYPES] = {
   "time",
   "startime",
   "lasttime",
   "trans",
   "dur",
   "avgdur",
   "saddr",
   "daddr",
   "proto",
   "sport",
   "dport",
   "stos",
   "dtos",
   "sttl",
   "dttl",
   "bytes",
   "sbytes",
   "dbytes",
   "pkts",
   "spkts",
   "dpkts",
   "load",
   "loss",
   "rate",
   "tranref",
   "seq",
   "srcid",
};

void RaProcessSOptions(void);

int RaSortSrcId (struct ArgusRecord *, struct ArgusRecord *);
int RaSortTime (struct ArgusRecord *, struct ArgusRecord *);
int RaSortStartTime (struct ArgusRecord *, struct ArgusRecord *);
int RaSortLastTime (struct ArgusRecord *, struct ArgusRecord *);
int RaSortCount (struct ArgusRecord *, struct ArgusRecord *);
int RaSortDuration (struct ArgusRecord *, struct ArgusRecord *);
int RaSortAvgDuration (struct ArgusRecord *, struct ArgusRecord *);
int RaSortSrcAddr (struct ArgusRecord *, struct ArgusRecord *);
int RaSortDstAddr (struct ArgusRecord *, struct ArgusRecord *);
int RaSortProtocol (struct ArgusRecord *, struct ArgusRecord *);
int RaSortSrcPort (struct ArgusRecord *, struct ArgusRecord *);
int RaSortDstPort (struct ArgusRecord *, struct ArgusRecord *);
int RaSortSrcTos (struct ArgusRecord *, struct ArgusRecord *);
int RaSortDstTos (struct ArgusRecord *, struct ArgusRecord *);
int RaSortSrcTtl (struct ArgusRecord *, struct ArgusRecord *);
int RaSortDstTtl (struct ArgusRecord *, struct ArgusRecord *);
int RaSortByteCount (struct ArgusRecord *, struct ArgusRecord *);
int RaSortLoad (struct ArgusRecord *, struct ArgusRecord *);
int RaSortLoss (struct ArgusRecord *, struct ArgusRecord *);
int RaSortRate (struct ArgusRecord *, struct ArgusRecord *);
int RaSortTranRef (struct ArgusRecord *, struct ArgusRecord *);
int RaSortSeq (struct ArgusRecord *, struct ArgusRecord *);
int RaSortSrcByteCount (struct ArgusRecord *, struct ArgusRecord *);
int RaSortDstByteCount (struct ArgusRecord *, struct ArgusRecord *);
int RaSortPktsCount (struct ArgusRecord *, struct ArgusRecord *);
int RaSortSrcPktsCount (struct ArgusRecord *, struct ArgusRecord *);
int RaSortDstPktsCount (struct ArgusRecord *, struct ArgusRecord *);
 
int (*RaSortFunction)(struct ArgusRecord  *, struct ArgusRecord *) = RaSortPktsCount;
int RaSortRoutine (const void *, const void *);

int (*RaSortAlgorithmTable[MAX_SORT_ALG_TYPES])(struct ArgusRecord *, struct ArgusRecord *) = {
   RaSortStartTime,
   RaSortStartTime,
   RaSortLastTime,
   RaSortCount,
   RaSortDuration,
   RaSortAvgDuration,
   RaSortSrcAddr,
   RaSortDstAddr,
   RaSortProtocol,
   RaSortSrcPort,
   RaSortDstPort,
   RaSortSrcTos,
   RaSortDstTos,
   RaSortSrcTtl,
   RaSortDstTtl,
   RaSortByteCount,
   RaSortSrcByteCount,
   RaSortDstByteCount,
   RaSortPktsCount,
   RaSortSrcPktsCount,
   RaSortDstPktsCount,
   RaSortLoad,
   RaSortLoss,
   RaSortRate,
   RaSortTranRef,
   RaSortSeq,
   RaSortSrcId,
};

extern void ArgusPrintDate (char *, struct ArgusRecord *);
extern void ArgusPrintStartDate (char *, struct ArgusRecord *);
extern void ArgusPrintLastDate (char *, struct ArgusRecord *);
extern void ArgusPrintSourceID (char *, struct ArgusRecord *);
extern void ArgusPrintIndicator (char *, struct ArgusRecord *);
extern void ArgusPrintMACAddress (char *, struct ArgusRecord *);
extern void ArgusPrintProto (char *, struct ArgusRecord *);
extern void ArgusPrintAddr (char *, unsigned short, void *);
extern void ArgusPrintSrcNet (char *, struct ArgusRecord *);
extern void ArgusPrintSrcAddr (char *, struct ArgusRecord *);
extern void ArgusPrintDstNet (char *, struct ArgusRecord *);
extern void ArgusPrintDstAddr (char *, struct ArgusRecord *);
extern void ArgusPrintPort (char *, struct ArgusRecord *, unsigned short);
extern void ArgusPrintSrcPort (char *, struct ArgusRecord *);
extern void ArgusPrintDstPort (char *, struct ArgusRecord *);
extern void ArgusPrintDir (char *, struct ArgusRecord *);
extern void ArgusPrintCounts (char *, struct ArgusRecord *);
extern void ArgusPrintSrcPackets (char *, struct ArgusRecord *);
extern void ArgusPrintDstPackets (char *, struct ArgusRecord *);
extern void ArgusPrintBytes (char *, struct ArgusRecord *);
extern void ArgusPrintSrcBytes (char *, struct ArgusRecord *);
extern void ArgusPrintDstBytes (char *, struct ArgusRecord *);
extern void ArgusPrintJitter (char *, struct ArgusRecord *);
extern void ArgusPrintSrcJitter (char *, struct ArgusRecord *);
extern void ArgusPrintDstJitter (char *, struct ArgusRecord *);
extern void ArgusPrintStatus (char *, struct ArgusRecord *);
extern void ArgusPrintUserData (char *, struct ArgusRecord *);
extern void ArgusPrintTCPExtensions (char *, struct ArgusRecord *);
extern void ArgusPrintSrcRate (char *, struct ArgusRecord *);
extern void ArgusPrintDstRate (char *, struct ArgusRecord *);
extern void ArgusPrintRate (char *, struct ArgusRecord *);
extern void ArgusPrintLoss (char *, struct ArgusRecord *);
extern void ArgusPrintSrcLoad (char *, struct ArgusRecord *);
extern void ArgusPrintDstLoad (char *, struct ArgusRecord *);
extern void ArgusPrintLoad (char *, struct ArgusRecord *);
extern void ArgusPrintSrcTTL (char *, struct ArgusRecord *);
extern void ArgusPrintDstTTL (char *, struct ArgusRecord *);
extern void ArgusPrintSrcTOS (char *, struct ArgusRecord *);
extern void ArgusPrintDstTOS (char *, struct ArgusRecord *);
extern void ArgusPrintVLAN (char *, struct ArgusRecord *);
extern void ArgusPrintMPLS (char *, struct ArgusRecord *);
extern void ArgusPrintWindow (char *, struct ArgusRecord *);
extern void ArgusPrintAvgDuration (char *, struct ArgusRecord *);
extern void ArgusPrintDuration (char *, struct ArgusRecord *);
extern void ArgusPrintTransactions (char *, struct ArgusRecord *);
extern void ArgusPrintSequenceNumber (char *, struct ArgusRecord *);
extern void ArgusPrintBinNumber (char *, struct ArgusRecord *);
extern void ArgusPrintBins (char *, struct ArgusRecord *);

extern void ArgusPrintDateLabel (char *);
extern void ArgusPrintStartDateLabel (char *);
extern void ArgusPrintLastDateLabel (char *);
extern void ArgusPrintSourceIDLabel (char *);
extern void ArgusPrintIndicatorLabel (char *);
extern void ArgusPrintMACAddressLabel (char *);
extern void ArgusPrintProtoLabel (char *);
extern void ArgusPrintAddrLabel (char *);
extern void ArgusPrintSrcNetLabel (char *);
extern void ArgusPrintSrcAddrLabel (char *);
extern void ArgusPrintDstNetLabel (char *);
extern void ArgusPrintDstAddrLabel (char *);
extern void ArgusPrintSrcPortLabel (char *);
extern void ArgusPrintDstPortLabel (char *);
extern void ArgusPrintIpIdLabel (char *);
extern void ArgusPrintDirLabel (char *);
extern void ArgusPrintCountsLabel (char *);
extern void ArgusPrintSrcPacketsLabel (char *);
extern void ArgusPrintDstPacketsLabel (char *);
extern void ArgusPrintBytesLabel (char *);
extern void ArgusPrintSrcBytesLabel (char *);
extern void ArgusPrintDstBytesLabel (char *);
extern void ArgusPrintJitterLabel (char *);
extern void ArgusPrintSrcJitterLabel (char *);
extern void ArgusPrintDstJitterLabel (char *);
extern void ArgusPrintStatusLabel (char *);
extern void ArgusPrintUserDataLabel (char *);
extern void ArgusPrintTCPExtensionsLabel (char *);
extern void ArgusPrintSrcRateLabel (char *);
extern void ArgusPrintDstRateLabel (char *);
extern void ArgusPrintSrcRateLabel (char *);
extern void ArgusPrintDstRateLabel (char *);
extern void ArgusPrintRateLabel (char *);
extern void ArgusPrintLossLabel (char *);
extern void ArgusPrintSrcLoadLabel (char *);
extern void ArgusPrintDstLoadLabel (char *);
extern void ArgusPrintLoadLabel (char *);
extern void ArgusPrintSrcTTLLabel (char *);
extern void ArgusPrintDstTTLLabel (char *);
extern void ArgusPrintSrcTOSLabel (char *);
extern void ArgusPrintDstTOSLabel (char *);
extern void ArgusPrintVLANLabel (char *);
extern void ArgusPrintMPLSLabel (char *);
extern void ArgusPrintWindowLabel (char *);
extern void ArgusPrintAvgDurationLabel (char *);
extern void ArgusPrintDurationLabel (char *);
extern void ArgusPrintTransactionsLabel (char *);
extern void ArgusPrintSequenceNumberLabel (char *);
extern void ArgusPrintBinNumberLabel (char *);
extern void ArgusPrintBinsLabel (char *);

void (*RaPrintAlgorithm)(char *, struct ArgusRecord *) = NULL;
void (*RaPrintAlgorithms[ARGUS_MAX_PRINT_ALG])(char *, struct ArgusRecord *) = {
   ArgusPrintStartDate,
   ArgusPrintIndicator,
   ArgusPrintProto,
   ArgusPrintSrcAddr,
   ArgusPrintSrcPort,
   ArgusPrintDir,
   ArgusPrintDstAddr,
   ArgusPrintDstPort,
   ArgusPrintCounts,
   ArgusPrintBytes,
   ArgusPrintStatus,
   NULL,
};


void (*RaPrintAlgorithmTable[MAX_PRINT_ALG_TYPES])(char *, struct ArgusRecord *) = {
   ArgusPrintDate,
   ArgusPrintStartDate,
   ArgusPrintLastDate,
   ArgusPrintTransactions,
   ArgusPrintDuration,
   ArgusPrintAvgDuration,
   ArgusPrintSrcNet,
   ArgusPrintSrcAddr,
   ArgusPrintDstNet,
   ArgusPrintDstAddr,
   ArgusPrintProto,
   ArgusPrintSrcPort,
   ArgusPrintDstPort,
   ArgusPrintSrcTOS,
   ArgusPrintDstTOS,
   ArgusPrintSrcTTL,
   ArgusPrintDstTTL,
   ArgusPrintBytes,
   ArgusPrintSrcBytes,
   ArgusPrintDstBytes,
   ArgusPrintCounts,
   ArgusPrintSrcPackets,
   ArgusPrintDstPackets,
   ArgusPrintSrcLoad,
   ArgusPrintDstLoad,
   ArgusPrintLoad,
   ArgusPrintLoss,
   ArgusPrintSrcRate,
   ArgusPrintDstRate,
   ArgusPrintRate,
   ArgusPrintSourceID,
   ArgusPrintIndicator,
   ArgusPrintMACAddress,
   ArgusPrintDir,
   ArgusPrintJitter,
   ArgusPrintSrcJitter,
   ArgusPrintDstJitter,
   ArgusPrintStatus,
   ArgusPrintUserData,
   ArgusPrintTCPExtensions,
   ArgusPrintWindow,
   ArgusPrintSequenceNumber,
   ArgusPrintBins,
   ArgusPrintBinNumber,
   ArgusPrintMPLS,
   ArgusPrintVLAN,
   ArgusPrintIpId,
};

void (*RaPrintLabelAlgorithmTable[MAX_PRINT_ALG_TYPES])(char *) = {
   ArgusPrintDateLabel,
   ArgusPrintStartDateLabel,
   ArgusPrintLastDateLabel,
   ArgusPrintTransactionsLabel,
   ArgusPrintDurationLabel,
   ArgusPrintAvgDurationLabel,
   ArgusPrintSrcNetLabel,
   ArgusPrintSrcAddrLabel,
   ArgusPrintDstNetLabel,
   ArgusPrintDstAddrLabel,
   ArgusPrintProtoLabel,
   ArgusPrintSrcPortLabel,
   ArgusPrintDstPortLabel,
   ArgusPrintSrcTOSLabel,
   ArgusPrintDstTOSLabel,
   ArgusPrintSrcTTLLabel,
   ArgusPrintDstTTLLabel,
   ArgusPrintBytesLabel,
   ArgusPrintSrcBytesLabel,
   ArgusPrintDstBytesLabel,
   ArgusPrintCountsLabel,
   ArgusPrintSrcPacketsLabel,
   ArgusPrintDstPacketsLabel,
   ArgusPrintSrcLoadLabel,
   ArgusPrintDstLoadLabel,
   ArgusPrintLoadLabel,
   ArgusPrintLossLabel,
   ArgusPrintSrcRateLabel,
   ArgusPrintDstRateLabel,
   ArgusPrintRateLabel,
   ArgusPrintSourceIDLabel,
   ArgusPrintIndicatorLabel,
   ArgusPrintMACAddressLabel,
   ArgusPrintDirLabel,
   ArgusPrintJitterLabel,
   ArgusPrintSrcJitterLabel,
   ArgusPrintDstJitterLabel,
   ArgusPrintStatusLabel,
   ArgusPrintUserDataLabel,
   ArgusPrintTCPExtensionsLabel,
   ArgusPrintWindowLabel,
   ArgusPrintSequenceNumberLabel,
   ArgusPrintBinsLabel,
   ArgusPrintBinNumberLabel,
   ArgusPrintMPLSLabel,
   ArgusPrintVLANLabel,
   ArgusPrintIpIdLabel,
};

char *RaPolicyError [] = {
   "Malformed String",
   NULL,
};

char RaLabelStr[1024], *RaLabel = NULL;
char *RaDefaultFlowLabel = "     SrcAddr  Sport  Dir         DstAddr  Dport  ";
char *RaMatrixFlowLabel  = "     SrcAddr         Dir         DstAddr         ";
char *RaTopNFlowLabel    = "     SrcAddr           ";
char *RacOptionLabel = " SrcPkt   Dstpkt    SrcBytes     DstBytes   ";
char *RAcOptionLabel = " SrcPkt   Dstpkt    SAppBytes    DAppBytes  ";
char *RaSrcRateOptionLabel = "  Src_pps ";
char *RaDstRateOptionLabel = "  Dst_pps ";
char *RaRateOptionLabel = "    Src_pps   Dst_pps ";
char *RaLossOptionLabel = "  Src_Loss  Dst_Loss  ";
char *RaSrcLoadOptionLabel =    "     Src_bps  ";
char *RaDstLoadOptionLabel =    "     Dst_bps  ";
char *RaLoadOptionLabel =       "      Src_bps      Dst_bps  ";
char *RaSrcAppLoadOptionLabel    = "   SrcApp_bps ";
char *RaDstAppLoadOptionLabel    = "   DstApp_bps ";
char *RaAppLoadOptionLabel    = "    SrcApp_bps   DstApp_bps ";
char *RaSrcJitterOptionLabel = "   SrcJitter(std)   ";
char *RaDstJitterOptionLabel = "   DstJitter(std)   ";
char *RaJitterOptionLabel = "   SrcJitter(std)      DstJitter(std)    ";
char *RaWindowOptionLabel = " SrcWin  DstWin ";
char *RamOptionLabel = "    SrcMacAddr       DstMacAddr     ";
char *RaROptionLabel = " SrcPkt   Dstpkt    Response Information    ";
char *RaTOptionLabel  = " Stos Dtos Sttl Dttl   SMinIdle    SMeanIdl    SMaxIdle     DMinIdle    DMeanIdl    DMaxIdle ";

int RaHistoTimeParse (char *);
int RaPolicyMatch (struct ArgusRecord *, struct RaPolicyStruct *);
int RaCIDRAddrMatches (u_int, struct RaCIDRAddr *);
int RaProtoMatches (u_short, int);
int RaPortMatches (u_short p1, u_short p2);
int RaTypeMatches (u_short p1, u_short p2);
unsigned int RaIndexArgusRecord (struct ArgusRecord *, struct ArgusFarHeaderStruct **);
struct RaPolicyStruct *RaFlowModelOverRides (struct ArgusRecord *, struct RaFlowModelStruct *);
void RaModifyFlow(struct RaPolicyStruct *, struct ArgusRecord *, struct RaFlowModelStruct *);
void RaModifyDefaultFlow(struct ArgusRecord *);
struct RaHashTableHeader *RaFindHashEntry (struct RaHashTableStruct *, struct RaHashStruct *);
struct RaHashTableHeader *RaAddHashEntry (struct RaHashTableStruct *, struct ArgusRecordStore *);
void RaRemoveHashEntry (struct RaHashTableStruct *, struct RaHashTableHeader *);
struct RaQueueStruct *RaNewQueue (void);
void RaDeleteQueue (struct RaQueueStruct *);
void RaCleanHashTable (struct RaHashTableStruct *);
void RaDeleteHashTable (struct RaHashTableStruct *);
int RaGetQueueCount(struct RaQueueStruct *);
void RaPushQueue(struct RaQueueStruct *, struct ArgusQueueHeader *);
int RaAddToQueue(struct RaQueueStruct *, struct ArgusQueueHeader *);
struct ArgusQueueHeader *RaPopQueue(struct RaQueueStruct *);
struct ArgusQueueHeader *RaRemoveFromQueue(struct RaQueueStruct *, struct ArgusQueueHeader *);
void RaEmptyQueue(struct RaQueueStruct *);
struct ArgusListStruct *ArgusNewList(void);
void ArgusDeleteList(struct ArgusListStruct *);
int ArgusListEmpty (struct ArgusListStruct *);
int ArgusGetListCount (struct ArgusListStruct *);
void *ArgusFrontList (struct ArgusListStruct *);
void *ArgusBackList (struct ArgusListStruct *);
void ArgusPushFrontList(struct ArgusListStruct *, void *);
void ArgusPopFrontList (struct ArgusListStruct *);
void ArgusPushBackList (struct ArgusListStruct *, void *);
void ArgusPopBackList  (struct ArgusListStruct *);
int RaCheckTimeout(struct ArgusRecordStore *obj, struct ArgusRecord *);
void RaEmptyQueue(struct RaQueueStruct *);
struct ArgusRecordStore *RaFindArgusRecord (struct RaHashTableStruct *, struct ArgusRecord *);
struct ArgusRecordStore *RaFindRevArgusRecord(struct RaHashTableStruct *, struct ArgusRecord *);
struct ArgusRecordStore *RaNewArgusStore (struct ArgusRecord *);

void RaDeleteArgusStore(struct ArgusRecordStore *);
void RaDeleteArgusData(struct ArgusRecordData *);

struct ArgusRecordData *RaNewArgusData (struct ArgusRecord *);
void RaRemoveArgusRecord (struct ArgusRecord *);
void RaMergeArgusRecord (struct ArgusRecord *, struct ArgusRecordStore *, int);
void RaUpdateArgusStore(struct ArgusRecord *, struct ArgusRecordStore *);
int RaCreatePolicyEntry (struct RaFlowModelStruct *, int, char *);
int RaTimeoutArgusStore (struct ArgusRecordStore *);

struct RaFlowModelStruct *RaReadFlowModelFile (char **);

struct RaPolicyStruct *RaParsePolicyEntry (char *);
struct RaCIDRAddr *RaParseCIDRAddr (char *);
int RaParseType (char *);

struct ArgusRecordStore *RaThisArgusStore = NULL;
struct ArgusHoldingBuffer *RaHoldingBuffer = NULL;

extern struct ARGUS_INPUT *ArgusInput;
extern struct ArgusRecord argusOriginal;
extern int major_version, minor_version;

extern char *ArgusProgramName;
extern char *ArgusProgramOptions;

extern u_int ArgusThisFarStatus;
extern struct ArgusFarHeaderStruct *ArgusThisFarHdrs[];

extern int ArgusGrepSource, ArgusGrepDestination;

extern long long tcp_dst_bytes, tcp_src_bytes;
extern long long udp_dst_bytes, udp_src_bytes;
extern long long icmp_dst_bytes, icmp_src_bytes;
extern long long ip_dst_bytes, ip_src_bytes;

extern void ArgusDebug (int, char *, ...);
extern int setArgusRemoteFilter(unsigned char *);

void ArgusClientInit(void);
void RaParseComplete (int);
void ArgusClientTimeout (void);
void parse_arg (int, char**);
void usage (void);

void RaProcessRecord (struct ArgusRecord *);

struct ArgusRecord * RaCopyArgusRecord (struct ArgusRecord *);
long long RaGetActiveDuration (struct ArgusRecord *);
long long RaGetuSecDuration (struct ArgusRecord *);
long long RaGetuSecAvgDuration (struct ArgusRecord *);

void RaProcessManRecord (struct ArgusRecord *);
void RaProcessFragRecord (struct ArgusRecord *);
void RaProcessTCPRecord (struct ArgusRecord *);
void RaProcessICMPRecord (struct ArgusRecord *);
void RaProcessUDPRecord (struct ArgusRecord *);
void RaProcessIPRecord (struct ArgusRecord *);
void RaProcessARPRecord (struct ArgusRecord *);
void RaProcessNonIPRecord (struct ArgusRecord *);

extern char *get_argus_string (struct ArgusRecord *);
extern char *get_man_string (struct ArgusRecord *);
extern char *get_tcp_string (struct ArgusRecord *);
extern char *get_ip_string (struct ArgusRecord *);
extern char *get_arp_string (struct ArgusRecord *);
extern char *get_icmp_string (struct ArgusRecord *);
extern char *get_nonip_string (struct ArgusRecord *);

extern char *getname(u_char *);
extern char *ArgusGetError (void);
extern void ArgusLog (int, char *, ...);
extern int RaSendArgusRecord(struct ArgusRecordStore *);

char *RaGenerateLabel(struct ArgusRecord *);



#else /* ArgusClient */


#if defined(ARGUS_SASL)
extern int ArgusMaxSsf;
extern int ArgusMinSsf;
#endif /* ARGUS_SASL */

extern int RaPrintDate;
extern int RaPrintTime;
extern int RaPrintDuration;
extern int RaPrintIndicator;
extern int RaPrintRate;
extern int RaPrintSource;
extern int RaPrintTransactions;
extern int RaPrintMACAddress;
extern int RaPrintProto;
extern int RaPrintSrcAddr;
extern int RaPrintSrcPort;
extern int RaPrintDir;
extern int RaPrintDstAddr;
extern int RaPrintDstPort;
extern int RaPrintCounts;
extern int RaPrintSrcPackets;
extern int RaPrintDstPackets;
extern int RaPrintBytes;
extern int RaPrintSrcBytes;
extern int RaPrintDstBytes;
extern int RaPrintJitter;
extern int RaPrintSrcJitter;
extern int RaPrintDstJitter;
extern int RaPrintSrcTime;
extern int RaPrintDstTime;
extern int RaPrintActTime;
extern int RaPrintIdleTime;
extern int RaPrintWindow;
extern int RaPrintDistance;
extern int RaPrintTTL;
extern int RaPrintSrcLoad;
extern int RaPrintDstLoad;
extern int RaPrintLoad;
extern int RaPrintLoss;
extern int RaPrintUserData;

void RaProcessSOptions(void);

extern int RaSortSrcId (struct ArgusRecord *, struct ArgusRecord *);
extern int RaSortTime (struct ArgusRecord *, struct ArgusRecord *);
extern int RaSortStartTime (struct ArgusRecord *, struct ArgusRecord *);
extern int RaSortLastTime (struct ArgusRecord *, struct ArgusRecord *);
extern int RaSortCount (struct ArgusRecord *, struct ArgusRecord *);
extern int RaSortDuration (struct ArgusRecord *, struct ArgusRecord *);
extern int RaSortAvgDuration (struct ArgusRecord *, struct ArgusRecord *);
extern int RaSortSrcAddr (struct ArgusRecord *, struct ArgusRecord *);
extern int RaSortDstAddr (struct ArgusRecord *, struct ArgusRecord *);
extern int RaSortProtocol (struct ArgusRecord *, struct ArgusRecord *);
extern int RaSortSrcPort (struct ArgusRecord *, struct ArgusRecord *);
extern int RaSortDstPort (struct ArgusRecord *, struct ArgusRecord *);
extern int RaSortSrcTos (struct ArgusRecord *, struct ArgusRecord *);
extern int RaSortDstTos (struct ArgusRecord *, struct ArgusRecord *);
extern int RaSortSrcTtl (struct ArgusRecord *, struct ArgusRecord *);
extern int RaSortDstTtl (struct ArgusRecord *, struct ArgusRecord *);
extern int RaSortByteCount (struct ArgusRecord *, struct ArgusRecord *);
extern int RaSortLoad (struct ArgusRecord *, struct ArgusRecord *);
extern int RaSortLoss (struct ArgusRecord *, struct ArgusRecord *);
extern int RaSortRate (struct ArgusRecord *, struct ArgusRecord *);
extern int RaSortTranRef (struct ArgusRecord *, struct ArgusRecord *);
extern int RaSortSeq (struct ArgusRecord *, struct ArgusRecord *);
extern int RaSortSrcByteCount (struct ArgusRecord *, struct ArgusRecord *);
extern int RaSortDstByteCount (struct ArgusRecord *, struct ArgusRecord *);
extern int RaSortPktsCount (struct ArgusRecord *, struct ArgusRecord *);
extern int RaSortSrcPktsCount (struct ArgusRecord *, struct ArgusRecord *);
extern int RaSortDstPktsCount (struct ArgusRecord *, struct ArgusRecord *);


extern void ArgusPrintDate (char *, struct ArgusRecord *);
extern void ArgusPrintStartDate (char *, struct ArgusRecord *);
extern void ArgusPrintLastDate (char *, struct ArgusRecord *);
extern void ArgusPrintSourceID (char *, struct ArgusRecord *);
extern void ArgusPrintIndicator (char *, struct ArgusRecord *);
extern void ArgusPrintMACAddress (char *, struct ArgusRecord *);
extern void ArgusPrintProto (char *, struct ArgusRecord *);
extern void ArgusPrintAddr (char *, unsigned short, void *);
extern void ArgusPrintSrcNet (char *, struct ArgusRecord *);
extern void ArgusPrintSrcAddr (char *, struct ArgusRecord *);
extern void ArgusPrintDstNet (char *, struct ArgusRecord *);
extern void ArgusPrintDstAddr (char *, struct ArgusRecord *);
extern void ArgusPrintPort (char *, struct ArgusRecord *, unsigned short);
extern void ArgusPrintSrcPort (char *, struct ArgusRecord *);
extern void ArgusPrintDstPort (char *, struct ArgusRecord *);
extern void ArgusPrintDir (char *, struct ArgusRecord *);
extern void ArgusPrintCounts (char *, struct ArgusRecord *);
extern void ArgusPrintSrcPackets (char *, struct ArgusRecord *);
extern void ArgusPrintDstPackets (char *, struct ArgusRecord *);
extern void ArgusPrintBytes (char *, struct ArgusRecord *);
extern void ArgusPrintSrcBytes (char *, struct ArgusRecord *);
extern void ArgusPrintDstBytes (char *, struct ArgusRecord *);
extern void ArgusPrintJitter (char *, struct ArgusRecord *);
extern void ArgusPrintSrcJitter (char *, struct ArgusRecord *);
extern void ArgusPrintDstJitter (char *, struct ArgusRecord *);
extern void ArgusPrintStatus (char *, struct ArgusRecord *);
extern void ArgusPrintUserData (char *, struct ArgusRecord *);
extern void ArgusPrintTCPExtensions (char *, struct ArgusRecord *);
extern void ArgusPrintSrcRate (char *, struct ArgusRecord *);
extern void ArgusPrintDstRate (char *, struct ArgusRecord *);
extern void ArgusPrintRate (char *, struct ArgusRecord *);
extern void ArgusPrintLoss (char *, struct ArgusRecord *);
extern void ArgusPrintSrcLoad (char *, struct ArgusRecord *);
extern void ArgusPrintDstLoad (char *, struct ArgusRecord *);
extern void ArgusPrintLoad (char *, struct ArgusRecord *);
extern void ArgusPrintSrcTTL (char *, struct ArgusRecord *);
extern void ArgusPrintDstTTL (char *, struct ArgusRecord *);
extern void ArgusPrintSrcTOS (char *, struct ArgusRecord *);
extern void ArgusPrintDstTOS (char *, struct ArgusRecord *);
extern void ArgusPrintVLAN (char *, struct ArgusRecord *);
extern void ArgusPrintMPLS (char *, struct ArgusRecord *);
extern void ArgusPrintWindow (char *, struct ArgusRecord *);
extern void ArgusPrintAvgDuration (char *, struct ArgusRecord *);
extern void ArgusPrintDuration (char *, struct ArgusRecord *);
extern void ArgusPrintTransactions (char *, struct ArgusRecord *);
extern void ArgusPrintSequenceNumber (char *, struct ArgusRecord *);
extern void ArgusPrintBinNumber (char *, struct ArgusRecord *);
extern void ArgusPrintBins (char *, struct ArgusRecord *);

extern void ArgusPrintDateLabel (char *);
extern void ArgusPrintStartDateLabel (char *);
extern void ArgusPrintLastDateLabel (char *);
extern void ArgusPrintSourceIDLabel (char *);
extern void ArgusPrintIndicatorLabel (char *);
extern void ArgusPrintMACAddressLabel (char *);
extern void ArgusPrintProtoLabel (char *);
extern void ArgusPrintAddrLabel (char *);
extern void ArgusPrintSrcNetLabel (char *);
extern void ArgusPrintSrcAddrLabel (char *);
extern void ArgusPrintDstNetLabel (char *);
extern void ArgusPrintDstAddrLabel (char *);
extern void ArgusPrintSrcPortLabel (char *);
extern void ArgusPrintDstPortLabel (char *);
extern void ArgusPrintDirLabel (char *);
extern void ArgusPrintIpIdLabel (char *);
extern void ArgusPrintCountsLabel (char *);
extern void ArgusPrintSrcPacketsLabel (char *);
extern void ArgusPrintDstPacketsLabel (char *);
extern void ArgusPrintBytesLabel (char *);
extern void ArgusPrintSrcBytesLabel (char *);
extern void ArgusPrintDstBytesLabel (char *);
extern void ArgusPrintJitterLabel (char *);
extern void ArgusPrintSrcJitterLabel (char *);
extern void ArgusPrintDstJitterLabel (char *);
extern void ArgusPrintStatusLabel (char *);
extern void ArgusPrintUserDataLabel (char *);
extern void ArgusPrintTCPExtensionsLabel (char *);
extern void ArgusPrintSrcRateLabel (char *);
extern void ArgusPrintDstRateLabel (char *);
extern void ArgusPrintSrcRateLabel (char *);
extern void ArgusPrintDstRateLabel (char *);
extern void ArgusPrintRateLabel (char *);
extern void ArgusPrintLossLabel (char *);
extern void ArgusPrintSrcLoadLabel (char *);
extern void ArgusPrintDstLoadLabel (char *);
extern void ArgusPrintLoadLabel (char *);
extern void ArgusPrintTTLLabel (char *);
extern void ArgusPrintWindowLabel (char *);
extern void ArgusPrintAvgDurationLabel (char *);
extern void ArgusPrintDurationLabel (char *);
extern void ArgusPrintTransactionsLabel (char *);
extern void ArgusPrintSequenceNumberLabel (char *);
extern void ArgusPrintVLANLabel (char *);
extern void ArgusPrintMPLSLabel (char *);

extern char **RaFlowModelFile;

extern char *appOptstring;
extern int RaPrintStatus;

extern int RaCumulativeMerge;
extern int RaPrintActualValues;
extern int RaReportAggregation;
extern int RaAutoCorrection;

extern int RaAllocHashTableHeaders;
extern int RaAllocArgusRecordStore;
extern int RaAllocArgusRecord;

extern int ArgusMinuteUpdate;
extern int ArgusHourlyUpdate;
extern int RaHistoTimeSeries;

extern struct ArgusFlow RaArgusFlowBuf, *RaArgusFlow;
extern long long RaThisActiveDuration;
extern int RaThisActiveIndex;

extern struct RaCIDRAddr RaCIDRBuffer, *RaCIDRPtr;

extern int RaThisFlowNum;
extern int RaThisModelNum;
extern int RaParseError;

extern struct RaHashTableStruct RaHashTable;

extern struct RaPolicyStruct *RaFlowModelPolicy;

extern struct RaQueueStruct *RaModelerQueue;
extern struct RaQueueStruct *RaAvailableQueue;
extern struct RaQueueStruct *RaFailedQueue;

extern struct RaQueueStruct *RaConnectedQueue;
extern struct RaQueueStruct *RaRejectedQueue;
extern struct RaQueueStruct *RaMulticastQueue;
extern struct RaQueueStruct *RaBeaconQueue;

extern int RaSOptionIndex;
extern void RaSortQueue (struct RaQueueStruct *);
extern char *RaSOptionStrings[ARGUS_MAX_S_OPTIONS];
extern int (*RaSortAlgorithm)(struct ArgusRecord *, struct ArgusRecord *);
extern int (*RaSortAlgorithms[ARGUS_MAX_SORT_ALG])(struct ArgusRecord *, struct ArgusRecord *);
extern void (*RaPrintAlgorithm)(char *, struct ArgusRecord *);
extern void (*RaPrintAlgorithms[ARGUS_MAX_PRINT_ALG])(char *, struct ArgusRecord *);

extern char *RaSortKeyWords[MAX_SORT_ALG_TYPES];
extern char *RaPrintKeyWords[MAX_PRINT_ALG_TYPES];

extern char *ArgusProgramName;
extern char *ArgusProgramOptions;

extern void ArgusDebug (int, char *, ...);
extern int setArgusRemoteFilter(unsigned char *);

extern void ArgusClientInit(void);
extern void RaParseComplete (int);
extern void ArgusClientTimeout (void);
extern void parse_arg (int, char**);
extern void usage (void);

extern struct ArgusRecord * RaCopyArgusRecord (struct ArgusRecord *);
extern long long RaGetActiveDuration (struct ArgusRecord *);
extern long long RaGetuSecDuration (struct ArgusRecord *);
extern long long RaGetuSecAvgDuration (struct ArgusRecord *);

extern struct ArgusRecordStore *RaThisArgusStore;
extern struct ArgusHoldingBuffer *RaHoldingBuffer;

extern char *RaPolicyError[];

extern char RaLabelStr[], *RaLabel;

extern int RaHistoTimeParse (char *);
extern int RaPolicyMatch (struct ArgusRecord *, struct RaPolicyStruct *);
extern int RaCIDRAddrMatches (u_int, struct RaCIDRAddr *);
extern int RaProtoMatches (u_short, int);
extern int RaPortMatches (u_short p1, u_short p2);
extern int RaTypeMatches (u_short p1, u_short p2);
extern unsigned int RaIndexArgusRecord (struct ArgusRecord *, struct ArgusFarHeaderStruct **);
extern struct RaPolicyStruct *RaFlowModelOverRides (struct ArgusRecord *, struct RaFlowModelStruct *);
extern void RaModifyFlow(struct RaPolicyStruct *, struct ArgusRecord *, struct RaFlowModelStruct *);
extern void RaModifyDefaultFlow(struct ArgusRecord *);
extern struct RaHashTableHeader *RaFindHashEntry (struct RaHashTableStruct *, struct RaHashStruct *);
extern struct RaHashTableHeader *RaAddHashEntry (struct RaHashTableStruct *, struct ArgusRecordStore *);
extern void RaRemoveHashEntry (struct RaHashTableStruct *, struct RaHashTableHeader *);
extern struct RaQueueStruct *RaNewQueue (void);
extern void RaDeleteQueue (struct RaQueueStruct *);
extern void RaCleanHashTable (struct RaHashTableStruct *);
extern void RaDeleteHashTable (struct RaHashTableStruct *);
extern int RaGetQueueCount(struct RaQueueStruct *);
extern void RaPushQueue(struct RaQueueStruct *, struct ArgusQueueHeader *);
extern int RaAddToQueue(struct RaQueueStruct *, struct ArgusQueueHeader *);
extern struct ArgusQueueHeader *RaPopQueue(struct RaQueueStruct *);
extern struct ArgusQueueHeader *RaRemoveFromQueue(struct RaQueueStruct *, struct ArgusQueueHeader *);
extern void RaEmptyQueue(struct RaQueueStruct *);
extern struct ArgusListStruct *ArgusNewList(void);
extern void ArgusDeleteList(struct ArgusListStruct *);
extern int ArgusListEmpty (struct ArgusListStruct *);
extern int ArgusGetListCount (struct ArgusListStruct *);
extern void *ArgusFrontList (struct ArgusListStruct *);
extern void *ArgusBackList (struct ArgusListStruct *);
extern void ArgusPushFrontList(struct ArgusListStruct *, void *);
extern void ArgusPopFrontList (struct ArgusListStruct *);
extern void ArgusPushBackList (struct ArgusListStruct *, void *);
extern void ArgusPopBackList  (struct ArgusListStruct *);
extern int RaCheckTimeout(struct ArgusRecordStore *obj, struct ArgusRecord *);
extern void RaEmptyQueue(struct RaQueueStruct *);
extern struct ArgusRecordStore *RaFindArgusRecord (struct RaHashTableStruct *, struct ArgusRecord *);
extern struct ArgusRecordStore *RaFindRevArgusRecord(struct RaHashTableStruct *, struct ArgusRecord *);
extern struct ArgusRecordStore *RaNewArgusStore (struct ArgusRecord *);


extern void RaDeleteArgusStore(struct ArgusRecordStore *);
extern void RaDeleteArgusData(struct ArgusRecordData *);

extern struct ArgusRecordData *RaNewArgusData (struct ArgusRecord *);
extern void RaRemoveArgusRecord (struct ArgusRecord *);
extern void RaMergeArgusRecord (struct ArgusRecord *, struct ArgusRecordStore *, int);
extern void RaUpdateArgusStore(struct ArgusRecord *, struct ArgusRecordStore *);
extern int RaCreatePolicyEntry (struct RaFlowModelStruct *, int, char *);
extern int RaTimeoutArgusStore (struct ArgusRecordStore *);

extern struct RaFlowModelStruct *RaReadFlowModelFile (char **);
extern struct RaPolicyStruct *RaParsePolicyEntry (char *);
extern struct RaCIDRAddr *RaParseCIDRAddr (char *);
extern int RaParseType (char *);

extern void RaProcessRecord (struct ArgusRecord *);
extern void RaProcessManRecord (struct ArgusRecord *);
extern void RaProcessFragRecord (struct ArgusRecord *);
extern void RaProcessTCPRecord (struct ArgusRecord *);
extern void RaProcessICMPRecord (struct ArgusRecord *);
extern void RaProcessUDPRecord (struct ArgusRecord *);
extern void RaProcessIPRecord (struct ArgusRecord *);
extern void RaProcessARPRecord (struct ArgusRecord *);
extern void RaProcessNonIPRecord (struct ArgusRecord *);

extern char *getname(u_char *);
extern char *ArgusGetError (void);
extern void ArgusLog (int, char *, ...);

extern int RaSortSrcId (struct ArgusRecord *, struct ArgusRecord *);
extern int RaSortStartTime (struct ArgusRecord *, struct ArgusRecord *);
extern int RaSortLastTime (struct ArgusRecord *, struct ArgusRecord *);
extern int RaSortDuration (struct ArgusRecord *, struct ArgusRecord *);
extern int RaSortSrcAddr (struct ArgusRecord *, struct ArgusRecord *);
extern int RaSortDstAddr (struct ArgusRecord *, struct ArgusRecord *);
extern int RaSortProtocol (struct ArgusRecord *, struct ArgusRecord *);
extern int RaSortSrcPort (struct ArgusRecord *, struct ArgusRecord *);
extern int RaSortDstPort (struct ArgusRecord *, struct ArgusRecord *);
extern int RaSortSrcTos (struct ArgusRecord *, struct ArgusRecord *);
extern int RaSortDstTos (struct ArgusRecord *, struct ArgusRecord *);
extern int RaSortSrcTtl (struct ArgusRecord *, struct ArgusRecord *);
extern int RaSortDstTtl (struct ArgusRecord *, struct ArgusRecord *);
extern int RaSortByteCount (struct ArgusRecord *, struct ArgusRecord *);
extern int RaSortLoad (struct ArgusRecord *, struct ArgusRecord *);
extern int RaSortLoss (struct ArgusRecord *, struct ArgusRecord *);
extern int RaSortRate (struct ArgusRecord *, struct ArgusRecord *);
extern int RaSortTrans (struct ArgusRecord *, struct ArgusRecord *);
extern int RaSortTranRef (struct ArgusRecord *, struct ArgusRecord *);
extern int RaSortSeq (struct ArgusRecord *, struct ArgusRecord *);
extern int RaSortSrcByteCount (struct ArgusRecord *, struct ArgusRecord *);
extern int RaSortDstByteCount (struct ArgusRecord *, struct ArgusRecord *);
extern int RaSortPktsCount (struct ArgusRecord *, struct ArgusRecord *);
extern int RaSortSrcPktsCount (struct ArgusRecord *, struct ArgusRecord *);
extern int RaSortDstPktsCount (struct ArgusRecord *, struct ArgusRecord *);

extern int (*RaSortFunction)(struct ArgusRecord  *, struct ArgusRecord *);
extern int RaSortRoutine (const void *, const void *);
extern int (*RaSortAlgorithmTable[MAX_SORT_ALG_TYPES])(struct ArgusRecord *, struct ArgusRecord *);

extern void (*RaPrintAlgorithmTable[MAX_PRINT_ALG_TYPES])(char *, struct ArgusRecord *);

extern char *RaGenerateLabel(struct ArgusRecord *);

int RaSendArgusRecord(struct ArgusRecordStore *);
#endif
#endif




