/* 
 * Copyright (C) 2008-2010 Andrea Di Pasquale <spikey.it@gmail.com>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice(s), this list of conditions and the following disclaimer as
 *    the first lines of this file unmodified other than the possible
 *    addition of one or more copyright notices.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice(s), this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDER(S) ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE COPYRIGHT HOLDER(S) BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH
 * DAMAGE.
 *
 * $ArpON: arpon.c,v 2.0 06/07/2010 15:38:40 Andrea Di Pasquale Exp $
 */

#include <sys/ioctl.h>
#include <sys/param.h>
#include <sys/types.h>
#include <sys/time.h>
#include <sys/stat.h>
#include <sys/wait.h>
#include <sys/resource.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#ifndef LINUX
#include <net/bpf.h>
#endif
#include <net/if.h>
#ifdef LINUX
#include <net/if_arp.h>
#endif
#ifdef NETBSD
#include <net/if_ether.h>
#elif !defined OPENBSD
#include <net/ethernet.h>
#endif
#ifdef LINUX 
#include <netinet/ether.h>
#endif
#ifdef OPENBSD
#include <netinet/if_ether.h>
#endif
#include <pthread.h>
#include <pcap.h>
#ifdef DEBIAN
#include <dumbnet.h>
#else
#include <dnet.h>
#endif
#include <libnet.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <stdarg.h>
#include <unistd.h>
#include <errno.h>
#include <time.h>
#include <getopt.h>
#include <fcntl.h>
#include <signal.h>
#include "include/queue.h"

#if defined NETBSD || defined OPENBSD || defined LINUX
#define	octet   ether_addr_octet
#endif /* struct ether addr. */

#define ERROR(str)  aprintf(stderr, 0, "  ERROR: %s:%d %s.\n\n",    \
                            __FILE__, __LINE__, str)	

enum iface_t { IFACE_MANUAL, IFACE_AUTO, IFACE_LIST, IFACE_NULL };
enum cache_t { SARPI_REFRESH, SARPI, DARPI_RESET, DARPI };
enum inspec_t { INSPEC_SARPI, INSPEC_DARPI, INSPEC_NULL };

struct iface {
    char dev[IF_NAMESIZE];
    int dev_offset;
    struct ether_addr dev_mac;
    struct in_addr dev_inet4;
    struct in_addr dev_netmask;
};

struct arp_header {	
    /* 
     * Arp header.
     */
    struct arp_hdr ah_header;
    /* 
     * Ethernet header. 
     */
    struct arp_ethip ah_addresses;
};

/*
 * For libdnet.
 * Required for arp_t fileno.
 */
struct arp_handle {
    int fd;
    int seq;
};

struct arp_cache {	
    TAILQ_ENTRY(arp_cache) entries;
    struct ether_addr ac_mac;
    struct in_addr ac_ip;
};

struct sarpi_cache {
    TAILQ_ENTRY(sarpi_cache) entries;
    struct ether_addr sc_mac;
    struct in_addr sc_ip;
};

struct darpi_cache {
    TAILQ_ENTRY(darpi_cache) entries;
    struct in_addr dc_ip;
};

static int       task_mode_cpu_priority(int, int, int);
static int       task_mode_daemon(void);
static FILE     *task_mode_pid_open(void);
static void      task_mode_pid_close(FILE *);
static int       task_mode_log(void);
static FILE     *task_mode_log_open(void);
static void      task_mode_log_close(FILE *);
static int       iface_manager(void);
static int       iface_check_uplink(void);
static void      iface_check_uplink_thread_sigusr1(int);
static void     *iface_check_uplink_thread(void *);
static void      iface_set_name(char *);
static void      iface_unset_name(void);
static int       iface_check_datalink(char *);
static int       iface_check_running(void);
static int       iface_check_online(void);
static void      iface_check_online_packet(unsigned char *,
    const struct pcap_pkthdr *, const unsigned char *); 
static int       iface_del_promisc(void);
static int       iface_get_mac_address(void);
static int       iface_get_inet4_addresses(void);
static void      iface_info_print(void);
static arp_t    *arp_cache_open(void);
static void      arp_cache_close(arp_t *);
static int       arp_cache_add(enum cache_t, char *);
static int       arp_cache_del(enum cache_t, char *);
static int       arp_cache_del_all(void);
static int       arp_cache_list_create(const struct arp_entry *, void *);
static void      arp_cache_list_destroy(void);
static void      sarpi_set_timeout(char *);
static void      sarpi_manager_thread_sigusr1(int);
static void     *sarpi_manager_thread(void *);
static int       sarpi_realtime(void);
static void      sarpi_realtime_thread_sigusr1(int);
static void     *sarpi_realtime_thread(void *);
static pcap_t   *sarpi_realtime_open_packets(void);
static void      sarpi_realtime_close_packets(pcap_t *);
static void      sarpi_realtime_read_packets(unsigned char *, 
    const struct pcap_pkthdr *, const unsigned char *);
static int       sarpi_cache_file_restore(void);
static int       sarpi_cache_list_create(const struct arp_entry *, void *);
static void      sarpi_cache_list_refresh_thread_sigusr1(int);
static void     *sarpi_cache_list_refresh_thread(void *);
static void      sarpi_cache_list_destroy(void);
static int       darpi_set_timeout(char *);
static void      darpi_manager_thread_sigusr1(int);
static void     *darpi_manager_thread(void *);
static int       darpi_realtime(void);
static void      darpi_realtime_thread_sigusr1(int);
static void     *darpi_realtime_thread(void *);
static pcap_t   *darpi_realtime_open_packets(void);
static void      darpi_realtime_close_packets(pcap_t *);
static void      darpi_realtime_read_packets(unsigned char *, 
    const struct pcap_pkthdr *, const unsigned char *);
static int       darpi_realtime_send_packet(struct ether_addr *, 
    struct in_addr *);
static int       darpi_cache_list_create(struct in_addr *);
static void      darpi_cache_list_check_thread_sigusr1(int);
static void     *darpi_cache_list_check_thread(void *);
static void      darpi_cache_list_destroy(void);
static void      aprintf(FILE *, int, char *, ...);
static void      license(void);
static void      version(void);
static void      help(void);
static int       main_signal(void);
static void     *main_signal_thread(void *);
static int       main_start(void);
static void      main_stop(void);

/*
 * iface informations.
 */
static struct iface dev;
/*
 * Arp cache.
 */
static struct arp_cache *arp_cache_begin, *arp_cache_next, *arp_cache_pos;
static TAILQ_HEAD(, arp_cache) arp_cache_head;
/*
 * SARPI cache.
 */
static struct sarpi_cache *sarpi_cache_begin, *sarpi_cache_next,
    *sarpi_cache_pos;
static TAILQ_HEAD(, sarpi_cache) sarpi_cache_head;
/*
 * DARPI cache.
 */
static struct darpi_cache *darpi_cache_begin, *darpi_cache_next, 
    *darpi_cache_pos;
static TAILQ_HEAD(, darpi_cache) darpi_cache_head;
/*
 * Signal handler.
 */
static sigset_t sigse;
/* 
 * Thread handler for:
 *  thread[0] = Signal handler
 *  thread[1] = SARPI/DARPI manager
 *  thread[2] = Iface uplink handler
 *  thread[3] = SARPI/DARPI
 *  thread[4] = SARPI/DARPI.
 */
static pthread_t thread[5];
static pthread_attr_t join_attr, detach_attr;
static pthread_rwlock_t rlock, wlock;
/* 
 * Default nice value for CPU priority.
 */
static int cpu_priority = 0;
/*
 * Default pid file.
 */
static char *pid_file = "/var/run/arpon.pid";
static pid_t pid_main;
/*
 * Default log file.
 */
static char *log_file = "/var/log/arpon.log";
/* 
 * Default log mode, with -1 is off.
 */
static int log_mode = -1;
/*
 * Default SARPI cache file.
 */
static char *sarpi_cache_file = "/etc/arpon.sarpi";
/* 
 * Default Arp cache refresh timeout, 10 minuts.
 */
static int sarpi_timeout = 10;
/* 
 * Default DARPI cache entry timeout, 500 ms.
 */
static int darpi_timeout = 500;
/*
 * Device name.
 */
static char *ddev = NULL;
/*
 * Default Iface type.
 */
static enum iface_t dif = IFACE_NULL;
/*
 * Default Inspection type.
 */
static enum inspec_t dinspec = INSPEC_NULL;

/**********************
 * Task mode handler: *
 **********************/

/* 
 * Sets CPU priority for who.
 */
static int 
task_mode_cpu_priority(int which, int who, int prio)
{
	
    if (cpu_priority != 0) {
        if (setpriority(which, who, prio) < 0) {
            ERROR(strerror(errno));
	    
            return (-1);
        }
    }

    return (0);
}

/* 
 * Demonize the process. 
 */
static int
task_mode_daemon(void)
{
    struct stat stats;
    FILE *pid_fstream;	
    int fd;

    if ((pid_fstream = task_mode_pid_open()) == NULL) {
    	return (-1);
    }

    if (stat(pid_file, &stats) < 0) {
        aprintf(stderr, 0, "  ERROR: %s\n\n", strerror(errno));
		
        return (-1);
    }

    task_mode_pid_close(pid_fstream);

    if (S_ISREG(stats.st_mode) == 0) {
        aprintf(stderr, 0, "  ERROR: %s is not a regular file.\n\n", pid_file);
		
        return (-1);
    }

    aprintf(stdout, 1, 
            "Task is forking to background, using %s pid file...\n\n", 
            pid_file);

    switch (fork()) { 
    case (-1):
        ERROR(strerror(errno));
        return (-1);
		
    case (0):
        break;
        	
    default:
        exit(EXIT_SUCCESS);	
    }

    if (setsid() < 0) {
        ERROR(strerror(errno));
		
        return (-1);
    }

    /* 
     * PID CPU Scheduling.
     */	
    if (task_mode_cpu_priority(PRIO_PROCESS, getpid(), cpu_priority) < 0) {
        return (-1);
    }

    /* 
     * Write arpon.pid file.
     */
    if ((pid_fstream = task_mode_pid_open()) == NULL) {
        kill(pid_main, SIGTERM);
		
        exit(EXIT_FAILURE);
    }
	
    fprintf(pid_fstream, "%d\n", (int)getpid());
    fflush(pid_fstream);
    task_mode_pid_close(pid_fstream);
	
    if ((fd = open("/dev/null", O_RDWR, 0)) < 0) {
        ERROR(strerror(errno));
		
        return (-1);
    }

    dup2(fd, STDIN_FILENO);
    dup2(fd, STDOUT_FILENO);
		
    if (fd > 2) {
        close(fd);
    }

    return (0);
}

/* 
 * Open pid file.
 */
static FILE *
task_mode_pid_open(void)
{
    FILE *pid_fstream;

    if ((pid_fstream = fopen(pid_file, "w+")) == NULL) {
        aprintf(stderr, 0, "  ERROR: %s: %s\n\n", pid_file, strerror(errno));
		
        return (NULL);
    }

    return (pid_fstream);
}

/* 
 * Close pid file.
 */
static void
task_mode_pid_close(FILE *pid_fstream)
{

    if (pid_fstream != NULL) {
        fclose(pid_fstream);
    }
}


/* 
 * Logging mode.
 */
static int 
task_mode_log(void)
{
    struct stat stats;
    FILE *log_fstream;

    if ((log_fstream = task_mode_log_open()) == NULL) {
        return (-1);
    }

    if (stat(log_file, &stats) < 0) {
        aprintf(stderr, 0, "  ERROR: %s: %s\n\n", log_file, strerror(errno));
		
        return (-1);
    }

    task_mode_log_close(log_fstream);

    if (S_ISREG(stats.st_mode) == 0) {
        aprintf(stderr, 0, "  ERROR: %s is not a regular file.\n\n", log_file);
		
        return (-1);
    }

    /* 
     * Logging mode:
     *  0: on
     * -1: off.
     */
    log_mode = 0;

    return (0);
}

/* 
 * Open log file.
 */
static FILE * 
task_mode_log_open(void)
{
    FILE *log_fstream;

    if ((log_fstream = fopen(log_file, "a+")) == NULL) {
        aprintf(stderr, 0, "  ERROR: %s: %s\n\n", log_file, strerror(errno));
		
        return (NULL);
    }

    return (log_fstream); 
}

/* 
 * Close log file.
 */
static void
task_mode_log_close(FILE *log_fstream)
{

    if (log_fstream != NULL) {
        fclose(log_fstream);
    }
}

/*******************
 * iface Handler: *
 *******************/

/* 
 * Handles the network iface with following modes:
 *  - Manual,
 *  - Automatic (First iface that can to be used), 
 *  - Listing eventual network ifaces.
 *  - boot/unplug/hibernation/suspension interface
 *
 * Doing the following operations:
 *  - Verifying Datalink 
 *  - Selecting the network iface to be used
 *  - Verifying iface running
 *  - Veryfying iface online ready
 *  - Putting down the promiscue flag if found set
 *  - Reading MAC Address
 *  - Reading IP, netmask inet4 addresses
 *  - Printing out network ifaces dates.
 */
static int
iface_manager(void)
{
    char errbuf[PCAP_ERRBUF_SIZE];
    pcap_if_t *ifc;
	
    if (pcap_findalldevs(&ifc, errbuf) < 0) { 
        ERROR(errbuf);
        
        return (-1);
    }	
	
    /*
     * For SARPI and DARPI.
     */
    if (dif == IFACE_NULL) {
        dif = IFACE_AUTO;
    }
	
    for (; ifc != NULL; ifc = ifc->next) { 
        switch (dif) {
        case (IFACE_MANUAL):
            if (strncmp(ddev, ifc->name, IF_NAMESIZE) == 0) {
                if ((ifc->flags & PCAP_IF_LOOPBACK) == 0) { 
                    if (iface_check_datalink(ifc->name) < 0) { 
                        aprintf(stderr, 0, "  ERROR: %s interface's datalink " \
                                "is not supported!\n\n", ddev);			
                        pcap_freealldevs(ifc);

                        return (-1); 
                    }			
						
                    iface_set_name(ifc->name);
						
                    if (iface_del_promisc() < 0 || 
                        iface_get_mac_address() < 0 ||
                        iface_get_inet4_addresses() < 0) {
                        pcap_freealldevs(ifc);

                        return (-1);
                    }
		    
                    pcap_freealldevs(ifc);		
                    
                    return (0);
                } else {
                    aprintf(stderr, 0, "  ERROR: %s interface is not " \
                            "supported!\n\n", ddev);
                    pcap_freealldevs(ifc);

                    return (-1);
                }
            }
            break;
				
        case (IFACE_AUTO):
            if ((ifc->flags & PCAP_IF_LOOPBACK) == 0) {
                if (iface_check_datalink(ifc->name) < 0) {
                    break;
                }
					
                iface_set_name(ifc->name);
					
                if (iface_del_promisc() < 0 ||
                    iface_get_mac_address() < 0 ||
                    iface_get_inet4_addresses() < 0) {
                    iface_unset_name();
						
                    break;
                }
					
                pcap_freealldevs(ifc);

                return (0);
            }
            break;
				
        case (IFACE_LIST):
            if ((ifc->flags & PCAP_IF_LOOPBACK) == 0) {
                if (iface_check_datalink(ifc->name) < 0) {
                    break;
                }
					
                iface_set_name(ifc->name);
					
                if (iface_del_promisc() < 0 || 
                    iface_get_mac_address() < 0 ||
                    iface_get_inet4_addresses() < 0) {
                    pcap_freealldevs(ifc);

                    return (-1);
                }
            }
            break;
				
        default:
            break;
        }   
    }
	
    if (dif == IFACE_MANUAL) {
        if (strncmp(ddev, dev.dev, IF_NAMESIZE) != 0) {
            aprintf(stderr, 0, "  ERROR: %s interface not found!\n\n", ddev);
            pcap_freealldevs(ifc);

            return (-1);
        }
    } else if (dif == IFACE_AUTO) {
        if (dev.dev[0] == '\0') {
            aprintf(stderr, 0, "  ERROR: Interface not found!\n\n");
            pcap_freealldevs(ifc);

            return (-1);
        }
    }
	
    pcap_freealldevs(ifc);
    return (0);
}

/*
 * Check automatically iface uplink:
 *  - boot
 *  - unplug ethernet/wireless
 *  - hibernation
 *  - suspension
 */
static int
iface_check_uplink(void)
{
	
    pthread_attr_init(&join_attr);
    pthread_attr_setschedpolicy(&join_attr, SCHED_RR);
	
    /*
     * Thread 3 joinabled, checks automatically iface uplink. 
     */
    if (pthread_create(&thread[2], &join_attr, 
        iface_check_uplink_thread, (void *) NULL) != 0) {
        ERROR(strerror(errno));
		
        pthread_attr_destroy(&join_attr);
        return (-1);
    }
	
    pthread_attr_destroy(&join_attr);
    return (0);
}

/* 
 * Iface uplink signal handler.
 */
static void
iface_check_uplink_thread_sigusr1(int sig)
{
	
    pthread_exit((void *) 0);
}

/*
 * Check automatically iface uplink.
 * If uplink is down stops daemon, waits uplink up
 * and restart the daemon.
 */
static void *
iface_check_uplink_thread(void *arg)
{
    struct sigaction saction;
    int ret, ret2;

#ifdef NETBSD
    saction.sa_flags = 0;
#endif
    saction.sa_handler = iface_check_uplink_thread_sigusr1;
	
    if (sigaction(SIGUSR1, &saction, NULL) < 0) {
        ERROR(strerror(errno));
		
        pthread_exit((void *) -1);
    }
	
    while (1) {
        if ((ret = iface_check_running()) < 0) {
            pthread_exit((void *) -1);
        } else if (ret > 0) {
            /*
             * iface down/suspend.
             */
			
            while (1) {
                if ((ret2 = iface_check_running()) < 0) {
                    pthread_exit((void *) -1);
                } else if (ret2 == 0) {
                    /*
                     * Iface up/restore.
                     * Reboot the daemon.
                     */
                    kill(0, SIGHUP);
                    break;
                }
				
                /*
                 * Check each 1 second.
                 */
#ifdef NETBSD
                sleep(1);
#else
                usleep(1000000);
#endif
            }
        }
		
        /*
         * Iface up.
         * Check each 1 second.
         */
#ifdef NETBSD
        sleep(1);
#else
        usleep(1000000);
#endif
    }
	
    pthread_exit((void *) 0);
}

/*
 * Sets the network iface.
 */
static void
iface_set_name(char *name)
{

    memset(dev.dev, '\0', IF_NAMESIZE);
    strncpy(dev.dev, name, IF_NAMESIZE);
}

/*
 * Unset the network iface.
 */
static void
iface_unset_name(void)
{

    memset(dev.dev, '\0', IF_NAMESIZE);
}

/*
 * Checks the datalink. Accepting EN10MB only 
 * (and so only ethernet and wireless ifaces).
 */
static int
iface_check_datalink(char *devarg)
{
    char errbuf[PCAP_ERRBUF_SIZE];
    pcap_t *pcap;
    int datalink;	

    if ((pcap = pcap_open_live(devarg, BUFSIZ, 0, 0, errbuf)) == NULL) {
        ERROR(errbuf);

        return (-1);
    }
	
    if ((datalink = pcap_datalink(pcap)) < 0) {
        ERROR(pcap_geterr(pcap));

        return (-1);
    }
	
    /* 
     * Set network offset if it is 
     * ethernet network iface.
     */
    if (datalink == DLT_EN10MB) {
        dev.dev_offset = 14;
        pcap_close(pcap);

        return (0);
    }
	
    pcap_close(pcap);

    return (-1);
}

/*
 * Checks if iface is running.
 */
static int
iface_check_running(void)
{
    struct ifreq ifr;
    int sd;
	
    if ((sd = socket(AF_INET, SOCK_DGRAM, 0)) < 0) {
        ERROR(strerror(errno));
		
        return (-1);
    }
	
    memset(ifr.ifr_name, '\0', IF_NAMESIZE);
    strncpy(ifr.ifr_name, dev.dev, IF_NAMESIZE);
		
    if (ioctl(sd, SIOCGIFFLAGS, &ifr) < 0) {
        ERROR(strerror(errno));
        close(sd);
			
        return (-1);
    }
	
    /*
     * iface is not running.
     */
    if ((ifr.ifr_flags & IFF_RUNNING) == 0) {
        close(sd);
		
        return (1);
    }
	
    /*
     * iface running.
     */
    close(sd);
	
    return (0);
}

/*
 * Wait first packet for online check.
 */
static int
iface_check_online(void)
{
    struct bpf_program compiled_filter;
    char errbuf[PCAP_ERRBUF_SIZE], *filter = "arp";
    pcap_t *pcap;
    int ret;
#ifndef LINUX
    unsigned int op = 1;
#endif
	
    if ((pcap = pcap_open_live(dev.dev, BUFSIZ, 0, 0, errbuf)) == NULL) {
        ERROR(errbuf);

        return (-1);
    }
	
#ifndef LINUX
    /*
     * BSD, differently from linux does not
     * support automatic socket soft real time
     * (Linux Socket Filter).
     * Therefore on BSD platform it's necessary
     * to use this I/O Control.
     */
    if (ioctl(pcap_fileno(pcap), BIOCIMMEDIATE, &op) < 0) {
        ERROR(strerror(errno));
        pcap_close(pcap);
	
        return (-1);
    }
#endif
	
    if (pcap_compile(pcap, &compiled_filter, 
        filter, 0, dev.dev_netmask.s_addr) < 0) {
        ERROR(pcap_geterr(pcap));
        pcap_close(pcap);
	
        return (-1);
    }
	
    if (pcap_setfilter(pcap, &compiled_filter) < 0) {
        ERROR(pcap_geterr(pcap));
        pcap_close(pcap);
		
        return (-1);
    }
	
    /*
     * Wait first packet.
     */
    if ((ret = pcap_loop(pcap, 1, iface_check_online_packet, NULL)) < 0) {
        pcap_freecode(&compiled_filter);

        switch (ret) {
        case (-1):
            ERROR(pcap_geterr(pcap));
            pcap_close(pcap);

            return (-1);

        case (-2):
        default:
            pcap_close(pcap);
            /* 
             * Offline.
             */
            return (1);
        }
    }

    /*
     * Online.
     */
	
    pcap_freecode(&compiled_filter);
    pcap_close(pcap);
		
    return (0);
}

/*
 * Wait an Arp packet.
 */
static void
iface_check_online_packet(unsigned char *arg, const struct pcap_pkthdr
    *header, const unsigned char *packet)
{

    /*
     * Arp traffic is ok.
     */
    return;
}

/*
 * Putting down the promiscue flag if 
 * found set in network iface.
 */
static int
iface_del_promisc(void)
{	
    struct ifreq ifr;
    int sd;
	
    if ((sd = socket(AF_INET, SOCK_STREAM, 0)) < 0) {
        ERROR(strerror(errno));
		
        return (-1);
    }
	
    strncpy(ifr.ifr_name, dev.dev, sizeof(char) * IF_NAMESIZE);
    ifr.ifr_name[sizeof(char) * strlen(ifr.ifr_name)] = '\0';
	
    if (ioctl(sd, SIOCGIFFLAGS, &ifr) < 0) {
        ERROR(strerror(errno));
        close(sd);

        return (-1);
    }

    if (ifr.ifr_flags & IFF_PROMISC) {
        /* 
         * Remove Promisc flag 
         */
        ifr.ifr_flags &= ~IFF_PROMISC;
		
        if (ioctl(sd, SIOCSIFFLAGS, &ifr) < 0) {
            ERROR(strerror(errno));
            close(sd);

            return (-1);
        }
    }

    close(sd);
    return (0);
}

/*
 * Reading network ifaces hw-MAC address. 
 */
static int
iface_get_mac_address(void)
{
    eth_addr_t mac;
    eth_t *if_eth;
    int i;
	
    if ((if_eth = eth_open(dev.dev)) == NULL) {
        ERROR(strerror(errno));
		
        return (-1);
    }
	
    if (eth_get(if_eth, &mac) < 0) {
        ERROR(strerror(errno));
        eth_close(if_eth);

        return (-1);
    }
	
    for (i = 0; i < ETHER_ADDR_LEN; i++) {
        dev.dev_mac.octet[i] = mac.data[i];
    }

    eth_close(if_eth);

    return (0);
}

/*
 * Checks and saves IPv4 and Netmask network adresses.
 */
static int 
iface_get_inet4_addresses(void)
{
    char errbuf_lnet[LIBNET_ERRBUF_SIZE], errbuf_pcap[PCAP_ERRBUF_SIZE];
    libnet_t *lnet;
    bpf_u_int32 network;
    int ret, ret2, state = 0;
	
    /* 
     * Get netmask and IPv4 addresses.
     * Case of errors, it waits:
     *	- running iface
     *	- IPv4 assigned
     *	- online iface.
     */
    while (1) {
        dev.dev_netmask.s_addr = 0;
		
        if (pcap_lookupnet(dev.dev, &network, 
            &(dev.dev_netmask.s_addr), errbuf_pcap) < 0) {
            if (dif == IFACE_LIST) {
                return (0);
            } else {
                if (errno == EADDRNOTAVAIL) {
                    /*
                     * Use the state for one first WARNING print.
                     */
                    if (state == 0) {
                        aprintf(stdout, 1, "WARNING: %s\n", errbuf_pcap);
						
                        if (dinspec == INSPEC_NULL) {
                            exit(EXIT_FAILURE);
                        }
						
                        state = 1;
                    }
                    continue;
                } else {
                    if (dif != IFACE_AUTO) {
                        ERROR(errbuf_pcap);
                    }
					
                    return (-1);
                }
            }
        } else {
            /* Checks iface running.
             * Return:
             *	-1: Error
             *	 0: iface running
             *	 1: iface is not running.
             */
            if ((ret = iface_check_running()) < 0) {
                return (-1);
            } 
            /*
             * iface running.
             */
            else if (ret == 0) {
                /*
                 * Check iface online with 
                 * first packet received.
                 */
                if (dif != IFACE_LIST && dinspec != INSPEC_NULL) {
                    aprintf(stdout, 1, "Wait link connection on %s...\n", 
                            dev.dev);
					
                    while (1) {
                        /*
                         * Check iface online.
                         * Return:
                         *  -1: Error
                         *   0: Online
                         *   1: Offline
                         */
                        if ((ret2 = iface_check_online()) < 0) {
                            return (-1);
                        } else if (ret2 == 1) {
                            sleep(1);
                            continue;
                        } else {
                            break;
                        }
                    }
                }
				
                /* 
                 * Get IPv4 address.
                 */
                if ((lnet = libnet_init(LIBNET_LINK, dev.dev, 
                    errbuf_lnet)) == NULL) {
                    if (dif == IFACE_LIST) {
                        return (0);
                    } else { 
                        ERROR(errbuf_lnet);

                        return (-1);
                    }
                }
			
                dev.dev_inet4.s_addr = 0;
			
                if ((dev.dev_inet4.s_addr = libnet_get_ipaddr4(lnet)) < 0) {
                    ERROR(libnet_geterror(lnet));

                    return (-1);	
                }
			
                libnet_destroy(lnet);
                break;
            }
			
            /*
             * iface is not running.
             */
        }
		
        /*
         * Check each 1 second.
         */
        sleep(1);
    }
	
    if (dinspec == INSPEC_NULL) {
        aprintf(stdout, 1, "dev(%s) ", dev.dev);
        aprintf(stdout, 0, "inet(%s) ", inet_ntoa(dev.dev_inet4));
        aprintf(stdout, 0, "hw(%s)\n", ether_ntoa(&(dev.dev_mac)));
	
        switch (dif) {
        case (IFACE_MANUAL):
        case (IFACE_AUTO):
            aprintf(stdout, 0, "\n");
            break;

        case (IFACE_LIST):
        case (IFACE_NULL):
            break;
        } 
    }
	
    return (0);
}

/*
 * Prints network ifaces informations.
 * For SARPI and DARPI.
 */
static void 
iface_info_print(void)
{
	
    switch (dinspec) {
    case (INSPEC_SARPI):
        aprintf(stdout, 1, "SARPI on ");
        break;
			
    case (INSPEC_DARPI):
        aprintf(stdout, 1, "DARPI on ");
        break;
		
    case (INSPEC_NULL):
        break;
    }
	
    aprintf(stdout, 0, "dev(%s) ", dev.dev);
    aprintf(stdout, 0, "inet(%s) ", inet_ntoa(dev.dev_inet4));
    aprintf(stdout, 0, "hw(%s)\n", ether_ntoa(&(dev.dev_mac)));
}

/********************
 * Arp Cache handler:
 ********************/

/*
 * Open Arp cache.
 */
static arp_t *
arp_cache_open(void)
{
    arp_t *arp;
	
    if ((arp = arp_open()) == NULL) {
        ERROR(strerror(errno));
        
        return (NULL);
    }
	
    return (arp);
}

/*
 * Close Arp cache.
 */
static void
arp_cache_close(arp_t *arp)
{
	
    arp_close(arp);
}

/*
 * Adds Arp cache entry.
 */
static int
arp_cache_add(enum cache_t type, char *arg)
{
    struct arp_entry entry;
    char c_mac[18], c_ip[16];
    arp_t *arp;
	
    if ((arp = arp_cache_open()) == NULL) {
    return (-1);
    }	

    sscanf(arg, "%15s %17s", c_ip, c_mac);
    c_mac[sizeof(char) * strlen(c_mac)] = '\0';
    c_ip[sizeof(char) * strlen(c_ip)] = '\0'; 
	
    if (addr_aton(c_mac, &entry.arp_ha) < 0) {
        ERROR(strerror(errno));
        arp_cache_close(arp);
		
        return (-1);
    }
	
    if (addr_aton(c_ip, &entry.arp_pa) < 0) {
        ERROR(strerror(errno));	
        arp_cache_close(arp);
		
        return (-1);
    }
	
    arp_add(arp, &entry);
	
    /*
     * These outputs depends by asked functionally.
     */	
    switch (type) {
    case (SARPI_REFRESH):
    case (SARPI):
    case (DARPI):
        aprintf(stdout, 0, "%s -> %s\n", c_ip, c_mac);
        break;
		
    case (DARPI_RESET):
        break;
    }
	
    arp_cache_close(arp);
	
    return (0);
}

/*
 * Search entry in Arp cache, if is found it will delete. 
 */
static int
arp_cache_del(enum cache_t type, char *arg)
{
    struct arp_entry entry;
    arp_t *arp;
    int i;
	
    if ((arp = arp_cache_open()) == NULL) {
        return (-1);
    }
	
    /* 
     * Read Arp cache. 
     */
    if (arp_loop(arp, arp_cache_list_create, NULL) < 0) {
        arp_cache_close(arp);
		
        return (-1);
    }
	
    /* 
     * Search IPv4 or hw-MAC in Arp Cache.
     */
    TAILQ_FOREACH(arp_cache_pos, &arp_cache_head, entries) {
        if (strcmp(arg, inet_ntoa(arp_cache_pos->ac_ip)) == 0 || 
            strcmp(arg, ether_ntoa(&arp_cache_pos->ac_mac)) == 0) { 
            for (i = 0; i < ETHER_ADDR_LEN; i++) {
                entry.arp_ha.addr_eth.data[i] = arp_cache_pos->ac_mac.octet[i];
            }

            if (addr_aton(inet_ntoa(arp_cache_pos->ac_ip), &entry.arp_pa) < 0) {
                ERROR(strerror(errno));
                arp_cache_list_destroy();               
                arp_cache_close(arp);

                return (-1);
            }
		
            arp_delete(arp, &entry);
			
            /*
             * These output depends by asked functionally.
             */	
            switch (type) {
            case (SARPI_REFRESH):
            case (SARPI):
            case (DARPI_RESET):
                break;
					
            case (DARPI):
                aprintf(stdout, 0, "%s -> %s\n",
                        inet_ntoa(arp_cache_pos->ac_ip), 
                        ether_ntoa(&arp_cache_pos->ac_mac));
                break;
            }
			
            arp_cache_list_destroy();               
            arp_cache_close(arp);

            return (0);
        }
    }

    if (type == DARPI) {
        aprintf(stdout, 0, "%s\n", arg);
		
        arp_cache_list_destroy();               
        arp_cache_close(arp);

        return (0);
    }
	
    aprintf(stderr, 0, "  ERROR: %s entry not found!\n\n", arg);
	
    arp_cache_list_destroy();               
    arp_cache_close(arp);

    return (-1);
}

/*
 * Delete all found entries in the Arp 
 * cache reading Arp cache tail.
 */
static int
arp_cache_del_all(void)
{
    struct arp_entry entry;
    arp_t *arp;
    int i, j = 1;
	
    if ((arp = arp_cache_open()) == NULL) {
        return (-1);
    }
	
    /* 
     * Read Arp cache. 
     */
    if (arp_loop(arp, arp_cache_list_create, NULL) < 0) {
        arp_cache_close(arp);
		
        return (-1);
    }
	
    TAILQ_FOREACH(arp_cache_pos, &arp_cache_head, entries) {
        for (i = 0; i < ETHER_ADDR_LEN; i++) {
            entry.arp_ha.addr_eth.data[i] = arp_cache_pos->ac_mac.octet[i];
        }
		
        if (addr_aton(inet_ntoa(arp_cache_pos->ac_ip), &entry.arp_pa) < 0) {
            ERROR(strerror(errno));
            arp_cache_list_destroy();               
            arp_cache_close(arp);
			
            return (-1);
        }
		
        arp_delete(arp, &entry);
		
        aprintf(stdout, 1, "%d) %15s -> %17s\n", j++, 
                inet_ntoa(arp_cache_pos->ac_ip), 
                ether_ntoa(&arp_cache_pos->ac_mac));
    }
	
    arp_cache_list_destroy();
    arp_cache_close(arp);
	
    return (0);
}

/*
 * Create Arp cache tail using one entry or it adds simply a new entry.
 */
static int
arp_cache_list_create(const struct arp_entry *entry, void *arg)
{
    struct ether_addr mac;
    struct in_addr ip;
    register int i;

    for (i = 0; i < ETHER_ADDR_LEN; i++) {
        mac.octet[i] = entry->arp_ha.addr_eth.data[i];
    }

    ip.s_addr = entry->arp_pa.addr_ip;
	
    if (arp_cache_begin == NULL) { 
        TAILQ_INIT(&arp_cache_head);
	
        if ((arp_cache_begin = malloc(sizeof(struct arp_cache))) == NULL) { 
            ERROR(strerror(errno));

            return (-1);
        }
		
        memcpy(&arp_cache_begin->ac_mac, &mac, sizeof(mac));
        memcpy(&arp_cache_begin->ac_ip, &ip, sizeof(ip));
        TAILQ_INSERT_HEAD(&arp_cache_head, arp_cache_begin, entries);
    } else {
        if ((arp_cache_next = malloc(sizeof(struct arp_cache))) == NULL) { 
            ERROR(strerror(errno));

            return (-1);
        }
		
        memcpy(&arp_cache_next->ac_mac, &mac, sizeof(mac));
        memcpy(&arp_cache_next->ac_ip, &ip, sizeof(ip));
        TAILQ_INSERT_TAIL(&arp_cache_head, arp_cache_next, entries);
    }
	
    return (0);
}

/*
 * Destroy Arp cache tail.
 */
static void
arp_cache_list_destroy(void)
{

    while (TAILQ_EMPTY(&arp_cache_head) == 0) {
        arp_cache_pos = TAILQ_FIRST(&arp_cache_head);
		
        TAILQ_REMOVE(&arp_cache_head, arp_cache_pos, entries);
        free(arp_cache_pos);
    }
}


/******************
 * SARPI handler: *
 ******************/

/*
 * Sets Arp cache timeout for automatic update.
 */
static void
sarpi_set_timeout(char *timeout)
{

    sarpi_timeout = atoi(timeout);
}

/*
 * SARPI manager signal handler.
 */
static void
sarpi_manager_thread_sigusr1(int sig)
{
	
    sarpi_cache_list_destroy();
    pthread_exit((void *) 0);
}

/*
 * Handles SARPI through two thread for parallelism:
 *  - 1:    Update automatically the Arp cache
 *  - 2:    Works in soft real time, in other words it 
 *          listens to the inbound/outbound arp packets.	
 */
static void *
sarpi_manager_thread(void *arg)
{
    struct sigaction saction;
	
#ifdef NETBSD
    saction.sa_flags = 0;
#endif
    saction.sa_handler = sarpi_manager_thread_sigusr1;
	
    if (sigaction(SIGUSR1, &saction, NULL) < 0) {
        ERROR(strerror(errno));
		
        pthread_exit((void *) -1);
    }
	
    /*
     * Iface work.
     */
    if (iface_manager() < 0) {
        pthread_exit((void *) -1);
    }
	
    if (iface_check_uplink() < 0) {
        pthread_exit((void *) -1);
    }
	
    iface_info_print();

    /* 
     * PID Parent CPU Scheduling. 
     */
    if (task_mode_cpu_priority(PRIO_PROCESS, getpid(), cpu_priority) < 0) {
        pthread_exit((void *) -1);
    }

    pthread_rwlock_init(&rlock, NULL);
    pthread_rwlock_init(&wlock, NULL);

    /* 
     * Arp Cache entries protected from file. 
     */
    if (sarpi_cache_file_restore() < 0) {
        exit(-1);
    }
	
    pthread_attr_init(&detach_attr);
    pthread_attr_setdetachstate(&detach_attr, PTHREAD_CREATE_DETACHED);
    pthread_attr_setschedpolicy(&detach_attr, SCHED_RR);
	
    /*
     * Thread 4 detached, update automatically the Arp cache. 
     */
    if (pthread_create(&thread[3], &detach_attr, 
        sarpi_cache_list_refresh_thread, (void *) NULL) != 0) {
        ERROR(strerror(errno));
        sarpi_cache_list_destroy();
		
        pthread_exit((void *) -1);
    }

    /*
     * Thread 5 detached, realtime inbound/outbound work.
     */
    if (sarpi_realtime() < 0) {
        sarpi_cache_list_destroy();
		
        pthread_exit((void *) -1);
    }

    pthread_exit((void *) 0);
}

/*
 * SARPI Realtime, process Arp reply 
 * inbound/outbound Arp packets.
 */
static int
sarpi_realtime(void)
{
	
    aprintf(stdout, 1, "Arp Cache refresh timeout: %d %s\n", 
            sarpi_timeout, sarpi_timeout == 1 ? "minut." : "minuts.");
	
    aprintf(stdout, 1, "Realtime Protect actived!\n");

    pthread_attr_init(&detach_attr);
    pthread_attr_setdetachstate(&detach_attr, PTHREAD_CREATE_DETACHED);
    pthread_attr_setschedpolicy(&detach_attr, SCHED_RR);

    /*
     * Thread 5 detached, Works in soft real time, in other 
     * words it listens to the inbound/outbound Arp packets.
     */
    if (pthread_create(&thread[4], &detach_attr, sarpi_realtime_thread, 
        (void *) NULL) != 0) {
        ERROR(strerror(errno));
	
        return (-1);
    }
	
    pthread_attr_destroy(&detach_attr);
	
    return (0);
}

/*
 * SARPI realtime signal handler.
 */
static void
sarpi_realtime_thread_sigusr1(int sig)
{
	
    pthread_exit((void *) 0);
}

/*
 * Use thread for non blocking to pcap_loop().
 */
static void *
sarpi_realtime_thread(void *arg)
{
    struct sigaction saction;
    pcap_t *pcap;
	
#ifdef NETBSD
    saction.sa_flags = 0;
#endif
    saction.sa_handler = sarpi_realtime_thread_sigusr1;
	
    if (sigaction(SIGUSR1, &saction, NULL) < 0) {
        ERROR(strerror(errno));
		
        pthread_exit((void *) -1);
    }
	
    if ((pcap = sarpi_realtime_open_packets()) == NULL) {
        pthread_exit((void *) -1);
    }

    while (1) {
        if (pcap_loop(pcap, 1, sarpi_realtime_read_packets, NULL) < 0) {
            ERROR(pcap_geterr(pcap));
            sarpi_realtime_close_packets(pcap);

            pthread_exit((void *) -1);
        }
    }

    sarpi_realtime_close_packets(pcap);

    pthread_exit((void *) 0);
}

/*
 * Open pcap file descriptor.
 */
static pcap_t *
sarpi_realtime_open_packets(void)
{
    struct bpf_program compiled_filter;
    char errbuf[PCAP_ERRBUF_SIZE], *filter = "arp";
    pcap_t *pcap;
#ifndef LINUX
    unsigned int op = 1;
#endif

    if ((pcap = pcap_open_live(dev.dev, BUFSIZ, 0, 0, errbuf)) == NULL) {
        ERROR(errbuf);
		
        return (NULL);
    }

#ifndef LINUX
    /* 
     * BSD, differently from linux does not 
     * support automatic socket soft real time 
     * (Linux Socket Filter). 
     * Therefore on BSD platform it's necessary 
     * to use this I/O Control.
     */
    if (ioctl(pcap_fileno(pcap), BIOCIMMEDIATE, &op) < 0) {
        ERROR(strerror(errno));
		
        return (NULL);
    }
#endif

    if (pcap_compile(pcap, &compiled_filter, filter, 0, 
        dev.dev_netmask.s_addr) < 0) {
        ERROR(pcap_geterr(pcap));
        sarpi_realtime_close_packets(pcap);

        return (NULL);
    }
	
    if (pcap_setfilter(pcap, &compiled_filter) < 0) {
        ERROR(pcap_geterr(pcap));
        sarpi_realtime_close_packets(pcap);

        return (NULL);
    }	
	
    pcap_freecode(&compiled_filter);
    return (pcap);
}

/*
 * Close pcap file descriptor.
 */
static void
sarpi_realtime_close_packets(pcap_t *pcap)
{

    pcap_close(pcap);
}

/*
 * Reads inbound/outbound packets such as: 
 *  - Arp reply/request packets 
 *      (it checks destionation/source address).
 *      If Arp reply/request's data like addresses 
 *      IPv4 and MAC match a static entry of Arp 
 *      cache protected by SARPI, update the 
 *      entry using the static one.
 */
static void
sarpi_realtime_read_packets(unsigned char *arg, const struct pcap_pkthdr 
    *header, const unsigned char *packet)
{
    struct ether_addr src_mac, dst_mac;
    struct arp_header *arp_packet;
    char c_src_ip[16], c_src_mac[18], c_dst_ip[16], c_dst_mac[18], 
         c_ap_ip[16], entry[34];
    int i;

    /* 
     * Arp Packet. 
     */	
    arp_packet = (struct arp_header *) (packet + dev.dev_offset);

    /*
     * Convert the source MAC/IPv4 to string.
     */
    snprintf(c_src_ip, 16, "%d.%d.%d.%d", 
             arp_packet->ah_addresses.ar_spa[0], 
             arp_packet->ah_addresses.ar_spa[1],
             arp_packet->ah_addresses.ar_spa[2], 
             arp_packet->ah_addresses.ar_spa[3]);
		
    for (i = 0; i < ETHER_ADDR_LEN; i++) {
        src_mac.octet[i] = arp_packet->ah_addresses.ar_sha[i];
    }	
    strncpy(c_src_mac, ether_ntoa(&src_mac), sizeof(char) * 18);
    c_src_mac[sizeof(char) * strlen(c_src_mac)] = '\0';

    /*
     * Convert the destination MAC/IPv4 to string
     */
    snprintf(c_dst_ip, 16, "%d.%d.%d.%d", 
             arp_packet->ah_addresses.ar_tpa[0], 
             arp_packet->ah_addresses.ar_tpa[1],
             arp_packet->ah_addresses.ar_tpa[2], 
             arp_packet->ah_addresses.ar_tpa[3]);

    for (i = 0; i < ETHER_ADDR_LEN; i++) {
        dst_mac.octet[i] = arp_packet->ah_addresses.ar_tha[i];
    }

    strncpy(c_dst_mac, ether_ntoa(&dst_mac), sizeof(char) * 18);
    c_dst_mac[sizeof(char) * strlen(c_dst_mac)] = '\0';

    /* 
     * In request, possible source address poisoned.
     * In reply, possible Arp Gratuitous for Arp poisoning.
     */
    if (ntohs(arp_packet->ah_header.ar_op) == ARP_OP_REPLY || 
        ntohs(arp_packet->ah_header.ar_op) == ARP_OP_REQUEST) {
        /* 
         * Check if we are the destionation address. 
         */
        if (strcmp(c_dst_ip, inet_ntoa(dev.dev_inet4)) == 0) {
            /* 
             * Search Arp reply/request source in SARPI (Arp cache) entries. 
             */
            TAILQ_FOREACH(sarpi_cache_pos, &sarpi_cache_head, entries) {
                /*
                 * Convert MAC/IPv4 to string form.
                 */
                memset(c_ap_ip, '\0', sizeof(char) * 16);
                strncpy(c_ap_ip, inet_ntoa(sarpi_cache_pos->sc_ip), 
                        sizeof(char) * 16);
                c_ap_ip[sizeof(char) * strlen(c_ap_ip)] = '\0';

                /* 
                 * In request, possible source address poisoned.
                 * In reply, possible Arp Gratuitous for Arp poisoning.
                 * Check if source is found in SARPI Cache. 
                 */
                if (strcmp(c_src_ip, c_ap_ip) == 0) {
                    /* 
                     * Source is Found! We reset static entry.
                     */

                    switch (ntohs(arp_packet->ah_header.ar_op)) {
                    case (ARP_OP_REPLY):
                        aprintf(stdout, 1, "Reply   << Refresh entry ");
                        break;

                    case (ARP_OP_REQUEST):
                        aprintf(stdout, 1, "Request << Refresh entry ");
                        break;
                    }

                    /* 
                     * Refresh static Arp Cache entry. 
                     */
                    snprintf(entry, 34, "%15s %17s", 
                             inet_ntoa(sarpi_cache_pos->sc_ip),
                             ether_ntoa(&(sarpi_cache_pos->sc_mac)));
                    entry[strlen(entry)] = '\0';
					
                    if (arp_cache_add(SARPI, entry) < 0) {
                        exit(EXIT_FAILURE);
                    } 
	
                    return;
                }
            }

            /* 
             * In request, possible source address poisoned.
             * In reply, possible Arp Gratuitous for Arp poisoning.
             * Source is not found in SARPI cache.
             * This entry is not protected by SARPI before
             * this is not present in SARPI cache.
             */
            switch (ntohs(arp_packet->ah_header.ar_op)) {
            case (ARP_OP_REPLY):	
                aprintf(stdout, 1, "Reply   << Ignore entry ");
                break;

            case (ARP_OP_REQUEST):
                aprintf(stdout, 1, "Request << Ignore entry ");
                break;
            }

            aprintf(stdout, 0, "%s -> %s\n", c_src_ip, c_src_mac);
        } 
        /* 
         * Check if we are the source address. 
         */
        else if (strcmp(c_src_mac, ether_ntoa(&(dev.dev_mac))) == 0 && 
                 strcmp(c_src_ip, inet_ntoa(dev.dev_inet4)) == 0) {
            switch (ntohs(arp_packet->ah_header.ar_op)) {
            case (ARP_OP_REPLY):	
                aprintf(stdout, 1, "Reply   >> Send to ");
                break;
			
            case (ARP_OP_REQUEST):
                aprintf(stdout, 1, "Request >> Send to ");
                break;
            }

            aprintf(stdout, 0, "%s -> %s\n", c_dst_ip, c_dst_mac);
        }
    }
}

/*
 * Sets static Arp cache entries from file.
 * Parsing of example:
 *
 * # Example of sarpi.cache
 * #
 * 192.168.1.1 aa:bb:cc:dd:ee:ff
 * ...
 */
static int
sarpi_cache_file_restore(void)
{
    struct addr aip, amac;
    struct arp_entry entry;
    char buf[100], ip[16], mac[18];
    FILE *fd;
    int i, j;
	
    if ((fd = fopen(sarpi_cache_file, "r")) == NULL) {
        aprintf(stderr, 0, "\n  ERROR: Configure static " \
                "Arp Cache entries in %s!\n\n", sarpi_cache_file);
		
        return (-1);
    }
	
    aprintf(stdout, 1, "Protects these Arp Cache's entries:\n");
	
    for (i = 1, j = 1; feof(fd) == 0; i++) {
        if (fgets(buf, 100, fd) == NULL) {
            break;
        }
		
        /*
         * Comment or new line.
         */
        if (buf[0] == '#' || buf[0] == '\n') {
            continue;
        }
		
        /*
         * Space line and Tab line with no entry.
         */
        if (buf[0] == ' ' || buf[0] == '\t') {
            if (buf[1] == '#' || buf[1] == '\n' || 
                buf[1] == ' ' || buf[1] == '\t')
                continue;
        }
		
        memset(ip, '\0', 16);
        memset(mac, '\0', 18);
	
        sscanf(buf, "%15s %17s", ip, mac);
		
        if (addr_pton(ip, &aip) < 0) {
            aprintf(stderr, 0, "\n  ERROR: " 
                    "It is not IPv4. Reconfigure %s:%d line!\n\n",
                    sarpi_cache_file, i);
            fclose(fd);
			
            return (-1);
        } else if (addr_pton(mac, &amac) < 0) {
            aprintf(stderr, 0, "\n  ERROR: "
                    "It is not Mac-hw. Reconfigure %s:%d line!\n\n",
                    sarpi_cache_file, i);
            fclose(fd);
			
            return (-1);
        }
		
        aprintf(stdout, 1, "%d) %15s -> %17s\n", j++, ip, mac);
		
        memcpy(&(entry.arp_pa), &aip, sizeof(aip));
        memcpy(&(entry.arp_ha), &amac, sizeof(amac));
		
        if (sarpi_cache_list_create(&entry, NULL) < 0) {
            fclose(fd);
			
            return (-1);
        }   
    }
	
    aprintf(stdout, 1, "Arp Cache restore from %s...\n", sarpi_cache_file);
	
    fclose(fd);
    return (0);
}

/*
 * Adds SARPI cache tail.
 */
static int
sarpi_cache_list_create(const struct arp_entry *entry, void *arg)
{
    struct ether_addr mac;
    struct in_addr ip;
    int i;
	
    for (i = 0; i < ETHER_ADDR_LEN; i++) {
        mac.octet[i] = entry->arp_ha.addr_eth.data[i];
    }
	
    ip.s_addr = entry->arp_pa.addr_ip;
	
    pthread_rwlock_wrlock(&wlock);
	
    if (sarpi_cache_begin == NULL) { 
        TAILQ_INIT(&sarpi_cache_head);
		
        if ((sarpi_cache_begin = malloc(sizeof(struct arp_cache))) == NULL) { 
            ERROR(strerror(errno));		
			
            pthread_rwlock_unlock(&wlock);
			
            return (-1);
        }
		
        memcpy(&sarpi_cache_begin->sc_mac, &mac, sizeof(mac));
        memcpy(&sarpi_cache_begin->sc_ip, &ip, sizeof(ip));
        TAILQ_INSERT_HEAD(&sarpi_cache_head, sarpi_cache_begin, entries);
		
        pthread_rwlock_unlock(&wlock);
		
        return (0);
    } 
	
    if ((sarpi_cache_next = malloc(sizeof(struct arp_cache))) == NULL) { 
        ERROR(strerror(errno));
		
        pthread_rwlock_unlock(&wlock);
		
        return (-1);
    }
	
    memcpy(&sarpi_cache_next->sc_mac, &mac, sizeof(mac));
    memcpy(&sarpi_cache_next->sc_ip, &ip, sizeof(ip));
    TAILQ_INSERT_TAIL(&sarpi_cache_head, sarpi_cache_next, entries);
	
    pthread_rwlock_unlock(&wlock);
	
    return (0);
}

/* 
 * SARPI cache list signal handler.
 */
static void
sarpi_cache_list_refresh_thread_sigusr1(int sig)
{
	
    pthread_exit((void *) 0);
}

/*
 * During every timeout it updates SARPI 
 * cache static entries in Arp cache.
 */
static void * 
sarpi_cache_list_refresh_thread(void *arg)
{
    struct sigaction saction;
    char entry[34];
    int i;
#ifdef NETBSD
    int ret = 0;
#endif
	
#ifdef NETBSD
    saction.sa_flags = 0;
#endif
    saction.sa_handler = sarpi_cache_list_refresh_thread_sigusr1;
	
    if (sigaction(SIGUSR1, &saction, NULL) < 0) {
        ERROR(strerror(errno));
		
        pthread_exit((void *) -1);
    }
	
    for (;;) {
        /* 
         * Sleep for thread suspend.
         */
#ifdef NETBSD
        /*
         * Convert by seconds to minuts.
         */
        if ((ret = sleep(sarpi_timeout * 60)) < 0) {
#else 
        /*
         * Convert by microseconds to minuts. 
         */
        if (usleep((sarpi_timeout * 1000000) * 60) < 0) {
#endif
            ERROR(strerror(errno));
				
            pthread_exit((void *) -1);
        }
			
#ifdef NETBSD
        /*
         * For SIGINT, SIGTERM, SIGQUIT,
         * It exited.
         */
			
        if (ret != 0) {
            pthread_exit((void *) -1);
        }
#endif
			
        aprintf(stdout, 1, "Refresh these Arp Cache entries:\n");
			
        /* 
         * Count entries. 
         */		
        i = 1;
			
        pthread_rwlock_rdlock(&rlock);
        pthread_rwlock_wrlock(&wlock);
			
        TAILQ_FOREACH(sarpi_cache_pos, &sarpi_cache_head, entries) {
            aprintf(stdout, 1, "%d) ", i);
				
            /* 
             * Refresh Arp Cache entries. 
             */
            snprintf(entry, 34, "%15s %17s", 
                     inet_ntoa(sarpi_cache_pos->sc_ip),
                     ether_ntoa(&(sarpi_cache_pos->sc_mac)));
            entry[strlen(entry)] = '\0';
				
            if (arp_cache_add(SARPI_REFRESH, entry) < 0) {
                pthread_rwlock_unlock(&wlock);
                pthread_rwlock_unlock(&rlock);
					
                pthread_exit((void *) -1);
            }
				
            i++;
        }
			
        pthread_rwlock_unlock(&wlock);
        pthread_rwlock_unlock(&rlock);
    }
		
    pthread_exit((void *) 0);
}
	
/*
 * Destroy SARPI cache tail.
 */
static void
sarpi_cache_list_destroy(void)
{
		
    pthread_rwlock_rdlock(&rlock);
		
    while (TAILQ_EMPTY(&sarpi_cache_head) == 0) {
        pthread_rwlock_wrlock(&wlock);
			
        sarpi_cache_pos = TAILQ_FIRST(&sarpi_cache_head);
			
        TAILQ_REMOVE(&sarpi_cache_head, sarpi_cache_pos, entries);
        free(sarpi_cache_pos);
			
        pthread_rwlock_unlock(&wlock);
    }
		
    pthread_rwlock_unlock(&rlock);
}	
	
/******************
 * DARPI handler: *
 ******************/

/*
 * Set DARPI Cache entry timeout.
 */
static int
darpi_set_timeout(char *timeout)
{
    int val = atoi(timeout);
	
#ifdef NETBSD
    /*
     * usleep() for NetBSD required < 1.000.000.
     */
    if ((val * 1000) >= 1000000) {
        aprintf(stderr, 0, "  ERROR: DARPI timeout %d is too large.\n\n", 
                val * 1000);
		
        return (-1);
    }
#endif
	
    darpi_timeout = val;
	
    return (0);
}

/*
 * DARPI manager signal handler.
 */
static void
darpi_manager_thread_sigusr1(int sig)
{
	
    darpi_cache_list_destroy();
    pthread_exit((void *) 0);
}
	
/* 
 * Handles DARPI, delete all found entries 
 * in the Arp cache to delete some poisoned 
 * hosts then it starts realtime execution
 * to reads the packets:
 *  - Arp request
 *  - Arp reply.
 */
static void *
darpi_manager_thread(void *arg)
{
    struct sigaction saction;
	
#ifdef NETBSD
    saction.sa_flags = 0;
#endif
    saction.sa_handler = darpi_manager_thread_sigusr1;
	
    if (sigaction(SIGUSR1, &saction, NULL) < 0) {
        ERROR(strerror(errno));
		
        pthread_exit((void *) -1);
    }
	
    /*
     * Iface work.
     */
    if (iface_manager() < 0) {
        pthread_exit((void *) -1);
    }
	
    if (iface_check_uplink() < 0) {
        pthread_exit((void *) -1);
    }
	
    iface_info_print();

    /* 
     * PID CPU Scheduling.
     */
    if (task_mode_cpu_priority(PRIO_PROCESS, getpid(), cpu_priority) < 0) {
        pthread_exit((void *) -1);
    }

    aprintf(stdout, 1, "Deletes these Arp Cache entries:\n");
	
    /* 
     * Delete all Arp Cache entries (Possible entries poisoned). 
     */
    if (arp_cache_del_all() < 0) {
        pthread_exit((void *) -1);
    }

    /*
     * Thread 4 detached, inbound/outbound work.
     */
    if (darpi_realtime() < 0) {
        pthread_exit((void *) -1);
    }

    pthread_exit((void *) 0);
}

/*
 * DARPI Realtime execution, process all 
 * inbound/outbound Arp packets.
 */
static int
darpi_realtime(void)
{
	
    aprintf(stdout, 1, "Cache entry timeout: %d %s\n", 
            darpi_timeout, darpi_timeout == 1 ? 
            "millisecond." : "milliseconds.");

    aprintf(stdout, 1, "Realtime Protect actived!\n");
	
    pthread_attr_init(&detach_attr);
    pthread_attr_setdetachstate(&detach_attr, PTHREAD_CREATE_DETACHED);
    pthread_attr_setschedpolicy(&detach_attr, SCHED_RR);

    /* 
     * Thread 4 detached, realtime inbound/outbound work.
     */
    if (pthread_create(&thread[3], &detach_attr, darpi_realtime_thread, 
        (void *) NULL) != 0) {
        ERROR(strerror(errno));

        return (-1);
    }
	
    pthread_attr_destroy(&detach_attr);
    darpi_cache_list_destroy();
	
    return (0);
}

/*
 * DARPI realtime signal handler.
 */
static void
darpi_realtime_thread_sigusr1(int sig)
{
		
    pthread_exit((void *) 0);
}
	
/*
 * Use thread for non blocking to pcap_loop().
 */
static void *
darpi_realtime_thread(void *arg)
{
    struct sigaction saction;
    pcap_t *pcap;
	
#ifdef NETBSD
    saction.sa_flags = 0;
#endif
    saction.sa_handler = darpi_realtime_thread_sigusr1;
	
    if (sigaction(SIGUSR1, &saction, NULL) < 0) {
        ERROR(strerror(errno));
		
        pthread_exit((void *) -1);
    }
	
    if ((pcap = darpi_realtime_open_packets()) == NULL) {
        pthread_exit((void *) -1);
    }
	
    pthread_rwlock_init(&rlock, NULL);
    pthread_rwlock_init(&wlock, NULL);

    while (1) {
        if (pcap_loop(pcap, 1, darpi_realtime_read_packets, NULL) < 0) {
            ERROR(pcap_geterr(pcap));

            pthread_rwlock_destroy(&wlock);
            pthread_rwlock_destroy(&rlock);
            darpi_realtime_close_packets(pcap);
			
            pthread_exit((void *) -1);
        }
    }

    pthread_rwlock_destroy(&wlock);
    pthread_rwlock_destroy(&rlock);
    darpi_realtime_close_packets(pcap);
	
    pthread_exit((void *) 0);
}

/*
 * Open pcap file descriptor.
 */
static pcap_t *
darpi_realtime_open_packets(void)
{
    struct bpf_program compiled_filter;
    char errbuf[PCAP_ERRBUF_SIZE], *filter = "arp";
    pcap_t *pcap;
#ifndef LINUX
    unsigned int op = 1;
#endif
	
    if ((pcap = pcap_open_live(dev.dev, BUFSIZ, 0, 0, errbuf)) == NULL) {
        ERROR(errbuf);

        return (NULL);
    }

#ifndef LINUX
    /* 
     * BSD, differently from linux does not 
     * support automatic socket soft real time 
     * (Linux Socket Filter). 
     * Therefore on BSD platform it's necessary 
     * to use this I/O Control.
     */
    if (ioctl(pcap_fileno(pcap), BIOCIMMEDIATE, &op) < 0) {
        ERROR(strerror(errno));

        return (NULL);
    }
#endif

    if (pcap_compile(pcap, &compiled_filter, filter, 0, 
        dev.dev_netmask.s_addr) < 0) {
        ERROR(pcap_geterr(pcap));
        darpi_realtime_close_packets(pcap);

        return (NULL);
    }
	
    if (pcap_setfilter(pcap, &compiled_filter) < 0) {
        ERROR(pcap_geterr(pcap));
        darpi_realtime_close_packets(pcap);

        return (NULL);
    }	
	
    pcap_freecode(&compiled_filter);
    return (pcap);
}

/*
 * Close pcap file descriptor.
 */
static void
darpi_realtime_close_packets(pcap_t *pcap)
{

    pcap_close(pcap);
}

/*
 * Reads inbound/outbound packets such as:
 *  - Arp request: 
 * 	- Checks the source, if it matches us:
 *	    Save the destination address in the 
 *	    DARPI Cache then it will checks the 
 *	    inbound Arp reply packet.
 *	- Or checks the destination, if it matches us:
 *	    Delete source address in Arp cache 
 *	    (possible address poisoned), 
 *	    resend Arp request to ex source.
 *
 *  - Arp reply:
 *	- Checks the source, if it matches us:
 *	    Sended.
 *	- Checks the destination, if matches us:
 *	    Checks the source address, if it matches 
 *	    in DARPI cache, they are accepted and 
 *	    inserted in Arp cache, otherwise they 
 *	    are rejected and deleted in Arp cache.
 */
static void
darpi_realtime_read_packets(unsigned char *arg, const struct pcap_pkthdr 
    *header, const unsigned char *packet)
{
    struct ether_addr src_mac, dst_mac, dst_mac_req;
    struct in_addr dst_ip, dst_ip_req;
    struct arp_header *arp_packet;
    char c_src_ip[16], c_src_mac[18], c_dst_ip[16], c_dst_mac[18], entry[34];
    int i;	

    /* 
     * Arp Packet.
     */
    arp_packet = (struct arp_header *) (packet + dev.dev_offset);

    /*
     * Convert source MAC/IPv4 string.
     */
    snprintf(c_src_ip, 16, "%d.%d.%d.%d", 
             arp_packet->ah_addresses.ar_spa[0], 
             arp_packet->ah_addresses.ar_spa[1],
             arp_packet->ah_addresses.ar_spa[2], 
             arp_packet->ah_addresses.ar_spa[3]);

    for (i = 0; i < ETHER_ADDR_LEN; i++) {
        src_mac.octet[i] = arp_packet->ah_addresses.ar_sha[i];
    }
    strncpy(c_src_mac, ether_ntoa(&src_mac), sizeof(char) * 18);
    c_src_mac[sizeof(char) * strlen(c_src_mac)] = '\0';

    /*
     * Convert destination MAC/IPv4 string.
     */
    snprintf(c_dst_ip, 16, "%d.%d.%d.%d", 
             arp_packet->ah_addresses.ar_tpa[0], 
             arp_packet->ah_addresses.ar_tpa[1],
             arp_packet->ah_addresses.ar_tpa[2], 
             arp_packet->ah_addresses.ar_tpa[3]);
    inet_aton(c_dst_ip, &dst_ip);

    for (i = 0; i < ETHER_ADDR_LEN; i++) {
        dst_mac.octet[i] = arp_packet->ah_addresses.ar_tha[i];
    }
    strncpy(c_dst_mac, ether_ntoa(&dst_mac), sizeof(char) * 18);
    c_dst_mac[sizeof(char) * strlen(c_dst_mac)] = '\0';

    if (ntohs(arp_packet->ah_header.ar_op) == ARP_OP_REQUEST) {
        /* 
         * Check if we are the source address. 
         */
        if (strcmp(c_src_mac, ether_ntoa(&(dev.dev_mac))) == 0 && 
            strcmp(c_src_ip, inet_ntoa(dev.dev_inet4)) == 0) {
            aprintf(stdout, 1, "Request >> Add entry ");

            /* 
             * DARPI Cache Add/Refresh. 
             */
            if (darpi_cache_list_create(&dst_ip) < 0)
                exit(EXIT_FAILURE);

            pthread_attr_init(&detach_attr);
            pthread_attr_setdetachstate(&detach_attr, PTHREAD_CREATE_DETACHED);
            pthread_attr_setschedpolicy(&detach_attr, SCHED_RR);
			
            /* 
             * Thread 5 detached, Check each DARPI Cache entry with 
             * timeout, possible host doesn't present in the network.
             */
            pthread_create(&thread[4], &detach_attr, 
                          darpi_cache_list_check_thread, (void *) &dst_ip);
			
            pthread_attr_destroy(&detach_attr);
        }
        /* 
         * Check if we are the destination address.
         */ 
        else if (strcmp(c_dst_ip, inet_ntoa(dev.dev_inet4)) == 0) {
            aprintf(stdout, 1, "Request << Delete entry ");

            /*
             * Possible source address poisoned.
             * Delete it from Arp cache and resend a 
             * request to ex source. 
             */
            if (arp_cache_del(DARPI, c_src_ip) < 0)
                exit(EXIT_FAILURE);
	
            memcpy(&dst_mac_req, ether_aton("ff:ff:ff:ff:ff:ff"), 
                sizeof(dst_mac_req));	
			
            if (inet_aton(c_src_ip, &dst_ip_req) == 0) {
                aprintf(stderr, 0, "  ERROR: %s inet4 address malformed!\n\n", 
                        c_src_ip);

                exit(EXIT_FAILURE);
            }
			
            /* 
             * After darpi will handlers this arp request and reply.
             */	
            if (darpi_realtime_send_packet(&dst_mac_req, &dst_ip_req) < 0)
                exit(EXIT_FAILURE);
        }
    } else if (ntohs(arp_packet->ah_header.ar_op) == ARP_OP_REPLY) {
        /* 
         * Check if we are the source address. 
         */
        if (strcmp(c_src_mac, ether_ntoa(&(dev.dev_mac))) == 0 && 
            strcmp(c_src_ip, inet_ntoa(dev.dev_inet4)) == 0) {
            aprintf(stdout, 1, "Reply   >> Send to ");
            aprintf(stdout, 0, "%s -> %s\n", c_dst_ip, c_dst_mac);	
        }	
        /* 
         * Check if we are the destination address. 
         */
        else if (strcmp(c_dst_mac, ether_ntoa(&(dev.dev_mac))) == 0 && 
                strcmp(c_dst_ip, inet_ntoa(dev.dev_inet4)) == 0) {
            aprintf(stdout, 1, "Reply   << ");

            /* 
             * Possible Arp Gratuitous for Arp Poisoning.
             * Search entry in DARPI cache, if it matches it 
             * is to inserted in Arp cache, else if it doesn't 
             * exist, it is to deleted from Arp cache.
             */
            TAILQ_FOREACH(darpi_cache_pos, &darpi_cache_head, entries) {
                if (strcmp(inet_ntoa(darpi_cache_pos->dc_ip), c_src_ip) == 0) {
                    TAILQ_REMOVE(&darpi_cache_head, darpi_cache_pos, entries);
                    free(darpi_cache_pos);
						
                    aprintf(stdout, 0, "Refresh entry ");
						
                    memset(entry, '\0', 34);
                    snprintf(entry, 34, "%15s %17s", c_src_ip, c_src_mac);
                    entry[strlen(entry)] = '\0';
					
                    if (arp_cache_add(DARPI, entry) < 0) {
                        exit(EXIT_FAILURE);
                    }
						
                    return;
                }
            }
			
            aprintf(stdout, 0, "Delete entry \n", c_src_ip);
			
            if (arp_cache_del(DARPI, c_src_ip) < 0) {
                exit(EXIT_FAILURE);
            }
        }
    }
}

/*
 * Sends arp request (See Arp request Inbound 
 * in darpi_realtime_read_packets()).
 */
static int
darpi_realtime_send_packet(struct ether_addr *dst_mac, struct in_addr *dst_ip)
{
    char errbuf[LIBNET_ERRBUF_SIZE];
    libnet_t *lnet;

    if ((lnet = libnet_init(LIBNET_LINK, dev.dev, errbuf)) == NULL) {
        ERROR(errbuf);
        
        return (-1);
    }
	
    if (libnet_autobuild_arp(ARPOP_REQUEST, 
        /* 
         * Source MAC address.
         */
        (u_int8_t *) &(dev.dev_mac),
        /* 
         * Source IPv4 address.
         */
        (u_int8_t *) &(dev.dev_inet4),
        /* 
         * Destination MAC address.
         */
        (u_int8_t *) dst_mac,
        /* 
         * Destination InetV4 address.
         */
        (u_int8_t *) dst_ip,			
        lnet ) < 0) {
        ERROR(libnet_geterror(lnet));
        libnet_destroy(lnet);

        return (-1);
    }
	
    if (libnet_autobuild_ethernet((u_int8_t *) dst_mac, 
        ETHERTYPE_ARP, lnet) < 0) {
        ERROR(libnet_geterror(lnet));
        libnet_destroy(lnet);

        return (-1);
    }
	
    /* 
     * Sends (Ethernet + Arp) Packet. 
     */
    if (libnet_write(lnet) < 0) {
        ERROR(libnet_geterror(lnet));
        libnet_destroy(lnet);

        return (-1);
    }

    libnet_destroy(lnet);
    return (0);
}

/*
 * Create or adds DARPI cache tail.
 */
static int 
darpi_cache_list_create(struct in_addr *entry)
{
    struct in_addr ip;
		
    ip.s_addr = entry->s_addr;
		
    aprintf(stdout, 0, "%s\n", inet_ntoa(*entry));
		
    pthread_rwlock_wrlock(&wlock);
		
    if (darpi_cache_begin == NULL) { 
        TAILQ_INIT(&darpi_cache_head);
			
        if ((darpi_cache_begin = malloc(sizeof(struct darpi_cache))) == NULL) {
            ERROR(strerror(errno));	
            pthread_rwlock_unlock(&wlock);
				
            return (-1);
        }
			
        memcpy(&darpi_cache_begin->dc_ip, &ip, sizeof(ip));
        TAILQ_INSERT_HEAD(&darpi_cache_head, darpi_cache_begin, entries);
			
        pthread_rwlock_unlock(&wlock);
			
        return (0);
    }
		
    if ((darpi_cache_next = malloc(sizeof(struct darpi_cache))) == NULL) {
        ERROR(strerror(errno));			
        pthread_rwlock_unlock(&wlock);
			
        return (-1);
    }
		
    memcpy(&darpi_cache_next->dc_ip, &ip, sizeof(ip));
    TAILQ_INSERT_TAIL(&darpi_cache_head, darpi_cache_next, entries);
		
    pthread_rwlock_unlock(&wlock);
		
    return (0);	
}
	
/* 
 * DARPI cache list signal handler.
 */
static void
darpi_cache_list_check_thread_sigusr1(int sig)
{
		
    pthread_exit((void *) 0);
}
	
/*
 * DARPI Cache entry timeout.
 * When DARPI reads Arp Request Outbound, it writes an 
 * entry in DARPI Cache, if this entry replies with Arp 
 * Reply Inbound it's ok, else if it doesn't replies, 
 * it probably doesn't exist. This timeout in this case, 
 * delete this entry from DARPI Cache. 
 */
static void *
darpi_cache_list_check_thread(void *arg)
{
    struct in_addr *entry = (struct in_addr *)arg;
    struct sigaction saction;
    char ip[16];
	
#ifdef NETBSD
    saction.sa_flags = 0;
#endif
    saction.sa_handler = darpi_cache_list_check_thread_sigusr1;
	
    if (sigaction(SIGUSR1, &saction, NULL) < 0) {
        ERROR(strerror(errno));
		
        pthread_exit((void *) -1);
    }
		
    snprintf(ip, sizeof(char) * 16, "%s", inet_ntoa(*entry));

    /* 
     * usleep() for thread suspend.
     * Convert by microseconds to milliseconds.
     */
    if (usleep(darpi_timeout * 1000) < 0) {
        ERROR(strerror(errno));
			
        pthread_exit((void *) -1);
    }
		
    pthread_rwlock_rdlock(&rlock);
    pthread_rwlock_wrlock(&wlock);
		
    TAILQ_FOREACH(darpi_cache_pos, &darpi_cache_head, entries) {
        if (strcmp(inet_ntoa(darpi_cache_pos->dc_ip), ip) == 0) {
            TAILQ_REMOVE(&darpi_cache_head, darpi_cache_pos, entries);
            free(darpi_cache_pos);
				
            aprintf(stdout, 1, "Reply   << Delete timeout entry %s\n", ip);
				
            pthread_rwlock_unlock(&wlock);
            pthread_rwlock_unlock(&rlock);
				
            pthread_exit((void *) 0);
        }
    }
		
    pthread_rwlock_unlock(&wlock);
    pthread_rwlock_unlock(&rlock);
		
    pthread_exit((void *) 0);
}
	
/*
 * Destroy DARPI cache tail.
 */
static void
darpi_cache_list_destroy(void)
{
		
    pthread_rwlock_rdlock(&rlock);
		
    while (TAILQ_EMPTY(&darpi_cache_head) == 0) {
        pthread_rwlock_wrlock(&wlock);
			
        darpi_cache_pos = TAILQ_FIRST(&darpi_cache_head);
			
        TAILQ_REMOVE(&darpi_cache_head, darpi_cache_pos, entries);
        free(darpi_cache_pos);
			
        pthread_rwlock_unlock(&wlock);
    }
		
    pthread_rwlock_unlock(&rlock);
}
	
/*****************
 * Misc handler: *
 *****************/

/* 
 * My printf() with logging mode.
 */
static void 
aprintf(FILE *stream, int ltime, char *fmt, ...)
{
    struct tm tm_cur;
    time_t time_cur;
    FILE *log_fstream;
    va_list ap;

    if (stream != NULL) {
        if (ltime == 1) {
            time_cur = time(NULL);
            tm_cur = *(struct tm *) localtime(&time_cur);
            fprintf(stream, "  %02d:%02d:%02d - ", 	
                    tm_cur.tm_hour, tm_cur.tm_min, tm_cur.tm_sec);	
        }

        va_start(ap, fmt);
        vfprintf(stream, fmt, ap);
        va_end(ap);
    }

    if (log_mode == 0) {
        if ((log_fstream = task_mode_log_open()) == NULL) {
            kill(pid_main, SIGTERM);
			
            exit(EXIT_FAILURE);
        }

        if (ltime == 1) {
            time_cur = time(NULL);
            tm_cur = *(struct tm *) localtime(&time_cur);
            fprintf(log_fstream, "  %02d:%02d:%02d - ", 	
                    tm_cur.tm_hour, tm_cur.tm_min, tm_cur.tm_sec);	
        }

        va_start(ap, fmt);
        vfprintf(log_fstream, fmt, ap);
        va_end(ap);
		
        fflush(log_fstream);

        task_mode_log_close(log_fstream);
    }
}

/*
 * Prints my license.
 */
static void
license(void)
{

#define COPYRIGHT                                                           \
"  Copyright (C) 2008-2010 Andrea Di Pasquale <spikey.it@gmail.com>\n"      \
"  All rights reserved.\n"                                                  \
"\n"                                                                        \
"  Redistribution and use in source and binary forms, with or without\n"    \
"  modification, are permitted provided that the following conditions\n"    \
"  are met:\n"                                                              \
"  1. Redistributions of source code must retain the above copyright\n"     \
"     notice(s), this list of conditions and the following disclaimer as\n" \
"     the first lines of this file unmodified other than the possible\n"    \
"     addition of one or more copyright notices.\n"                         \
"  2. Redistributions in binary form must reproduce the above copyright\n"  \
"     notice(s), this list of conditions and the following disclaimer in "  \
"the\n"                                                                     \
"     documentation and/or other materials provided with the "              \
"distribution.\n"                                                           \
"\n"                                                                        \
"  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDER(S) ``AS IS'' AND ANY\n"\
"  EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE "      \
"IMPLIED\n"                                                                 \
"  WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR "             \
"PURPOSE ARE\n"                                                             \
"  DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER(S) BE "               \
"LIABLE FOR ANY\n"                                                          \
"  DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL "     \
"DAMAGES\n"                                                                 \
"  (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR\n"    \
"  SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) "     \
"HOWEVER\n"                                                                 \
"  CAUSED AND ON ANY THEORY OF LIABILITY, WHETER IN CONTRACT, STRICT\n"     \
"  LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY "  \
"WAY\n"                                                                     \
"  OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF " \
"SUCH\n"                                                                    \
"  DAMAGE.\n\n"

    printf(COPYRIGHT);

#undef COPYRIGHT
}

/*
 * Prints version.
 */
static void
version(void)
{
	
    printf("\n  ArpON \"Arp handler inspectiON\" 2.0 " \
           "(http://arpon.sourceforge.net)\n\n");
}

/*
 * Prints help summary page.
 */
static void
help(void)
{

#define HELP                                                                \
"  Usage: arpon [Options] {SARPI | DARPI}\n"                                \
"\n"                                                                        \
"  TASK MODE:\n"                                                            \
"    -n, --nice             <Nice>         Sets PID's CPU priority\n"       \
"                                          (Default Nice: %d)\n"            \
"    -p, --pid-file         <Pid file>     Sets the pid file\n"             \
"                                          (Default: %s)\n"                 \
"    -q, --quiet                           Works in background task\n"      \
"\n"                                                                        \
"  LOG MODE:\n"                                                             \
"    -f, --log-file         <Log file>     Sets the log file\n"             \
"                                          (Default: %s)\n"                 \
"    -g, --log                             Works in logging mode\n"         \
"\n"                                                                        \
"  INTERFACE:\n"                                                            \
"    -i, --iface            <Iface>        Sets valid interface manually\n" \
"    -o, --iface-auto                      Sets interface automatically\n"  \
"    -l, --iface-list                      Prints all valid interfaces\n"   \
"\n"                                                                        \
"  STATIC ARP INSPECTION:\n"                                                \
"    -c, --sarpi-cache      <Cache file>   Sets Arp Cache entries from file\n"\
"                                          (Default: %s)\n"                 \
"    -x, --sarpi-timeout    <Timeout>      Sets Arp Cache refresh timeout\n"\
"                                          (Default: %d minuts)\n"          \
"    -s, --sarpi                           Manages Arp Cache statically\n"  \
"\n"                                                                        \
"  DYNAMIC ARP INSPECTION:\n"                                               \
"    -y, --darpi-timeout    <Timeout>      Sets Arp Cache entries timeout\n"\
"                                          (Default: %d milliseconds)\n"    \
"    -d, --darpi                           Manages Arp Cache dynamically\n" \
"\n"                                                                        \
"  MISC:\n"                                                                 \
"    -e, --license                         Prints license page\n"           \
"    -v, --version                         Prints version number\n"         \
"    -h, --help                            Prints help summary page\n"      \
"\n"                                                                        \
"  SEE THE MAN PAGE FOR MANY DESCRIPTIONS AND EXAMPLES\n\n"

    printf(HELP, cpu_priority, pid_file, log_file, 
           sarpi_cache_file, sarpi_timeout, darpi_timeout);
	
#undef HELP
}

/*****************
 * Main handler: *
 *****************/

/*
 * Main signal handler.
 */
static int
main_signal(void)
{
		
    pthread_attr_init(&join_attr);
    pthread_attr_setschedpolicy(&join_attr, SCHED_RR);
		
    /*
     * Thread 1 joinabled, signal handler. 
     */
    if (pthread_create(&thread[0], &join_attr, 
        main_signal_thread, (void *) NULL) != 0) {
        ERROR(strerror(errno));
        pthread_attr_destroy(&join_attr);
		
        return (-1);
    }
		
    pthread_join(thread[0], NULL);
    pthread_attr_destroy(&join_attr);
	
    return (0);
}
	
/*
 * This thread is main signal handler of:
 *  - SIGINT, SIGTERM, SIGQUIT: Exit
 *  - SIGHUP, SIGCONT: Reboot
 * and handler for all threads.
 */
static void *
main_signal_thread(void *arg)
{
    int sig;
	
    while (1) {
        sigwait(&sigse, &sig);
		
        switch (sig) {
        case (SIGINT):
        case (SIGTERM):
        case (SIGQUIT):
            switch (dinspec) {
            case (INSPEC_SARPI):
                aprintf(stdout, 0, "\r\n  SARPI Interrupt...\n\n");
                break;
			
            case (INSPEC_DARPI):
                aprintf(stdout, 0, "\r\n  DARPI Interrupt...\n\n");
                break;
			
            case (INSPEC_NULL):
                break;
            }
			
            exit(-1);
				
        case (SIGHUP):
        case (SIGCONT):
            /*
             * Stop all threads and start their (Reboot).
             * Thread[0] = Signal is not restarted
             * Thread[1] = SARPI/DARPI manager yes
             * Thread[2] = Iface uplink yes
             * Thread[3] = SARPI/DARPI 1° Thread yes
             * Thread[4] = SARPI/DARPI 2° Thread yes
             */
            main_stop();
			
            if (main_start() < 0) {
                pthread_exit((void *) -1);
            }
            break;
				
        default:
            break;
        }
    }
	
    pthread_exit((void *) 0);
}
	
/*
 * Start all threads in this order:
 *  - SARPI:
 *      SARPI manager -> Iface uplink -> SARPI 1° & 2°
 *
 *  - DARPI:
 *      DARPI manager -> Iface uplink -> DARPI 1° & 2°
 */
static int
main_start(void)
{
		
    pthread_attr_init(&detach_attr);
    pthread_attr_setdetachstate(&detach_attr, PTHREAD_CREATE_DETACHED);
    pthread_attr_setschedpolicy(&detach_attr, SCHED_RR);
		
    switch (dinspec) {
    case (INSPEC_SARPI):
        /*
         * Thread 2 detached, start SARPI manager.
         */
        if (pthread_create(&thread[1], &detach_attr, sarpi_manager_thread, 
            (void *) NULL) != 0) {
            ERROR(strerror(errno));
					
            return (-1);
        }
			
        pthread_attr_destroy(&detach_attr);
        break;
				
    case (INSPEC_DARPI):
        /*
         * Thread 2 detached, start DARPI manager.
         */
        if (pthread_create(&thread[1], &detach_attr, darpi_manager_thread, 
            (void *) NULL) != 0) {
            ERROR(strerror(errno));
				
            return (-1);
        }
				
        pthread_attr_destroy(&detach_attr);
        break;
				
    case (INSPEC_NULL):
        break;
    }
		
    return (0);
}
	
/*
 * Stop these threads:
 * Thread[0] = Signal handler is not stopped
 * Thread[1] = SARPI/DARPI manager yes
 * Thread[2] = Iface uplink yes
 * Thread[3] = SARPI/DARPI 1°thread yes
 * Thread[4] = SARPI/DARPI 2° thread yes.
 */
static void 
main_stop(void)
{
    int i;
		
    for (i = 1; i < 5; i++) {
        pthread_kill(thread[i], SIGUSR1);
    }
}
	
/*
 * Main. 
 */
int
main(int argc, char *argv[], char *envp[])
{
    struct option longopts[] = {
        { "nice",           required_argument,  NULL,   'n' },
        { "pid-file",       required_argument,  NULL,   'p' },
        { "quiet",          no_argument,        NULL,   'q' },	
        { "log-file",       required_argument,  NULL,   'f' },
        { "log",            no_argument,        NULL,   'g' },
        { "iface",          required_argument,  NULL,   'i' },
        { "iface-auto",     no_argument,        NULL,   'o' },
        { "iface-list",     no_argument,        NULL,   'l' },
        { "sarpi-cache",    required_argument,  NULL,   'c' },
        { "sarpi-timeout",  required_argument,  NULL,   'x' },
        { "sarpi",          no_argument,        NULL,   's' },
        { "darpi-timeout",  required_argument,  NULL,   'y' },	
        { "darpi",          no_argument,        NULL,   'd' },
        { "license",        no_argument,        NULL,   'e' },
        { "version",        no_argument,        NULL,   'v' },
        { "help",           no_argument,        NULL,   'h' },
        { NULL,             0,                  NULL,    0  }	    
    };
    int gopt, j;

    /*
     * Check if SARPI or DARPI timeout is setted without option.
     */
    j = 0;

    /* 
     * Sanitize environment from LD_PRELOAD attacks. 
     */
    if (getenv("LD_PRELOAD") != NULL) {
        unsetenv("LD_PRELOAD");
        execve(argv[0], argv, envp);
    }
	
    aprintf(stdout, 0, "\n");
	
    if (getuid() != 0x0) {
        aprintf(stderr, 0, "  ERROR: must run as root!\n\n");
        return (-1);
    }
	
    if (argc == 1) {
        aprintf(stderr, 0, 
                "  ERROR: Use -h or --help for more information.\n\n");
		
        return (0);
    }
			  
    pid_main = getpid();
	
    while ((gopt = getopt_long(argc, argv, "n:p:qf:gi:olc:x:sy:devh", 
            longopts, NULL)) != -1) { 
        switch(gopt) {
        case ('n'):
            cpu_priority = atoi(optarg);
            break;

        case ('p'):
            pid_file = optarg;
            break;
				
        case ('q'):
            if (task_mode_daemon() < 0) {
                return (-1);
            }
            break;
	
        case ('f'):
            log_file = optarg;
            break;

        case ('g'):
            if (task_mode_log() < 0) {
                return (-1);
            }
            break;
				
        case ('i'):
            dif = IFACE_MANUAL;
            ddev = optarg;
            break;
				
        case ('o'):
            dif = IFACE_AUTO;
            break;
				
        case ('l'):
            dif = IFACE_LIST;
				
            if (iface_manager() < 0) {
                return (-1);
            }
				
            aprintf(stdout, 0, "\n");
            return (0);
				
        case ('c'):
            sarpi_cache_file = optarg;
	
            j = 1;
            break;
				
        case ('x'):
            sarpi_set_timeout(optarg);
	
            j = 1;
            break;
				
        case ('s'):
            if (dinspec != INSPEC_NULL) {
                aprintf(stderr, 0, "  ERROR: Can't use both DARPI & " \
                        "SARPI on same interface!\n\n");

                return (-1);
            }
			
            dinspec = INSPEC_SARPI;
            break;
			
        case ('y'):
            if (darpi_set_timeout(optarg) < 0) {
                return (-1);
            }
			
            j = 2;	
            break;
	
        case ('d'):
            if (dinspec != INSPEC_NULL) {
                aprintf(stderr, 0, "  ERROR: Can't use both SARPI & " \
                        "DARPI on same interface!\n\n");

                return (-1);
            }

            dinspec = INSPEC_DARPI;
            break;
				
        case ('e'):
            license();
            return (0);
	
        case ('v'):
            version();
            return (0);
	
        case ('h'):
            help();
            return (0);
	
        case (':'):
        case ('?'):
            aprintf(stderr, 0, "\n");
            return (-1);
				
        default:
            break;
        }
    }
    argc -= optind;
    argv += optind;
	
    if (dinspec == INSPEC_NULL) {
        if (j != 0) {
            switch (j) {
            case (1):
                aprintf(stderr, 0, "  ERROR: SARPI required " \
                        "-s option to work!\n\n");
                break;
					
            case (2):
                aprintf(stderr, 0, "  ERROR: DARPI required " \
                        "-d option to work!\n\n");
                break;
            }
        }
    } else {
        switch (dinspec) {
        case (INSPEC_SARPI):
            if (j == 2) {
                aprintf(stderr, 0, "  ERROR: SARPI doesn't " \
                        "required DARPI options!\n\n");
                return (-1);
            }
            break;

        case (INSPEC_DARPI):
            if (j == 1) {
                aprintf(stderr, 0, "  ERROR: DARPI doesn't " \
                        "required SARPI options!\n\n");
                return (-1);
            }
            break;

        case (INSPEC_NULL):
        default:
            break;
        }
    }
	
    sigemptyset(&sigse);
    sigaddset(&sigse, SIGINT);
    sigaddset(&sigse, SIGTERM);
    sigaddset(&sigse, SIGQUIT);
    sigaddset(&sigse, SIGHUP);
    sigaddset(&sigse, SIGCONT);
	
    if (pthread_sigmask(SIG_BLOCK, &sigse, NULL) < 0) {
        ERROR(strerror(errno));
		
        return (-1);
    }
	
    if (main_start() < 0) {
        return (-1);
    }
	
    if (main_signal() < 0) {
        return (-1);
    }
	
    return (0);
}

/* 
 * EOF. 
 */ 
