//  ************************************************************************************************
//
//  BornAgain: simulate and fit reflection and scattering
//
//! @file      Device/Mask/DetectorMask.h
//! @brief     Defines class DetectorMask.
//!
//! @homepage  http://www.bornagainproject.org
//! @license   GNU General Public License v3 or higher (see COPYING)
//! @copyright Forschungszentrum Jülich GmbH 2018
//! @authors   Scientific Computing Group at MLZ (see CITATION, AUTHORS)
//
//  ************************************************************************************************

#ifndef BORNAGAIN_DEVICE_MASK_DETECTORMASK_H
#define BORNAGAIN_DEVICE_MASK_DETECTORMASK_H

#include <memory>

#ifndef SWIG
#include "Base/Types/CloneableVector.h"
#endif

class Scale;
class IShape2D;

class Datafield;

class MaskPattern {
public:
    MaskPattern(IShape2D* shape_, bool doMask_);
    MaskPattern(const MaskPattern&) = delete;
    ~MaskPattern();
    MaskPattern* clone() const;
    IShape2D* shape; // cloneable
    bool doMask;
};

//! Collection of detector masks.

class DetectorMask {
public:
    DetectorMask(const Scale& xAxis, const Scale& yAxis);
    ~DetectorMask();
    DetectorMask(const DetectorMask& other);

    //! Add mask to the stack of detector masks.
    //! The value "true" means that the area will be excluded from the analysis.
    //! @param shape The shape of mask.
    //! @param mask_value The value of mask
    void addMask(const IShape2D& shape, bool mask_value);

    bool isMasked(size_t i_flat) const;

    //! Returns true if has masks
    bool hasMasks() const;

    int numberOfMaskedChannels() const { return m_number_of_masked_channels; }

    size_t numberOfMasks() const;

    const MaskPattern* patternAt(size_t iMask) const;

private:
    void process_masks();

    // primary data:
#ifndef SWIG
    const Scale* m_xAxis;
    const Scale* m_yAxis;
    CloneableVector<MaskPattern> m_stack;
#endif

    // cached secondary data:
    std::vector<bool> m_masked;
    int m_number_of_masked_channels;
};

#endif // BORNAGAIN_DEVICE_MASK_DETECTORMASK_H
