/* IMSpector - Instant Messenger Transparent Proxy Service
 * http://www.imspector.org/
 * (c) Lawrence Manning <lawrence@aslak.net>, 2006
 *          
 * Released under the GPL v2. */

#include "imspector.h"

#define PLUGIN_NAME "Yahoo IMSpector protocol plugin"
#define PROTOCOL_NAME "Yahoo"
#define PROTOCOL_PORT 5050

extern "C"
{
	bool initprotocolplugin(struct protocolplugininfo &pprotocolplugininfo,
		class Options &options, bool debugmode);
	void closeprotocolplugin(void);
	int processpacket(bool outgoing, class Socket &incomingsock, char *replybuffer,
		int *replybufferlength, std::vector<struct imevent> &imevents, std::string &clientaddress);
	int generatemessagepacket(struct response &response, char *replybuffer, int *replybufferlength);
};

#pragma pack(2)

struct header
{
	uint32_t version;
	uint16_t pkt_len;
	uint16_t service;
	uint32_t status;
	uint32_t session_id;
};

#pragma pack()

struct tagvalue
{
	std::string text;
	struct messageextent messageextent;
};

#define YAHOO_VERSION_UNKNOWN 0
#define YAHOO_VERSION_OLD 1
#define YAHOO_VERSION_FLASH 2

#define YAHOO_SERVICE_NULL 0
#define YAHOO_SERVICE_MESSAGE 0x06
#define YAHOO_SERVICE_GROUP_MESSAGE 0x1d
#define YAHOO_SERVICE_FILETRANSFER 0x46
#define YAHOO_SERVICE_P2PFILEXFER 0x4d
#define YAHOO_SERVICE_P2PFILEXFER_ALT 0xdc
#define YAHOO_SERVICE_MISC 0x4b

#define YAHOO_OLD_MARKER1 0xc0
#define YAHOO_OLD_MARKER2 0x80
#define YAHOO_FLASH_MARKER1 '^'
#define YAHOO_FLASH_MARKER2 '$'

int yahooversion = YAHOO_VERSION_UNKNOWN;
uint32_t sessionid;
std::string localid = "Unknown";
std::string remoteid = "Unknown";
bool groupchat = false;
int packetcount = 0;
bool tracing = false;
bool localdebugmode = false;

int gettagsandvalues(uint8_t *buffer, int length, std::map<std::string,
	struct tagvalue> &tagvalues, int messageextentoffset);
bool addtagvalue(char *buffer, int *bufferlength, std::string tag, std::string value);
int recvuntil(Socket &sock, char *string, int length, char end);
	
bool initprotocolplugin(struct protocolplugininfo &protocolplugininfo,
	class Options &options, bool debugmode)
{
	if (options["yahoo_protocol"] != "on") return false;

	localdebugmode = debugmode;
	
	protocolplugininfo.pluginname = PLUGIN_NAME;
	protocolplugininfo.protocolname = PROTOCOL_NAME;
	protocolplugininfo.port = htons(PROTOCOL_PORT);
	
	if (options["yahoo_trace"] == "on") tracing = true;
	
	return true;
}

void closeprotocolplugin(void)
{
	return;
}

/* The main plugin function. See protocolplugin.cpp. */
int processpacket(bool outgoing, class Socket &incomingsock, char *replybuffer, 
	int *replybufferlength, std::vector<struct imevent> &imevents, std::string &clientaddress)
{
	char ymsg[4]; /* No NULL terminator. */

	*replybufferlength = 0;

	/* Get the ymsg. This may be YMSG for the old protocol, or the start
	 * of the content-length for the new. */
	if (!incomingsock.recvalldata(ymsg, 4)) return 1;
		
	memcpy(replybuffer, ymsg, 4);
	*replybufferlength += 4;
	
	int service = YAHOO_SERVICE_NULL;
	std::map<std::string, struct tagvalue> tagvalues;
	
	if (ymsg[0] == 'Y' && ymsg[1] == 'M' && ymsg[2] == 'S' && ymsg[3] == 'G')
	{
		yahooversion = YAHOO_VERSION_OLD;		
		
		debugprint(localdebugmode, PROTOCOL_NAME ": Original protocol");

		debugprint(localdebugmode, PROTOCOL_NAME ": ID: %c %c %c %c",
			ymsg[0], ymsg[1], ymsg[2], ymsg[3]);

		struct header header;
	
		memset(&header, 0, sizeof(struct header));

		/* Get the fix sized header and ensure endianess. */	
		if (!incomingsock.recvalldata((char *) &header, sizeof(struct header))) return 1;
	
		memcpy(replybuffer + *replybufferlength, &header, sizeof(struct header));
		*replybufferlength += sizeof(struct header);
	
		header.version = ntohl(header.version);
		header.pkt_len = ntohs(header.pkt_len);
		header.service = ntohs(header.service);
		header.status = ntohs(header.status);
		header.session_id = ntohl(header.session_id);
		
		sessionid = header.session_id;
		
		debugprint(localdebugmode, PROTOCOL_NAME ": verion: %x pkt_len: %d " \
			"service: %04x status: %d session_id: %d",
			header.version, header.pkt_len, header.service, header.status, header.session_id);
		
		uint8_t buffer[BUFFER_SIZE];
		
		/* Get the following data assuming the header indicates there is some. */
		if (header.pkt_len)
		{
			if (!incomingsock.recvalldata((char *) buffer, header.pkt_len)) return 1;
			
			/* Split out the buffer into a list of tag and values. */
			gettagsandvalues(buffer, header.pkt_len, tagvalues, *replybufferlength);
			
			memcpy(replybuffer + *replybufferlength, buffer, header.pkt_len);
			*replybufferlength += header.pkt_len;
		}

		/* This service variable is shared between the old and the flash style
		 * versions of the protocol. */
		service = header.service;
	}
	else
	{
		yahooversion = YAHOO_VERSION_FLASH;

		debugprint(localdebugmode, PROTOCOL_NAME ": Flash protocol");
	
		int len;
		
		if ((len = recvuntil(incomingsock, (char *) replybuffer + 4, BUFFER_SIZE, '\0')) < 1) 
				return 1;
		*replybufferlength += len;
	
		char *p = replybuffer;		
		bool needmore = true;
		char *endtag = NULL;
		
		do
		{
			std::string payload; int payloadlength;
			std::string tag;
			bool closing;
			std::map<std::string, std::string> params;
			
			/* Parse out the simple yahoo XML using our crufy hand made parser. */
			p = parsexmltag(localdebugmode, p, payload, payloadlength, tag, closing, params);
	
			if (tag == "Ymsg")
			{
				service = atol(params["Command"].c_str());
				sessionid = atol(params["SessionId"].c_str());
				/* Note where the tag ends, as this is needed to work out the
				 * messageextent. */
				endtag = p;
			}
				
			if (tag == "/Ymsg")
			{
				/* The "yahoo style" tags are in the payload of the closing tag. */
				gettagsandvalues((uint8_t *) payload.c_str(), payloadlength,
					tagvalues, endtag - replybuffer);
				needmore = false;
			}
		}
		while (needmore);		
	}
	
	/* Simple stuff: build the imevent and push it onto the list. */
	struct imevent imevent;
	
	imevent.outgoing = outgoing;
	imevent.type = TYPE_NULL;
	imevent.filtered = false;
	imevent.messageextent.start = 0;
	imevent.messageextent.length = 0;
	
	if (service == YAHOO_SERVICE_MESSAGE)
	{
		imevent.type = TYPE_MSG;
		imevent.eventdata = tagvalues["14"].text;
		
		imevent.messageextent = tagvalues["14"].messageextent;
	}
	else if (service == YAHOO_SERVICE_GROUP_MESSAGE)
	{
		imevent.type = TYPE_MSG;
		if (outgoing)
			imevent.eventdata = tagvalues["14"].text;
		else
			imevent.eventdata = tagvalues["3"].text + ": " + tagvalues["14"].text;
		
		imevent.messageextent = tagvalues["14"].messageextent;
	}	
	else if (service == YAHOO_SERVICE_P2PFILEXFER ||
		service == YAHOO_SERVICE_P2PFILEXFER_ALT)
	{
		imevent.type = TYPE_FILE;
		imevent.eventdata = tagvalues["27"].text + " " + tagvalues["28"].text + " bytes";
	}
	else if (service == YAHOO_SERVICE_MISC)
	{
		if (tagvalues["49"].text == "TYPING")
		{
			imevent.type = TYPE_TYPING;
			imevent.eventdata = "";
		}
		if (tagvalues["49"].text == "WEBCAMINVITE")
		{
			imevent.type = TYPE_WEBCAM;
			imevent.eventdata = "";
		}
	}
	
	if (imevent.type != TYPE_NULL)
	{
		imevent.timestamp = time(NULL);
		imevent.clientaddress = clientaddress;
		imevent.protocolname = PROTOCOL_NAME;

		if (service == YAHOO_SERVICE_GROUP_MESSAGE)
		{
			if (outgoing)
			{
				localid = tagvalues["1"].text;
				remoteid = tagvalues["57"].text;
			}
			else
			{
				localid = tagvalues["1"].text;
				remoteid = tagvalues["57"].text;
			}
			
			groupchat = true;
		}
		else	
		{
			if (outgoing)
			{
				localid = tagvalues["1"].text;
				remoteid = tagvalues["5"].text;
			}
			else
			{
				localid = tagvalues["5"].text;
				remoteid = tagvalues["4"].text;
			}
			
			groupchat = false;
		}
		
		if (!localid.empty() && !remoteid.empty())
		{
			imevent.localid = localid;
			imevent.remoteid = remoteid;
	
			std::transform(imevent.localid.begin(), imevent.localid.end(), imevent.localid.begin(), tolower);
			std::transform(imevent.remoteid.begin(), imevent.remoteid.end(), imevent.remoteid.begin(), tolower);
			
			imevents.push_back(imevent);
		}
	}
	
	/* Write out trace packets if enabled. */
	if (tracing) tracepacket("yahoo", packetcount, replybuffer, *replybufferlength);
	
	packetcount++;
	
	return 0;
}

int generatemessagepacket(struct response &response, char *replybuffer, int *replybufferlength)
{
	if (groupchat || localid.empty() || remoteid.empty()) return 1;

	if (response.text.length() > STRING_SIZE) return 1;

	/* First of all, make the yahoo tag buffer. addtagvalue will add the right
	 * type of tag cos its look in the yahooversion global. */
	char tagvaluebuffer[BUFFER_SIZE]; int tagvaluebufferlength = 0;
	
	memset(tagvaluebuffer, 0, BUFFER_SIZE);
	
	if (response.outgoing)
	{
		addtagvalue(tagvaluebuffer, &tagvaluebufferlength, "1", localid);
		addtagvalue(tagvaluebuffer, &tagvaluebufferlength, "5", remoteid);
		addtagvalue(tagvaluebuffer, &tagvaluebufferlength, "14", response.text);
		addtagvalue(tagvaluebuffer, &tagvaluebufferlength, "97", "1");
	}
	else
	{
		addtagvalue(tagvaluebuffer, &tagvaluebufferlength, "4", remoteid);
		addtagvalue(tagvaluebuffer, &tagvaluebufferlength, "1", remoteid);
		addtagvalue(tagvaluebuffer, &tagvaluebufferlength, "5", localid);
		addtagvalue(tagvaluebuffer, &tagvaluebufferlength, "97", "1");
		addtagvalue(tagvaluebuffer, &tagvaluebufferlength, "14", response.text);
		addtagvalue(tagvaluebuffer, &tagvaluebufferlength, "63", ";0");
		addtagvalue(tagvaluebuffer, &tagvaluebufferlength, "64", "0");
		addtagvalue(tagvaluebuffer, &tagvaluebufferlength, "1002", "1");
		addtagvalue(tagvaluebuffer, &tagvaluebufferlength, "206", "2");
		addtagvalue(tagvaluebuffer, &tagvaluebufferlength, "10093", "4");
	}

	if (yahooversion == YAHOO_VERSION_OLD)
	{
		/* Old protocol: YMSG, followed by header (with length), followed by tags. */
	
		char ymsg[4];
	
		ymsg[0] = 'Y'; ymsg[1] = 'M'; ymsg[2] = 'S'; ymsg[3] = 'G';

		memcpy(replybuffer, ymsg, 4);

		struct header header;

		header.version = htonl(0xa0000);
		header.pkt_len = htons(tagvaluebufferlength);
		header.service = htons(YAHOO_SERVICE_MESSAGE);
		header.status = htonl(1);
		header.session_id = htonl(sessionid);

		memcpy(replybuffer + 4, &header, sizeof(struct header));
		
		memcpy(replybuffer + 4 + sizeof(header), tagvaluebuffer, tagvaluebufferlength);
	}
	else
	{
		/* Flash protocol: optional content length, followed by Ymsg XML tag, with
		 * the tags inside it. */
		 
		char xmlbuffer[BUFFER_SIZE];
		char contentlengthbuffer[BUFFER_SIZE];
		
		memset(xmlbuffer, 0, BUFFER_SIZE);
		memset(contentlengthbuffer, 0, BUFFER_SIZE);
		
		snprintf(xmlbuffer, BUFFER_SIZE - 1,
			"<Ymsg Command=\"%d\" Status=\"1\" Version=\"102\" VendorId=\"402\" SessionId=\"0\">%s</Ymsg>",
			YAHOO_SERVICE_MESSAGE, tagvaluebuffer);

		if (response.outgoing)
			snprintf(contentlengthbuffer, BUFFER_SIZE - 1, "content-length: %d\r\n\r\n", strlen(xmlbuffer));

		snprintf(replybuffer, BUFFER_SIZE, "%s%s", contentlengthbuffer, xmlbuffer);
	
		*replybufferlength = strlen(replybuffer) + 1; /* For the NULL at the end. */
	}
	
	if (tracing) tracepacket("yahoo-out", packetcount, replybuffer, *replybufferlength);
	
	packetcount++;
	
	return 0;
}

/* Builds up a map of tag and values using the different markers. */
int gettagsandvalues(uint8_t *buffer, int length, std::map<std::string,
	struct tagvalue> &tagvalues, int messageextentoffset)
{
	int counter = 0;
	uint8_t *p = buffer;
	uint8_t marker1 = YAHOO_OLD_MARKER1;
	uint8_t marker2 = YAHOO_OLD_MARKER2;

	if (yahooversion == YAHOO_VERSION_FLASH)
	{
		marker1 = YAHOO_FLASH_MARKER1;
		marker2 = YAHOO_FLASH_MARKER2;
	}
		
	while (p - buffer < length)
	{
		std::string tag; struct tagvalue tagvalue;	

		while (!(p[0] == marker1 && p[1] == marker2))
		{
			if (p - buffer >= length) break;
			tag += *p;
			p++;
		}
		p += 2;
		
		/* Build the message extent now. */
		tagvalue.messageextent.start = p - buffer + messageextentoffset;
		tagvalue.messageextent.length = 0; /* NULL terminate. */
		
		while (!(p[0] == marker1 && p[1] == marker2))
		{
			if (p - buffer >= length) break;
			tagvalue.text += *p;
			tagvalue.messageextent.length++;
			p++;
		}
		p += 2;
		
		tagvalues[tag] = tagvalue;
		counter++;
		
		debugprint(localdebugmode, PROTOCOL_NAME ": Tag: %s Value: %s", tag.c_str(), tagvalue.text.c_str());
	}
	
	return counter;
}

bool addtagvalue(char *buffer, int *bufferlength, std::string tag, std::string value)
{
	uint8_t marker1 = YAHOO_OLD_MARKER1;
	uint8_t marker2 = YAHOO_OLD_MARKER2;

	if (yahooversion == YAHOO_VERSION_FLASH)
	{
		marker1 = YAHOO_FLASH_MARKER1;
		marker2 = YAHOO_FLASH_MARKER2;
	}

	/* Check that the tag and value (and markers) will fit. */
	if (*bufferlength > BUFFER_SIZE - (int)(tag.length()) + (int)(value.length()) + 4)
	{
		syslog(LOG_INFO, PROTOCOL_NAME ": Tag and Value will not fit");
		return false;
	}

	memcpy(buffer + *bufferlength, tag.c_str(), tag.length());
	*bufferlength += tag.length();
	
	buffer[(*bufferlength)++] = marker1;
	buffer[(*bufferlength)++] = marker2;
			
	memcpy(buffer + *bufferlength, value.c_str(), value.length());
	*bufferlength += value.length();

	buffer[(*bufferlength)++] = marker1;
	buffer[(*bufferlength)++] = marker2;
	
	debugprint(localdebugmode, PROTOCOL_NAME ": Added: Tag: %s Value: %s", tag.c_str(), value.c_str());
	
	return true;
}

/* This is stolen out of socket.cpp. */
int recvuntil(Socket &sock, char *string, int length, char end)
{
	int totalreceved = 0;
	int receved = 0;
		
	while (totalreceved < length)
	{
		if (!(receved = sock.recvdata(&string[totalreceved], 1)) > 0) return -1;
		if (string[totalreceved] == end) return totalreceved + 1;
		totalreceved += receved;
	}
	
	/* It was too long, but nevermind. */
	return totalreceved;
}
