# Copyright (c) 2015 SUSE Linux GmbH.  All rights reserved.
#
# This file is part of kiwi.
#
# kiwi is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# kiwi is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with kiwi.  If not, see <http://www.gnu.org/licenses/>
#
import importlib
from typing import (
    Dict, Any
)
import os
import logging
from xml.dom import minidom
from lxml import etree

# project
from kiwi.utils.temporary import Temporary
from kiwi.markup import Markup
from kiwi.defaults import Defaults
from kiwi import xml_parse
from kiwi.command import Command

from kiwi.exceptions import (
    KiwiCommandError,
    KiwiSchemaImportError,
    KiwiValidationError,
    KiwiDescriptionInvalid,
    KiwiDataStructureError,
    KiwiExtensionError,
    KiwiCommandNotFound
)

log = logging.getLogger('kiwi')


class XMLDescription:
    """
    **Implements data management for the image description**

    Supported description markup languages are XML, YAML, JSON and INI.
    The provided input file is converted into XML, transformed to the
    current RelaxNG schema via XSLT and validated against this result.

    * XSLT Style Sheet processing to apply on this version of kiwi
    * Schema Validation based on RelaxNG schema
    * Loading XML data into internal data structures

    Attributes

    :param str description: path to description file
    :param str derived_from: path to base description file
    """
    def __init__(
        self, description: str = '', derived_from: str = None
    ):
        self.markup = Markup.new(description)
        self.description = self.markup.get_xml_description()
        self.derived_from = derived_from
        self.description_origin = description
        self.extension_data: Dict = {}

    def load(self) -> Any:
        """
        Read XML description, validate it against the schema
        and the schematron rules and pass it to the
        autogenerated(generateDS) parser.

        :return: instance of XML toplevel domain (image)

        :rtype: object
        """
        isoschematron = None
        schematron = None
        try:
            isoschematron = importlib.import_module(
                Defaults.get_schematron_module_name()
            )
        except Exception as error:
            log.warning(f"schematron validation skipped: {error}")
        try:
            schema_doc = etree.parse(Defaults.get_schema_file())
            relaxng = etree.RelaxNG(schema_doc)
            if isoschematron:
                schematron = isoschematron.Schematron(
                    schema_doc, store_report=True
                )
        except Exception as issue:
            raise KiwiSchemaImportError(issue)
        try:
            description = etree.parse(self.description)
            validation_rng = relaxng.validate(description)
            if schematron:
                validation_schematron = schematron.validate(description)
        except Exception as issue:
            raise KiwiValidationError(issue)
        if not validation_rng:
            XMLDescription._get_relaxng_validation_details(
                Defaults.get_schema_file(),
                self.description,
                relaxng.error_log
            )
        if schematron and not validation_schematron:
            XMLDescription._get_schematron_validation_details(
                schematron.validation_report
            )
        if not validation_rng or (schematron and not validation_schematron):
            log.debug(open(self.description).read())
            raise KiwiDescriptionInvalid(
                'Failed to validate schema and/or schematron rules. '
                'Use --debug for more details'
            )

        parse_result = self._parse()

        if parse_result.get_extension():
            extension_namespace_map = \
                description.getroot().xpath('extension')[0].nsmap

            for namespace_name in extension_namespace_map:
                extensions_for_namespace = description.getroot().xpath(
                    'extension/{namespace}:*'.format(namespace=namespace_name),
                    namespaces=extension_namespace_map
                )
                if extensions_for_namespace:
                    # one toplevel entry point per extension via xmlns
                    if len(extensions_for_namespace) > 1:
                        raise KiwiExtensionError(
                            'Multiple toplevel sections for "{0}" found'.format(
                                namespace_name
                            )
                        )

                    # store extension xml data parse tree for this namespace
                    self.extension_data[namespace_name] = \
                        etree.ElementTree(extensions_for_namespace[0])

                    # validate extension xml data
                    try:
                        xml_catalog = Command.run(
                            [
                                'xmlcatalog', '/etc/xml/catalog',
                                extension_namespace_map[namespace_name]
                            ]
                        )
                        extension_schema = xml_catalog.output.rstrip().replace(
                            'file://', ''
                        )
                        extension_relaxng = etree.RelaxNG(
                            etree.parse(extension_schema)
                        )
                    except Exception as issue:
                        raise KiwiExtensionError(
                            'Extension schema error: {0}'.format(issue)
                        )
                    validation_result = extension_relaxng.validate(
                        self.extension_data[namespace_name]
                    )
                    if not validation_result:
                        xml_data_unformatted = etree.tostring(
                            self.extension_data[namespace_name],
                            encoding='utf-8'
                        )
                        xml_data_domtree = minidom.parseString(
                            xml_data_unformatted
                        )
                        extension_file = Temporary().new_file()
                        with open(extension_file.name, 'w') as xml_data:
                            xml_data.write(xml_data_domtree.toprettyxml())
                        XMLDescription._get_relaxng_validation_details(
                            extension_schema,
                            extension_file.name,
                            extension_relaxng.error_log
                        )
                        raise KiwiExtensionError(
                            'Schema validation for extension XML data failed'
                        )

        return parse_result

    def get_extension_xml_data(self, namespace_name: str) -> Any:
        """
        Return the xml etree parse result for the specified extension namespace

        :param str namespace_name: name of the extension namespace

        :return: result of etree.parse

        :rtype: object
        """
        return self.extension_data.get(namespace_name)

    @staticmethod
    def _get_relaxng_validation_details(
        schema_file, description_file, error_log
    ):
        """
        Run jing program to validate description against the schema

        Jing provides detailed error information in case of a schema
        validation failure. If jing is not present the standard
        error_log as provided from the raw XML libraries is used
        """
        try:
            Command.run(
                ['jing', schema_file, description_file]
            )
        except KiwiCommandError as issue:
            log.info('RelaxNG validation failed. See jing report:')
            log.info('--> {0}'.format(issue))
        except KiwiCommandNotFound as issue:
            log.warning(issue)
            log.warning(
                'For detailed schema validation report, install: jing'
            )
            log.info('Showing only raw library error log:')
            log.info('--> {0}'.format(error_log))

    @staticmethod
    def _get_schematron_validation_details(validation_report):
        """
        Extract error message form the schematron validation report

        :param etree validation_report: the schematron validation report
        """
        nspaces = validation_report.getroot().nsmap
        log.info('Schematron validation failed:')
        for msg in validation_report.xpath(
            '//svrl:failed-assert/svrl:text', namespaces=nspaces
        ):
            log.info('--> %s', msg.text)

    def _parse(self):
        try:
            parse = xml_parse.parse(
                self.description, True
            )
            parse.description_dir = self.description_origin and os.path.dirname(
                self.description_origin
            )
            parse.derived_description_dir = self.derived_from
            return parse
        except Exception as issue:
            raise KiwiDataStructureError(issue)
