/**************************************************************************
*   Copyright (C) 2007-2010 by Thomas Thelliez aka jblud                  *
*   Contact : <admin.kontrol@gmail.com>                                   *
*                                                                         *
* This program is free software; you can redistribute it and/or           *
* modify it either under the terms of the GNU Lesser General Public       *
* License version 3 as published by the Free Software Foundation          *
* (the "LGPL") or, at your option, any later version.                     *
* If you do not alter this notice, a recipient may use your version       *
* of this file under the LGPL.                                            *
*                                                                         *
* You should have received a copy of the LGPL along with this library     *
* in the file COPYING; if not, write to the Free Software                 *
* Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA *
*                                                                         *
* This software is distributed on an "AS IS" basis, WITHOUT WARRANTY      *
* OF ANY KIND, either express or implied. See the LGPL for                *
* the specific language governing rights and limitations.                 *
**************************************************************************/

#include "auth_tcp_client.h"

auth_tcp_client::auth_tcp_client(
        QString server_password,
        QObject *parent)
            : password(server_password),
            QObject(parent)
{
    tcp_socket = new QTcpSocket(this);
    auth_status = TOKEN_WELCOME;

    OpenSSL_add_all_algorithms();
    ERR_load_crypto_strings();

    rsa_key_pair = RSA_generate_key (
            RSA_KEY_SIZE, 65537, NULL, NULL );
    if (rsa_key_pair == NULL) {
        fprintf (stderr, "%s:%d Creation of keys failed\n",
                 __FILE__, __LINE__);
    }

    // Check key structure
    int valid = RSA_check_key(rsa_key_pair);
    if (valid == 0) {
        fprintf (stderr, "%s:%d Invalid RSA Key.\n",
                 __FILE__, __LINE__);
    }

    pubk = RSA_Key_to_char_array(rsa_key_pair,
                                 PUBLIC_KEY, NULL);
    if (pubk == NULL) {
        fprintf (stderr, "%s:%d Problem converting Key.\n",
                 __FILE__, __LINE__);
    }

    char * prvk = RSA_Key_to_char_array(rsa_key_pair,
                                        PRIVATE_KEY, NULL);
    if (prvk == NULL) {
        fprintf (stderr, "%s:%d Problem converting Key.\n",
                 __FILE__, __LINE__);
    }

    rsa_prv = char_array_to_RSA_key(prvk,
                                    PRIVATE_KEY, NULL);
    if (rsa_prv == NULL) {
        fprintf (stderr, "%s:%d Problem converting Key.\n",
                 __FILE__, __LINE__);
    }

    qstring_pub_key = QString(pubk);
    receiving = false;
    QObject::connect(tcp_socket, SIGNAL(readyRead()),
                     this, SLOT(read_from_server()),
                     Qt::DirectConnection);
    QObject::connect(tcp_socket, SIGNAL(disconnected()),
                     this, SLOT(disconnected_client()));
    QObject::connect(tcp_socket, SIGNAL(error(QAbstractSocket::SocketError)),
                     this, SLOT(socket_error(QAbstractSocket::SocketError)));
}

void auth_tcp_client::read_from_server()
{
    if (auth_status == AUTH_CLIENT) {
        QString data = get_data_from_socket();
        if (!data.isEmpty()) {
            QString decrypted = auth_aes_decrypt(data);
            emit received_stream(decrypted);
        }
    } else {
        handle_crypted_auth();
    }
}

void auth_tcp_client::connect_to_server(
        const char* ip_server, int port_server)
{
    tcp_socket->connectToHost(ip_server,
                              port_server);
}

void auth_tcp_client::stop_client()
{
    if (tcp_socket->state() != QAbstractSocket::UnconnectedState && tcp_socket->state() != QAbstractSocket::ClosingState) {
        tcp_socket->close();
        tcp_socket->abort();
    }
}

QString auth_tcp_client::get_data_from_socket()
{
    if (!receiving) {
        QString sLine = QString("");
        if (tcp_socket->bytesAvailable() > HEADER_SIZE) {
            sLine.append(tcp_socket->read(HEADER_SIZE));
        }
        QStringList list = sLine.split(SIZE_DELIMITER);
        if (list.size() > 1) {
            packet_size = remove_header_padding(list.at(0).toInt());
            current_bytes = new QByteArray;
            receiving = true;
            current_bytes->append(QByteArray(list.at(1).toUtf8()));
            current_size = current_bytes->length();
        } else {send_error(); return "";}
    }
    if (packet_size > current_size) {
        while (tcp_socket->bytesAvailable() > 0) {
            current_bytes->append(tcp_socket->read(10000));
            current_size = current_bytes->length();
            QApplication::processEvents();
            emit update_progress_size(current_size, packet_size);
        }
    }
    if (packet_size == current_size) {
        QString result = QString(current_bytes->data());
        delete current_bytes;
        receiving = false;
        current_bytes = NULL;
        current_size = 0;
        packet_size = 0;
        if (result.contains(ERROR, Qt::CaseInsensitive))
            send_error();
        return result;
    }
    return "";
}

void auth_tcp_client::write_data_to_socket(QString data)
{
    tcp_socket->write(QString(add_header_padding(data.length()) + QString(SIZE_DELIMITER) + data).toUtf8());
    tcp_socket->flush();
}

/*
    #define TOKEN_WELCOME 0
    #define RSA_PUBLIC_KEY_EXCHANGE 1
    #define PASSWORD_VALIDATION 2
    #define AES_KEY_RECEPTION 3
    #define AUTH_SUCCESS 4
*/
void auth_tcp_client::handle_crypted_auth()
{
    QString result;
    switch( auth_status )
    {
    case TOKEN_WELCOME: {
            // decoding from base64.
            result = get_data_from_socket();
            if (!result.isEmpty()) {
                if (result.contains(XML_START_ELEMENT,
                                    Qt::CaseInsensitive)
                    && result.contains(XML_END_ELEMENT,
                                       Qt::CaseInsensitive)) {
                    token = welcome_token_exchange_parser(
                            result.toUtf8().data());

                    pub_key_final = qstring_pub_key.mid(
                            0,
                            qstring_pub_key.indexOf(
                                    END_RSA_PUBLIC_KEY));
                    pub_key_final.append(END_RSA_PUBLIC_KEY);

                    write_data_to_socket(QString(
                            rsa_public_key_exchange(
                                    pub_key_final.toUtf8().data())));
                    emit emit_tcp_state(TOKEN_WELCOME);
                    auth_status = AES_KEY_RECEPTION;
                } else {
                    fprintf (stderr,
                             "%s:%d TOKEN welcome XML : bad format..\n",
                             __FILE__, __LINE__);
                    send_error();
                }
            }
        }
        break;
    case AES_KEY_RECEPTION : {
            result = get_data_from_socket();
            if (!result.isEmpty() && !result.contains(ERROR,
                                 Qt::CaseInsensitive)) {
                unsigned char decrypted[2560] = { 0 };
                if (RSA_private_decrypt(RSA_size(rsa_key_pair),
                                        (unsigned char *)
                    QByteArray::fromBase64(
                            QByteArray(
                                    result.toUtf8().data())).data(),
                            decrypted, rsa_prv,
                            RSA_PKCS1_PADDING)==-1) {
                    int error = ERR_get_error();
                    fprintf (stderr, "%s %s:%d \n", ERR_lib_error_string(
                            error ),
                             __FILE__, __LINE__);
                    fprintf (stderr, "%s %s:%d \n", ERR_func_error_string(
                            error ),
                             __FILE__, __LINE__);
                    fprintf (stderr, "%s %s:%d \n", ERR_reason_error_string(
                            error ),
                             __FILE__, __LINE__);
                    send_error();
                } else if (strlen((char*) decrypted) == 0) {
                    fprintf (stderr, "%s:%d Error decrypting data.\n",
                             __FILE__, __LINE__);
                    send_error();
                }
                else {
                    if (QString((char*)decrypted).contains(XML_START_ELEMENT,
                                                           Qt::CaseInsensitive) &&
                        QString((char*)decrypted).contains(
                                XML_END_ELEMENT, Qt::CaseInsensitive)) {
                        QString message_final = QString((char*)decrypted);
                        message_final = message_final.mid(0,
                                                          message_final.indexOf(
                                                                  XML_END_ELEMENT));
                        message_final.append(XML_END_ELEMENT);
                        aes = aes_key_exchange_parser(
                                (char*) message_final.toUtf8().data());

                        // SHA1 hash of token + password.
                        unsigned char sha1sum[SHA_DIGEST_LENGTH];
                        strcat(token, password.toUtf8().data());
                        SHA1((unsigned char*)token, strlen(token), sha1sum);
                        b64_hashed_password = base64(
                                (unsigned char*)sha1sum,
                                sizeof(sha1sum));

                        write_data_to_socket(auth_aes_encrypt(password_exchange(
                                b64_hashed_password)));

                        auth_status = AUTH_SUCCESS;
                        emit emit_tcp_state(AES_KEY_RECEPTION);

                    } else {
                        fprintf (stderr,
                                 "%s:%d AES key reception XML : bad format.\n",
                                 __FILE__, __LINE__);
                        send_error();
                    }
                }
            }
            break;
        }
    case AUTH_SUCCESS :
        result = get_data_from_socket();
        // Receiving encrypted stream with AES symetric algorithm
        if (!result.isEmpty()) {
            QString decrypted = auth_aes_decrypt(result);
            if (decrypted.contains(SUCCESS,
                                   Qt::CaseInsensitive)) {
                auth_status = AUTH_CLIENT;
                emit auth_suceeded();
                emit emit_tcp_state(AUTH_SUCCESS);
            } else {
                tcp_socket->close();
                tcp_socket->disconnect();
                emit wrong_password();
            }
        }
        break;
    }
}

QString auth_tcp_client::auth_aes_encrypt(char* message)
{
    // AES Encryption + base64 encoding
    std::string key_encrypt((char*) aes->key);
    std::string std_xml((char*) message);
    string str = END_DELIMITER;
    std_xml.append(str);

    std::string encrypted(aes_encrypt(encode(std_xml), key_encrypt));
    return QString::fromStdString(encrypted);
}

QString auth_tcp_client::auth_aes_decrypt(QString message)
{
    // AES Decryption after base64 decoding
    std::string key_decrypt((char*) aes->key);
    std::string decrypted(aes_decrypt(message.toStdString(),
                                      key_decrypt));
    QString message_final = QString(decode(decrypted).c_str());
    message_final = message_final.mid(0, message_final.indexOf(END));
    return message_final;
}

int auth_tcp_client::write_encrypted_data_to_socket(QString data)
{
    if (auth_status == AUTH_CLIENT) {
       this->write_data_to_socket(auth_aes_encrypt(data.toUtf8().data()));
       return 1;
    }
    return 0;
}

void auth_tcp_client::send_error()
{
    emit emit_error();
    write_data_to_socket(QString(ERROR));
    tcp_socket->close();
    tcp_socket->disconnect();
}

void auth_tcp_client::disconnected_client()
{
    emit disconnected_socket();
}

/*
QAbstractSocket::ConnectionRefusedError	0	The connection was refused by the peer (or timed out).
QAbstractSocket::RemoteHostClosedError	1	The remote host closed the connection. Note that the client socket (i.e., this socket) will be closed after the remote close notification has been sent.
QAbstractSocket::HostNotFoundError	2	The host address was not found.
QAbstractSocket::SocketAccessError	3	The socket operation failed because the application lacked the required privileges.
QAbstractSocket::SocketResourceError	4	The local system ran out of resources (e.g., too many sockets).
QAbstractSocket::SocketTimeoutError	5	The socket operation timed out.
QAbstractSocket::DatagramTooLargeError	6	The datagram was larger than the operating system's limit (which can be as low as 8192 bytes).
QAbstractSocket::NetworkError	7	An error occurred with the network (e.g., the network cable was accidentally plugged out).
QAbstractSocket::AddressInUseError	8	The address specified to QUdpSocket::bind() is already in use and was set to be exclusive.
QAbstractSocket::SocketAddressNotAvailableError	9	The address specified to QUdpSocket::bind() does not belong to the host.
QAbstractSocket::UnsupportedSocketOperationError	10	The requested socket operation is not supported by the local operating system (e.g., lack of IPv6 support).
QAbstractSocket::ProxyAuthenticationRequiredError	12	The socket is using a proxy, and the proxy requires authentication.
QAbstractSocket::SslHandshakeFailedError	13	The SSL/TLS handshake failed, so the connection was closed (only used in QSslSocket)
QAbstractSocket::UnfinishedSocketOperationError	11	Used by QAbstractSocketEngine only, The last operation attempted has not finished yet (still in progress in the background).
QAbstractSocket::ProxyConnectionRefusedError	14	Could not contact the proxy server because the connection to that server was denied
QAbstractSocket::ProxyConnectionClosedError	15	The connection to the proxy server was closed unexpectedly (before the connection to the final peer was established)
QAbstractSocket::ProxyConnectionTimeoutError	16	The connection to the proxy server timed out or the proxy server stopped responding in the authentication phase.
QAbstractSocket::ProxyNotFoundError	17	The proxy address set with setProxy() (or the application proxy) was not found.
QAbstractSocket::ProxyProtocolError	18	The connection negotiation with the proxy server because the response from the proxy server could not be understood.
QAbstractSocket::UnknownSocketError
*/
void auth_tcp_client::socket_error(QAbstractSocket::SocketError socketError)
{
    switch (socketError) {
    case QAbstractSocket::RemoteHostClosedError:
        emit disconnected_client(QAbstractSocket::RemoteHostClosedError);
        break;
    case QAbstractSocket::HostNotFoundError:
        qDebug("The host was not found. Please check the "
               "host name and port settings.");
        emit disconnected_client(QAbstractSocket::HostNotFoundError);
        break;
    case QAbstractSocket::ConnectionRefusedError:
        qDebug("Make sure the KontrolPack server is running, "
               "and check that the host name and port "
               "settings are correct.");

        emit disconnected_client(QAbstractSocket::ConnectionRefusedError);
        break;
    case QAbstractSocket::SocketAccessError:
        emit disconnected_client(QAbstractSocket::SocketAccessError);
        break;
    case QAbstractSocket::SocketResourceError:
        emit disconnected_client(QAbstractSocket::SocketResourceError);
        break;
    case QAbstractSocket::SocketTimeoutError:
        emit disconnected_client(QAbstractSocket::SocketTimeoutError);
        break;
    case QAbstractSocket::DatagramTooLargeError:
        emit disconnected_client(QAbstractSocket::DatagramTooLargeError);
        break;
    case QAbstractSocket::NetworkError:
        emit disconnected_client(QAbstractSocket::NetworkError);
        break;
    case QAbstractSocket::AddressInUseError:
        emit disconnected_client(QAbstractSocket::AddressInUseError);
        break;
    case QAbstractSocket::SocketAddressNotAvailableError:
        emit disconnected_client(QAbstractSocket::SocketAddressNotAvailableError);
        break;
    case QAbstractSocket::UnsupportedSocketOperationError:
        emit disconnected_client(QAbstractSocket::UnsupportedSocketOperationError);
        break;
    case QAbstractSocket::ProxyAuthenticationRequiredError:
        emit disconnected_client(QAbstractSocket::ProxyAuthenticationRequiredError);
        break;
    case QAbstractSocket::SslHandshakeFailedError:
        emit disconnected_client(QAbstractSocket::SslHandshakeFailedError);
        break;
    case QAbstractSocket::UnfinishedSocketOperationError:
        emit disconnected_client(QAbstractSocket::UnfinishedSocketOperationError);
        break;
    case QAbstractSocket::ProxyConnectionRefusedError:
        emit disconnected_client(QAbstractSocket::ProxyConnectionRefusedError);
        break;
    case QAbstractSocket::ProxyConnectionClosedError:
        emit disconnected_client(QAbstractSocket::ProxyConnectionClosedError);
        break;
    case QAbstractSocket::ProxyConnectionTimeoutError:
        emit disconnected_client(QAbstractSocket::ProxyConnectionTimeoutError);
        break;
    case QAbstractSocket::ProxyNotFoundError:
        emit disconnected_client(QAbstractSocket::ProxyNotFoundError);
        break;
    case QAbstractSocket::ProxyProtocolError:
        emit disconnected_client(QAbstractSocket::ProxyProtocolError);
        break;
    case QAbstractSocket::UnknownSocketError:
        emit disconnected_client(QAbstractSocket::UnknownSocketError);
        break;
    default:
        ;
    }
}
