/**
 * Copyright (c) Members of the EGEE Collaboration. 2004-2010. 
 * See http://www.eu-egee.org/partners/ for details on the copyright
 * holders.  
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); 
 * you may not use this file except in compliance with the License. 
 * You may obtain a copy of the License at 
 * 
 *     http://www.apache.org/licenses/LICENSE-2.0 
 * 
 * Unless required by applicable law or agreed to in writing, software 
 * distributed under the License is distributed on an "AS IS" BASIS, 
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. 
 * See the License for the specific language governing permissions and 
 * limitations under the License.
 *
 *
 *  Authors:
 *  2009-
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     Mischa Sall\'e <msalle@nikhef.nl>
 *     David Groep <davidg@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *     <grid-mw-security@nikhef.nl> 
 *
 *  2007-2009
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     David Groep <davidg@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *
 *  2003-2007
 *     Martijn Steenbakkers <martijn@nikhef.nl>
 *     Gerben Venekamp <venekamp@nikhef.nl>
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     David Groep <davidg@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *
 */


/*!
 *  \file pdl_rule.h
 *
 *  \brief Include file for using the pdl rules.
 *
 *  \author  G.M. Venekamp  (venekamp@nikhef.nl)
 *  \version $Revision: 14868 $
 *  \date    $Date: 2011-03-04 21:55:13 +0100 (Fri, 04 Mar 2011) $
 *
 */


#ifndef _PDL_RULE_H
#define _PDL_RULE_H

#include "pdl.h"


/*!
 *  \brief Structure keeps track of the state and the true/false
 *         braches.
 *
 */
typedef struct rule_s {
  char* state;         //!<  Name of the state.
  char* true_branch;   //!<  Name of the true_branch, or 0 if none.
  char* false_branch;  //!<  Name of the false_branch, or 0 if none.
  unsigned int lineno;       //!<  Line number where rule appeared.
  struct rule_s* next;       //!<  Next rule, or 0 if none.
} rule_t;


/*!
 *  \brief Which type is the current rule.
 */
typedef enum {
  STATE,          //!<  State
  TRUE_BRANCH,    //!<  True branch
  FALSE_BRANCH    //!<  False branch
} rule_type_t;


/*!
 *  \brief Tell something about recursion in rules.
 */
typedef enum {
  NO_RECURSION      = 0x00,   //!<  There is no known recursion.
  RECURSION         = 0x01,   //!<  Recursion has been found.
  RECURSION_HANDLED = 0x02    //!<  Recursion has been found and handled/reported.
} recursion_t;


/*!
 *  \brief  Given a rule, which side of the rule are we working on.
 */
typedef enum {
  left_side,   //!<  left side, i.e. state part of the rule.
  right_side   //!<  right side, i.e. either true or false branch.
} side_t;


extern rule_t* lcmaps_add_rule(record_t* state, record_t* true_branch, record_t* false_branch);
extern void    lcmaps_free_rules(rule_t* rule);
extern rule_t* lcmaps_find_state(rule_t* rule, char* state);
extern void    lcmaps_show_rules(rule_t* rule);
extern void    lcmaps_start_new_rules(void);
extern rule_t* lcmaps_get_top_rule(void);
extern void    lcmaps_allow_new_rules(BOOL allow);
extern rule_t* lcmaps_get_rule(char* rule, side_t side);
extern void    lcmaps_set_top_rule(rule_t* rule);
extern BOOL    lcmaps_check_rule_for_recursion(rule_t* rule);
extern void    lcmaps_reduce_rule(rule_t* rule);

#endif
