/*
 * @(#)MarkRecord.java
 *
 * Copyright (C) 2002-2003 Matt Albrecht
 * groboclown@users.sourceforge.net
 * http://groboutils.sourceforge.net
 *
 *  Permission is hereby granted, free of charge, to any person obtaining a
 *  copy of this software and associated documentation files (the "Software"),
 *  to deal in the Software without restriction, including without limitation
 *  the rights to use, copy, modify, merge, publish, distribute, sublicense,
 *  and/or sell copies of the Software, and to permit persons to whom the 
 *  Software is furnished to do so, subject to the following conditions:
 *
 *  The above copyright notice and this permission notice shall be included in 
 *  all copies or substantial portions of the Software. 
 *
 *  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR 
 *  IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, 
 *  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL 
 *  THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER 
 *  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING 
 *  FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER 
 *  DEALINGS IN THE SOFTWARE.
 */

package net.sourceforge.groboutils.codecoverage.v2.datastore;

import net.sourceforge.groboutils.codecoverage.v2.IAnalysisMetaData;

/**
 * Contains data associated with an analysis module's bytecode instruction
 * mark.  It should be associated with a particular class.
 *
 * @author    Matt Albrecht <a href="mailto:groboclown@users.sourceforge.net">groboclown@users.sourceforge.net</a>
 * @version   $Date: 2003/11/25 13:10:41 $
 * @since     December 15, 2002
 */
public class MarkRecord
{
    private IAnalysisMetaData amd;
    private String analysisModule;
    private String methodSignature;
    private short markIndex;
    
    private short amIndex = -1;
    private short methIndex = -1;
    private int lineNo = -1;
    
    
    /**
     * @param data cannot be <tt>null</tt>.
     * @param am the analysis module that made this mark.
     * @param methSig the method signature where this mark was placed.
     * @param lineNumber the Java source line corresponding to this mark.
     */
    public MarkRecord( IAnalysisMetaData data, String am, String methSig,
            short markId, int lineNumber )
    {
        if (data == null || am == null || methSig == null)
        {
            throw new IllegalArgumentException( "No null args." );
        }
        if (markId < 0)
        {
            throw new IllegalArgumentException(
                "Indicies cannot be less than 0.");
        }
        this.amd = data;
        this.analysisModule = am;
        this.methodSignature = methSig;
        this.markIndex = markId;
        this.lineNo = lineNumber;
    }
    
    
    /**
     * @see ClassRecord#addMark( MarkRecord )
     */
    void processMark( ClassRecord cr, AnalysisModuleSet ams )
    {
        if (cr == null)
        {
            throw new IllegalArgumentException( "No null args." );
        }
        if (ams == null)
        {
            ams = cr.getAnalysisModuleSet();
        }
        this.amIndex = ams.getMeasureIndex( getAnalysisModule() );
        this.methIndex = cr.getMethodIndex( getMethodSignature() );
    }
    
    
    public boolean equals( Object o )
    {
        if (o == null || !(o instanceof MarkRecord))
        {
            return false;
        }
        if (this == o)
        {
            return true;
        }
        
        // line number doesn't add into the equals.
        MarkRecord mr = (MarkRecord)o;
        if (mr.analysisModule.equals( this.analysisModule ) &&
            mr.methodSignature.equals( this.methodSignature ) &&
            mr.markIndex == this.markIndex)
        {
            // the amd object doesn't have to match.
            return true;
        }
        // else
        return false;
    }
    
    
    public int hashCode()
    {
        return (int)this.markIndex +
            this.analysisModule.hashCode() +
            this.methodSignature.hashCode();
    }
    
    
    public IAnalysisMetaData getAnalysisMetaData()
    {
        return this.amd;
    }
    
    
    public String getAnalysisModule()
    {
        return this.analysisModule;
    }
    
    
    public short getAnalysisModuleIndex()
    {
        if (this.amIndex < 0)
        {
            throw new IllegalStateException(
                "Never processed or unknown module '"+this.analysisModule+
                "'." );
        }
        return this.amIndex;
    }
    
    
    public String getMethodSignature()
    {
        return this.methodSignature;
    }
    
    
    public short getMethodIndex()
    {
        if (this.methIndex < 0)
        {
            throw new IllegalStateException(
                "Never processed or unknown method '"+this.methodSignature+
                "'." );
        }
        return this.methIndex;
    }
    
    
    public short getMarkIndex()
    {
        return this.markIndex;
    }
    
    
    public int getLineNumber()
    {
        return this.lineNo;
    }
    
    
    public String toString()
    {
        return "Mark Record: "+getMethodSignature()+", mark "+getMarkIndex();
    }
}

