/*---------------------------------------------------------------------------*\
  =========                 |
  \\      /  F ield         | OpenFOAM: The Open Source CFD Toolbox
   \\    /   O peration     |
    \\  /    A nd           | www.openfoam.com
     \\/     M anipulation  |
-------------------------------------------------------------------------------
    Released 2009-2011 OpenCFD Ltd.
    Copyright (C) 2011-2013 OpenFOAM Foundation
    Modified code Copyright (C) 2017-2019 OpenCFD Ltd.
-------------------------------------------------------------------------------
License
    This file is part of OpenFOAM.

    OpenFOAM is free software: you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    OpenFOAM is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
    for more details.

    You should have received a copy of the GNU General Public License
    along with OpenFOAM.  If not, see <http://www.gnu.org/licenses/>.

\*---------------------------------------------------------------------------*/

#include "argList.H"

// * * * * * * * * * * * * * * * Member Functions  * * * * * * * * * * * * * //

template<class T>
inline void Foam::argList::readList(ITstream& is, List<T>& list)
{
    if (is.size() == 1)
    {
        // Single token - treat like List with one entry
        list.resize(1);
        is >> list.first();
    }
    else
    {
        is >> list;
    }
}


// * * * * * * * * * * * * * * * Member Functions  * * * * * * * * * * * * * //

inline const Foam::word& Foam::argList::executable() const
{
    return executable_;
}


inline const Foam::string& Foam::argList::commandLine() const
{
    return commandLine_;
}


inline const Foam::fileName& Foam::argList::rootPath() const
{
    return rootPath_;
}


inline const Foam::fileName& Foam::argList::caseName() const
{
    return case_;
}


inline const Foam::fileName& Foam::argList::globalCaseName() const
{
    return globalCase_;
}


inline Foam::fileName Foam::argList::path() const
{
    return rootPath()/caseName();
}


inline Foam::fileName Foam::argList::globalPath() const
{
    return rootPath()/globalCaseName();
}


inline Foam::fileName Foam::argList::relativePath
(
    const fileName& input,
    const bool caseTag
) const
{
    return input.relative(globalPath(), caseTag);
}


inline bool Foam::argList::distributed() const
{
    return parRunControl_.distributed();
}


inline const Foam::ParRunControl& Foam::argList::parRunControl() const
{
    return parRunControl_;
}


inline Foam::label Foam::argList::size() const
{
    return args_.size();
}


inline const Foam::stringList& Foam::argList::args() const
{
    return args_;
}


inline Foam::stringList& Foam::argList::args()
{
    return args_;
}


inline const Foam::HashTable<Foam::string>& Foam::argList::options() const
{
    return options_;
}


inline Foam::HashTable<Foam::string>& Foam::argList::options()
{
    return options_;
}


inline bool Foam::argList::found(const word& optName) const
{
    return options_.found(optName);
}


inline Foam::ITstream Foam::argList::lookup(const word& optName) const
{
    return ITstream(optName, options_[optName]);
}


// * * * * * * * * * * * * Template Specializations  * * * * * * * * * * * * //

namespace Foam
{
    template<> inline int32_t argList::get<int32_t>(const label index) const
    {
        return Foam::readInt32(args_[index]);
    }

    template<> inline int64_t argList::get<int64_t>(const label index) const
    {
        return Foam::readInt64(args_[index]);
    }

    template<> inline float argList::get<float>(const label index) const
    {
        return Foam::readFloat(args_[index]);
    }

    template<> inline double argList::get<double>(const label index) const
    {
        return Foam::readDouble(args_[index]);
    }


    template<> inline int32_t argList::opt<int32_t>(const word& optName) const
    {
        return Foam::readInt32(options_[optName]);
    }

    template<> inline int64_t argList::opt<int64_t>(const word& optName) const
    {
        return Foam::readInt64(options_[optName]);
    }

    template<> inline float argList::opt<float>(const word& optName) const
    {
        return Foam::readFloat(options_[optName]);
    }

    template<> inline double argList::opt<double>(const word& optName) const
    {
        return Foam::readDouble(options_[optName]);
    }


    template<>
    inline string argList::get<Foam::string>(const label index) const
    {
        return args_[index];
    }

    template<>
    inline word argList::get<Foam::word>(const label index) const
    {
        return args_[index];
    }

    template<>
    inline fileName argList::get<Foam::fileName>(const label index) const
    {
        return args_[index];
    }


    template<>
    inline string argList::opt<Foam::string>(const word& optName) const
    {
        return options_[optName];
    }

    template<>
    inline word argList::opt<Foam::word>(const word& optName) const
    {
        return options_[optName];
    }

    template<>
    inline fileName argList::opt<Foam::fileName>(const word& optName) const
    {
        return options_[optName];
    }
}


// * * * * * * * * * * * * * * * Member Functions  * * * * * * * * * * * * * //

template<class T>
inline T Foam::argList::get(const label index) const
{
    ITstream is(Foam::name(index), args_[index]);

    T val;
    is >> val;

    checkITstream(is, index);

    return val;
}


template<class T>
inline T Foam::argList::opt(const word& optName) const
{
    ITstream is(optName, options_[optName]);

    T val;
    is >> val;

    checkITstream(is, optName);

    return val;
}


template<class T>
inline T Foam::argList::opt(const word& optName, const T& deflt) const
{
    if (found(optName))
    {
        return opt<T>(optName);
    }

    return deflt;
}


template<class T>
inline bool Foam::argList::readIfPresent
(
    const word& optName,
    T& val
) const
{
    if (found(optName))
    {
        val = opt<T>(optName);
        return true;
    }

    return false;
}


template<class T>
inline bool Foam::argList::readIfPresent
(
    const word& optName,
    T& val,
    const T& deflt
) const
{
    if (readIfPresent<T>(optName, val))
    {
        return true;
    }

    val = deflt;
    return false;
}


template<class T>
inline T Foam::argList::getOrDefault
(
    const word& optName,
    const T& deflt
) const
{
    if (found(optName))
    {
        return opt<T>(optName);
    }

    return deflt;
}


template<class T>
inline Foam::List<T> Foam::argList::getList(const label index) const
{
    ITstream is(Foam::name(index), args_[index]);

    List<T> list;
    readList(is, list);

    checkITstream(is, index);

    return list;
}


template<class T>
inline Foam::List<T> Foam::argList::getList
(
    const word& optName,
    bool mandatory
) const
{
    List<T> list;

    if (mandatory || found(optName))
    {
        ITstream is(optName, options_[optName]);

        readList(is, list);

        checkITstream(is, optName);
    }

    return list;
}


template<class T>
inline bool Foam::argList::readListIfPresent
(
    const word& optName,
    List<T>& list
) const
{
    if (found(optName))
    {
        ITstream is(optName, options_[optName]);

        readList(is, list);

        checkITstream(is, optName);

        return true;
    }

    return false;
}


// * * * * * * * * * * * * * * * Member Operators  * * * * * * * * * * * * * //

inline const Foam::string& Foam::argList::operator[](const label index) const
{
    return args_[index];
}


inline const Foam::string& Foam::argList::operator[](const word& optName) const
{
    return options_[optName];
}


// ************************************************************************* //
