/*
 *  Copyright (c) 2007-2009 Cyrille Berger <cberger@cberger.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation;
 * either version 2, or (at your option) any later version of the License.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#ifndef _GTLCORE_CODE_GENERATOR_H_
#define _GTLCORE_CODE_GENERATOR_H_

#include <list>
#include <vector>

#include <GTLCore/Macros.h>
#include <GTLCore/String.h>
#include <GTLCore/StdTypes.h>

namespace llvm {
  class BasicBlock;
  class Constant;
  class Function;
  class FunctionType;
  class Module;
  class Value;
  class Type;
  class LLVMContext;
}

// TODO: all functions should be static and take a GenerationContext as argument

namespace GTLCore {
  class ModuleData;
  class Type;
  class Value;
  class VariableNG;
  class Function;
  namespace AST {
    class Expression;
  }
}

namespace LLVMBackend {
  class ExpressionResult;
  class ExpressionGenerationContext;
  class GenerationContext;
  /**
   * @internal
   * This class is private to OpenGTL and shouldn't be use outside.
   * 
   * This class provides helper functions to create llvm code.
   * 
   * @ingroup GTLCore
   */
  class GTLCORE_EXPORT CodeGenerator {
    public:
      /**
       * Create a new \ref CodeGenerator for the given llvm module.
       */
      CodeGenerator(GTLCore::ModuleData* module);
      ~CodeGenerator();
      /**
       * Create a null terminated string
       */
      static llvm::Constant* stringToConstant(llvm::LLVMContext& _llvmContext, const GTLCore::String& string);
      /**
       * @return an integer constant
       */
      static llvm::Constant* integerToConstant(llvm::LLVMContext& _llvmContext, gtl_int32 v);
      static llvm::Constant* integerToConstant(llvm::LLVMContext& _llvmContext, gtl_uint32 v);
      static llvm::Constant* integerToConstant(llvm::LLVMContext& _llvmContext, gtl_int64 v);
      /**
       * @return a boolean constant
       */
      static llvm::Constant* boolToConstant(llvm::LLVMContext& _llvmContext, bool v);
      /**
       * @return a float constant
       */
      static llvm::Constant* floatToConstant(llvm::LLVMContext& _llvmContext, float v);
      static llvm::Constant* nullPointerToConstant(llvm::LLVMContext& _llvmContext);
      /**
       * @return convert a \ref Value to a llvm constant
       */
      static llvm::Constant* valueToConstant( LLVMBackend::GenerationContext& _gc, const GTLCore::Value& v);
      static llvm::Constant* constantsToArray( LLVMBackend::GenerationContext& _gc, const std::vector< llvm::Constant* >& _constants, const GTLCore::Type* _type );
      static llvm::Constant* constantsToStructure( LLVMBackend::GenerationContext& _gc, const std::vector< llvm::Constant* >& _constants, const GTLCore::Type* _type );
      static llvm::Constant* constantsToVector( LLVMBackend::GenerationContext& _gc, const std::vector< llvm::Constant* >& _constants, const GTLCore::Type* _type );
      /**
       * @return a \ref Value from a llvm constant
       */
      static GTLCore::Value constantToValue(llvm::Constant* constant, const GTLCore::Type* _type);
      /**
       * Create a vector of type @p _vecType
       */
      static llvm::Value* createVector(llvm::LLVMContext& _llvmContext, const GTLCore::Type* _vecType);
      /**
       * Create a vector of type @p _vecType
       */
      static llvm::Value* createVector( const GTLCore::Type* _vecType, llvm::Constant* _val);
      /**
       * Create a vector of @p _size values equal to @p _val
       * @param _valType the type of the @p _val
       */
      static llvm::Value* createVector( int _size, llvm::Constant* _val);
      /**
       * Create a function and add it to the module.
       * @param name name of the function
       * @param type type of the function
       * @return a pointer to the llvm function
       */
      static llvm::Function* createFunction( llvm::Module* _module, llvm::FunctionType* type, const GTLCore::String& name);
      static llvm::Value* convertPointerToCharP(llvm::BasicBlock* currentBlock, llvm::Value* value);
      /**
       * Convert the pointer contained in @p value to a pointer of type @p type
       */
      static llvm::Value* convertPointerTo(llvm::BasicBlock* _currentBlock, llvm::Value* _value, llvm::Type* _type);
      static llvm::Value* convertValueTo(llvm::BasicBlock* currentBlock, llvm::Value* value, const GTLCore::Type* valueType, const GTLCore::Type* type);
      static llvm::Constant* convertConstantTo(llvm::Constant* constant, const GTLCore::Type* constantType, const GTLCore::Type* type);
      static llvm::Value* convertToHalf( LLVMBackend::GenerationContext& generationContext, llvm::BasicBlock* currentBlock, llvm::Value* value, const GTLCore::Type* _valueType);
      static llvm::Value* convertFromHalf( LLVMBackend::GenerationContext& generationContext, llvm::BasicBlock* currentBlock, llvm::Value* value);
      static llvm::Value* convertArrayToVector(LLVMBackend::GenerationContext& gc, LLVMBackend::ExpressionGenerationContext& _egc, llvm::Value* value, const GTLCore::Type* valueType, const GTLCore::Type* type);
      static llvm::Constant* convertConstantArrayToVector(llvm::Constant* constant, const GTLCore::Type* constantType, const GTLCore::Type* type);
      static llvm::Value* vectorValueAt( llvm::BasicBlock* _currentBlock, llvm::Value* _vector, llvm::Value* _index);
      static llvm::Value* createRound( llvm::BasicBlock* _currentBlock, llvm::Value* _val );
    public: // Boolean Expression
      llvm::Value* createOrExpression(llvm::BasicBlock* currentBlock, llvm::Value* lhs, const GTLCore::Type* lhsType, llvm::Value* rhs, const GTLCore::Type* rhsType);
      ExpressionResult createOrExpression( llvm::BasicBlock* currentBlock, ExpressionResult lhs, ExpressionResult rhs);
      llvm::Constant* createOrExpression( llvm::Constant* lhs, const GTLCore::Type* lhsType, llvm::Constant* rhs, const GTLCore::Type* rhsType);
      static llvm::Value* createAndExpression(llvm::BasicBlock* currentBlock, llvm::Value* lhs, const GTLCore::Type* lhsType, llvm::Value* rhs, const GTLCore::Type* rhsType);
      static ExpressionResult createAndExpression( llvm::BasicBlock* currentBlock, ExpressionResult lhs, ExpressionResult rhs);
      static llvm::Constant* createAndExpression( llvm::Constant* lhs, const GTLCore::Type* lhsType, llvm::Constant* rhs, const GTLCore::Type* rhsType);
    public: // Bit expressions
      llvm::Value* createBitXorExpression(llvm::BasicBlock* currentBlock, llvm::Value* lhs, const GTLCore::Type* lhsType, llvm::Value* rhs, const GTLCore::Type* rhsType);
      ExpressionResult createBitXorExpression( llvm::BasicBlock* currentBlock, ExpressionResult lhs, const GTLCore::Type* lhsType, ExpressionResult rhs, const GTLCore::Type* rhsType);
      llvm::Constant* createBitXorExpression( llvm::Constant* lhs, const GTLCore::Type* lhsType, llvm::Constant* rhs, const GTLCore::Type* rhsType);
      llvm::Value* createBitOrExpression(llvm::BasicBlock* currentBlock, llvm::Value* lhs, const GTLCore::Type* lhsType, llvm::Value* rhs, const GTLCore::Type* rhsType);
      ExpressionResult createBitOrExpression( llvm::BasicBlock* currentBlock, ExpressionResult lhs, const GTLCore::Type* lhsType, ExpressionResult rhs, const GTLCore::Type* rhsType);
      llvm::Constant* createBitOrExpression( llvm::Constant* lhs, const GTLCore::Type* lhsType, llvm::Constant* rhs, const GTLCore::Type* rhsType);
      static llvm::Value* createBitAndExpression(llvm::BasicBlock* currentBlock, llvm::Value* lhs, const GTLCore::Type* lhsType, llvm::Value* rhs, const GTLCore::Type* rhsType);
      static ExpressionResult createBitAndExpression( llvm::BasicBlock* currentBlock, ExpressionResult lhs, const GTLCore::Type* lhsType, ExpressionResult rhs, const GTLCore::Type* rhsType);
      static llvm::Constant* createBitAndExpression( llvm::Constant* lhs, const GTLCore::Type* lhsType, llvm::Constant* rhs, const GTLCore::Type* rhsType);
    public: // Arithmetic expressions
      // Addition
      static llvm::Value* createAdditionExpression(llvm::BasicBlock* currentBlock, llvm::Value* lhs, const GTLCore::Type* lhsType, llvm::Value* rhs, const GTLCore::Type* rhsType);
      static ExpressionResult createAdditionExpression( llvm::BasicBlock* currentBlock, ExpressionResult lhs, ExpressionResult rhs);
      static llvm::Constant* createAdditionExpression( llvm::Constant* lhs, const GTLCore::Type* lhsType, llvm::Constant* rhs, const GTLCore::Type* rhsType);
      // Substraction
      static llvm::Value* createSubstractionExpression(llvm::BasicBlock* currentBlock, llvm::Value* lhs, const GTLCore::Type* lhsType, llvm::Value* rhs, const GTLCore::Type* rhsType);
      static ExpressionResult createSubstractionExpression( llvm::BasicBlock* currentBlock, ExpressionResult lhs, ExpressionResult rhs);
      static llvm::Constant* createSubstractionExpression( llvm::Constant* lhs, const GTLCore::Type* lhsType, llvm::Constant* rhs, const GTLCore::Type* rhsType);
      // Multiplication
      static llvm::Value* createMultiplicationExpression(llvm::BasicBlock* currentBlock, llvm::Value* lhs, const GTLCore::Type* lhsType, llvm::Value* rhs, const GTLCore::Type* rhsType);
      static ExpressionResult createMultiplicationExpression( llvm::BasicBlock* currentBlock, ExpressionResult lhs, ExpressionResult rhs );
      static llvm::Constant* createMultiplicationExpression( llvm::Constant* lhs, const GTLCore::Type* lhsType, llvm::Constant* rhs, const GTLCore::Type* rhsType);
      // Division
      static llvm::Value* createDivisionExpression(llvm::BasicBlock* currentBlock, llvm::Value* lhs, const GTLCore::Type* lhsType, llvm::Value* rhs, const GTLCore::Type* rhsType);
      static llvm::Constant* createDivisionExpression( llvm::Constant* lhs, const GTLCore::Type* lhsType, llvm::Constant* rhs, const GTLCore::Type* rhsType);
      static ExpressionResult createDivisionExpression( llvm::BasicBlock* currentBlock, ExpressionResult lhs, ExpressionResult rhs);
      // Modulo
      llvm::Value* createModuloExpression(llvm::BasicBlock* currentBlock, llvm::Value* lhs, const GTLCore::Type* lhsType, llvm::Value* rhs, const GTLCore::Type* rhsType);
      llvm::Constant* createModuloExpression( llvm::Constant* lhs, const GTLCore::Type* lhsType, llvm::Constant* rhs, const GTLCore::Type* rhsType);
      ExpressionResult createModuloExpression( llvm::BasicBlock* currentBlock, ExpressionResult lhs, const GTLCore::Type* lhsType, ExpressionResult rhs, const GTLCore::Type* rhsType);
    public: // Shift expressions
      llvm::Value* createRightShiftExpression(llvm::BasicBlock* currentBlock, llvm::Value* lhs, const GTLCore::Type* lhsType, llvm::Value* rhs, const GTLCore::Type* rhsType);
      ExpressionResult createRightShiftExpression( llvm::BasicBlock* currentBlock, ExpressionResult lhs,  ExpressionResult rhs);
      llvm::Constant* createRightShiftExpression( llvm::Constant* lhs, const GTLCore::Type* lhsType, llvm::Constant* rhs, const GTLCore::Type* rhsType);
      llvm::Value* createLeftShiftExpression(llvm::BasicBlock* currentBlock, llvm::Value* lhs, const GTLCore::Type* lhsType, llvm::Value* rhs, const GTLCore::Type* rhsType);
      ExpressionResult createLeftShiftExpression( llvm::BasicBlock* currentBlock, ExpressionResult lhs, ExpressionResult rhs);
      llvm::Constant* createLeftShiftExpression( llvm::Constant* lhs, const GTLCore::Type* lhsType, llvm::Constant* rhs, const GTLCore::Type* rhsType);
    public: // Comparison Expressions
      static llvm::Value* createEqualExpression(llvm::BasicBlock* currentBlock, llvm::Value* lhs, const GTLCore::Type* lhsType, llvm::Value* rhs, const GTLCore::Type* rhsType);
      static ExpressionResult createEqualExpression( llvm::BasicBlock* currentBlock, ExpressionResult lhs, const GTLCore::Type* lhsType, ExpressionResult rhs, const GTLCore::Type* rhsType);
      static llvm::Constant* createEqualExpression( llvm::Constant* lhs, const GTLCore::Type* lhsType, llvm::Constant* rhs, const GTLCore::Type* rhsType);
      //
      static llvm::Value* createDifferentExpression(llvm::BasicBlock* currentBlock, llvm::Value* lhs, const GTLCore::Type* lhsType, llvm::Value* rhs, const GTLCore::Type* rhsType);
      static ExpressionResult createDifferentExpression( llvm::BasicBlock* currentBlock, ExpressionResult lhs, const GTLCore::Type* lhsType, ExpressionResult rhs, const GTLCore::Type* rhsType);
      static llvm::Constant* createDifferentExpression( llvm::Constant* lhs, const GTLCore::Type* lhsType, llvm::Constant* rhs, const GTLCore::Type* rhsType);
      //
      static llvm::Value* createStrictInferiorExpression(llvm::BasicBlock* currentBlock, llvm::Value* lhs, const GTLCore::Type* lhsType, llvm::Value* rhs, const GTLCore::Type* rhsType);
      static ExpressionResult createStrictInferiorExpression( llvm::BasicBlock* currentBlock, ExpressionResult lhs, const GTLCore::Type* lhsType, ExpressionResult rhs, const GTLCore::Type* rhsType);
      static llvm::Constant* createStrictInferiorExpression( llvm::Constant* lhs, const GTLCore::Type* lhsType, llvm::Constant* rhs, const GTLCore::Type* rhsType);
      // createInferiorExpression
      static llvm::Value* createInferiorOrEqualExpression(llvm::BasicBlock* currentBlock, llvm::Value* lhs, const GTLCore::Type* lhsType, llvm::Value* rhs, const GTLCore::Type* rhsType);
      static ExpressionResult createInferiorOrEqualExpression( llvm::BasicBlock* currentBlock, ExpressionResult lhs, const GTLCore::Type* lhsType, ExpressionResult rhs, const GTLCore::Type* rhsType);
      static llvm::Constant* createInferiorOrEqualExpression( llvm::Constant* lhs, const GTLCore::Type* lhsType, llvm::Constant* rhs, const GTLCore::Type* rhsType);
      //
      static llvm::Value* createStrictSupperiorExpression(llvm::BasicBlock* currentBlock, llvm::Value* lhs, const GTLCore::Type* lhsType, llvm::Value* rhs, const GTLCore::Type* rhsType);
      static ExpressionResult createStrictSupperiorExpression( llvm::BasicBlock* currentBlock, ExpressionResult lhs, const GTLCore::Type* lhsType, ExpressionResult rhs, const GTLCore::Type* rhsType);
      static llvm::Constant* createStrictSupperiorExpression( llvm::Constant* lhs, const GTLCore::Type* lhsType, llvm::Constant* rhs, const GTLCore::Type* rhsType);
      //
      static llvm::Value* createSupperiorOrEqualExpression(llvm::BasicBlock* currentBlock, llvm::Value* lhs, const GTLCore::Type* lhsType, llvm::Value* rhs, const GTLCore::Type* rhsType);
      static ExpressionResult createSupperiorOrEqualExpression( llvm::BasicBlock* currentBlock, ExpressionResult lhs, const GTLCore::Type* lhsType, ExpressionResult rhs, const GTLCore::Type* rhsType);
      static llvm::Constant* createSupperiorOrEqualExpression( llvm::Constant* lhs, const GTLCore::Type* lhsType, llvm::Constant* rhs, const GTLCore::Type* rhsType);
    public: // Unary expression
      // Minus
      ExpressionResult createMinusExpression( llvm::BasicBlock* currentBlock, ExpressionResult rhs, const GTLCore::Type* rhsType);
      llvm::Value* createMinusExpression(llvm::BasicBlock* currentBlock, llvm::Value* rhs, const GTLCore::Type* rhsType);
      llvm::Constant* createMinusExpression( llvm::Constant* rhs, const GTLCore::Type* rhsType);
      // Not
      ExpressionResult createNotExpression( llvm::BasicBlock* currentBlock, ExpressionResult rhs, const GTLCore::Type* rhsType);
      llvm::Value* createNotExpression(llvm::BasicBlock* currentBlock, llvm::Value* rhs, const GTLCore::Type* rhsType);
      llvm::Constant* createNotExpression( llvm::Constant* rhs, const GTLCore::Type* rhsType);
      // Tilde
      ExpressionResult createTildeExpression( llvm::BasicBlock* currentBlock, ExpressionResult rhs, const GTLCore::Type* rhsType);
      llvm::Value* createTildeExpression(llvm::BasicBlock* currentBlock, llvm::Value* rhs, const GTLCore::Type* rhsType);
      llvm::Constant* createTildeExpression( llvm::Constant* rhs, const GTLCore::Type* rhsType);
      // Increment
      static void createIncrementExpression( LLVMBackend::GenerationContext& gc, llvm::BasicBlock* currentBlock, GTLCore::VariableNG* var);
      static void createIncrementExpression( llvm::BasicBlock* currentBlock, llvm::Value* pointer);
      void createDecrementExpression( LLVMBackend::GenerationContext& gc, llvm::BasicBlock* currentBlock, GTLCore::VariableNG* var);
      void createDecrementExpression( LLVMBackend::GenerationContext& gc, llvm::BasicBlock* currentBlock, llvm::Value* pointer);
      
    public: // Statements
      /**
       * Create an if statement.
       * @param before the basic block before the if statement
       * @param test the test
       * @param firstAction the block with the first action in case if is true
       * @param lastAction the block with the last action in case if is true
       * @param after the basic block after the if statement
       */
      static void createIfStatement( llvm::BasicBlock* before, llvm::Value* test, const GTLCore::Type* testType, llvm::BasicBlock* firstAction, llvm::BasicBlock* lastAction, llvm::BasicBlock* after);
      static void createIfElseStatement( llvm::BasicBlock* before, llvm::Value* test, const GTLCore::Type* testType, llvm::BasicBlock* firstAction, llvm::BasicBlock* lastAction, llvm::BasicBlock* firstElseAction, llvm::BasicBlock* lastElseAction, llvm::BasicBlock* after);
      /**
       * Create a for statement.
       * @param before the basic block before the for statement
       * @param test the basic block which hold the computation of the test
       * @param testResult the value with the result of the test
       * @param update the basic block which hold the computation of the update
       * @param firstAction the block with the first action of the loop
       * @param lastAction the block with the last action of the loop
       * @param after the basic block after the for statement
       */
      static void createForStatement(llvm::BasicBlock* before, llvm::BasicBlock* beginTest, llvm::BasicBlock* endTest, llvm::Value* testResult, const GTLCore::Type* testType, llvm::BasicBlock* update, llvm::BasicBlock* firstAction, llvm::BasicBlock* lastAction,  llvm::BasicBlock* after);
      
      /**
       * Create an iteration for statement.
       * This is a stament that looks like :
       * @code
       * for(int variable = 0; variable \< maxValue ; ++variable )
       * {
       *   // firstAction
       *   // lastAction
       * }
       * @endcode
       * 
       * @param before the basic block before the for statement
       * @param variable the variable to increment
       * @param maxValue the (maximum value + 1) reached by the variable
       * @param firstAction the block with the first action of the loop
       * @param lastAction the block with the last action of the loop
       * @return the exit block of the loop
       */
      static llvm::BasicBlock* createIterationForStatement( LLVMBackend::GenerationContext&, llvm::BasicBlock* before, GTLCore::VariableNG* variable, llvm::Value* maxValue, const GTLCore::Type* maxValueType, llvm::BasicBlock* firstAction, llvm::BasicBlock* lastAction);
      
      void createWhileStatement( llvm::BasicBlock* before, llvm::BasicBlock* test, llvm::Value* testResult, const GTLCore::Type* testType, llvm::BasicBlock* firstAction, llvm::BasicBlock* lastAction,  llvm::BasicBlock* after);
    public:
      /**
       * @param _valuePtr pointer to the value
       */
      static llvm::BasicBlock* createClampExpression( LLVMBackend::GenerationContext& _context, llvm::BasicBlock* _current, llvm::Value* _valuePtr, const GTLCore::Type* _type, llvm::Value* _min, llvm::Value* _max);
      /**
       * Clamp a value
       */
      static llvm::Value* clampValue( LLVMBackend::GenerationContext& _context, LLVMBackend::ExpressionGenerationContext& _expressionGenerationContext, llvm::Value* _value, llvm::Value* _min, llvm::Value* _max);
    public:
      /**
       * Set the count field of a Structure/Array to a given value.
       */
      static void setCountFieldOf( llvm::BasicBlock* currentBlock, llvm::Value* ptr, llvm::Value* value );
      /**
       * @return the count field of a Structure/Array
       */
      static llvm::Value* getCountFieldOf( llvm::BasicBlock* currentBlock, llvm::Value* ptr );
      /**
       * Increment the count field.
       */
      static llvm::Value* incrementCountFieldOf( llvm::BasicBlock* currentBlock, llvm::Value* ptr, llvm::Value* _increment );
    private:
      static llvm::Value* countFieldPointer( llvm::BasicBlock* currentBlock, llvm::Value* ptr );
    public:
      /**
       * @param _currentBlock the current basic block
       * @param _pointer a pointer to the array structure
       * @param _index the index of the value in the array
       * @return a pointer on the value of an array
       */
      static llvm::Value* accessArrayValueClamp( LLVMBackend::GenerationContext& generationContext, LLVMBackend::ExpressionGenerationContext& _expressionGenerationContext, llvm::Value* _pointer, llvm::Value* _index );
      static llvm::Value* accessArrayValueNoClamp( llvm::BasicBlock* _currentBlock, llvm::Value* _pointer, llvm::Value* _index );
      static llvm::Value* accessArraySizePointer( llvm::BasicBlock* _currentBlock, llvm::Value* _pointer );
      static llvm::Value* accessArraySize( llvm::BasicBlock* _currentBlock, llvm::Value* _pointer );
    public:
      ExpressionResult callFunction( LLVMBackend::GenerationContext& _gc, LLVMBackend::ExpressionGenerationContext& egc, const GTLCore::Function* _function, const std::list<GTLCore::AST::Expression*>& m_arguments );
    public:
      /**
       * This function allocate memory using the @ref MemoryManager (or malloc depending on compilation options).
       * For safety reason, only use @ref freeMemory to free memory allocated using @ref allocateMemory.
       * For thread safety, only allocate/free a given memory block from the same thread, however the memory can
       * be used from different threads, given that the usual threading precautions are taken.
       *
       * @param _type the type of the object to allocate
       * @param _a the number of elements to allocate
       * @return a pointer to the allocated memory
       */
      static llvm::Value* allocateMemory( LLVMBackend::GenerationContext& _gc, llvm::Type* _type, llvm::Value* _size, llvm::BasicBlock* _bb);
      /**
       * This function free memory using the @ref MemoryManager (or malloc depending on compilation options).
       * For safety reason, only use @ref freeMemory to free memory allocated using @ref allocateMemory.
       * For thread safety, only allocate/free a given memory block from the same thread, however the memory can
       * be used from different threads, given that the usual threading precautions are taken.
       *
       * @param _ptr to the memory
       */
      static void freeMemory( LLVMBackend::GenerationContext& _gc, llvm::Value* _ptr, llvm::BasicBlock* _bb);
    public:
      static llvm::BasicBlock* callProgressReportNextRow( LLVMBackend::GenerationContext& _gc, llvm::Value* _a, llvm::BasicBlock* _bb);
      static llvm::BasicBlock* callProgressReportNextPixel( LLVMBackend::GenerationContext& _gc, llvm::Value* _a, llvm::BasicBlock* _bb);
    private:
      static llvm::Value* createComparisonExpression(llvm::BasicBlock* currentBlock, llvm::Value* lhs, const GTLCore::Type* lhsType, llvm::Value* rhs, const GTLCore::Type* rhsType, unsigned int unsignedIntegerPred, unsigned int signedIntegerPred, unsigned int floatPred);
      static ExpressionResult createComparisonExpression( llvm::BasicBlock* currentBlock, ExpressionResult lhs, const GTLCore::Type* lhsType, ExpressionResult rhs, const GTLCore::Type* rhsType, unsigned int unsignedIntegerPred, unsigned int signedIntegerPred, unsigned int floatPred);
      static llvm::Constant* createComparisonExpression( llvm::Constant* lhs, const GTLCore::Type* lhsType, llvm::Constant* rhs, const GTLCore::Type* rhsType, unsigned int unsignedIntegerPred, unsigned int signedIntegerPred, unsigned int floatPred);
    private:
      struct Private;
      Private* const d;
  };
}

#endif
