///////////////////////////////////////////////////////////////////////////////
//
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#include <core/Core.h>
#include <base/utilities/Color.h>
#include <core/scene/SceneNode.h>
#include <core/scene/animation/controller/StandardControllers.h>
#include <core/scene/animation/controller/LookAtController.h>
#include <core/undo/UndoManager.h>
#include <core/data/DataSetManager.h>
#include <core/data/ObjectLoadStream.h>
#include <core/data/ObjectSaveStream.h>
#include <core/reference/CloneHelper.h>
#include <core/scene/GroupNode.h>
#include <core/scene/SelectionSet.h>

namespace Core {

IMPLEMENT_ABSTRACT_PLUGIN_CLASS(SceneNode, RefTarget)
DEFINE_FLAGS_REFERENCE_FIELD(SceneNode, TransformationController, "Transform", PROPERTY_FIELD_ALWAYS_DEEP_COPY, _transformation);
DEFINE_FLAGS_REFERENCE_FIELD(SceneNode, SceneNode, "TargetNode", PROPERTY_FIELD_ALWAYS_CLONE, _targetNode);
DEFINE_FLAGS_VECTOR_REFERENCE_FIELD(SceneNode, SceneNode, "Children", PROPERTY_FIELD_ALWAYS_CLONE, _children);
DEFINE_PROPERTY_FIELD(SceneNode, "NodeName", _nodeName);
DEFINE_PROPERTY_FIELD(SceneNode, "DisplayColor", _displayColor);
SET_PROPERTY_FIELD_LABEL(SceneNode, _transformation, "Transformation")
SET_PROPERTY_FIELD_LABEL(SceneNode, _targetNode, "Target")
SET_PROPERTY_FIELD_LABEL(SceneNode, _children, "Children")
SET_PROPERTY_FIELD_LABEL(SceneNode, _nodeName, "Name")
SET_PROPERTY_FIELD_LABEL(SceneNode, _displayColor, "Display color")

/******************************************************************************
* Default constructor.
******************************************************************************/
SceneNode::SceneNode(bool isLoading) : RefTarget(isLoading), _parentNode(NULL), worldTransform(IDENTITY),
	worldTransformValidity(TimeNever), worldBBTime(TimeNegativeInfinity), _displayColor(0,0,0),
	_flags(SCENENODE_NOFLAGS)
{
	INIT_PROPERTY_FIELD(SceneNode, _transformation);
	INIT_PROPERTY_FIELD(SceneNode, _targetNode);
	INIT_PROPERTY_FIELD(SceneNode, _children);
	INIT_PROPERTY_FIELD(SceneNode, _nodeName);
	INIT_PROPERTY_FIELD(SceneNode, _displayColor);

	if(!isLoading) {
		// Generate random color for node.
		static int seed = 1;
		srand(seed++);
		_displayColor = Color::fromHSV((FloatType)rand() / (FloatType)RAND_MAX, 1, 1);
		// Create a transformation controller for the node.
		_transformation = CONTROLLER_MANAGER.createDefaultController<TransformationController>();
	}
}

/******************************************************************************
* Returns this node's world transformation matrix.
* This matrix contains the transformation of the parent node.
******************************************************************************/
const AffineTransformation& SceneNode::getWorldTransform(TimeTicks time, TimeInterval& validityInterval)
{
	if(!worldTransformValidity.contains(time)) {
		worldTransformValidity.setInfinite();
		worldTransform = IDENTITY;
		// Get parent node's tm.
		if(parentNode() != NULL && !parentNode()->isRootNode()) {
			worldTransform = worldTransform * parentNode()->getWorldTransform(time, worldTransformValidity);
		}
		// apply own tm
		if(transformationController())
			transformationController()->applyValue(time, worldTransform, worldTransformValidity);
	}
	validityInterval.intersect(worldTransformValidity);
	return worldTransform;
}

/******************************************************************************
* Returns this node's local transformation matrix.
* This matrix  does not contain the ObjectTransform of this node and
* does not contain the transformation of the parent node.
******************************************************************************/
AffineTransformation SceneNode::getLocalTransform(TimeTicks time, TimeInterval& validityInterval)
{
	AffineTransformation result = IDENTITY;
	if(transformationController())
		transformationController()->applyValue(time, result, validityInterval);
	return result;
}

/******************************************************************************
* This method marks the world transformation cache as invalid,
* so it will be rebuilt during the next call to GetWorldTransform().
******************************************************************************/
void SceneNode::invalidateWorldTransformation()
{
	worldTransformValidity.setEmpty();
	worldBBTime = TimeNegativeInfinity;
	Q_FOREACH(SceneNode* child, children())
		child->invalidateWorldTransformation();
}

/******************************************************************************
* Deletes this node from the scene. This will also delete all child nodes.
******************************************************************************/
void SceneNode::deleteNode()
{
	// Delete target too.
	SceneNode::SmartPtr tn = targetNode();
	if(tn) {
		// Clear reference first to prevent infinite recursion.
        _targetNode = NULL;
		tn->deleteNode();
	}

	// Delete all child nodes recursively.
	Q_FOREACH(SceneNode* child, children())
		child->deleteNode();

	OVITO_ASSERT(children().empty());

	// Delete node itself.
	autoDeleteObject();
}

/******************************************************************************
* Binds this scene node to a target node and creates a look at controller
* that lets this scene node look at the target. The target will automatically
* be deleted if this scene node is deleted and vice versa.
* Returns the newly created LookAtController assigned as rotation controller for this node.
******************************************************************************/
LookAtController* SceneNode::bindToTarget(SceneNode* targetNode)
{
	this->_targetNode = targetNode;

	// Let this node look at the target.
	PRSTransformationController* prs = dynamic_object_cast<PRSTransformationController>(transformationController());
	if(prs != NULL) {
		if(targetNode != NULL) {
			CHECK_OBJECT_POINTER(targetNode);

			// Create a look at controller.
			LookAtController::SmartPtr lookAtCtrl = dynamic_object_cast<LookAtController>(prs->rotationController());
			if(!lookAtCtrl)
				lookAtCtrl = new LookAtController();
			lookAtCtrl->setTargetNode(targetNode);

			// Assign it as rotation sub-controller.
			prs->setRotationController(lookAtCtrl);

			return lookAtCtrl.get();
		}
		else {
			// Reset to default rotation controller.
			prs->setRotationController(CONTROLLER_MANAGER.createDefaultController<RotationController>());
		}
	}

	return NULL;
}

/******************************************************************************
* From RefMaker.
******************************************************************************/
bool SceneNode::onRefTargetMessage(RefTarget* source, RefTargetMessage* msg)
{
	if(msg->type() == REFTARGET_CHANGED) {
		if(source == transformationController()) {
			// TM has changed -> rebuild world tm cache.
			invalidateWorldTransformation();
		}
		// The bounding box might have changed if the object has changed.
		worldBBTime = TimeNegativeInfinity;
	}
	else if(msg->type() == REFTARGET_DELETED && source == targetNode()) {
		// Target node has been deleted -> delete this node too.
		if(!UNDO_MANAGER.isUndoingOrRedoing())
			deleteNode();
	}
	return RefTarget::onRefTargetMessage(source, msg);
}

/******************************************************************************
* From RefMaker.
******************************************************************************/
void SceneNode::onRefTargetInserted(const PropertyFieldDescriptor& field, RefTarget* newTarget, int listIndex)
{
	if(field == PROPERTY_FIELD_DESCRIPTOR(SceneNode, _children)) {
		// A new child node has been added.
		SceneNode* child = static_object_cast<SceneNode>(newTarget);
		CHECK_OBJECT_POINTER(child);
		OVITO_ASSERT(child->parentNode() == NULL);
		child->_parentNode = this;
		// Invalidate cached world bounding box of this parent node.
		worldBBTime = TimeNegativeInfinity;
	}
	RefTarget::onRefTargetInserted(field, newTarget, listIndex);
}

/******************************************************************************
* From RefMaker.
******************************************************************************/
void SceneNode::onRefTargetRemoved(const PropertyFieldDescriptor& field, RefTarget* oldTarget, int listIndex)
{
	if(field == PROPERTY_FIELD_DESCRIPTOR(SceneNode, _children)) {
		// A child node has been removed.
		SceneNode* child = static_object_cast<SceneNode>(oldTarget);
		OVITO_ASSERT(child->parentNode() == this);
		child->_parentNode = NULL;
		// Invalidate cached world bounding box of this parent node.
		worldBBTime = TimeNegativeInfinity;
	}
	RefTarget::onRefTargetRemoved(field, oldTarget, listIndex);
}

/******************************************************************************
* Adds a child scene node to this node.
******************************************************************************/
void SceneNode::addChild(SceneNode* newChild)
{
	CHECK_OBJECT_POINTER(newChild);

	// Check whether it is already a child of this parent.
	if(newChild->parentNode() == this) {
		OVITO_ASSERT(children().contains(newChild));
		return;
	}

	// Remove new child from old parent node first.
	if(newChild->parentNode() != NULL)
		newChild->parentNode()->removeChild(newChild);
	OVITO_ASSERT(newChild->parentNode() == NULL);

	// Insert into children array of this parent.
	_children.push_back(newChild);
	// This parent should be automatically filled into the child's parent pointer.
	OVITO_ASSERT(newChild->parentNode() == this);

	// Adjust transformation to preserve world position.
	TimeInterval iv(TimeForever);
	const AffineTransformation& newParentTM = getWorldTransform(ANIM_MANAGER.time(), iv);
	if(newParentTM != IDENTITY)
		newChild->transformationController()->changeParent(ANIM_MANAGER.time(), IDENTITY, newParentTM, newChild);
	newChild->invalidateWorldTransformation();
}

/******************************************************************************
* Removes a child node from this parent node.
******************************************************************************/
void SceneNode::removeChild(SceneNode* child)
{
	CHECK_OBJECT_POINTER(child);
	OVITO_ASSERT_MSG(child->parentNode() == this, "SceneNode::removeChild()", "The given node is not a child of this parent node.");

	int index = _children.indexOf(child);
	OVITO_ASSERT(index != -1);

	// Remove child node from array.
	_children.remove(index);
	OVITO_ASSERT(_children.contains(child) == false);
	OVITO_ASSERT(child->parentNode() == NULL);

	// Update child node.
	TimeInterval iv(TimeForever);
	AffineTransformation oldParentTM = getWorldTransform(ANIM_MANAGER.time(), iv);
	if(oldParentTM != IDENTITY)
		child->transformationController()->changeParent(ANIM_MANAGER.time(), oldParentTM, IDENTITY, child);
	child->invalidateWorldTransformation();
}

/******************************************************************************
* Returns the child of this node with the given index.
******************************************************************************/
SceneNode* SceneNode::childNode(int index) const
{
	OVITO_ASSERT_MSG(index >= 0 && index < childCount(), "SceneNode::childNode()", "Node index out of range.");
	OVITO_ASSERT(_children[index]->parentNode() == this);
	return _children[index];
}

/******************************************************************************
* Returns the bounding box of the scene node in world coordinates.
*    time - The time at which the bounding box should be returned.
******************************************************************************/
const Box3& SceneNode::worldBoundingBox(TimeTicks time)
{
    if(worldBBTime == time)
		return worldBB;

	worldBBTime = time;
	TimeInterval iv(TimeForever);
	AffineTransformation tm = getWorldTransform(time, iv);
	worldBB = localBoundingBox(time).transformed(tm);

	Q_FOREACH(SceneNode* child, children())
		worldBB.addBox(child->worldBoundingBox(time));

	return worldBB;
}


/******************************************************************************
* Returns true if this node is currently selected.
******************************************************************************/
bool SceneNode::isSelected() const
{
	OVITO_ASSERT_MSG(this->isReferencedBy(DATASET_MANAGER.currentSet()), "SceneNode::isSelected()", "This method may only be used for scene nodes that are part of the current dataset.");

	if(DATASET_MANAGER.currentSelection()->contains(const_cast<SceneNode*>(this)))
		return true;

	GroupNode* gn = closedParentGroup();
	if(gn == NULL) return false;
	return gn->isSelected();
}

/******************************************************************************
* Controls if this node is currently selected.
******************************************************************************/
void SceneNode::setSelected(bool selected)
{
	OVITO_ASSERT_MSG(this->isReferencedBy(DATASET_MANAGER.currentSet()), "SceneNode::setSelected()", "This method may only be used for scene nodes that are part of the current dataset.");

	if(selected)
		DATASET_MANAGER.currentSet()->selection()->add(this);
	else
		DATASET_MANAGER.currentSet()->selection()->remove(this);
}

/******************************************************************************
* Gets the upper most (closed) group node this node is part of or
* NULL if this node is not part of a closed group node.
******************************************************************************/
GroupNode* SceneNode::closedParentGroup() const
{
	const SceneNode* n = this;
	const GroupNode* gn = NULL;
	while((n = n->parentNode()) != NULL) {
		if(n->isGroupNode() && !static_object_cast<GroupNode>(n)->isGroupOpen()) {
            gn = static_object_cast<GroupNode>(n);
		}
	}
	return const_cast<GroupNode*>(gn);
}

/******************************************************************************
* Saves the class' contents to the given stream.
******************************************************************************/
void SceneNode::saveToStream(ObjectSaveStream& stream)
{
	RefTarget::saveToStream(stream);

	stream.beginChunk(0x001);
	stream.writeEnum(_flags);
	stream.endChunk();
}

/******************************************************************************
* Loads the class' contents from the given stream.
******************************************************************************/
void SceneNode::loadFromStream(ObjectLoadStream& stream)
{
	RefTarget::loadFromStream(stream);

	stream.expectChunk(0x001);
	stream.readEnum(_flags);
	stream.closeChunk();
	Q_FOREACH(SceneNode* child, children())
		child->_parentNode = this;
}

/******************************************************************************
* Creates a copy of this object.
******************************************************************************/
RefTarget::SmartPtr SceneNode::clone(bool deepCopy, CloneHelper& cloneHelper)
{
	// Let the base class create an instance of this class.
	SceneNode::SmartPtr clone = static_object_cast<SceneNode>(RefTarget::clone(deepCopy, cloneHelper));

	// Copt the node flags.
	clone->_flags = this->_flags;

	// Clone target too.
	if(clone->targetNode()) {
		OVITO_ASSERT(targetNode() != NULL);

		// Insert cloned target into scene.
		if(!clone->targetNode()->parentNode()) {
			CHECK_OBJECT_POINTER(targetNode()->parentNode());
			targetNode()->parentNode()->addChild(clone->targetNode());
		}

		// Set new target for lookat controller.
		clone->bindToTarget(clone->targetNode());
	}

	return clone;
}

};
