// Copyright (C) 1999-2015
// Smithsonian Astrophysical Observatory, Cambridge, MA, USA
// For conditions of distribution and use, see copyright notice in "copyright"

#include <tk.h>

#include "polygon.h"
#include "fitsimage.h"

Polygon::Polygon(Base* p, const Vector& ctr,
		 const Vector& b)
  : Marker(p, ctr, 0)
{
  angle = 0;
  strcpy(type_, "polygon");

  Vector bb = b;
  vertex.append(new Vertex(-bb[0],-bb[1]));
  vertex.append(new Vertex( bb[0],-bb[1]));
  vertex.append(new Vertex( bb[0], bb[1]));
  vertex.append(new Vertex(-bb[0], bb[1]));

  updateBBox();
}

Polygon::Polygon(Base* p, const Vector& ctr,
		 const Vector& b,
		 const char* clr, int* dsh,
		 int wth, const char* fnt, const char* txt,
		 unsigned short prop, const char* cmt,
		 const List<Tag>& tg, const List<CallBack>& cb)
  : Marker(p, ctr, 0, clr, dsh, wth, fnt, txt, prop, cmt, tg, cb)
{
  angle = 0;
  strcpy(type_, "polygon");

  Vector bb = b;
  vertex.append(new Vertex(-bb[0],-bb[1]));
  vertex.append(new Vertex( bb[0],-bb[1]));
  vertex.append(new Vertex( bb[0], bb[1]));
  vertex.append(new Vertex(-bb[0], bb[1]));

  updateBBox();
}

Polygon::Polygon(Base* p, const List<Vertex>& v, 
		 const char* clr, int* dsh,
		 int wth, const char* fnt, const char* txt,
		 unsigned short prop, const char* cmt,
		 const List<Tag>& tg, const List<CallBack>& cb)
  : Marker(p, Vector(0,0), 0, clr, dsh, wth, fnt, txt, prop, cmt, tg, cb)
{
  // Vertex list is in ref coords
  angle = 0;
  strcpy(type_, "polygon");
  vertex = v;

  // check to see if the first and last node are the same
  if (vertex.head()->vector[0] == vertex.tail()->vector[0] &&
      vertex.head()->vector[1] == vertex.tail()->vector[1])
    delete vertex.pop();

  // find center
  center = Vector(0,0);
  vertex.head();
  do
    center += vertex.current()->vector;
  while (vertex.next());
  center /= vertex.count();

  // vertices are relative
  vertex.head();
  do
    vertex.current()->vector *= Translate(-center) * FlipY(); // no rotation
  while (vertex.next());

  updateBBox();
}

Polygon::Polygon(const Polygon& a) : Marker(a)
{
  vertex = a.vertex;
}

void Polygon::renderX(Drawable drawable, Coord::InternalSystem sys, RenderMode mode)
{
  GC lgc = renderXGC(mode);

  vertex.head();
  Vector v1;
  Vector v2 = fwdMap(vertex.current()->vector,sys);
  int done = 0;

  do {
    if (!vertex.next()) {
      done = 1;
      vertex.head();
    }
    v1 = v2;
    v2 = fwdMap(vertex.current()->vector,sys);
    XDrawLine(display, drawable, lgc, v1[0], v1[1], v2[0], v2[1]);
  } while (!done);
}

void Polygon::renderPS(int mode)
{
  renderPSGC(mode);

  vertex.head();
  int first = 1;
  do {
    ostringstream str;
    Vector v =  fwdMap(vertex.current()->vector,Coord::CANVAS);
    if (first) {
      str << "newpath " << endl
	  << v.TkCanvasPs(parent->canvas) << " moveto" << endl << ends;
      first = 0;
    }
    else
      str << v.TkCanvasPs(parent->canvas) << " lineto" << endl << ends;

    Tcl_AppendResult(parent->interp, str.str().c_str(), NULL);
  } while (vertex.next());

  ostringstream str;
  str << "closepath stroke" << endl << ends;
  Tcl_AppendResult(parent->interp, str.str().c_str(), NULL);
}

#ifdef MAC_OSX_TK
void Polygon::renderMACOSX()
{
  renderMACOSXGC();

  vertex.head();
  Vector v1;
  Vector v2 = fwdMap(vertex.current()->vector,Coord::CANVAS);
  int done = 0;

  do {
    if (!vertex.next()) {
      done = 1;
      vertex.head();
    }
    v1 = v2;
    v2 = fwdMap(vertex.current()->vector,Coord::CANVAS);
    macosxDrawLine(v1,v2);
  } while (!done);
}
#endif

#ifdef __WIN32
void Polygon::renderWIN32()
{
  renderWIN32GC();

  vertex.head();
  Vector v1;
  Vector v2 =  fwdMap(vertex.current()->vector,Coord::CANVAS);
  int done = 0;

  do {
    if (!vertex.next()) {
      done = 1;
      vertex.head();
    }
    v1 = v2;
    v2 = fwdMap(vertex.current()->vector,Coord::CANVAS);
    win32DrawLine(v1,v2);
  } while (!done);
}
#endif

// Support

void Polygon::updateHandles()
{
  // generate handles
  numHandle = 4 + vertex.count();
  if (handle)
    delete [] handle;
  handle = new Vector[numHandle];

  // the first four are our control handles
  BBox bb;
  vertex.head();
  do
    bb.bound(vertex.current()->vector);
  while (vertex.next());

  Vector zz = parent->zoom();
  float r = 10/zz.length();
  bb.expand(r); // give us more room

  handle[0] = fwdMap(bb.ll,Coord::CANVAS);
  handle[1] = fwdMap(bb.lr(),Coord::CANVAS);
  handle[2] = fwdMap(bb.ur,Coord::CANVAS);
  handle[3] = fwdMap(bb.ul(),Coord::CANVAS);

  // and the rest are vertices
  int i=4;
  vertex.head();
  do
    handle[i++] = fwdMap(vertex.current()->vector,Coord::CANVAS);
  while (vertex.next());
}

void Polygon::updateCoords(const Matrix& mx)
{
  Scale s(mx);
  vertex.head();
  do
    vertex.current()->vector *= s;
  while (vertex.next());
  
  Marker::updateCoords(mx);
}

void Polygon::edit(const Vector& v, int h)
{
  if (h < 5) {
    Vector s1 = v * bckMatrix();
    Vector s2 = bckMap(handle[h-1],Coord::CANVAS);

    if (s1[0] != 0 && s1[1] != 0 && s2[0] != 0 && s2[1] != 0) {
      double a = fabs(s1[0]/s2[0]);
      double b = fabs(s1[1]/s2[1]);
      double s = a > b ? a : b;

      vertex.head();
      do
	vertex.current()->vector *= Scale(s);
      while (vertex.next());
    }

    updateBBox();
    doCallBack(CallBack::EDITCB);
  }
  else {
    moveVertex(v,h);

    updateBBox();
    doCallBack(CallBack::EDITCB);
    doCallBack(CallBack::MOVECB); // center can change
  }
}

void Polygon::rotate(const Vector& v, int h)
{
  if (h < 5)
    Marker::rotate(v,h);
  else {
    // we need to check this here, because we are really rotating
    if (canEdit()) { 
      moveVertex(v,h);

      updateBBox();
      doCallBack(CallBack::EDITCB);
      doCallBack(CallBack::MOVECB); // center can change
    }
  }
}

void Polygon::reset(const Vector& b)
{
  angle = 0;
  vertex.deleteAll();

  Vector bb = b;
  vertex.append(new Vertex(-bb[0],-bb[1]));
  vertex.append(new Vertex( bb[0],-bb[1]));
  vertex.append(new Vertex( bb[0], bb[1]));
  vertex.append(new Vertex(-bb[0], bb[1]));

  updateBBox();
}

int Polygon::isInRef(const Vector& v)
{
  /*
    v[0]-- x value of point being tested
    v[1]-- y value of point being tested

    This algorithm is from "An Introduction to Ray Tracing", Academic Press,
    1989, edited by Andrew Glassner, pg 53
    -- a point lies in a polygon if a line is extended from the point to 
    infinite in any direction and the number of intersections with the 
    polygon is odd.
    This is valid for both concave and convex polygons.
    Points on a vertex are considered inside.
    Points on a edge are considered inside.
  */

  int crossings = 0;   // number of crossings

  vertex.head();
  Vector v1;
  Vector v2 = vertex.current()->vector - v;

  int sign = ((v2[1])>=0) ? 1 : -1; // init sign

  // for all edges
  int done = 0;

  do {
    // look at next two vertices
    v1 = v2;
    if (!vertex.next()) {
      done = 1;
      vertex.head();
    }

    v2 = vertex.current()->vector - v;

    int nextSign = (v2[1]>=0) ? 1 : -1; // sign holder for p2

    if (sign != nextSign) {
      if (v1[0]>0 && v2[0]>0)
	crossings++;
      else if (v1[0]>0 || v2[0]>0) {
	if (v1[0]-(v1[1]*(v2[0]-v1[0])/(v2[1]-v1[1])) > 0)
	  crossings++;
      }
      sign = nextSign;
    }
  } while (!done);

  return fmod(float(crossings),float(2)) ? 1 : 0; // if odd, point is inside
}

void Polygon::createVertex(int which, const Vector& v)
{
  // which segment (1 to n)
  // v is in ref coords
  Matrix mm = bckMatrix();

  int seg = which-1;
  if (seg>=0 && seg<vertex.count()) {
    Vertex* n = new Vertex(v * mm);
    vertex.insert(seg,n);

    recalcCenter();

    updateBBox();
    doCallBack(CallBack::EDITCB);
    doCallBack(CallBack::MOVECB); // center can change
  }
}

void Polygon::deleteVertex(int h)
{
  if (h>4) {
    int hh = h-4-1;

    if (vertex.count() > 3) {
      Vertex* v = vertex[hh];
      if (v) {
	vertex.extractNext(v);
	delete v;

	recalcCenter();

	updateBBox();
	doCallBack(CallBack::EDITCB);
	doCallBack(CallBack::MOVECB); // center can change
      }  
    }
  }
}

int Polygon::getSegment(const Vector& v)
{
  // v is in canvas coords
  Matrix mm = fwdMatrix();

  vertex.head();
  Vector v1;
  Vector v2 = vertex.current()->vector * mm;
  int done = 0;

  int ii = 1;
  do {
    v1 = v2;
    if (!vertex.next()) {
      vertex.head();
      done = 1;
    }
    v2 = vertex.current()->vector * mm;

    Vector l1 = parent->mapFromRef(v1,Coord::CANVAS);
    Vector l2 = parent->mapFromRef(v2,Coord::CANVAS);
    double a = (l2-l1).angle();
    Matrix mx = Translate(-l1) * FlipY() * Rotate(-a); 
    Vector end = l2*mx;
    Vector vv = v*mx;
    
    if (vv[0]>0 && vv[0]<end[0] && 
	vv[1]>-parent->markerEpsilon && vv[1]<parent->markerEpsilon)
      return ii;

    ii++;
  } while (!done);

  return 0;
}

void Polygon::moveVertex(const Vector& v, int h)
{
  Matrix mm = bckMatrix();

  if (vertex[h-5])
    vertex.current()->vector = v * mm;

  recalcCenter();
}

void Polygon::recalcCenter()
{
  // recalculate center

  Vector nc;
  vertex.head();
  do
    nc += vertex.current()->vector * Rotate(angle) * FlipY();
  while (vertex.next());
  nc /= vertex.count();

  center += nc;

  // update all vertices

  vertex.head();
  do
    vertex.current()->vector -= nc * FlipY() * Rotate(-angle);
  while (vertex.next());
}

void Polygon::analysis(AnalysisTask mm, int which)
{
  switch (mm) {
  case HISTOGRAM:
    if (!analysisHistogram_ && which) {
      addCallBack(CallBack::MOVECB, analysisHistogramCB_[0], 
		  parent->options->cmdName);
      addCallBack(CallBack::EDITCB, analysisHistogramCB_[0], 
		  parent->options->cmdName);
      addCallBack(CallBack::ROTATECB, analysisHistogramCB_[0], 
		  parent->options->cmdName);
      addCallBack(CallBack::DELETECB, analysisHistogramCB_[1], 
		  parent->options->cmdName);
    }
    if (analysisHistogram_ && !which) {
      deleteCallBack(CallBack::MOVECB, analysisHistogramCB_[0]);
      deleteCallBack(CallBack::EDITCB, analysisHistogramCB_[0]);
      deleteCallBack(CallBack::ROTATECB, analysisHistogramCB_[0]);
      deleteCallBack(CallBack::DELETECB, analysisHistogramCB_[1]);
    }

    analysisHistogram_ = which;
    break;
  case PLOT3D:
    if (!analysisPlot3d_ && which) {
      addCallBack(CallBack::MOVECB, analysisPlot3dCB_[0], 
		  parent->options->cmdName);
      addCallBack(CallBack::EDITCB, analysisPlot3dCB_[0], 
		  parent->options->cmdName);
      addCallBack(CallBack::ROTATECB, analysisPlot3dCB_[0], 
		  parent->options->cmdName);
      addCallBack(CallBack::DELETECB, analysisPlot3dCB_[1], 
		  parent->options->cmdName);
    }
    if (analysisPlot3d_ && !which) {
      deleteCallBack(CallBack::MOVECB, analysisPlot3dCB_[0]);
      deleteCallBack(CallBack::EDITCB, analysisPlot3dCB_[0]);
      deleteCallBack(CallBack::ROTATECB, analysisPlot3dCB_[0]);
      deleteCallBack(CallBack::DELETECB, analysisPlot3dCB_[1]);
    }

    analysisPlot3d_ = which;
    break;
  case STATS:
    if (!analysisStats_ && which) {
      addCallBack(CallBack::MOVECB, analysisStatsCB_[0], 
		  parent->options->cmdName);
      addCallBack(CallBack::EDITCB, analysisStatsCB_[0], 
		  parent->options->cmdName);
      addCallBack(CallBack::ROTATECB, analysisStatsCB_[0], 
		  parent->options->cmdName);
      addCallBack(CallBack::UPDATECB, analysisStatsCB_[0], 
		  parent->options->cmdName);
      addCallBack(CallBack::DELETECB, analysisStatsCB_[1], 
		  parent->options->cmdName);
    }
    if (analysisStats_ && !which) {
      deleteCallBack(CallBack::MOVECB, analysisStatsCB_[0]);
      deleteCallBack(CallBack::EDITCB, analysisStatsCB_[0]);
      deleteCallBack(CallBack::ROTATECB, analysisStatsCB_[0]);
      deleteCallBack(CallBack::UPDATECB, analysisStatsCB_[0]);
      deleteCallBack(CallBack::DELETECB, analysisStatsCB_[1]);
    }

    analysisStats_ = which;
    break;
  default:
    // na
    break;
  }
}

void Polygon::analysisHistogram(char* xname, char* yname, int num)
{
  double* x;
  double* y;
  BBox bb(center);

  Matrix mm = Rotate(angle) * Translate(center);
  vertex.head();
  do
    bb.bound(vertex.current()->vector * mm);
  while (vertex.next());

  parent->markerAnalysisHistogram(this, &x, &y, bb, num);
  analysisXYResult(xname, yname, x, y, num+1);
}

void Polygon::analysisPlot3d(char* xname, char* yname,
			    Coord::CoordSystem sys, 
			    Marker::AnalysisMethod method)
{
  double* x;
  double* y;
  BBox bb(center);

  Matrix mm = Rotate(angle) * Translate(center);
  vertex.head();
  do
    bb.bound(vertex.current()->vector * mm);
  while (vertex.next());

  int num = parent->markerAnalysisPlot3d(this, &x, &y, bb, sys, method);
  analysisXYResult(xname, yname, x, y, num);
}

void Polygon::analysisStats(Coord::CoordSystem sys, Coord::SkyFrame sky)
{
  ostringstream str;
  BBox bb(center);

  Matrix mm = Rotate(angle) * Translate(center);
  vertex.head();
  do
    bb.bound(vertex.current()->vector * mm);
  while (vertex.next());

  parent->markerAnalysisStats(this, str, bb, sys, sky);
  Tcl_AppendResult(parent->interp, str.str().c_str(), NULL);
}

// list

void Polygon::list(ostream& str, Coord::CoordSystem sys, Coord::SkyFrame sky,
		   Coord::SkyFormat format, int conj, int strip)
{
  FitsImage* ptr = parent->findFits(sys,center);
  listPre(str, sys, sky, ptr, strip, 0);

  Matrix mm = fwdMatrix();
  switch (sys) {
  case Coord::IMAGE:
  case Coord::PHYSICAL:
  case Coord::DETECTOR:
  case Coord::AMPLIFIER:
    {
      str << type_ << '(';
      int first=1;
      vertex.head();
      do {
	if (!first)
	  str << ',';
	first=0;

	Vector v = ptr->mapFromRef(vertex.current()->vector*mm,sys);
	str << setprecision(8) << v[0] << ',' << v[1];
      }
      while (vertex.next());
      str << ')';
    }
    break;
  default:
    if (ptr->hasWCS(sys)) {
      if (ptr->hasWCSCel(sys)) {
	switch (format) {
	case Coord::DEGREES:
	  {
	    str << type_ << '(';
	    int first=1;
	    vertex.head();
	    do {
	      if (!first)
		str << ',';
	      first=0;

	      Vector v = ptr->mapFromRef(vertex.current()->vector*mm,sys,sky);
	      str << setprecision(8) << v[0] << ',' << v[1];
	    }
	    while (vertex.next());
	    str << ')';
	  }
	  break;
	case Coord::SEXAGESIMAL:
	  {
	    char buf[64];
	    char ra[16];
	    char dec[16];

	    str << type_ << '(';
	    int first=1;
	    vertex.head();
	    do {
	      if (!first)
		str << ',';
	      first=0;

	      ptr->mapFromRef(vertex.current()->vector*mm,sys,sky,format,buf,64);
	      string x(buf);
	      istringstream wcs(x);
	      wcs >> ra >> dec;
	      str << ra << ',' << dec;
	    }
	    while (vertex.next());
	    str << ')';
	  }
	  break;
	}
      }
      else {
	str << type_ << '(';
	int first=1;
	vertex.head();
	do {
	  if (!first)
	    str << ',';
	  first=0;

	  Vector v = ptr->mapFromRef(vertex.current()->vector*mm,sys);
	  str << setprecision(8) << v[0] << ',' << v[1];
	}
	while (vertex.next());
	str << ')';
      }
    }
  }

  listPost(str, conj, strip);
}

void Polygon::listXML(ostream& str, Coord::CoordSystem sys, Coord::SkyFrame sky, 
		      Coord::SkyFormat format)
{
  FitsImage* ptr = parent->findFits(sys,center);
  Matrix mm = fwdMatrix();
  Vector* vv = new Vector[vertex.count()];

  XMLRowInit();
  XMLRow(XMLSHAPE,type_);

  vertex.head();
  int cnt =0;
  do
    vv[cnt++] =vertex.current()->vector*mm;
  while (vertex.next());
  XMLRowPoint(ptr,sys,sky,format,vv,vertex.count());
  delete [] vv;

  XMLRowProps(ptr,sys);
  XMLRowEnd(str);
}

void Polygon::listCiao(ostream& str, Coord::CoordSystem sys, int strip)
{
  FitsImage* ptr = parent->findFits();
  Matrix mm = fwdMatrix();
  listCiaoPre(str);

  switch (sys) {
  case Coord::IMAGE:
  case Coord::PHYSICAL:
  case Coord::DETECTOR:
  case Coord::AMPLIFIER:
    {
      str << type_ << '(';
      int first=1;
      vertex.head();
      do {
	if (!first)
	  str << ',';
	first=0;

	Vector v = ptr->mapFromRef(vertex.current()->vector * mm,Coord::PHYSICAL);
	str << setprecision(8) << v[0] << ',' << v[1];
      }
      while (vertex.next());
      str << ')';
    }
    break;
  default:
    if (ptr->hasWCSCel(sys)) {
      char buf[64];
      char ra[16];
      char dec[16];

      str << type_ << '(';
      int first=1;
      vertex.head();
      do {
	if (!first)
	  str << ',';
	first=0;

	ptr->mapFromRef(vertex.current()->vector*mm,sys,Coord::FK5,Coord::SEXAGESIMAL,buf,64);
	string x(buf);
	istringstream wcs(x);
	wcs >> ra >> dec;
	str << ra << ',' << dec;
      }
      while (vertex.next());
      str << ')';
    }
  }

  listCiaoPost(str, strip);
}

void Polygon::listPros(ostream& str, Coord::CoordSystem sys, Coord::SkyFrame sky,
		       Coord::SkyFormat format, int strip)
{
  FitsImage* ptr = parent->findFits();
  Matrix mm = fwdMatrix();

  switch (sys) {
  case Coord::IMAGE:
  case Coord::DETECTOR:
  case Coord::AMPLIFIER:
    sys = Coord::IMAGE;
  case Coord::PHYSICAL:
    {
      coord.listProsCoordSystem(str,sys,sky);
      str << "; " << type_;
      vertex.head();
      do {
	Vector v = ptr->mapFromRef(vertex.current()->vector*mm,sys);
	str << setprecision(8) << v;
      }
      while (vertex.next());
    }
    break;
  default:
    if (ptr->hasWCSCel(sys)) {
      coord.listProsCoordSystem(str,sys,sky);
      str << "; " << type_ << ' ';

      switch (format) {
      case Coord::DEGREES:
	{
	  vertex.head();
	  do {
	    Vector v = ptr->mapFromRef(vertex.current()->vector*mm,sys,sky);
	    str << setprecision(8) << v[0] << "d " << v[1] << "d ";
	  }
	  while (vertex.next());
	}
	break;
      case Coord::SEXAGESIMAL:
	{
	  char buf[64];
	  char ra[16];
	  char dec[16];

	  vertex.head();
	  do {
	    ptr->mapFromRef(vertex.current()->vector*mm,sys,sky,format,buf,64);
	string x(buf);
	istringstream wcs(x);
	    wcs >> ra >> dec;
	    if (dec[0]=='+')
	      str << ' ' << ra << ' ' << dec+1;
	    else
	      str << ' ' << ra << ' ' << dec;
	  }
	  while (vertex.next());
	}
	break;
      }
    }
  }

  listProsPost(str, strip);
}

void Polygon::listSAOtng(ostream& str, Coord::CoordSystem sys, Coord::SkyFrame sky,
			 Coord::SkyFormat format, int strip)
{
  FitsImage* ptr = parent->findFits();
  Matrix mm = fwdMatrix();
  listSAOtngPre(str, strip);

  switch (sys) {
  case Coord::IMAGE:
  case Coord::PHYSICAL:
  case Coord::DETECTOR:
  case Coord::AMPLIFIER:
    {
      str << type_ << '(';
      int first=1;
      vertex.head();
      do {
	if (!first)
	  str << ',';
	first=0;

	Vector v = ptr->mapFromRef(vertex.current()->vector*mm,sys);
	str << setprecision(8) << v[0] << ',' << v[1];
      }
      while (vertex.next());
      str << ')';
    }
    break;
  default:
    if (ptr->hasWCSCel(sys)) {
      switch (format) {
      case Coord::DEGREES:
	{
	  str << type_ << '(';
	  int first=1;
	  vertex.head();
	  do {
	    if (!first)
	      str << ',';
	    first=0;

	    Vector v = ptr->mapFromRef(vertex.current()->vector*mm,sys,sky);
	    str << setprecision(8) << v[0] << ',' << v[1];
	  }
	  while (vertex.next());
	  str << ')';
	}
	break;
      case Coord::SEXAGESIMAL:
	{
	  char buf[64];
	  char ra[16];
	  char dec[16];

	  str << type_ << '(';
	  int first=1;
	  vertex.head();
	  do {
	    if (!first)
	      str << ',';
	    first=0;

	    ptr->mapFromRef(vertex.current()->vector*mm,sys,sky,format,buf,64);
	    string x(buf);
	    istringstream wcs(x);
	    wcs >> ra >> dec;
	    str << ra << ',' << dec;
	  }
	  while (vertex.next());
	  str << ')';
	}
	break;
      }

    }
  }

  listSAOtngPost(str, strip);
}

void Polygon::listSAOimage(ostream& str, int strip)
{
  FitsImage* ptr = parent->findFits();
  listSAOimagePre(str);
  Matrix mm = fwdMatrix();

  str << type_ << '(';
  int first=1;
  vertex.head();
  do {
    if (!first)
      str << ',';
    first=0;

    Vector v = ptr->mapFromRef(vertex.current()->vector*mm,Coord::IMAGE);
    str << setprecision(8) << v[0] << ',' << v[1];
  }
  while (vertex.next());

  str << ')';
  listSAOimagePost(str, strip);
}

