#include "SwiftUnsafeTypes.h"
#include "SwiftBasicTypes.h"

#include "Plugins/LanguageRuntime/Swift/SwiftLanguageRuntime.h"
#include "lldb/DataFormatters/TypeSynthetic.h"
#include "lldb/Symbol/CompilerType.h"
#include "lldb/Utility/LLDBLog.h"
#include "lldb/Utility/Log.h"
#include "lldb/ValueObject/ValueObject.h"
#include "lldb/lldb-enumerations.h"
#include "llvm/ADT/StringRef.h"

#include <utility>

using namespace lldb;
using namespace lldb_private;

namespace {

class SwiftUnsafeType {
public:
  enum class UnsafePointerKind {
    eSwiftUnsafePointer,
    eSwiftUnsafeRawPointer,
    eSwiftUnsafeBufferPointer,
    eSwiftUnsafeRawBufferPointer,
  };

  static std::unique_ptr<SwiftUnsafeType> Create(ValueObject &valobj);
  size_t GetCount() const { return m_count; }
  addr_t GetStartAddress() const { return m_start_addr; }
  CompilerType GetElementType() const { return m_elem_type; }
  UnsafePointerKind GetKind() const { return m_kind; }
  bool HasPointee() const {
    return m_count == 1 && m_kind == UnsafePointerKind::eSwiftUnsafePointer;
  }
  virtual lldb::ChildCacheState Update() = 0;

  virtual ~SwiftUnsafeType() = default;

protected:
  SwiftUnsafeType(ValueObject &valobj, UnsafePointerKind kind);
  addr_t GetAddress(llvm::StringRef child_name);
  std::optional<size_t> GetCountValue(llvm::StringRef child_name);
  CompilerType GetArgumentType();

  ValueObject &m_valobj;
  const UnsafePointerKind m_kind;
  size_t m_count;
  addr_t m_start_addr;
  CompilerType m_elem_type;
};

SwiftUnsafeType::SwiftUnsafeType(ValueObject &valobj, UnsafePointerKind kind)
    : m_valobj(*valobj.GetNonSyntheticValue().get()), m_kind(kind) {}

lldb::addr_t SwiftUnsafeType::GetAddress(llvm::StringRef child_name) {
  ConstString name(child_name);
  ValueObjectSP optional_value_sp(m_valobj.GetChildMemberWithName(name, true));
  assert(optional_value_sp && optional_value_sp->GetCompilerType()
                                  .GetTypeName()
                                  .GetStringRef()
                                  .starts_with("Swift.Optional"));

  if (!optional_value_sp || !optional_value_sp->GetSyntheticValue()) {
    LLDB_LOG(GetLog(LLDBLog::DataFormatters),
             "{0}: Couldn't unwrap the 'Swift.Optional' ValueObject child "
             "named {1}.",
             __FUNCTION__, name);
    return false;
  }

  lldb::SyntheticChildrenSP synth_children_sp =
      optional_value_sp->GetSyntheticChildren();
  if (!synth_children_sp)
    return false;
  auto frontend = synth_children_sp->GetFrontEnd(*optional_value_sp);
  if (!frontend)
    return false;
  ValueObjectSP synth_optional_sp = frontend->GetSyntheticValue();

  if (!synth_optional_sp || !synth_optional_sp->GetNumChildren()) {
    LLDB_LOG(GetLog(LLDBLog::DataFormatters),
             "{0}: Couldn't unwrap the 'Swift.UnsafePointer' ValueObject child "
             "named 'some'.",
             __FUNCTION__);
    return false;
  }

  CompilerType type = synth_optional_sp->GetCompilerType();
  ValueObjectSP unsafe_ptr_value_sp = synth_optional_sp;
  // Work around a bug in GetSyntheticValue(), which doesn't return
  // the result of SyntheticFrontend::GetSyntheticValue().
  if (type.GetTypeName().GetStringRef().starts_with("Swift.Optional")) {
    // type = TypeSystemSwiftTypeRef::GetOptionalType(type);

    // if (auto v = synth_optional_sp->GetChildAtIndex(0))
    //   unsafe_ptr_value_sp = v->GetSP();
    // type = unsafe_ptr_value_sp->GetCompilerType();
  }
  assert(type.GetTypeName().GetStringRef().starts_with("Swift.Unsafe"));
  if (!type.IsValid()) {
    LLDB_LOG(GetLog(LLDBLog::DataFormatters),
             "{0}: Couldn't get the compiler type for the "
             "'Swift.UnsafePointer' ValueObject.",
             __FUNCTION__, type.GetTypeName());
    return false;
  }

  auto type_system = type.GetTypeSystem().dyn_cast_or_null<TypeSystemSwift>();
  if (!type_system) {
    LLDB_LOG(GetLog(LLDBLog::DataFormatters),
             "{0}: Couldn't get {1} type system.", __FUNCTION__,
             type.GetTypeName());
    return false;
  }

  CompilerType argument_type =
      type_system->GetGenericArgumentType(type.GetOpaqueQualType(), 0);

  if (argument_type.IsValid())
    m_elem_type = argument_type;

  assert(
      !m_elem_type.GetTypeName().GetStringRef().starts_with("Swift.Optional"));
  assert(!m_elem_type.GetTypeName().GetStringRef().starts_with("Swift.Unsafe"));

  ValueObjectSP pointer_value_sp =
      unsafe_ptr_value_sp->GetChildAtIndex(0, true);
  if (!pointer_value_sp) {
    LLDB_LOG(GetLog(LLDBLog::DataFormatters),
             "{0}: Couldn't unwrap the 'Swift.Int' ValueObject named "
             "'pointerValue'.",
             __FUNCTION__);
    return false;
  }

  return pointer_value_sp->GetValueAsUnsigned(LLDB_INVALID_ADDRESS);
}

std::optional<size_t>
SwiftUnsafeType::GetCountValue(llvm::StringRef child_name) {
  ValueObjectSP count_value_sp(
      m_valobj.GetChildMemberWithName(child_name, true));
  if (!count_value_sp) {
    LLDB_LOG(GetLog(LLDBLog::DataFormatters),
             "{0}: Couldn't find ValueObject child member named '{1}'.",
             __FUNCTION__, child_name);
    return std::nullopt;
  }

  ValueObjectSP value_provided_child_sp;

  // Implement Swift's 'value-providing synthetic children' workaround.
  // Depending on whether the ValueObject type is a primitive or a structure,
  // lldb should prioritize the synthetic value children.
  // If it has no synthetic children then fallback to non synthetic children.
  ValueObjectSP synthetic = count_value_sp->GetSyntheticValue();
  if (synthetic)
    value_provided_child_sp = synthetic->GetChildAtIndex(0, true);
  if (!value_provided_child_sp)
    value_provided_child_sp = count_value_sp->GetChildAtIndex(0, true);

  // If neither child exists, fail.
  if (!value_provided_child_sp) {
    LLDB_LOG(GetLog(LLDBLog::DataFormatters),
             "{0}: Couldn't extract 'value-providing synthetic children' from "
             "ValueObject '{1}'.",
             __FUNCTION__, child_name);
    return std::nullopt;
  }

  size_t count = value_provided_child_sp->GetValueAsUnsigned(UINT64_MAX);

  if (count == UINT64_MAX) {
    LLDB_LOG(GetLog(LLDBLog::DataFormatters),
             "{0}: Couldn't get a valid value for ValueObject '{1}'.",
             __FUNCTION__, child_name);
    return std::nullopt;
  }

  return count;
}

CompilerType SwiftUnsafeType::GetArgumentType() {
  CompilerType type = m_valobj.GetCompilerType();
  if (!type.IsValid()) {
    LLDB_LOG(GetLog(LLDBLog::DataFormatters),
             "{0}: Couldn't get the compiler type for the '{1}' ValueObject.",
             __FUNCTION__, type.GetTypeName());
    return {};
  }

  auto type_system = type.GetTypeSystem().dyn_cast_or_null<TypeSystemSwift>();
  if (!type_system) {
    LLDB_LOG(GetLog(LLDBLog::DataFormatters),
             "{0}: Couldn't get {1} type system.", __FUNCTION__,
             type.GetTypeName());
    return {};
  }

  return type_system->GetGenericArgumentType(type.GetOpaqueQualType(), 0);
}

class SwiftUnsafeBufferPointer final : public SwiftUnsafeType {
public:
  SwiftUnsafeBufferPointer(ValueObject &valobj);
  lldb::ChildCacheState Update() override;
};

SwiftUnsafeBufferPointer::SwiftUnsafeBufferPointer(ValueObject &valobj)
    : SwiftUnsafeType(valobj, UnsafePointerKind::eSwiftUnsafeBufferPointer) {}

lldb::ChildCacheState SwiftUnsafeBufferPointer::Update() {
  if (!m_valobj.GetNumChildren())
    return ChildCacheState::eRefetch;

  // Here is the layout of Swift's Unsafe[Mutable]BufferPointer.
  //
  //  ▿ UnsafeBufferPointer
  //    ▿ _position : Optional<UnsafePointer<Int>>
  //      ▿ some : UnsafePointer<Int>
  //        - pointerValue : Int
  //    - count : Int
  //
  // The structure has 2 children:
  //  1. The buffer `count` child stored as a Swift `Int` type. This entry is a
  // "value-providing synthetic children", so lldb need to access to its
  // children in order to get the  actual value.
  //  2. An Optional UnsafePointer to the buffer start address. To access the
  // pointer address, lldb unfolds every ValueObject child until reaching
  // `pointerValue`.

  if (auto count = GetCountValue("count"))
    m_count = *count;
  else
    return ChildCacheState::eRefetch;

  addr_t start_addr = GetAddress("_position");

  if (!start_addr || start_addr == LLDB_INVALID_ADDRESS) {
    LLDB_LOG(GetLog(LLDBLog::DataFormatters),
             "{0}: Couldn't get a valid address for ValueObject '_position'.",
             __FUNCTION__);
    return ChildCacheState::eRefetch;
  }

  m_start_addr = start_addr;

  return ChildCacheState::eReuse;
}

class SwiftUnsafeRawBufferPointer final : public SwiftUnsafeType {
public:
  SwiftUnsafeRawBufferPointer(ValueObject &valobj);
  lldb::ChildCacheState Update() override;

private:
  addr_t m_end_addr;
};

SwiftUnsafeRawBufferPointer::SwiftUnsafeRawBufferPointer(ValueObject &valobj)
    : SwiftUnsafeType(valobj, UnsafePointerKind::eSwiftUnsafeRawBufferPointer) {
}

lldb::ChildCacheState SwiftUnsafeRawBufferPointer::Update() {
  auto num_or_error = m_valobj.GetNumChildren();
  if (!num_or_error) {
    llvm::consumeError(num_or_error.takeError());
    return ChildCacheState::eRefetch;
  }

  // Here is the layout of Swift's UnsafeRaw[Mutable]BufferPointer.
  // It's a view of the raw bytes of the pointee object. Each byte is viewed as
  // a `UInt8` value independent of the type of values held in that memory.
  //
  // ▿ UnsafeRawBufferPointer
  //   ▿ _position : Optional<UnsafeRawPointer<Int>>
  //     ▿ some : UnsafeRawPointer<Int>
  //       - pointerValue : Int
  //   ▿ _end : Optional<UnsafeRawPointer<Int>>
  //     ▿ some : UnsafeRawPointer<Int>
  //       - pointerValue : Int
  //
  // The structure has 2 Optional UnsafePointers to the buffer's start address
  // and end address. To access the pointer address, lldb unfolds every
  // ValueObject child until reaching `pointerValue`.

  addr_t addr = GetAddress("_position");
  if (!addr || addr == LLDB_INVALID_ADDRESS) {
    LLDB_LOG(GetLog(LLDBLog::DataFormatters),
             "{0}: Couldn't get a valid address for ValueObject '_position'.",
             __FUNCTION__);
    return ChildCacheState::eRefetch;
  }
  m_start_addr = addr;

  addr = GetAddress("_end");
  if (!addr || addr == LLDB_INVALID_ADDRESS) {
    LLDB_LOG(GetLog(LLDBLog::DataFormatters),
             "{0}: Couldn't get a valid address for ValueObject '_end'.",
             __FUNCTION__);
    return ChildCacheState::eRefetch;
  }
  m_end_addr = addr;

  if (!m_elem_type.IsValid()) {
    CompilerType type = m_valobj.GetCompilerType();
    if (!type.IsValid()) {
      LLDB_LOG(GetLog(LLDBLog::DataFormatters),
               "{0}: Couldn't get a valid base compiler type.", __FUNCTION__);
      return ChildCacheState::eRefetch;
    }

    auto type_system = type.GetTypeSystem().dyn_cast_or_null<TypeSystemSwift>();
    if (!type_system) {
      LLDB_LOG(GetLog(LLDBLog::DataFormatters),
               "{0}: Couldn't get {1} type system.", __FUNCTION__,
               type.GetTypeName());
      return ChildCacheState::eRefetch;
    }

    CompilerType compiler_type =
        type_system->GetTypeFromMangledTypename(ConstString("$ss5UInt8VD"));
    if (!compiler_type.IsValid()) {
      LLDB_LOG(GetLog(LLDBLog::DataFormatters),
               "{0}: Couldn't get a valid compiler type for 'Swift.UInt8'.",
               __FUNCTION__);
      return ChildCacheState::eRefetch;
    }

    m_elem_type = compiler_type;
  }

  auto opt_type_size = m_elem_type.GetByteSize(m_valobj.GetTargetSP().get());

  if (!opt_type_size) {
    LLDB_LOG(GetLog(LLDBLog::DataFormatters),
             "{0}: Couldn't get element byte size.", __FUNCTION__);
    return ChildCacheState::eRefetch;
  }
  m_count = (m_end_addr - m_start_addr) / *opt_type_size;

  return ChildCacheState::eReuse;
}

class SwiftUnsafePointer final : public SwiftUnsafeType {
public:
  SwiftUnsafePointer(ValueObject &valobj, UnsafePointerKind kind);
  lldb::ChildCacheState Update() override;
};

SwiftUnsafePointer::SwiftUnsafePointer(ValueObject &valobj,
                                       UnsafePointerKind kind)
    : SwiftUnsafeType(valobj, kind) {}

lldb::ChildCacheState SwiftUnsafePointer::Update() {
  if (!m_valobj.GetNumChildren())
    return ChildCacheState::eRefetch;

  // Here is the layout of Swift's Unsafe[Mutable]Pointer.
  //
  // ▿ Unsafe[Raw]Pointer
  //   ▿ _rawValue : Int
  //       - pointerValue : Int
  //

  CompilerType argument_type = GetArgumentType();
  if (argument_type.IsValid())
    m_elem_type = argument_type;

  assert(
      !m_elem_type.GetTypeName().GetStringRef().starts_with("Swift.Optional"));

  ValueObjectSP pointer_value_sp(m_valobj.GetChildAtIndex(0, true));
  if (!pointer_value_sp) {
    LLDB_LOG(GetLog(LLDBLog::DataFormatters),
             "{0}: Couldn't unwrap the 'Swift.Int' ValueObject named "
             "'pointerValue'.",
             __FUNCTION__);
    return ChildCacheState::eRefetch;
  }

  addr_t addr = pointer_value_sp->GetValueAsUnsigned(LLDB_INVALID_ADDRESS);

  if (!addr || addr == LLDB_INVALID_ADDRESS)
    return ChildCacheState::eRefetch;

  m_start_addr = addr;
  m_count = (m_elem_type.IsValid()) ? 1 : 0;

  return ChildCacheState::eReuse;
}

class SwiftSpan final : public SwiftUnsafeType {
public:
  SwiftSpan(ValueObject &valobj);
  lldb::ChildCacheState Update() override;
};

SwiftSpan::SwiftSpan(ValueObject &valobj)
    : SwiftUnsafeType(valobj, UnsafePointerKind::eSwiftUnsafeRawBufferPointer) {
}

lldb::ChildCacheState SwiftSpan::Update() {
  if (auto count = GetCountValue("_count"))
    m_count = *count;
  else
    return ChildCacheState::eRefetch;

  addr_t start_addr = GetAddress("_pointer");
  if (!start_addr || start_addr == LLDB_INVALID_ADDRESS) {
    LLDB_LOG(GetLog(LLDBLog::DataFormatters),
             "{0}: Couldn't get a valid address for ValueObject '_pointer'.",
             __FUNCTION__);
    return ChildCacheState::eRefetch;
  }
  m_start_addr = start_addr;

  CompilerType argument_type = GetArgumentType();
  if (argument_type.IsValid())
    m_elem_type = argument_type;
  else
    return ChildCacheState::eRefetch;

  return ChildCacheState::eReuse;
}

std::unique_ptr<SwiftUnsafeType> SwiftUnsafeType::Create(ValueObject &valobj) {
  CompilerType type = valobj.GetCompilerType();
  if (!type.IsValid()) {
    LLDB_LOG(GetLog(LLDBLog::DataFormatters),
             "{0}: Couldn't get a valid base compiler type.", __FUNCTION__);
    return nullptr;
  }

  // Resolve the base Typedefed CompilerType.
  while (true) {
    opaque_compiler_type_t qual_type = type.GetOpaqueQualType();
    if (!qual_type) {
      LLDB_LOG(GetLog(LLDBLog::DataFormatters),
               "{0}: Couldn't get {1} opaque compiler type.", __FUNCTION__,
               type.GetTypeName());
      return nullptr;
    }

    auto type_system = type.GetTypeSystem().dyn_cast_or_null<TypeSystemSwift>();
    if (!type_system) {
      LLDB_LOG(GetLog(LLDBLog::DataFormatters),
               "{0}: Couldn't get {1} type system.", __FUNCTION__,
               type.GetTypeName());
      return nullptr;
    }

    if (!type_system->IsTypedefType(qual_type))
      break;

    type = type_system->GetTypedefedType(qual_type);
  }

  if (!type.IsValid()) {
    LLDB_LOG(GetLog(LLDBLog::DataFormatters),
             "{0}: Couldn't resolve a valid base compiler type.", __FUNCTION__);
    return nullptr;
  }

  llvm::StringRef valobj_type_name(type.GetTypeName().GetCString());
  valobj_type_name.consume_front("Swift.");
  bool is_unsafe = valobj_type_name.consume_front("Unsafe");
  valobj_type_name.consume_front("Mutable");

  if (!is_unsafe && valobj_type_name.consume_front("Span"))
    return std::make_unique<SwiftSpan>(valobj);

  bool is_raw = valobj_type_name.consume_front("Raw");
  bool is_buffer_ptr = valobj_type_name.consume_front("Buffer");
  UnsafePointerKind kind =
      static_cast<UnsafePointerKind>(is_buffer_ptr << 1 | is_raw);

  switch (kind) {
  case UnsafePointerKind::eSwiftUnsafePointer:
  case UnsafePointerKind::eSwiftUnsafeRawPointer:
    return std::make_unique<SwiftUnsafePointer>(valobj, kind);
  case UnsafePointerKind::eSwiftUnsafeBufferPointer:
    return std::make_unique<SwiftUnsafeBufferPointer>(valobj);
  case UnsafePointerKind::eSwiftUnsafeRawBufferPointer:
    return std::make_unique<SwiftUnsafeRawBufferPointer>(valobj);
  }
}

} // namespace

bool lldb_private::formatters::swift::UnsafeTypeSummaryProvider(
    ValueObject &valobj, Stream &stream, const TypeSummaryOptions &options) {
  std::unique_ptr<SwiftUnsafeType> unsafe_ptr = SwiftUnsafeType::Create(valobj);

  if (!unsafe_ptr || unsafe_ptr->Update() == ChildCacheState::eRefetch)
    return false;
  size_t count = unsafe_ptr->GetCount();
  addr_t addr = unsafe_ptr->GetStartAddress();

  auto is_buffer_ptr = [](SwiftUnsafeType::UnsafePointerKind kind) {
    return static_cast<int>(kind) & (1 << 1);
  };

  // Hide the number of children if not BufferPointer type.
  if (!is_buffer_ptr(unsafe_ptr->GetKind()))
    stream.Printf("0x%" PRIx64, addr);
  else
    stream.Printf("%zu %s (0x%" PRIx64 ")", count,
                  (count == 1) ? "value" : "values", addr);

  return true;
}

static std::vector<ValueObjectSP>
ExtractChildrenFromSwiftPointerValueObject(ValueObjectSP valobj_sp,
                                           SwiftUnsafeType &unsafe_ptr) {
  if (!valobj_sp)
    return {};

  lldb::ProcessSP process_sp(valobj_sp->GetProcessSP());

  if (!process_sp)
    return {};

  const addr_t start_addr = unsafe_ptr.GetStartAddress();
  const size_t num_children = unsafe_ptr.GetCount();
  const CompilerType element_type = unsafe_ptr.GetElementType();

  // Performance optimization. Give up if this is a forward-declared
  // Clang type.  In that case GetByteStride will attempt to fall back
  // to SwiftASTContext, which is generally correct because it could
  // be a type imported from an expression, but in this data formatter
  // the potential cost of triggering SwiftASTContext is not the right
  // trade-off.
  auto tss = element_type.GetTypeSystem().dyn_cast_or_null<TypeSystemSwift>();
  CompilerType clang_type;
  if (tss && tss->IsImportedType(element_type.GetOpaqueQualType(), &clang_type))
    if (!clang_type.IsCompleteType())
      return {};    
  
  auto stride = element_type.GetByteStride(process_sp.get());
  if (!stride)
    return {};

  auto element_stride = *stride;
  size_t buffer_size = num_children * element_stride;
  if (buffer_size > 512 * 1024 * 1024) {
    LLDB_LOG(GetLog(LLDBLog::DataFormatters),
             "Suspiciously large object: num_children={0}, stride={1}",
             num_children, element_stride);
    return {};
  }

  std::unique_ptr<lldb_private::WritableDataBuffer> buffer_up(
      new DataBufferHeap(buffer_size, 0));
  Status error;
  size_t read_bytes = process_sp->ReadMemory(start_addr, buffer_up->GetBytes(),
                                             buffer_size, error);

  if (!read_bytes || error.Fail())
    return {};

  auto ptr_size = process_sp->GetAddressByteSize();
  auto order = process_sp->GetByteOrder();
  DataExtractor buffer_data(buffer_up->GetBytes(), buffer_up->GetByteSize(),
                            order, ptr_size);

  std::vector<ValueObjectSP> children;
  auto exe_ctx_ref = valobj_sp->GetExecutionContextRef();
  // UnsafePointer/UnsafeMutablePointer have a `pointee` property.
  if (unsafe_ptr.HasPointee()) {
    DataExtractor data(buffer_data, 0, element_stride);
    children.push_back(ValueObject::CreateValueObjectFromData(
        "pointee", data, exe_ctx_ref, element_type));
    return children;
  }

  for (size_t i = 0; i < num_children; i++) {
    StreamString idx_name;
    idx_name.Printf("[%zu]", i);
    DataExtractor data(buffer_data, i * element_stride, element_stride);
    children.push_back(ValueObject::CreateValueObjectFromData(
        idx_name.GetString(), data, exe_ctx_ref, element_type));
  }

  return children;
}

namespace lldb_private {
namespace formatters {
namespace swift {
std::vector<ValueObjectSP>
ExtractChildrenFromSwiftPointerValueObject(ValueObjectSP valobj_sp) {
  auto unsafe_ptr = ::SwiftUnsafeType::Create(*valobj_sp.get());
  unsafe_ptr->Update();
  return ::ExtractChildrenFromSwiftPointerValueObject(valobj_sp,
                                                      *unsafe_ptr.get());
}

class UnsafeTypeSyntheticFrontEnd : public SwiftBasicTypeSyntheticFrontEnd {
public:
  UnsafeTypeSyntheticFrontEnd(lldb::ValueObjectSP valobj_sp);

  llvm::Expected<uint32_t> CalculateNumChildren() override;
  lldb::ValueObjectSP GetChildAtIndex(uint32_t idx) override;
  lldb::ChildCacheState Update() override;
  bool MightHaveChildren() override;
  size_t GetIndexOfChildWithName(ConstString name) override;

private:
  std::unique_ptr<SwiftUnsafeType> m_unsafe_ptr;
  WritableDataBufferSP m_buffer_sp;
  std::vector<ValueObjectSP> m_children;
};
} // namespace swift
} // namespace formatters
} // namespace lldb_private

lldb_private::formatters::swift::UnsafeTypeSyntheticFrontEnd::
    UnsafeTypeSyntheticFrontEnd(lldb::ValueObjectSP valobj_sp)
    : SwiftBasicTypeSyntheticFrontEnd(*valobj_sp.get()) {

  ProcessSP process_sp = valobj_sp->GetProcessSP();
  if (!process_sp)
    return;


  m_unsafe_ptr = ::SwiftUnsafeType::Create(*valobj_sp.get());

  lldbassert(m_unsafe_ptr != nullptr && "Could not create Swift Unsafe Type");

  if (valobj_sp)
    Update();
}

llvm::Expected<uint32_t> lldb_private::formatters::swift::
    UnsafeTypeSyntheticFrontEnd::CalculateNumChildren() {
  return m_unsafe_ptr->GetCount();
}

lldb::ValueObjectSP
lldb_private::formatters::swift::UnsafeTypeSyntheticFrontEnd::GetChildAtIndex(
    uint32_t idx) {
  const uint32_t num_children = CalculateNumChildrenIgnoringErrors();

  if (idx >= num_children || idx >= m_children.size())
    return lldb::ValueObjectSP();

  return m_children[idx];
}

lldb::ChildCacheState
lldb_private::formatters::swift::UnsafeTypeSyntheticFrontEnd::Update() {
  m_children.clear();
  ValueObjectSP valobj_sp = m_backend.GetSP();
  if (!valobj_sp)
    return ChildCacheState::eRefetch;

  if (!m_unsafe_ptr)
    return ChildCacheState::eRefetch;
  if (m_unsafe_ptr->Update() == ChildCacheState::eRefetch)
    return ChildCacheState::eRefetch;

  const uint32_t num_children = CalculateNumChildrenIgnoringErrors();
  m_children = ::ExtractChildrenFromSwiftPointerValueObject(valobj_sp,
                                                          *m_unsafe_ptr.get());
  return m_children.size() == num_children ? ChildCacheState::eReuse
                                           : ChildCacheState::eRefetch;
}

bool lldb_private::formatters::swift::UnsafeTypeSyntheticFrontEnd::
    MightHaveChildren() {
  return m_unsafe_ptr && m_unsafe_ptr->GetCount();
}

size_t lldb_private::formatters::swift::UnsafeTypeSyntheticFrontEnd::
    GetIndexOfChildWithName(ConstString name) {
  if (m_unsafe_ptr && m_unsafe_ptr->HasPointee() && name == "pointee")
    return 0;
  return UINT32_MAX;
}

SyntheticChildrenFrontEnd *
lldb_private::formatters::swift::UnsafeTypeSyntheticFrontEndCreator(
    CXXSyntheticChildren *, lldb::ValueObjectSP valobj_sp) {
  if (!valobj_sp)
    return nullptr;
  return (new UnsafeTypeSyntheticFrontEnd(valobj_sp));
}
