/*******************************************************************************
 * Part of "Intel(R) Active Management Technology (Intel(R) AMT)
 *                   User Notification Service (UNS)"
 *
 * Copyright (c) 2007 Intel Corp.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions, and the following disclaimer,
 *    without modification.
 * 2. Redistributions in binary form must reproduce at minimum a disclaimer
 *    substantially similar to the "NO WARRANTY" disclaimer below
 *    ("Disclaimer") and any redistribution must be conditioned upon
 *    including a substantially similar Disclaimer requirement for further
 *    binary redistribution.
 * 3. Neither the names of the above-listed copyright holders nor the names
 *    of any contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * Alternatively, this software may be distributed under the terms of the
 * GNU General Public License ("GPL") version 2 as published by the Free
 * Software Foundation.
 *
 * NO WARRANTY
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTIBILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 * HOLDERS OR CONTRIBUTORS BE LIABLE FOR SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING
 * IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGES.
 *******************************************************************************/

//----------------------------------------------------------------------------
//
//  File:       Options.cpp
//
//  contents:   This file contains the implementation of a generic class for command line options
//              parsing.
//
//----------------------------------------------------------------------------

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif
#include "Options.h"

#include <string.h>
#include <set>

#ifndef _WIN32
#include <strings.h>
#define stricmp strcasecmp
#endif

using namespace std;

const char Options::DEFAULT_SWITCH_CHAR = '-';

bool Options::StrLess::operator()(const char* left, const char* right) const
{
	return (stricmp(left, right) < 0);
}

Options::Format::Format(const Options::FormatNode rawFormat[],
						int rawFormatSize, std::size_t numMandatoryParams,
						char switchChar)
{
	for (int i=0; i<rawFormatSize; i++)
		addOption(rawFormat[i].option, rawFormat[i].bHasParam, rawFormat[i].bShouldRead);
	m_numMandatoryParams = numMandatoryParams;
	m_switchChar = switchChar;
}

Options::Format::Format()
{
	m_numMandatoryParams = 0;
	m_switchChar = Options::DEFAULT_SWITCH_CHAR;
}

void Options::Format::addOption(const char* option, bool bHasParam, bool bShouldRead)
{
	m_formatMap[option] = bHasParam;
	m_inputMap[option] = bShouldRead;
}

bool Options::Format::optionExists(const char *option) const
{
	return (m_formatMap.find(option) != m_formatMap.end());
}

bool Options::Format::optionHasParam(const char* option) const
{
	FormatMap::const_iterator it = m_formatMap.find(option);
	if (it == m_formatMap.end())
	{
		//Option not found - so it doesn't have a parameter.
		return false;
	}

	return it->second;
}

bool Options::Format::optionShouldRead(const char* option) const
{
	FormatMap::const_iterator it = m_inputMap.find(option);
	if (it == m_inputMap.end())
	{
		//Option not found - so it doesn't have a parameter.
		return false;
	}
	return it->second;
}

bool Options::optionExists(const char* option) const
{
	return m_optionsMap.find(option) != m_optionsMap.end();
}


bool Options::setError(Error err) const
{
	m_error = err;
	return (m_error == ERROR_NONE);
}

/*
This parsing function goes over the command line, looking for words that begin with
the switch char ('-' by default). When the switch char is found, we check if this
is a valid option. If it is, and it has a parameter, the next word in the command
line is assumed to be its value.
*/
bool Options::parse(int argc, const char* argv[], const Format &format)
{
	const char* optionName = NULL;
	const char* optionVal = NULL;
	MandatoryParamsVector mandatoryParams;

	if ((argv == NULL) || (argc < 0))
		return setError(ERROR_ARGC_ARGV);

	//We start from the second parameter, since the first is the program name.
	for (int i=1; i<argc; i++)
	{
		if (argv[i][0] == format.switchChar())
		{
			optionName = argv[i] + 1;
			if (optionExists(optionName))
			{
				/*The option was already specified in this command line.*/
				return setError(ERROR_OPTION_ALREADY_SPECIFIED);
			}

			if (!format.optionExists(optionName))
			{
				/*This option was not specified in the format.*/
				return setError(ERROR_INVALID_OPTION);
			}

			if (format.optionHasParam(optionName))
			{
				//If this option has a parameter, read it and skip it.
				if (++i >= argc)
				{
					//This option should have a parameter.
					return setError(ERROR_NO_OPTION_PARAM);
				}
				optionVal = argv[i];
			}
			else 
			{
			    if (format.optionShouldRead(optionName))
			    {
#define MAX_USER_INPUT 32
			        char *option = new char[MAX_USER_INPUT];
			        char *newline = NULL;
				while (newline == NULL)
			        {
				    printf("Please provide value for option '%s':\n", optionName);
				    fgets(option, MAX_USER_INPUT, stdin);
				
				    if (option == NULL)
				    {
				        printf("Invalid value. Try again.\n");
					continue;
				    }

				    newline = strchr(option, '\n');
				    if (newline == NULL)
				    { 
					printf("Option string too long. Upper bound is %d characters.\n", MAX_USER_INPUT-1);
					// clear buffer until EOL
					while (newline == NULL)
					{
					    fgets(option, MAX_USER_INPUT, stdin);
					    if (option == NULL)
					    {
					        break;
					    }
					    newline = strchr(option, '\n');
					}; // while
					// set marker to continue outer while loop
					newline = NULL;
				    }
						else
						{
							*newline = '\0'; 
						}
						
				}
				optionVal = option;
#undef MAX_USER_INPUT
			    }
			    else
			    {
				optionVal = "";
			    }
			}
			m_optionsMap[optionName] = optionVal;
		}
		else //	if (argv[i][0] == format.switchChar())
		{
			//This is assumed to be a mandatory parameter
			if (mandatoryParams.size() == format.numMandatoryParams())
			{
				/*Too many 'mandatory' parameters found - this is an error.
				Note that if 0 mandatory params was specified (the default),
				only options (i.e. words that start with the switch char)
				can appear in the command line.*/
				return setError(ERROR_TOO_MANY_PARAMS);
			} //if
			mandatoryParams.push_back(argv[i]);
		} //else
	} //for

	if (mandatoryParams.size() < format.numMandatoryParams())
	{
		/*Not enough mandatory parameters found - error.*/
		return setError(ERROR_NOT_ENOUGH_PARAMS);
	}

	/*We only set the m_MandatoryParams member in the end so that we'll have a
	  transaction, and we won't mess this member up in case of errors.*/
	m_MandatoryParams = mandatoryParams;
	m_parsed = true;
	return setError(ERROR_NONE);
}

bool Options::parse(int argc, const char* argv[],
					const FormatNode rawFormat[], int rawFormatSize,
					std::size_t numMandatoryParams, char switchChar)
{
	return parse(argc, argv, Format(rawFormat, rawFormatSize,
		numMandatoryParams, switchChar));
}

Options::Options(int argc, const char* argv[],
				 const Format &format)
{
	reset();
	parse(argc, argv, format);
}

Options::Options(int argc, const char* argv[], const FormatNode rawFormat[],
				 int rawFormatSize, std::size_t numMandatoryParams, char switchChar)
{
	reset();
	parse(argc, argv, rawFormat, rawFormatSize, numMandatoryParams, switchChar);
}

Options::Options()
{
	reset();
}

void Options::reset()
{
	setError(ERROR_NONE);
	m_parsed = false;
}

const char* Options::getOption(const char* option) const
{
	OptionsMap::const_iterator it = m_optionsMap.find(option);
	if (it == m_optionsMap.end())
	{
		setError(ERROR_INVALID_OPTION);
		return NULL;
	}
	else
	{
		setError(ERROR_NONE);
		return it->second;
	}
}

std::size_t Options::numMandatoryParams() const
{
    return m_MandatoryParams.size();
}

const char* Options::getMandatoryParam(unsigned int index) const
{
	if (index > m_MandatoryParams.size())
	{
		setError(ERROR_NOT_ENOUGH_PARAMS);
		return NULL;
	}
	else
	{
		setError(ERROR_NONE);
		return m_MandatoryParams[index];
	}
}

const char* Options::errorStr() const
{
    return errorToStr(error());
}

const char* Options::errorToStr(Error err)
{
    switch (err)
	{
		case ERROR_NONE:
			return "No error";
		case ERROR_ARGC_ARGV:
			return "Internal error in argc or argv";
		case ERROR_OPTION_ALREADY_SPECIFIED:
			return "An option was specified more than once";
		case ERROR_INVALID_OPTION:
			return "Invalid option";
		case ERROR_NO_OPTION_PARAM:
			return "An option was not given a parameter";
		case ERROR_TOO_MANY_PARAMS:
			return "Too many non-option parameters";
		case ERROR_NOT_ENOUGH_PARAMS:
			return "Not enough non-option parameters";
		default:
			return "Unknown";
	}
}
