/* 	session.c:
 *		core <------> stream channels
 *
 *	part of the martian_modem
 *
 *	description:
 *		an implementation of reader/writer part of modem
 *
 *	note:
 *		both terms reader/writer related to modem and its core
 *	
 *
 *	current implementation:
 *
 *		core state checked. if there's something to read from core
 *		out-stream is added to watchlist, if there's a room to transmit
 *		in-stream is added to watchlist.
 *
 *		Next wait on watched streams, pause if none to watch. 
 *
 *		This wait either succeeds resulting in streams io operations
 *		or might be interrupted by core initiated timer tick. 
 *		In the latter case core state is rescanned.
 *
 *
 * 	supposed multithreaded implementation for SMP:
 *
 *		supposed to be 2 threaded thing
 *
 *		reader: listens to virtual uart, reads from core internal 
 *		   buffers and writes to socket,
 *
 *		writer: listens to pty if anything to be written it will pass 
 *		   the data in chunks to the virtual uart
 *
 *		for reader core virtual interrupt will pulse data here event
 *
 *		access to core should be serialized for MT implementation
 *
 *	created 6.11.2005
 *
 *	Author: A. Chentsov
 * 	Copying: LGPL
 *
 */

/*** pselect implies */
#define _GNU_SOURCE

#include <stdio.h>
#include <unistd.h>		/*** close, unlink */
#include <syscall.h>		/*** for gettid() definition */
#include <linux/unistd.h>

/*** fd sets, pselect */
#include <sys/select.h>

/*** MIN */
#include <sys/param.h>

/*** __u-types */
#include <asm/types.h>		

/*** sigaction */
#include <signal.h>

/*** errno, EINTR */
#include <errno.h>

/*** nanosleep */
#include <time.h>
#include <sys/time.h>

#include <string.h> 

#define MSIGVI (SIGRTMIN + 1)

#define CORE_USLICE		(10 * 1000)
#define CORE_SLICE		(1000 * CORE_USLICE)
#define PULSE_CHECK		1

#include "mport.h"
extern mport_t mport;

#define PREFIX "session"
#include "log.h"
#include "sysdep.h"

#include "watch.h"
#include "core.h"
#include <sys/ioctl.h>
#include "../martian.h"

/***** core's symbols *****/


//extern void timertick_function (int num);

/**************************/

/***** pin symbols *****/
extern int pin_recv (int pin, char *buf, int length);
extern int pin_send (int pin, char *buf, int length);
extern int pin_get_send_size (int pin);
extern int pin_check_hangup (int pin);

/**************************/
struct _actionwatch pulse_latency;
struct _actionwatch core_virtual_watch;
struct _actionwatch write_watch, read_watch;
struct _actionwatch read_dsp_watch;

struct _stream {
	int bytes, calls;
};

static struct {
	time_t start;
	
	/* statistics */
	int pulsesAll, pulsesDone;
	struct _stream up, down;
	struct _stream vi_up, vi_down;

	int pin;
	fd_set vi_readers, vi_writers;
	struct timeval immediately;

	unsigned in_vi	: 1;
	unsigned in_pulse	: 1;
	unsigned closed : 1;
} session;

/*** timer processing ***/

/*int pulse_core;
int no_irq = 0;
int pending = 0;*/


static void inline do_pulse_port (mport_t *port) {
	session.in_pulse = 1;

	session.pulsesDone++;
	if (Debugged (Timings))
		watch_start (&core_virtual_watch);

	port->do_pulse (port);

	if (Debugged (Timings)) {
		watch_stop (&core_virtual_watch);
		if ((watch_nsec (core_virtual_watch, last)) > 100000000) 
			logdebugadd (2, "d");
	}

	session.in_pulse = 0;
}

typedef enum event_t {
	Fault, Pulse, RW, Closed, CarrierLost
} event_t;

#define CHECKDOPULSE(port)				\
	do if ((port)->pulse) {				\
		/*(port)->do_pulse (port);*/		\
		do_pulse_port (port);			\
		logdebugadd (Mechanism, "p");			\
		if (session.closed) 			\
			return Closed;			\
		if ((port)->carrier == CLost) 		\
			return CarrierLost;		\
		/* update sets after timer */		\
		return Pulse;				\
	} while (0)

static inline event_t wait_events (int fd, fd_set *preaders, fd_set *pwriters)
{
	struct timespec timeout;
	timeout.tv_sec 	= 0;
	timeout.tv_nsec	= PULSE_CHECK * CORE_SLICE;

	
	CHECKDOPULSE (&mport);
	/*
	if (mport.pulse) {
		do_pulse_port (&mport);
		mport.do_pulse (&mport);
		if (mport.carrier == CLost) 
			return CarrierLost;
	}
	*/
	/* there's very little chance that pulse come between this point
	 * and the point when we enter the kernel in syscall
	 * for this reason there's a timeout latency comes into play
	 */

	if (preaders || pwriters)
	{
		int num;

		logdebugadd (5, "s");
		while ( (num = pselect (fd + 1, preaders, pwriters, NULL, & timeout, NULL)) <= 0) {
			if (num == 0)  
				logdebugadd (5, "0"); 

			if ((num == -1 && errno == EINTR) ||
			    (num == 0)) {
				/* check & pulse core */
				CHECKDOPULSE (&mport);
				if (!mport.pulse && num != 0) 
					LOGDEBUG (2, "interrupted spuriously\n");
				
				
				if (preaders) FD_SET (fd, preaders);
				if (pwriters) FD_SET (fd, pwriters);
			} 

			// that's bad thing with io
			else if (errno == EIO) {
				LOGERR ("pty io error; closing session\n");
				return Fault;
			} 
			else {
				LOGSYSERR (Error, "select");
				LOGERR ("closing session\n");
				return Fault;
			} 
		}
		logdebugadd (5, "e");
		return RW;
	} 
	/* no descriptors to watch */
	/* no core room to write or carrier lost */
	else {
		/* assert */
		ASSERT (mport.carrier == COk, return CarrierLost);

		while (1) {
			int result = nanosleep (&timeout, NULL);
			logdebugadd (Mechanism, "n");
			if (result == 0 || (result == -1 && errno == EINTR)) {
				CHECKDOPULSE (&mport);
			}
			else {
				LOGSYSERR (Error, "nanosleep");
				LOGERR ("closing session\n");
				return Fault;
			}
		}
	}
}


/*** pty <-- core ***/
static inline int downstream (int fd)
{
	char buf[0x400];
	int bytes = MIN (mport.available, sizeof buf);

	mport_read (&mport, buf, bytes);

	/* write to stream; what to do if it block? 
	 * well waterline do the trick 
	 * there's always enough room 
	 */

	char *ptr = buf;
	int left = bytes;
	do {
		int sent = pin_send (fd, ptr, left); 
		if (sent == -1) {
			if (errno == EINTR) {
				if (mport.carrier == CLost)
					return -1 /*out*/;
				continue;
			}
			/* otherwise */
			LOGSYSERR (Debug, "write");
			session.closed = 1;
			return -1; 
		}
		else if (sent == 0) {
			LOGDEBUG (2, "downstream: written 0\n");
			session.closed = 1;
			return -1; 
		}

		LOGQDEBUG (3, " <-- %d\n", sent);
		LOGQDEBUG (4, " <-- %*.*s\n", sent, sent, ptr);
		/*LOGQDEBUG (3, "pty <-- core: %d\n", sent);
		LOGQDEBUG (4, "pty <-- core: %*.*s\n", sent, sent, ptr); */
		/* logdebugadd (3, "<%d<", sent);
		logdebugadd (4, "%*.*s<", sent, sent, ptr); */

		/* statistics */
		session.down.bytes += sent;
		session.down.calls++;
		if (session.in_vi) {
			session.vi_down.bytes += sent;
			session.vi_down.calls++;
		}

		left -= sent;
		if (left) 
			LOGDEBUG (2, "downstream: sent less than supplied. repeat\n");

		ptr += sent;

	} while (left); 

	/* perhaps check the pulse
	*/
	return bytes;
}

/*** pty --> core ***/
static inline int upstream (int fd)
{
	char buf[0x100];
	// we prefer to jump
	int received, bytes;

	bytes = MIN (mport.room, sizeof buf);

	while ( (received = pin_recv (	
				fd, 
				buf, 
				bytes)) == -1) {
				
		if (errno == EINTR) 
			continue;
		else if (errno == EIO) {
			/* client closed */
			session.closed = 1;
			LOGDEBUG (2, "reader: EIO\n");
			return -1;
		}
		else {
			/* error, quit */
			LOGSYSERR (Debug, "upstream:");
			return -1;
		}
	}
	if (received == 0) {
		/* client closed */
		session.closed = 1;
		LOGDEBUG (2, "reader closed\n");
		return -1;
	}

	/* precond: received <= bytes */
	bytes = mport_write (&mport, buf, received);

	if (received != bytes) {
		LOGWARN ("upstream: written to core %d < %d (supplied)\n", bytes, received);
	}

	/* statistics */
	session.up.bytes += bytes;
	session.up.calls++;

	
	/*LOGQDEBUG (3, "pty --> core: %d\n", bytes);
	LOGQDEBUG (4, "pty --> core: %*.*s\n", bytes, bytes, buf);*/
	LOGQDEBUG (3, " --> %d\n", bytes);
	LOGQDEBUG (4, " --> %*.*s\n", bytes, bytes, buf);
	/*
	logdebugadd (3, ">%d>", bytes);
	logdebugadd (4, "%*.*s", bytes, bytes, buf); */

	return bytes;
}

static inline void 
session_init (int pin, fd_set *preaders, fd_set *pwriters)
{
	FD_ZERO (preaders);
	FD_ZERO (pwriters);

	FD_ZERO (&session.vi_readers);
	FD_ZERO (&session.vi_writers);

	session.pin = pin;

	session.immediately.tv_sec  =
	session.immediately.tv_usec = 0;

	session.in_vi	= 
	session.closed	= 0;

	session.in_pulse	= 0;

	/* reset statistics */
	session.pulsesAll 	=
	session.pulsesDone	= 0;

	session.down.bytes	= 0;
	session.down.calls	= 0;
	
	session.up 	= session.down;
	session.vi_down	= session.down;
	session.vi_up	= session.down;

	if (Debugged (Timings)) {
		watch_init (&pulse_latency);
		watch_init (&core_virtual_watch);
		watch_init (&write_watch); watch_init (&read_watch);
		watch_init (&read_dsp_watch);
		watch_set_thresh (&core_virtual_watch, 100 * 1000000);
	}
}

/*** two funcs to decide what should be watched ***/
static inline fd_set *checkset_upstream (int fd, fd_set *preaders)
{
	if (mport.carrier == CLost) 
		return NULL;

	if (mport.ready) {
		FD_SET (fd, preaders);
		return preaders;
	}
	else 
		return NULL;
}

static inline fd_set *checkset_downstream (int fd, fd_set *pwriters)
{
	if (mport.has_data) {
		FD_SET (fd, pwriters);
		return pwriters;
	}
	else 
		return NULL;
}

void virtual_rs_interrupt (int a) {
	fd_set *pwriters;

	session.in_vi = 1;

	/* notify port */
	mport_vi (&mport);

	if (session.pin < 0)
		return ;

	/* watching only for writing - downstream */
	pwriters = checkset_downstream (session.pin, &session.vi_writers);

	/*** pick events ***/
	if (select (session.pin + 1, NULL, pwriters, NULL, &session.immediately) <= 0) 
		goto l_quit;

	/*** write ***/
	if ( pwriters && FD_ISSET (session.pin, pwriters) ) {
		if (Debugged(Timings)) watch_start (&write_watch);
		if (downstream (session.pin) == -1) 
			goto l_quit /*out*/;
		if (Debugged(Timings)) {
			watch_stop (&write_watch);
			unsigned ms = watch_nsec (write_watch, last) / 1000000;
			if (ms > 5) logdebugadd (Timings, "w%dms", ms);
		}
	}

	if (mport.carrier == CLost) {
		LOGDEBUG (Note, "virtual int: carrier lost\n");
		goto l_quit /*out*/;
	}

	/* Usually CTS also checked. 
	 * This bits are not set in UART layer of core. 
	 */

	/* perhaps gather some statistics or monitor modem state */
l_quit:
	session.in_vi = 0;
}


#define SHOW_TIMINGS(w, str) \
	LOGQ (Debug, str " min = %.2f %s, max = %.2f %s; %.2f %s average\n", \
		watch_format(&w,Min), 	\
		watch_unit (w,Min),	\
		watch_format(&w,Max), 	\
		watch_unit (w,Max),	\
		watch_format (&w, Avg),	\
		watch_unit (w, Avg)	\
		);


/*** diagnostics, chronometry */
static void dump_stats ()
{
	if (config.debuglevel < Stats)
		return;

	LOGQ (Debug, "\n");
	LOGQ (Debug, "\t\t*** session stats ***\n");
	LOGQ (Debug, "\n");

	{
		char timestr[25], *ptr;
		unsigned duration, seconds;
		struct timeval now;
		gettimeofday (&now, NULL);

		ptr = timestr;

		duration = now.tv_sec - session.start;
		seconds = duration % 60;
		duration = duration / 60;
		if (duration) {
			int minutes = duration % 60;
			duration /= 60;

			if (duration) 
				ptr += sprintf (ptr, "%d h ", duration);

			ptr += sprintf (ptr, "%d min ", minutes);
		}
		ptr += sprintf (ptr, "%d sec", seconds);
		LOGQ (Debug, "duration %s\n", timestr);
	}

	// LOGQ (Debug, "Pulses, effective/total :   %d/%d\n", session.pulsesDone, session.pulsesAll); 
	LOGQ (Debug, "Pulses, effective/total :   %d/%d\n", session.pulsesDone, mport.pulsesAll); 
	LOGQ (Debug, ">>> out: %d bytes in %d chunks, %d average\n", 
		session.up.bytes, session.up.calls,
		session.up.calls ? session.up.bytes / session.up.calls : 0); 

	LOGQ (Debug, "in <<< : %d bytes in %d chunks, %d average\n", 
		session.down.bytes, session.down.calls, 
		session.down.calls? session.down.bytes / session.down.calls : 0); 
	/*
	LOGQ (Debug, "(%db / %d inside virtual interrupt)\n", 
		session.vi_down.bytes, session.vi_down.calls); */

	SHOW_TIMINGS (pulse_latency, "Pulse latency");
/*
	LOGQ (Debug, "Pulse latency min = %.2f %s, max = %.2f %s; %.2f %s average\n", 
		watch_format(&pulse_latency,Min), 
		watch_unit (pulse_latency,Min),
		watch_format(&pulse_latency,Max), 
		watch_unit (pulse_latency,Max),
		watch_format (&pulse_latency, Avg),
		watch_unit (pulse_latency, Avg));
*/

	LOGQ (Debug, "Servicing virtual irq min = %.2f %s, max = %.2f %s; %.2f %s average, with %d threshed\n", 
		watch_format(&core_virtual_watch,Min), 
		watch_unit (core_virtual_watch,Min),
		watch_format(&core_virtual_watch,Max), 
		watch_unit (core_virtual_watch,Max),
		watch_format (&core_virtual_watch, Avg),
		watch_unit (core_virtual_watch, Avg),
		core_virtual_watch.threshnum);

	SHOW_TIMINGS (write_watch, "Write expenses");
	SHOW_TIMINGS (read_watch, "Read expenses");

	LOGQ (Debug, "Rates: connection/last. Up: %s/%s. Down: %s/%s\n",
		sRate[x_status[1]], sRate[x_status[3]], 
		sRate[x_status[2]], sRate[x_status[4]]);

	if (x_status[0x12] + x_status[0x13] != 0)
		LOGQ (Debug, "Retrains: %d local, %d remote\n", x_status[0x12], x_status[0x13]);
	
	LOGQ (Debug, "Frames: total/erronous. Transmit: %d/%d. Receive: %d/%d\n",

		x_status[0xe], x_status[0xf], x_status[0x10], x_status[0x11]);

	LOGQ (Debug, "\n");
	LOGQ (Debug, "\t\t*********************\n");
	LOGQ (Debug, "\n");
}

extern void pin_start_timer (void );
extern void pin_stop_timer (void );
extern unsigned pin_check_mctrl (int changes);
/***** session_run: the engine *****/

void session_run (int fd) {
	fd_set readers, writers;
	fd_set *preaders, *pwriters;

	session_init (fd, &readers, &writers);
	mport_open (&mport); /*, &session);*/
	/* perhaps should be in mport_open) */
	mport_mctrl_set (&mport, pin_check_mctrl (0)); 

	pin_start_timer();

	/* stream output buffer is expected to be more than 0x1000 
		i.e. core read ring */

	{
		struct timeval now;
		gettimeofday (&now, NULL);
		session.start = now.tv_sec;
	}

	LOGINFO ("Started %s", ctime (&session.start));

	while (1) {
		/*** set up sets to watch ***/
		/* check on cts to be added */

		preaders = checkset_upstream (fd, &readers);
		/* 
		 * writer depends on data availability in core 
		 * this way it will be checked only on the second encounter
		 * so check in virtual interrupt is more effective
		 * unless we are absolutely sure that with main loop
		 * the buffering is better
		 */
		/* pwriters = checkset_downstream (fd, &writers); */
		pwriters = NULL;

		/*** wait on pin for events ***/
		switch (wait_events (fd, preaders, pwriters)) {
		case Closed:
			LOGINFO ("client's gone\n");
			goto l_escape;

		case CarrierLost:
			LOGINFO ("carrier lost\n");
			if (config.check_carrier) 
				goto l_escape;
			else {
				LOGINFO ("ignoring\n");
				mport.carrier = COk; continue;
			}

		case Fault:
			goto l_escape;

		case Pulse:
			continue;

		case RW:
			break;

		default:
			LOGWARN ("wait_events: unexpected result\n");
			continue;
		}
		

		/*** write ***/
		/* assert */
		if ( pwriters && FD_ISSET (fd, pwriters) ) {
			LOGDEBUG (2, "mainloop: down is available!!!!!\n");
			if (downstream (fd) == -1) 
				break /*out*/;
		}

		/*** read ***/
		if (preaders && FD_ISSET (fd, preaders)) {
			if (Debugged(Timings)) watch_start (&read_watch);
			if (upstream (fd) == -1) {
				if (session.closed) LOGINFO ("client's gone\n");
				break /*out*/;
			}
			if (Debugged(Timings)) {
				watch_stop (&read_watch);
				unsigned ms = watch_nsec (read_watch, last) / 1000000;
				if (ms > 5) logdebugadd (Timings, "r%dms", ms);
			}
		}
	}
l_escape:
	pin_stop_timer();
	session.pin = -1;

	mport_close (&mport);

	{
		struct timeval now;
		gettimeofday (&now, NULL);
		LOGINFO ("Closed %s", ctime (&now.tv_sec));
	}

	dump_stats();
}
