//  ************************************************************************************************
//
//  BornAgain: simulate and fit reflection and scattering
//
//! @file      GUI/View/Fit/FitObjectiveBuilder.cpp
//! @brief     Implements class SimulationBuilder
//!
//! @homepage  http://www.bornagainproject.org
//! @license   GNU General Public License v3 or higher (see COPYING)
//! @copyright Forschungszentrum Jülich GmbH 2018
//! @authors   Scientific Computing Group at MLZ (see CITATION, AUTHORS)
//
//  ************************************************************************************************

#include "GUI/View/Fit/FitObjectiveBuilder.h"
#include "Base/Util/Assert.h"
#include "Device/Data/Datafield.h"
#include "Fit/Kernel/Minimizer.h"
#include "Fit/Minimizer/IMinimizer.h"
#include "GUI/Model/Data/DataItem.h"
#include "GUI/Model/Device/RealItem.h"
#include "GUI/Model/Job/FitParameterContainerItem.h"
#include "GUI/Model/Job/FitSuiteItem.h"
#include "GUI/Model/Job/JobItem.h"
#include "GUI/Model/Job/MinimizerItem.h"
#include "GUI/Model/ToCore/SimulationToCore.h"
#include "GUI/View/Fit/GUIFitObserver.h"
#include "Sim/Fitting/FitObjective.h"
#include "Sim/Fitting/ObjectiveMetric.h"
#include "Sim/Simulation/ISimulation.h"

FitObjectiveBuilder::FitObjectiveBuilder(JobItem* jobItem)
    : m_jobItem(jobItem)
{
    ASSERT(m_jobItem->fitSuiteItem());
}

FitObjectiveBuilder::~FitObjectiveBuilder() = default;

void FitObjectiveBuilder::runFit()
{
    m_fit_objective = createFitObjective();

    auto module = m_jobItem->fitSuiteItem()->minimizerContainerItem()->createMetric();
    m_fit_objective->setObjectiveMetric(std::move(module));

    if (m_observer) {
        fit_observer_t plot_observer = [&](const FitObjective& obj) { m_observer->update(&obj); };
        m_fit_objective->initPlot(1, std::move(plot_observer));
    }

    auto minimizer_impl = createMinimizer();
    const bool requires_residuals = minimizer_impl->requiresResiduals();

    mumufit::Minimizer minimizer;
    minimizer.setMinimizer(minimizer_impl.release());

    auto result = requires_residuals ? minimizer.minimize(
                      [&](const mumufit::Parameters& params) {
                          return m_fit_objective->evaluate_residuals(params);
                      },
                      createParameters())
                                     : minimizer.minimize(
                                         [&](const mumufit::Parameters& params) {
                                             return m_fit_objective->evaluate(params);
                                         },
                                         createParameters());
    m_fit_objective->finalize(result);
}

std::unique_ptr<FitObjective> FitObjectiveBuilder::createFitObjective() const
{
    auto result = std::make_unique<FitObjective>();

    simulation_builder_t builder = [&](const mumufit::Parameters& params) {
        return buildSimulation(params);
    };

    auto* realItem = m_jobItem->realItem();
    ASSERT(realItem);

    const DataItem* intensity_item = realItem->dataItem();
    ASSERT(intensity_item);
    ASSERT(intensity_item->c_field());

    std::unique_ptr<Datafield> data(intensity_item->c_field()->clone());

    result->execAddSimulationAndData(builder, *data, nullptr, 1.0);

    return result;
}

std::unique_ptr<IMinimizer> FitObjectiveBuilder::createMinimizer() const
{
    return m_jobItem->fitSuiteItem()->minimizerContainerItem()->createMinimizer();
}

mumufit::Parameters FitObjectiveBuilder::createParameters() const
{
    return m_jobItem->fitSuiteItem()->fitParameterContainerItem()->createParameters();
}

void FitObjectiveBuilder::attachObserver(std::shared_ptr<GUIFitObserver> observer)
{
    m_observer = observer;
}

void FitObjectiveBuilder::interruptFitting()
{
    m_fit_objective->interruptFitting();
}

std::unique_ptr<ISimulation>
FitObjectiveBuilder::buildSimulation(const mumufit::Parameters& params) const
{
    static std::mutex build_simulation_mutex;
    std::unique_lock<std::mutex> lock(build_simulation_mutex);

    update_fit_parameters(params);
    return GUI::ToCore::itemsToSimulation(*m_jobItem->sampleItem(), *m_jobItem->instrumentItem(),
                                          m_jobItem->simulationOptionsItem());
}

void FitObjectiveBuilder::update_fit_parameters(const mumufit::Parameters& params) const
{
    auto* fitParContainer = m_jobItem->fitParameterContainerItem();
    fitParContainer->setValuesInParameterContainer(params.values(),
                                                   m_jobItem->parameterContainerItem());
}
