//  ************************************************************************************************
//
//  BornAgain: simulate and fit reflection and scattering
//
//! @file      GUI/View/Mask/MaskGraphicsScene.h
//! @brief     Defines class MaskGraphicsScene
//!
//! @homepage  http://www.bornagainproject.org
//! @license   GNU General Public License v3 or higher (see COPYING)
//! @copyright Forschungszentrum Jülich GmbH 2018
//! @authors   Scientific Computing Group at MLZ (see CITATION, AUTHORS)
//
//  ************************************************************************************************

#ifndef BORNAGAIN_GUI_VIEW_MASK_MASKGRAPHICSSCENE_H
#define BORNAGAIN_GUI_VIEW_MASK_MASKGRAPHICSSCENE_H

#include "GUI/View/Mask/MaskDrawingContext.h"
#include "GUI/View/Mask/MaskEditorHelper.h"
#include <QGraphicsScene>
#include <QItemSelection>
#include <QItemSelectionModel>

class ISceneAdaptor;
class IShape2DView;
class IntensityDataItem;
class MaskContainerItem;
class MaskContainerModel;
class MaskEditorAction;
class MaskGraphicsProxy;
class MaskItem;
class MaskItemObject;
class PolygonView;

class ColorMap;

//! Graphics scene for MaskEditorCanvas to draw masks on top of intensity data widgets.

class MaskGraphicsScene : public QGraphicsScene {
    Q_OBJECT
public:
    MaskGraphicsScene(QObject* parent = nullptr);
    ~MaskGraphicsScene() override;

    void setMaskContext(IntensityDataItem* intensityItem, MaskContainerItem* maskContainerItem);

    void resetContext();

    void setSelectionModel(QItemSelectionModel* selectionModel);

    ColorMap* colorMap();
signals:
    void itemContextMenuRequest(const QPoint& point);
    void lineItemProcessed();
    void lineItemMoved(MaskItemObject* sender);
    void lineItemDeleted(MaskItemObject* sender);

public slots:
    void onActivityModeChanged(MaskEditorFlags::Activity value);
    void onMaskValueChanged(MaskEditorFlags::MaskValue value);
    void onRowsInserted(const QModelIndex&, int, int);
    void onRowsAboutToBeRemoved(const QModelIndex&, int first, int last);
    void cancelCurrentDrawing();
    void resetScene();
    void updateScene();

private slots:
    void onSessionSelectionChanged(const QItemSelection& /* selected */,
                                   const QItemSelection& /* deselected */);
    void onSceneSelectionChanged();
    void onMaskMove();

protected:
    void mousePressEvent(QGraphicsSceneMouseEvent* event) override;
    void mouseMoveEvent(QGraphicsSceneMouseEvent* event) override;
    void mouseReleaseEvent(QGraphicsSceneMouseEvent* event) override;
    void drawForeground(QPainter* painter, const QRectF& rect) override;
    void contextMenuEvent(QGraphicsSceneContextMenuEvent* event) override;

private:
    void updateProxyWidget();
    void updateViews();
    IShape2DView* addViewForItem(MaskItemObject* item);
    void removeItemViewFromScene(MaskItemObject* item);

    void connectMaskContainer(MaskContainerModel* maskContainerModel);
    void disconnectMaskContainer(MaskContainerModel* maskContainerModel);

    bool isValidMouseClick(QGraphicsSceneMouseEvent* event);
    bool isValidForRectangleShapeDrawing(QGraphicsSceneMouseEvent* event);
    bool isValidForPolygonDrawing(QGraphicsSceneMouseEvent* event);
    bool isValidForLineDrawing(QGraphicsSceneMouseEvent* event);
    bool isValidForMaskAllDrawing(QGraphicsSceneMouseEvent* event);

    bool isAreaContains(QGraphicsSceneMouseEvent* event, MaskEditorHelper::EViewTypes viewType);
    bool isDrawingInProgress() const;
    void setDrawingInProgress(bool value);
    void setInPanAndZoomMode(bool value);
    void updateCursors();

    void makeViewAtMousePosSelected(QGraphicsSceneMouseEvent* event);

    void processRectangleOrEllipseItem(QGraphicsSceneMouseEvent* event);
    void processPolygonItem(QGraphicsSceneMouseEvent* event);
    void processLineItem(QGraphicsSceneMouseEvent* event);
    void processVerticalLineItem(const QPointF& pos);
    void processHorizontalLineItem(const QPointF& pos);
    void processMaskAllItem(QGraphicsSceneMouseEvent* event);
    void subscribeMaskItem(MaskItemObject* item);

    void setZValues();
    PolygonView* currentPolygon() const;

    MaskContainerModel* m_maskContainerModel = nullptr;
    MaskContainerItem* m_maskContainerItem = nullptr;
    QItemSelectionModel* m_selectionModel = nullptr;
    QMap<MaskItemObject*, IShape2DView*> m_ItemToView;
    MaskGraphicsProxy* m_proxy = nullptr;
    QSharedPointer<ISceneAdaptor> m_adaptor;
    bool m_block_selection = false;
    bool m_mouse_is_pressed = false;
    IntensityDataItem* m_intensityItem = nullptr;
    MaskItemObject* m_currentItem = nullptr;
    QPointF m_currentMousePosition;
    MaskDrawingContext m_context;
};

#endif // BORNAGAIN_GUI_VIEW_MASK_MASKGRAPHICSSCENE_H
