/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2012 Nick Gnedin 
All rights reserved.

This file may be distributed and/or modified under the terms of the
GNU General Public License version 2 as published by the Free Software
Foundation and appearing in the file LICENSE.GPL included in the
packaging of this file.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/


#include "iconfigure.h"
#if ISHELL_INCLUDED(ISHELL_GG)


#include "iggpageview.h"


#include "ianimator.h"
#include "icamera.h"
#include "icolorbars.h"
#include "icontrolmodule.h"
#include "idata.h"
#include "idatareader.h"
#include "ierror.h"
#include "ierrorstatus.h"
#include "iimagefactory.h"
#include "imarker.h"
#include "imarkerfamily.h"
#include "ipalette.h"
#include "ipaletteset.h"
#include "iparallelmanager.h"
#include "ipointglyph.h"
#include "irendertool.h"
#include "ishell.h"
#include "iviewmodule.h"
#include "ivtk.h"
#include "iwriter.h"

#include "iggcommoneventobservers.h"
#include "iggdatatypeprovider.h"
#include "iggdialoganimatingprogress.h"
#include "iggdialogauto.h"
#include "iggdialogimagecomposer.h"
#include "iggdialogscriptdebugger.h"
#include "iggextensionfactory.h"
#include "iggframedatatypeselector.h"
#include "iggframedatavariablelist.h"
#include "iggframedoublebutton.h"
#include "iggframematerialproperties.h"
#include "iggframeobjectcontrols.h"
#include "iggframepaletteselection.h"
#include "iggframeposition.h"
#include "iggmainwindow.h"
#include "iggrenderwindowobserver.h"
#include "iggwidgetarea.h"
#include "iggwidgetkeybutton.h"
#include "iggwidgetkeycolorselection.h"
#include "iggwidgetkeylineedit.h"
#include "iggwidgetkeyselectionbox.h"
#include "iggwidgetkeyslider.h"
#include "iggwidgetkeytrackball.h"
#include "iggwidgetotherbutton.h"
#include "iggwidgetmisc.h"
#include "iggwidgettext.h"

#include "ibgwidgetareasubject.h"
#include "ibgwidgetbuttonsubject.h"
#include "ibgwidgetselectionboxsubject.h"
#include "ibgwidgethelper.h"

#include <vtkCamera.h>
#include <vtkRenderer.h>
#include <vtkRenderWindow.h>

#include "iggsubjectfactory.h"

//
//  Templates (needed for some compilers)
//
#include "iarraytemplate.h"
#include "iggwidgetkeyhandlertemplate.h"
#include "iggwidgetkeyslidertemplate.h"


using namespace iParameter;


namespace iggPageView_Private
{
	//
	//  Show color bar check box
	//
	class ShowColorBarCheckBox : public iggWidgetKeyHandler<int>
	{
		
	public:
		
		ShowColorBarCheckBox(const iObjectKey &key, iggFrame *parent) : iggWidgetKeyHandler<int>(WidgetType::Other,key,RenderMode::Immediate,parent,3,0)
		{
			mSubject = iggSubjectFactory::CreateWidgetButtonSubject(this,ButtonType::CheckBox,"Show",1);
		}
		
	protected:

		virtual void OnVoid1Body()
		{
			this->ExecuteControl(true);
		}

		virtual void QueryValue(int &v) const
		{
			if(mSubject->IsDown()) v = 1; else v = 0;
		}
		
		virtual void UpdateValue(int v)
		{
			mSubject->SetDown(v != 0);
		}
		
		ibgWidgetButtonSubject *mSubject;
	};


	//
	//  Color bar frame
	//
	class ColorBarFrame : public iggFrame
	{
		
	public:
		
		ColorBarFrame(iggWidgetKeyCheckBox *autobox, const iString &title, const iObjectKey &key, iggFrame *parent) : iggFrame(title,parent,1), mKey(key)
		{
			mProvider = new iggKeywordDataTypeProvider("scalars",this); 

			mAutomaticBox = autobox;
			autobox->AddDependent(this);

			//this->AddLine(new ShowColorBarCheckBox(key,this));
			this->AddLine(new iggWidgetTextArea("Palette",this));
			mPaletteList = new iggWidgetKeyComboBox("",0,key,this,2);
			this->AddLine(mPaletteList);

			iggFrameDataVariableList *vl = new iggFrameDataVariableList(mProvider,"Variable",key,0,this,0);
			vl->Complete(-1);
			this->AddLine(vl);
			this->AddSpace(2);

			this->AddLine(new iggWidgetTextArea("Data type",this));
			this->AddLine(new iggFrameDataTypeSelector(mProvider,false,"",this));
		}

		virtual ~ColorBarFrame()
		{
			delete mProvider;
		}

	protected:

		virtual void UpdateWidgetBody()
		{
			//
			//  Are we enabled?
			//
			bool b;
			mAutomaticBox->UpdateWidget();
			mAutomaticBox->QueryValue(b);
			this->Enable(!b);

			if(!b)
			{
				//
				//  Update palettes
				//
				int i, np = iPaletteSet::Default()->GetNumberOfPalettes();
				mPaletteList->Clear();
				mPaletteList->SetOffset(np-1);
				for(i=np-1; i>=0; i--)
				{
					mPaletteList->InsertItem(iPaletteSet::Default()->GetPalette(i)->GetName()+" reversed");
				}
				mPaletteList->InsertItem("Do not show");
				for(i=0; i<np; i++)
				{
					mPaletteList->InsertItem(iPaletteSet::Default()->GetPalette(i)->GetName());
				}
				//
				//  Update data type
				//
				iString ws;
				this->GetShell()->GetControlModule()->PackCommand(ws,mKey,1,mProvider->GetActiveDataType().GetId());
				if(!this->GetShell()->GetControlModule()->Execute(ws,false)) return this->UpdateFailed();
			}
			//
			//  Update the rest
			//
			this->iggFrame::UpdateWidgetBody();
		}

		const iObjectKey &mKey;
		iggWidgetKeyComboBox *mPaletteList;
		iggWidgetKeyCheckBox *mAutomaticBox;
		iggKeywordDataTypeProvider *mProvider;
	};


	//
	//  Animation controls
	//
	class AnimationStyleBox : public iggWidgetKeyRadioBox
	{

	public:

		AnimationStyleBox(iggFrame *parent) : iggWidgetKeyRadioBox(1,"Style",0,iAnimator::KeyStyle(),parent){}

	protected:

		virtual void UpdateWidgetBody()
		{
			this->iggWidgetKeyRadioBox::UpdateWidgetBody();
			this->Enable(!this->GetShell()->GetControlModule()->GetViewModule()->GetAnimator()->UsingScript());
		}
	};


	class AnimationFramesSlider : public iggWidgetKeyLargeIntSlider
	{
		
	public:

		AnimationFramesSlider(iggFrame *parent) : iggWidgetKeyLargeIntSlider(1,150,5,"Frames per file",iAnimator::KeyNumberOfFrames(),RenderMode::NoRender,parent){}

	protected:

		virtual void UpdateWidgetBody()
		{
			this->iggWidgetKeyLargeIntSlider::UpdateWidgetBody();
			this->Enable(!this->GetShell()->GetControlModule()->GetViewModule()->GetAnimator()->UsingScript());
		}
	};


	class AnimateButton : public iggWidgetSimpleButton
	{

	public:

		AnimateButton(iggFrame *parent) : iggWidgetSimpleButton("Animate",parent)
		{
			this->SetBaloonHelp("Start animation");
		}

	protected:

		virtual void UpdateWidgetBody()
		{
			mWidgetHelper->Enable(this->GetShell()->GetControlModule()->GetViewModule()->GetReader()->IsFileAnimatable());
		}

		virtual void Execute()
		{
			this->GetMainWindow()->GetDialogAnimatingProgress()->Animate();
			this->GetShell()->GetControlModule()->Render();

			if(this->GetShell()->GetControlModule()->GetViewModule()->GetAnimator()->GetErrorStatus()->IsError())
			{
				this->GetMainWindow()->PopupWindow("Animation completed with the following message:\n"+this->GetShell()->GetControlModule()->GetViewModule()->GetAnimator()->GetErrorStatus()->Message());
			}
		}
	};


	//
	//  Window manipulation widgets
	//
	//
	//  A small dialog with the list of existing windows
	//
	class WindowList : public iggWidgetListView
	{

	public:

		WindowList(iggFrame *parent) : iggWidgetListView(false,parent)
		{
			this->SetBaloonHelp("Select the current window","This control selects a visualization window to be the \"current\" one. The current window is labeled by a marker placed in the upper left corner of the window, and is the window controled by the widgets.");
        }

	protected:

		virtual void UpdateWidgetBody()
		{
			int i, n = this->GetShell()->GetControlModule()->GetNumberOfViewModules();
			if(n != this->Count())
			{
				this->Clear();
				iString ws;
				for(i=0; i<n; i++)
				{
					ws = "Window " + iString::FromNumber(i+1);
					if(this->GetShell()->GetControlModule()->GetViewModule(i)->IsClone())
					{
						ws += " (Clone of #" + iString::FromNumber(this->GetShell()->GetControlModule()->GetViewModule(i)->GetCloneOfWindow()+1) + ")";
					}
					this->InsertItem(ws);
				}
			}
			this->Select(this->GetShell()->GetControlModule()->GetCurrentViewModuleIndex(),this->GetShell()->GetControlModule()->GetCurrentViewModuleIndex());
		}

		virtual void OnCursorPositionChanged(int line, int index)
		{
			if(this->GetShell()->GetControlModule()->SetCurrentViewModuleIndex(line)) this->GetMainWindow()->UpdateAll();
		}
	};


	class WindowListDialog : public iggDialogAuto
	{

	public:

		WindowListDialog(iggMainWindow *parent) : iggDialogAuto(parent,"Windows",1)
		{
			if(this->ImmediateConstruction()) this->CompleteConstruction();
		}

		void CompleteConstructionBody()
		{
			mFrame->AddLine(new WindowList(mFrame));
			this->ResizeContents(100,200);
		}
	};


	class WindowNumberComboBox : public iggWidget
	{

	public:

		WindowNumberComboBox(iggFrame *parent, iggPageView *page) : iggWidget(parent), mPage(page)
		{
			mSubject = iggSubjectFactory::CreateWidgetComboBoxSubject(this,"Set current window");
			this->SetBaloonHelp("Select the current window","This control selects a visualization window to be the \"current\" one. The current window is labeled by a marker placed in the upper left corner of the window, and is the window controled by the widgets.");
        }

	protected:

		virtual void UpdateWidgetBody()
		{
			mPage->GetWindowListDialog()->UpdateDialog();

			mSubject->Clear();

			int i, n = this->GetShell()->GetControlModule()->GetNumberOfViewModules();
			for(i=0; i<n; i++) mSubject->InsertItem("Window "+iString::FromNumber(i+1));
			mSubject->SetValue(this->GetShell()->GetControlModule()->GetCurrentViewModuleIndex());
		}

		void OnInt1Body(int v)
		{
			if(this->GetShell()->GetControlModule()->SetCurrentViewModuleIndex(v)) this->GetMainWindow()->UpdateAll();
		}

		ibgWidgetComboBoxSubject *mSubject;
		iggPageView *mPage;
	};


	class CopyFromWindowComboBox : public iggWidget
	{

	public:

		CopyFromWindowComboBox(iggFrame *parent) : iggWidget(parent)
		{
			mSubject = iggSubjectFactory::CreateWidgetComboBoxSubject(this,"Copy settings from...");
			this->SetBaloonHelp("Copy all internal settings from another window");
        }

	protected:

		virtual void UpdateWidgetBody()
		{
			mSubject->Clear();

			int i, n = this->GetShell()->GetControlModule()->GetNumberOfViewModules();
			mSubject->InsertItem("...");
			if(n > 1)
			{
				this->Enable(true);
				for(i=0; i<n; i++) if(i != this->GetShell()->GetControlModule()->GetCurrentViewModuleIndex())
				{
					mSubject->InsertItem("Window "+iString::FromNumber(i+1));
				}
			}
			else this->Enable(false);
			mSubject->SetValue(0);
		}

		void OnInt1Body(int v)
		{
			if(v > 0)
			{
				if(v > this->GetShell()->GetControlModule()->GetCurrentViewModuleIndex()) v++;

				iViewModule *vm1 = this->GetShell()->GetControlModule()->GetViewModule();
				iViewModule *vm2 = this->GetShell()->GetControlModule()->GetViewModule(v-1);
				if(vm1!=0 && vm2!=0 && vm1!=vm2)
				{
					this->GetMainWindow()->Block(true);
					vm1->CopySettings(vm2);
					vm1->Render();
					this->GetMainWindow()->Block(false);
				}
			}
			mSubject->SetValue(0);
		}

		ibgWidgetComboBoxSubject *mSubject;
	};


	class DialogWait : public iggDialog
	{

	public:

		DialogWait(bool create, iggMainWindow *parent) : iggDialog(parent,DialogFlag::Blocking|DialogFlag::NoTitleBar,0,"Working...",0,3,0)
		{
			mFlipper = new iggWidgetLogoFlipper(mFrame);
			mFrame->AddLine(0,mFlipper);
			mFrame->AddLine(new iggWidgetTextArea(iString("%b%+")+(create?"Creat":"Delet")+"ing window...",mFrame),3);

			this->CompleteConstruction();
		}

		void CompleteConstructionBody()
		{
		}

		virtual void ShowBody(bool s)
		{
			if(s) mFlipper->Start(); else mFlipper->Abort();
			iggDialog::ShowBody(s);
		}

	protected:

		iggWidgetLogoFlipper *mFlipper;
	};


	class CreateWindowButton : public iggWidgetSimpleButton
	{

	public:

		CreateWindowButton(int type, iggFrame *parent) : iggWidgetSimpleButton("",parent)
		{
			mType = type;
			mDialog = new DialogWait(true,this->GetMainWindow());

			switch(mType)
			{
			case ModuleType::New:
				{
					mSubject->SetText("Create new window");
					this->SetBaloonHelp("Create a new independent window","This button creates a new visualization window. The new window is independent of other windows and has default values of all parameters.");
					break;
				}
			case ModuleType::Copy:
				{
					mSubject->SetText("Copy current window");
					this->SetBaloonHelp("Copy the current window","This button creates a new visualization window which is a copy of the current visualization window, i.e. it has the same values of all parameters for all objects. The new window is independent of other windows, and has no data associated with it.");
					break;
				}
			case ModuleType::Clone:
				{
					mSubject->SetText("Clone current window");
					this->SetBaloonHelp("Clone the current window","This button creates a clone of the current visualization window. The clone shares the data with the parent window, and at birth has the same values of all parameters for all visualization objects. It does have a separate set of visualization objects however, so the parameters of the parent and the clone window can be changed independently.");
					break;
				}
			default: IERROR_FATAL("Incorrect button type");
			}
        }

		virtual ~CreateWindowButton()
		{
			delete mDialog;
		}

	protected:

		virtual void Execute()
		{
			this->GetMainWindow()->Block(true);
			mDialog->Show(true);
			if(!this->GetShell()->GetControlModule()->CreateObject(iViewModule::Type(),true,-1,mType))
			{
				this->GetMainWindow()->PopupWindow("Unable to create new window.");
			}
			else
			{
				if(this->GetShell()->GetControlModule()->SetCurrentViewModuleIndex(this->GetShell()->GetControlModule()->GetNumberOfViewModules()-1)) this->GetMainWindow()->UpdateAll();
			}
			mDialog->Show(false);
			this->GetMainWindow()->Block(false);
		}

		int mType;
		DialogWait *mDialog;
	};


	class DeleteWindowButton : public iggWidgetSimpleButton
	{

	public:

		DeleteWindowButton(iggFrame *parent) : iggWidgetSimpleButton("Delete current window",parent)
		{
			mDialog = new DialogWait(false,this->GetMainWindow());

			this->SetBaloonHelp("Delete the current window","This button deletes the current visualization window and all the objects and the data associated with it.");
        }

		virtual ~DeleteWindowButton()
		{
			delete mDialog;
		}

	protected:

		virtual void UpdateWidgetBody()
		{
			mWidgetHelper->Enable(this->GetShell()->GetControlModule()->GetNumberOfViewModules() > 1);
		}

		virtual void Execute()
		{
			this->GetMainWindow()->Block(true);
			mDialog->Show(true);
			if(!this->GetShell()->GetControlModule()->DeleteObject(iViewModule::Type(),true))
			{
				this->GetMainWindow()->PopupWindow("Unable to delete current window.");
			}
			else
			{
				if(this->GetShell()->GetControlModule()->SetCurrentViewModuleIndex(this->GetShell()->GetControlModule()->GetNumberOfViewModules()-1)) this->GetMainWindow()->UpdateAll();
			}
			mDialog->Show(false);
			this->GetMainWindow()->Block(false);
		}

		DialogWait *mDialog;
	};


	class WindowUnderFocusCurrentCheckBox : public iggWidget
	{

	public:

		WindowUnderFocusCurrentCheckBox(iggFrame *parent) : iggWidget(parent)
		{
			mSubject = iggSubjectFactory::CreateWidgetButtonSubject(this,ButtonType::CheckBox,"Make window under focus current",1);
			this->SetBaloonHelp("Toggles whether the visualization window with the keyboard focus becomes the current one","When this box is checked, moving the leyboard focus into a visualization window will make that window the current one.");
        }

	protected:

		virtual void UpdateWidgetBody()
		{
			mSubject->SetDown(this->GetMainWindow()->GetWindowUnderFocusCurrent());
		}

		void OnVoid1Body()
		{
			this->GetMainWindow()->SetWindowUnderFocusCurrent(mSubject->IsDown());
		}

		ibgWidgetButtonSubject *mSubject;
	};


	class WidgetsControlAllWindowsCheckBox : public iggWidget
	{

	public:

		WidgetsControlAllWindowsCheckBox(iggFrame *parent) : iggWidget(parent)
		{
			mSubject = iggSubjectFactory::CreateWidgetButtonSubject(this,ButtonType::CheckBox,"Widgets control all windows",1);
			this->SetBaloonHelp("Toggles whether widgets control all visualization windows or only the current one","When this box is checked, widget controls in the gui window will affect all visualization windows, not just the current one. This option should be used with care, since it only makes sense if different visualization windows display similar scenes (for example, the same visualization object for different data files). If the scenes in different visualization windows are unlike each other, the effects of this option may be quite unexpected.");
        }

	protected:

		virtual void UpdateWidgetBody()
		{
		}

		void OnVoid1Body()
		{
			if(mSubject->IsDown())
			{
				iggWidgetKeyHandlerBase::SetGlobalExecuteFlags(ModuleOption::All|RenderOption::All);
			}
			else
			{
				//
				//  Go back to default ones
				//
				iggWidgetKeyHandlerBase::SetGlobalExecuteFlags(0);
			}
		}

		ibgWidgetButtonSubject *mSubject;
	};


	class WindowStatusIcon : public iggFrame
	{

	public:

		WindowStatusIcon(iggFrame *parent) : iggFrame("Status",parent,1)
		{
			mIcon1 = iImageFactory::FindIcon("winindiv.png");
			mIcon2 = iImageFactory::FindIcon("winclone.png");

			mText1 = "Individual";
			mText2 = "Clone of #";

			mIconView = new iggWidgetImageArea(*mIcon1,false,this);
			mTextView = new iggWidgetTextArea(mText1,this);
			this->AddLine(mIconView);
			this->AddLine(mTextView);
//			this->AddSpace(10);
		}

	protected:

		virtual void UpdateWidgetBody()
		{
			if(this->GetShell()->GetControlModule()->GetViewModule()->IsClone())
			{
				mIconView->SetImage(*mIcon2,false);
				mTextView->SetText(mText2+iString::FromNumber(this->GetShell()->GetControlModule()->GetViewModule()->GetCloneOfWindow()+1));
			}
			else
			{
				mIconView->SetImage(*mIcon1,false);
				mTextView->SetText(mText1);
			}
			this->iggFrame::UpdateWidgetBody();
		}

		const iImage *mIcon1, *mIcon2;
		iString mText1, mText2;
		iggWidgetImageArea *mIconView;
		iggWidgetTextArea *mTextView;
	};


	//
	//  Marker widgets
	//
	class MarkerFrameBox : public iggFrame
	{

	public:

		MarkerFrameBox(iggFrame *parent) : iggFrame("Marker",parent,2)
		{
		}

	protected:

		virtual void UpdateWidgetBody()
		{
			if(this->GetShell()->GetControlModule()->GetViewModule()->GetMarkerFamily()->GetMaxMemberIndex()>0 || this->GetShell()->GetControlModule()->GetViewModule()->GetMarkerFamily()->IsVisible())
			{
				this->Enable(true);
				this->iggFrame::UpdateWidgetBody();
			}
			else
			{
				this->Enable(false);
			}
		}
	};


	class CurrentMarkerBox : public iggWidgetKeyComboBox
	{

	public:

		CurrentMarkerBox(iggFrame *parent) : iggWidgetKeyComboBox("",0,iViewModule::KeyMarkerCurrent(),parent)
		{
		}

	protected:

		virtual void UpdateWidgetBody()
		{
			int i, n = this->GetShell()->GetControlModule()->GetViewModule()->GetMarkerFamily()->GetMaxMemberIndex() + 1;

			if(n==1 && !this->GetShell()->GetControlModule()->GetViewModule()->GetMarkerFamily()->IsVisible())
			{
				mSubject->Clear();
			}
			else
			{
				if(n != mSubject->Count())
				{
					mSubject->Clear();
					for(i=0; i<n; i++)
					{
						mSubject->InsertItem("#"+iString::FromNumber(i+1)+" "+this->GetShell()->GetControlModule()->GetViewModule()->GetMarkerFamily()->GetMember(i)->GetTypeAsString());
					}
				}
				else
				{
					i = this->GetShell()->GetControlModule()->GetViewModule()->GetMarkerFamily()->GetCurrentMemberIndex();
					mSubject->SetItem("#"+iString::FromNumber(i+1)+" "+this->GetShell()->GetControlModule()->GetViewModule()->GetMarkerFamily()->GetMember(i)->GetTypeAsString(),i,true);
				}
			}
			this->iggWidgetKeyComboBox::UpdateWidgetBody();
		}
	};


	class MarkerPositionFrame : public iggFramePosition, public iggRenderWindowObserver
	{

	public:

		MarkerPositionFrame(iggFrame *parent) : iggFramePosition("Position",iMarker::KeyPosition(true),parent,true), iggRenderWindowObserver(parent)
		{
		}

	protected:

		virtual void OnRenderWindowModified()
		{
			this->iggFramePosition::UpdateWidgetBody();
		}
	};


	//
	//  A small dialog with the line edit to set the ruler scale exactly
	//
	class RulerScaleDialog : public iggDialogAuto, public iggRenderWindowObserver
	{

	public:

		RulerScaleDialog(iggMainWindow *parent) : iggDialogAuto(parent,"Ruler Scale",2), iggRenderWindowObserver(mFrame)
		{
			if(this->ImmediateConstruction()) this->CompleteConstruction();
		}

		void CompleteConstructionBody()
		{
			mFrame->AddLine(new iggWidgetTextArea("Scale",mFrame),new iggWidgetKeyFloatLineEdit("",iViewModule::KeyRulerScale(),RenderMode::UseGlobal,mFrame));
			mFrame->AddLine(new iggWidgetTextArea("Title",mFrame),new iggWidgetKeyTextLineEdit(false,"",iViewModule::KeyRulerTitle(),RenderMode::UseGlobal,mFrame));
			mFrame->AddSpace(10);
		}

	protected:

		virtual void OnRenderWindowModified()
		{
			this->UpdateDialog();
		}
	};

	class MarkerMoveCaptionsButton : public iggWidgetKeyExecButton
	{

	public:

		MarkerMoveCaptionsButton(const iString &text, const iObjectKey &key, iggFrame *parent) : iggWidgetKeyExecButton(text,key,parent,-1,0)
		{
		}

	protected:

		virtual void UpdateWidgetBody()
		{
			this->iggWidgetKeyExecButton::UpdateWidgetBody();

			bool hasCaptions = false;
			int i, n = this->GetShell()->GetControlModule()->GetViewModule()->GetMarkerFamily()->GetMaxMemberIndex() + 1;
			for(i=0; !hasCaptions && i<n; i++)
			{
				if(!this->GetShell()->GetControlModule()->GetViewModule()->GetMarkerFamily()->GetMember(i)->GetCaptionText().IsEmpty()) hasCaptions = true;
			}
			this->Enable(hasCaptions);
		}
	};


	class ShowRulerCheckBox : public iggWidgetKeyCheckBox
	{

	public:

		ShowRulerCheckBox(iggFrame *parent) : iggWidgetKeyCheckBox("Ruler",iViewModule::KeyRuler(),parent)
		{
			mDialog = new RulerScaleDialog(this->GetMainWindow());
		}

		virtual ~ShowRulerCheckBox()
		{
			delete mDialog;
		}

	protected:

		void OnVoid1Body()
		{
			if(mDialog != 0) mDialog->Show(mSubject->IsDown());
			iggWidgetKeyCheckBox::OnVoid1Body();
		}

		RulerScaleDialog *mDialog;
	};


	//
	//  A small dialog with several widgets to set the record label parameters
	//
	class RecordLabelDialog : public iggDialogAuto
	{

	public:

		RecordLabelDialog(iggMainWindow *parent) : iggDialogAuto(parent,"Record Label Properties",3)
		{
			if(this->ImmediateConstruction()) this->CompleteConstruction();
		}

		void CompleteConstructionBody()
		{
			mFrame->AddLine(new iggWidgetTextArea("Name",mFrame),new iggWidgetKeyTextLineEdit(false,"",iViewModule::KeyLabelName(),RenderMode::UseGlobal,mFrame),static_cast<iggWidget*>(0));
			mFrame->AddLine(new iggWidgetTextArea("Unit",mFrame),new iggWidgetKeyTextLineEdit(false,"",iViewModule::KeyLabelUnit(),RenderMode::UseGlobal,mFrame),static_cast<iggWidget*>(0));
			mFrame->AddLine(new iggWidgetTextArea("Scale",mFrame),new iggWidgetKeyFloatLineEdit("",iViewModule::KeyLabelScale(),RenderMode::UseGlobal,mFrame),static_cast<iggWidget*>(0));
			mFrame->AddLine(new iggWidgetTextArea("Offset",mFrame),new iggWidgetKeyFloatLineEdit("",iViewModule::KeyLabelOffset(),RenderMode::UseGlobal,mFrame),static_cast<iggWidget*>(0));
			mFrame->AddLine(new iggWidgetKeySpinBox(0,7,"Number of digits after the point",0,iViewModule::KeyLabelDigits(),mFrame),3);
			mFrame->AddSpace(10);
		}
	};


	class ShowRecordLabelCheckBox : public iggWidgetKeyCheckBox
	{

	public:

		ShowRecordLabelCheckBox(iggFrame *parent) : iggWidgetKeyCheckBox("Record label",iViewModule::KeyLabel(),parent)
		{
			mDialog = new RecordLabelDialog(this->GetMainWindow());
		}

		virtual ~ShowRecordLabelCheckBox()
		{
			delete mDialog;
		}

	protected:

		void OnVoid1Body()
		{
			if(mDialog != 0) mDialog->Show(mSubject->IsDown());
			iggWidgetKeyCheckBox::OnVoid1Body();
		}

		RecordLabelDialog *mDialog;
	};


	class ClippingRangeButton : public iggWidgetSimpleButton
	{

	public:

		ClippingRangeButton(int type, iggWidgetKeyCheckBox *buddy, iggFrame *parent) : iggWidgetSimpleButton("",parent,true)
		{
			mBuddy = buddy;
			mBuddy->AddDependent(this);
			mType = type;

			if(mType%2 == 0)
			{
				mSubject->SetIcon(*iImageFactory::FindIcon("moveleft.png"));
			}
			else
			{
				mSubject->SetIcon(*iImageFactory::FindIcon("moveright.png"));
			}
			this->SetBaloonHelp("Adjust camera clipping range","This button adjust the clipping range of the current camera.");
        }

	protected:

		virtual void UpdateWidgetBody()
		{
			bool v;
			mBuddy->QueryValue(v);
			this->Enable(!v);
		}

		virtual void Execute()
		{
			double cr[2];

			if(this->GetShell()->GetControlModule()->QueryValue(iCamera::KeyClippingRange(),cr,2))
			{
				switch(mType)
				{
				case 0:
					{
						cr[0] *= 0.5;
						break;
					}
				case 1:
					{
						cr[0] *= 2.0;
						break;
					}
				case 2:
					{
						cr[1] *= 0.5;
						break;
					}
				case 3:
					{
						cr[1] *= 2.0;
						break;
					}
				}
				iString ws;
				this->GetShell()->GetControlModule()->PackCommand(ws,iCamera::KeyClippingRange(),cr,2);
				this->GetShell()->GetControlModule()->Execute(ws,this->GetMainWindow()->IsAutoRender());
			}
		}

		int mType;
		iggWidgetKeyCheckBox *mBuddy;
	};


	class ClippingRangeDisplay : public iggWidgetNumberDisplay, protected iggRenderWindowObserver
	{

	public:

		ClippingRangeDisplay(int type, iggFrame *parent) : iggWidgetNumberDisplay("",parent), iggRenderWindowObserver(parent)
		{
			mType = type;
		}

	protected:

		virtual void UpdateWidgetBody()
		{
			double cr[2];

			if(this->GetShell()->GetControlModule()->QueryValue(iCamera::KeyClippingRange(),cr,2))
			{
				switch(mType)
				{
				case 0:
					{
						this->Display(cr[0]);
						break;
					}
				case 1:
					{
						this->Display(cr[1]);
						break;
					}
				}
			}
		}

		virtual void OnRenderWindowModified()
		{
			this->UpdateWidget();
		}

		int mType;
	};


	//
	//  Window size dialog and its widgets
	//
	class WindowSizeButton : public iggWidgetSimpleButton
	{

	public:

		WindowSizeButton(iggWidgetSimpleSpinBox **boxes, const iString &title, float scale, iggFrame *parent) : iggWidgetSimpleButton(title,parent)
		{
			mBoxes = boxes;
			mScale = scale;
			this->SetBaloonHelp(title+" the window size","This button "+title+"s the size of the current view window.");
        }

	protected:

		virtual void Execute()
		{
			int i;
			for(i=0; i<2; i++) mBoxes[i]->SetValue(round(mScale*mBoxes[i]->GetValue()));
		}

		float mScale;
		iggWidgetSimpleSpinBox **mBoxes;
	};

	class WindowSizeList : public iggWidget
	{

	public:

		WindowSizeList(iggWidgetSimpleSpinBox **boxes, iggFrame *parent) : iggWidget(parent)
		{
			mBoxes = boxes;
			mSubject = iggSubjectFactory::CreateWidgetComboBoxSubject(this,"");

			this->SetBaloonHelp("Select the size for the visualization window");
		}

	protected:

		virtual void OnInt1Body(int n)
		{
			iString s = mSubject->GetText(n);

			int j, k;
			bool ok;
			for(j=0; j<2; j++)
			{
				k = s.Section(" x ",j,j).ToInt(ok);
				if(ok) mBoxes[j]->SetValue(k);
			}
		}

		virtual void UpdateWidgetBody()
		{
		}

		iggWidgetSimpleSpinBox **mBoxes;
		ibgWidgetComboBoxSubject *mSubject;
	};

	class WindowSizeList1 : public WindowSizeList
	{

	public:

		WindowSizeList1(iggWidgetSimpleSpinBox **boxes, iggFrame *parent) : WindowSizeList(boxes,parent)
		{
			mSubject->InsertItem("Rectangle...");
			mSubject->InsertItem("300 x 200");
			mSubject->InsertItem("450 x 300");
			mSubject->InsertItem("700 x 450");
			mSubject->InsertItem("1000 x 700");
			mSubject->InsertItem("1500 x 1000");
		}
	};

	class WindowSizeList2 : public WindowSizeList
	{

	public:

		WindowSizeList2(iggWidgetSimpleSpinBox **boxes, iggFrame *parent) : WindowSizeList(boxes,parent)
		{
			mSubject->InsertItem("Square...");
			mSubject->InsertItem("200 x 200");
			mSubject->InsertItem("300 x 300");
			mSubject->InsertItem("450 x 450");
			mSubject->InsertItem("700 x 700");
			mSubject->InsertItem("1000 x 1000");
		}
	};

	class WindowSizeDialog : public iggDialog
	{

	public:

		WindowSizeDialog(iggMainWindow *parent) : iggDialog(parent,DialogFlag::Modal,0,"Visualization Window Size",0,2,"Set")
		{
			if(this->ImmediateConstruction()) this->CompleteConstruction();
		}

		void CompleteConstructionBody()
		{
			mBoxes[0] = new iggWidgetSimpleSpinBox(120,32768,1,"",mFrame);
			mBoxes[1] = new iggWidgetSimpleSpinBox(120,32768,1,"",mFrame);

			iggFrame* tmp = new iggFrame(mFrame,2);
			tmp->AddLine(new WindowSizeButton(mBoxes,"half",0.5f,tmp),new WindowSizeButton(mBoxes,"double",2.0f,tmp));
			mFrame->AddLine(tmp,2);
			mFrame->AddLine(new WindowSizeList1(mBoxes,mFrame),new WindowSizeList2(mBoxes,mFrame));
			mFrame->AddLine(new iggWidgetTextArea("%bWidth:",mFrame),mBoxes[0]);
			mFrame->AddLine(new iggWidgetTextArea("%bHeight:",mFrame),mBoxes[1]);
			mFrame->AddSpace(10);

			mBoxes[0]->SetBaloonHelp("Set window width","Set the width of the current visualization window.");
			mBoxes[1]->SetBaloonHelp("Set window height","Set the height of the current visualization window.");
		}

		virtual void ShowBody(bool s)
		{
			if(s)
			{
                const int *size = this->GetShell()->GetControlModule()->GetViewModule()->GetRenderWindowSize();
				mBoxes[0]->SetValue(size[0]);
				mBoxes[1]->SetValue(size[1]);
			}
			iggDialog::ShowBody(s);
		}

	protected:

		virtual bool CanBeClosed()
		{
			const int *s = this->GetShell()->GetControlModule()->GetViewModule()->GetRenderWindowSize();
			if(s[0]!=mBoxes[0]->GetValue() || s[1]!=mBoxes[1]->GetValue())
			{
				this->GetShell()->GetControlModule()->GetViewModule()->SetRenderWindowSize(mBoxes[0]->GetValue(),mBoxes[1]->GetValue());
			}
			return true;
		}

		iggWidgetSimpleSpinBox *mBoxes[2];
	};


	//
	//  Misc buttons
	//
	class ShowLogCheckBox : public iggWidgetSimpleCheckBox
	{

	public:

		ShowLogCheckBox(iggFrame *parent) : iggWidgetSimpleCheckBox("Show log  ",parent)
		{
			this->SetChecked(true);
			this->SetBaloonHelp("Show the log window","This check box shows/hides the log window");
		}

	protected:

		virtual void UpdateWidgetBody()
		{
			this->SetChecked(this->GetMainWindow()->IsLogVisible());
		}

		virtual void OnChecked(bool s)
		{
			this->GetMainWindow()->ShowLog(s);
		}
	};


	class ClearLogButton : public iggWidgetSimpleButton
	{

	public:

		ClearLogButton(iggFrame *parent) : iggWidgetSimpleButton("Clear log",parent)
		{
			this->SetBaloonHelp("Clear the log window","This button removes all the text in the log window.");
        }

	protected:

		virtual void Execute()
		{
			this->GetMainWindow()->ClearLog();
		}
	};


	class ResetAnimationImageIndexButton : public iggWidgetSimpleButton
	{

	public:

		ResetAnimationImageIndexButton(iggFrame *parent) : iggWidgetSimpleButton("Reset image counter",parent)
		{
			this->SetBaloonHelp("Reset image counter back to 1","This button resets the counter of animation image files to 1. The animation files created so far will be overwritten in the next animation.");
        }

	protected:

		virtual void Execute()
		{
			if(this->GetMainWindow()->PopupWindow("Please confirm: reseting the counter will overwrite the animation files created so far.",PopupWindow::Message,"Reset","Cancel") == 0)
			{
				//this->GetShell()->GetControlModule()->GetViewModule()->GetWriter()->ResetAnimationImageIndex();
			}
		}
	};


	class MoveCameraButton : public iggWidgetSimpleButton
	{

	public:

		MoveCameraButton(int dir, iggFrame *parent) : iggWidgetSimpleButton("",parent,true)
		{
			mDir = dir;
			switch(mDir)
			{
			case -1:
				{
					mSubject->SetIcon(*iImageFactory::FindIcon("moveleft.png"));
					break;
				}
			case 1:
				{
					mSubject->SetIcon(*iImageFactory::FindIcon("moveright.png"));
					break;
				}
			case -2:
				{
					mSubject->SetIcon(*iImageFactory::FindIcon("movedown.png"));
					break;
				}
			case 2:
				{
					mSubject->SetIcon(*iImageFactory::FindIcon("moveup.png"));
					break;
				}
			default:
				{
					IERROR_LOW("Invalid direction.");
				}
			}
			this->SetBaloonHelp("Move the camera focal point","This button moves the camera focal point by 10 degrees in the specified direction.");
		}

	protected:

		virtual void Execute()
		{
			static const double angle = 10.0;
			vtkCamera *cam = this->GetShell()->GetControlModule()->GetViewModule()->GetRenderer()->GetActiveCamera();

			switch(mDir)
			{
			case -1:
				{
					cam->Yaw(angle);
					break;
				}
			case 1:
				{
					cam->Yaw(-angle);
					break;
				}
			case -2:
				{
					cam->Pitch(-angle);
					break;
				}
			case 2:
				{
					cam->Pitch(angle);
					break;
				}
			default:
				{
					IERROR_LOW("Invalid direction.");
				}
			}
			cam->OrthogonalizeViewUp();
			this->GetShell()->GetControlModule()->GetViewModule()->Render();
		}

		int mDir;
	};


	class StereoModeSelector : public iggWidgetKeyComboBox 
	//class StereoModeSelector : public iggWidgetKeyRadioBox
	{

	public:

		StereoModeSelector(iggFrame *parent) : 
		  iggWidgetKeyComboBox("Method",0,iViewModule::KeyStereoMode(),parent)
		  //iggWidgetKeyRadioBox(1,"Method",0,iViewModule::KeyStereoMode(),parent)
		  {
			  this->InsertItem("Mono");
			  this->InsertItem("Dual windows");
			  this->InsertItem("Crystal eyes");
			  this->InsertItem("Blue-red");
			  this->InsertItem("Interlaced");
			  this->InsertItem("Left eye only");
			  this->InsertItem("Right eye only");
			  this->InsertItem("Dresden display");
			  this->InsertItem("Anaglyph");
#ifndef IVTK_PRE54
			  this->InsertItem("Checkerboard");
#endif
		  }

	protected:

		virtual void UpdateWidgetBody()
		{
			this->SetItem("Crystal eyes",2,this->GetShell()->GetControlModule()->GetViewModule()->GetRenderWindow()->GetStereoCapableWindow()!=0);
		}
	};

	
	class StereoHardwareStatus : public iggWidgetTextDisplay
	{

	public:

		StereoHardwareStatus(iggFrame *parent) : iggWidgetTextDisplay("Stereo harware status: ",parent)
		{
		}

	protected:

		virtual void UpdateWidgetBody()
		{
			if(this->GetShell()->GetControlModule()->GetViewModule()->GetRenderWindow()->GetStereoCapableWindow() != 0)
			{
				this->Display("DETECTED");
			}
			else
			{
				this->Display("NOT FOUND");
			}
		}
	};


	class ParallelExecutionBox : public iggWidgetSimpleRadioBox
	{

	public:

		ParallelExecutionBox(iggFrame *parent) : iggWidgetSimpleRadioBox(1,"Execution",parent)
		{
			this->SetBaloonHelp("Specifiy serial or parallel execution mode","Use this control to quickly switch from a serial (single core) to parallel (multiple cores/processors). More elaborate control of parallel execution id provided by the Parallel Controller dialog.");
			this->InsertItem("Serial");
			this->InsertItem("Parallel");
		}

	protected:

		virtual void UpdateWidgetBody()
		{
			mSubject->SetValue((this->GetShell()->GetControlModule()->GetParallelManager()->GetNumberOfProcessors()==1)?0:1);
		}

		virtual void OnInt1Body(int i)
		{
			this->GetShell()->GetControlModule()->GetParallelManager()->SetNumberOfProcessors((i==0)?this->GetShell()->GetControlModule()->GetParallelManager()->GetMinNumberOfProcessors():this->GetShell()->GetControlModule()->GetParallelManager()->GetMaxNumberOfProcessors());
		}
	};

	//
	//  Lights page
	//
	class SceneLightsPage : public iggFrameDC
	{

	public:
	
		SceneLightsPage(iggFrameBase *parent) : iggFrameDC(parent,3)
		{
			if(this->ImmediateConstruction()) this->CompleteConstruction();
		}

	protected:

		virtual void CompleteConstructionBody()
		{
			iggFrame *db = new iggFrame("Main light direction",this,1);
			db->AddLine(new iggWidgetKeyTrackBall(iViewModule::KeyLightAngles(),true,RenderMode::UseGlobal,db));
			this->AddLine(db);

			iggFrame *ib = new iggFrame("Intensity",this,1);
			iggWidgetKeyFloatSlider *mli = new iggWidgetKeyFloatSlider(0.01f,10.0f,200,1,0,"Main light",iViewModule::KeyLightIntensity(),RenderMode::UseGlobal,ib,0);
			iggWidgetKeyFloatSlider *fli = new iggWidgetKeyFloatSlider(0.01f,10.0f,200,1,0,"Fill light",iViewModule::KeyLightIntensity(),RenderMode::UseGlobal,ib,1);
			iggWidgetKeyFloatSlider *hli = new iggWidgetKeyFloatSlider(0.01f,10.0f,200,1,0,"Head light",iViewModule::KeyLightIntensity(),RenderMode::UseGlobal,ib,2);
			hli->AddBuddy(mli);
			fli->AddBuddy(mli);
			mli->SetStretch(3,10);
			hli->SetStretch(3,10);
			fli->SetStretch(3,10);
			ib->AddLine(mli);
			ib->AddLine(fli);
			ib->AddLine(hli);
			this->AddLine(ib,2);

			this->AddSpace(10);
			this->SetColStretch(1,10);
			this->SetColStretch(2,3);
		}
	};


	//
	//  Multi-view modes page
	//
	class SceneModesPage : public iggFrameDC
	{

	public:
	
		SceneModesPage(iggFrameBase *parent) : iggFrameDC(parent,3)
		{
			if(this->ImmediateConstruction()) this->CompleteConstruction();
		}

	protected:

		virtual void CompleteConstructionBody()
		{
			const iImage *icon = iImageFactory::FindIcon("view.png");

			//
			//  Book
			//
			iggFrameBook *vb = new iggFrameBook(this);
			this->AddLine(vb);
			//
			//  Stereo mode page
			//
			iggFrame *vbpage0 = new iggFrame(vb,3);
			vb->AddPage("Stereo",icon,vbpage0);
			{
				StereoModeSelector *sm = new StereoModeSelector(vbpage0);
				vbpage0->AddLine(sm,new iggWidgetTextArea("",vbpage0));
				vbpage0->AddLine(new iggWidgetKeyCheckBox("Show alighnment marks in a dual window mode",iViewModule::KeyStereoAlignmentMarks(),vbpage0),2);
				vbpage0->AddLine(new iggWidgetKeyFloatSlider(0.1f,10.0f,20,1,4,"Angle between eyes",iCamera::KeyEyeAngle(),RenderMode::UseGlobal,vbpage0),2);

				vbpage0->AddSpace(2);

				vbpage0->AddLine(new StereoHardwareStatus(vbpage0),2);

				vbpage0->AddSpace(10);
				vbpage0->SetColStretch(0,0);
				vbpage0->SetColStretch(1,10);
				vbpage0->SetColStretch(2,3);
			}
			//
			//  Multi view mode page
			//
			iggFrame *vbpage1 = iggExtensionFactory::CreateMultiViewFrame(vb);
			if(vbpage1 != 0) vb->AddPage("Multi view",icon,vbpage1);
		}
	};


	//
	//  Color bars page
	//
	class SceneColorBarsPage : public iggFrameDC
	{

	public:
	
		SceneColorBarsPage(iggFrameBase *parent) : iggFrameDC(parent,3)
		{
			if(this->ImmediateConstruction()) this->CompleteConstruction();
		}

	protected:

		virtual void CompleteConstructionBody()
		{
			iggWidgetKeyCheckBox *acb = new iggWidgetKeyCheckBox("Automatic",iColorBars::KeyAutomatic(),this);

			this->AddLine(acb,2);
			this->AddLine(new ColorBarFrame(acb,"Left bar",iColorBars::KeyBarLeft(),this),new ColorBarFrame(acb,"Right bar",iColorBars::KeyBarRight(),this));

			this->AddSpace(10);
			iggFrame *tmp = new iggFrame(this,2);
			tmp->AddLine(new iggWidgetTextArea("Side offset",tmp),new iggWidgetKeyFloatSlider(0.01,0.3,29,0,0,"",iColorBars::KeySideOffset(),RenderMode::UseGlobal,tmp));
			tmp->AddLine(new iggWidgetTextArea("Bar size",tmp),new iggWidgetKeyFloatSlider(0.2,1.0,8,0,0,"",iColorBars::KeySize(),RenderMode::UseGlobal,tmp));
			tmp->SetColStretch(1,10);
			this->AddLine(tmp,3);

			this->SetColStretch(2,10);
		}
	};


	//
	//  Tools page
	//
	class SceneToolsPage : public iggFrameDC
	{

	public:
	
		SceneToolsPage(iggFrameBase *parent) : iggFrameDC(parent,3)
		{
			if(this->ImmediateConstruction()) this->CompleteConstruction();
		}

	protected:

		virtual void CompleteConstructionBody()
		{
			//
			//  Clip plane
			//
			iggFrame *cpf = new iggFrame("Clip plane",this,3);
			cpf->AddLine(new iggWidgetKeyCheckBox("Use clip plane",iViewModule::KeyClipPlane(),cpf));
			cpf->AddSpace(2);

			iggFrame *cb = new iggFrame("Clip plane direction",cpf,1);
			cb->AddLine(new iggWidgetKeyTrackBall(iViewModule::KeyClipPlaneDirection(),false,RenderMode::UseGlobal,cb));
			cpf->AddLine(cb);
			cpf->AddLine(new iggWidgetKeyFloatSlider(-2.0f,2.0f,400,0,0,"Clip plane distance",iViewModule::KeyClipPlaneDistance(),RenderMode::UseGlobal,cpf),2);
			cpf->AddLine(new iggWidgetKeyCheckBox("Show as glass plane",iViewModule::KeyGlassClipPlane(),cpf));

			this->AddLine(cpf);
			this->AddSpace(2);

			//
			//  Measuring box
			//
			iggFrame *mbf = new iggFrame("Measuring box",this,3);
			mbf->AddLine(new iggWidgetKeyCheckBox("Measuring box",iViewModule::KeyMeasuringBox(),mbf));
			this->AddLine(mbf);

			this->AddSpace(10);
			this->SetColStretch(0,10);
			this->SetColStretch(1,3);
		}
	};

	//
	//  Scene page
	// ************************************************
	//
	class ScenePage : public iggFrameDC
	{

	public:
	
		ScenePage(iggFrameBase *parent) : iggFrameDC(parent,1)
		{
			if(this->ImmediateConstruction()) this->CompleteConstruction();
		}

	protected:

		virtual void CompleteConstructionBody()
		{
			const iImage *icon = iImageFactory::FindIcon("view.png");

			//
			//  Book
			//
			iggFrameBook *sb = new iggFrameBook(this);
			this->AddLine(sb);
			//
			//  Camera page (created immediately)
			//
			iggFrame *sbpage0 = new iggFrame(sb,3);
			sb->AddPage("Camera",icon,sbpage0);
			{
				iggFrame *vf = new iggFrame("View angle",sbpage0,1);
				vf->AddSpace(1);
				vf->AddLine(new iggWidgetKeyFloatSlider(10.0,120.0,110,0,3,"",iCamera::KeyViewAngle(),RenderMode::UseGlobal,vf));
				vf->AddLine(new iggWidgetKeyCheckBox("Vertical",iCamera::KeyViewAngleVertical(),vf));
				vf->AddSpace(10);

				iggFrame *cb = new iggFrame("Camera orientatioin",sbpage0,3);
				cb->AddLine(static_cast<iggWidget*>(0),new MoveCameraButton(2,cb),static_cast<iggWidget*>(0));
				cb->AddLine(new MoveCameraButton(-1,cb),static_cast<iggWidget*>(0),new MoveCameraButton(1,cb));
				cb->AddLine(static_cast<iggWidget*>(0),new MoveCameraButton(-2,cb),static_cast<iggWidget*>(0));

				sbpage0->AddLine(vf,cb);
				sbpage0->AddSpace(2);

				iggFrame *crf = new iggFrame("Camera clipping range",sbpage0,5);
				iggWidgetKeyCheckBox *cfa = new iggWidgetKeyCheckBox("Automatic",iCamera::KeyClippingRangeAuto(),crf);
				crf->AddLine(cfa,3);
				crf->AddLine(new ClippingRangeButton(0,cfa,crf),new ClippingRangeButton(1,cfa,crf),new iggWidgetTextArea("Adjust clipping range",crf),new ClippingRangeButton(2,cfa,crf),new ClippingRangeButton(3,cfa,crf));
				crf->AddLine(new ClippingRangeDisplay(0,crf),2,new iggWidgetTextArea("(in units of focal distance)",crf),1,new ClippingRangeDisplay(1,crf),2);
				sbpage0->AddLine(crf,2);
				sbpage0->AddSpace(2);

				sbpage0->AddSpace(4);
				sbpage0->SetColStretch(2,10);
			}
			//
			//  Lights page
			//
			SceneLightsPage *sbpage1 = new SceneLightsPage(sb);
			sb->AddPage("Lights",icon,sbpage1);

			//
			//  Multi-view modes page
			//
			SceneModesPage *sbpage2 = new SceneModesPage(sb);
			sb->AddPage("Modes",icon,sbpage2);

			//
			//  Color bars page
			//
			SceneColorBarsPage *sbpage3 = new SceneColorBarsPage(sb);
			sb->AddPage("Color bars",icon,sbpage3);

			//
			//  Tools page
			//
			SceneToolsPage *sbpage4 = new SceneToolsPage(sb);
			sb->AddPage("Tools",icon,sbpage4);
		}
	};


	//
	//  Animation page
	// ************************************************
	//
	class AnimationPage : public iggFrameDC
	{

	public:
	
		AnimationPage(iggFrameBase *parent) : iggFrameDC(parent,1)
		{
			if(this->ImmediateConstruction()) this->CompleteConstruction();
		}

	protected:

		virtual void CompleteConstructionBody()
		{
			const iImage *icon = iImageFactory::FindIcon("view.png");

			//
			//  Book
			//
			iggFrameBook *ab = new iggFrameBook(this);
			this->AddLine(ab);
			//
			//  Style page
			//
			iggFrame *abpage0 = new iggFrame(ab,3);
			ab->AddPage("Style",icon,abpage0);
			{
				AnimationStyleBox *as = new AnimationStyleBox(abpage0);
				as->InsertItem("Static");
				as->InsertItem("Rotate/scale");
				as->InsertItem("Tumble");
				as->InsertItem("Flyby");

				iggFrame *tmp = new iggFrame(abpage0);
				iggWidgetKeyRadioBox *ao = new iggWidgetKeyRadioBox(1,"Animation output",0,iViewModule::KeyAnimationOutput(),tmp);
				ao->InsertItem("Series of images");
				ao->InsertItem("MPEG movie");

#ifdef IVTK_SUPPORTS_MPEG
#ifdef IVTK_501
				ao->SetItem("MPEG movie",1,false); //  disable MPEG generation for VTK 5.0.1 due to a bug in vtkMPEGWriter
#endif
#else
				ao->SetItem("MPEG movie",1,false); //  disable MPEG generation for VTK 5.0.1 due to a bug in vtkMPEGWriter
#endif

				ao->InsertItem("AVI movie");
#ifndef IVTK_SUPPORTS_AVI
				ao->SetItem("AVI movie",2,false);
#endif

				tmp->AddLine(ao);
				tmp->AddSpace(10);
				abpage0->AddLine(as,tmp);

				abpage0->AddSpace(10);

				AnimationFramesSlider *af = new AnimationFramesSlider(abpage0);
				af->SetRange(1,150,1);
				abpage0->AddLine(af,2);

				iggWidgetKeyFileNameLineEdit *aq = new iggWidgetKeyFileNameLineEdit(false,"IFrIT Animation Script",this->GetShell()->GetEnvironment(Environment::Script),"(*.ias)",true,"Current script",true,iAnimator::KeyScriptFile(),abpage0);
				aq->AddDependent(af);
				aq->AddDependent(as);
				abpage0->AddLine(aq,2);
				abpage0->AddLine(new iggWidgetLaunchButton(this->GetMainWindow()->GetDialogScriptDebugger(),"Launch script debugger",abpage0),2);

				abpage0->AddSpace(10);

				abpage0->AddLine(new iggWidgetKeyFileNameLineEdit(false,"IFrIT Animation Output",this->GetShell()->GetEnvironment(Environment::Image),"(*)",false,"Destination",false,iViewModule::KeyAnimationRootName(),abpage0),2);

				abpage0->AddSpace(10);

				iggFrame *abut = new iggFrame(abpage0,3);
				abut->AddLine(0,new AnimateButton(abut));
				abut->SetColStretch(0,10);
				abut->SetColStretch(2,10);
				abpage0->AddLine(abut,3);

				abpage0->SetColStretch(1,10);
				abpage0->SetColStretch(2,3);
			}
			//
			//  Settings page
			//
			iggFrame *abpage1 = new iggFrame(ab,2);
			ab->AddPage("Settings",icon,abpage1);
			{

				iggWidgetKeyFloatSlider *tmpfs = new iggWidgetKeyFloatSlider(1.0e-4,1.0,40,1,0,"Flyby speed",iAnimator::KeyFlybySpeed(),RenderMode::NoRender,abpage1);
				tmpfs->SetStretch(2,10);
				abpage1->AddLine(tmpfs);
				tmpfs = new iggWidgetKeyFloatSlider(0.0,0.1,30,0,0,"Cross section speed",iAnimator::KeyCrossSectionSpeed(),RenderMode::NoRender,abpage1);
				tmpfs->SetStretch(2,10);
				abpage1->AddLine(tmpfs);

				abpage1->AddSpace(2);

				iggFrame *ar = new iggFrame("Rotations",abpage1,1);
				tmpfs = new iggWidgetKeyFloatSlider(-10.0,10.0,200,0,4,"Azimuth (phi)",iAnimator::KeyPhi(),RenderMode::NoRender,ar);
				tmpfs->SetStretch(3,10,0);
				ar->AddLine(tmpfs);
				tmpfs = new iggWidgetKeyFloatSlider(-10.0,10.0,200,0,4,"Elevation (theta)",iAnimator::KeyTheta(),RenderMode::NoRender,ar);
				tmpfs->SetStretch(3,10,0);
				ar->AddLine(tmpfs);
				tmpfs = new iggWidgetKeyFloatSlider(-10.0,10.0,200,0,4,"Roll angle",iAnimator::KeyRoll(),RenderMode::NoRender,ar);
				tmpfs->SetStretch(3,10,0);
				ar->AddLine(tmpfs);
				abpage1->AddLine(ar);
				abpage1->AddLine(new iggWidgetKeyFloatSlider(0.98,1.02,400,1,5,"Zoom",iAnimator::KeyZoom(),RenderMode::NoRender,abpage1));

				abpage1->AddSpace(3);

				iggWidgetKeySpinBox *tmpsb = new iggWidgetKeySpinBox(0,999,"Number of blended frames",0,iAnimator::KeyNumberOfBlendedFrames(),abpage1);
				tmpsb->SetStretch(10,0);
				abpage1->AddLine(tmpsb);
				tmpsb = new iggWidgetKeySpinBox(0,999,"Number of transition frames",0,iAnimator::KeyNumberOfTransitionFrames(),abpage1);
				tmpsb->SetStretch(10,0);
				abpage1->AddLine(tmpsb);

				abpage1->AddSpace(10);
				abpage1->SetColStretch(0,10);
				abpage1->SetColStretch(1,3);
			}
			//
			//  Title/logo page
			//
			iggFrame *abpage4 = new iggFrame(ab,2);
			ab->AddPage("Title/logo",icon,abpage4);
			{
				iggFrame *atbox = new iggFrame("Title",abpage4,3);
				iggWidgetKeyFileNameLineEdit *atle = new iggWidgetKeyFileNameLineEdit(false,"IFrIT Title Page File",this->GetMainWindow()->GetShell()->GetEnvironment(Environment::Base),"Images (*.jpg *.jpeg *.pnm *.bmp *.png *.tif *.tiff)",true,"",true,iAnimator::KeyTitlePageFile(),atbox);
				atbox->AddLine(atle,3);

				iggWidgetKeySpinBox *tmpsb = new iggWidgetKeySpinBox(0,1000,"Frames to show title",0,iAnimator::KeyNumberOfTitlePageFrames(),atbox);
				tmpsb->SetStretch(10,0);
				atbox->AddLine(tmpsb,2);
				tmpsb = new iggWidgetKeySpinBox(0,1000,"Frames to dissolve title",0,iAnimator::KeyNumberOfTitlePageBlendedFrames(),atbox);
				tmpsb->SetStretch(10,0);
				atbox->AddLine(tmpsb,2);

				atbox->SetColStretch(1,1);
				atbox->SetColStretch(2,10);
				abpage4->AddLine(atbox);

				iggFrame *albox = new iggFrame("Logo",abpage4,3);
				iggWidgetKeyFileNameLineEdit *alle = new iggWidgetKeyFileNameLineEdit(false,"IFrIT Logo File",this->GetMainWindow()->GetShell()->GetEnvironment(Environment::Base),"Images (*.jpg *.jpeg *.pnm *.bmp *.png *.tif *.tiff)",true,"",true,iAnimator::KeyLogoFile(),albox);
				//avb = new iggWidgetLaunchButton(static_cast<iggDialog*>(0),"View",parent,trueViewButton(alia,alf);
				//alf->AddLine(arb,avb);
				albox->AddLine(alle,3);

				iggWidgetKeyComboBox *tmpcb = new iggWidgetKeyComboBox("Location",0,iAnimator::KeyLogoLocation(),albox);
				tmpcb->InsertItem("Upper left");
				tmpcb->InsertItem("Upper right");
				tmpcb->InsertItem("Lower left");
				tmpcb->InsertItem("Lower rigt");
				tmpcb->SetStretch(10,0);
				albox->AddLine(tmpcb,2);
				iggWidgetKeyFloatSlider *tmpfs = new iggWidgetKeyFloatSlider(0,1.0,10,0,4,"Opacity",iAnimator::KeyLogoOpacity(),RenderMode::NoRender,albox);
				tmpfs->SetStretch(10,0);
				albox->AddLine(tmpfs,2);

				albox->SetColStretch(1,1);
				albox->SetColStretch(2,10);
				abpage4->AddLine(albox);
				abpage4->AddSpace(10);
				abpage4->SetColStretch(0,10);
				abpage4->SetColStretch(1,3);
			}
		}
	};


	//
	//  Markers page
	// ************************************************
	//
	class MarkersPage : public iggFrameDC
	{

	public:
	
		MarkersPage(iggFrameBase *parent) : iggFrameDC(parent,3)
		{
			if(this->ImmediateConstruction()) this->CompleteConstruction();
		}

	protected:

		virtual void CompleteConstructionBody()
		{
			int i;
			const iImage *icon = iImageFactory::FindIcon("view.png");
			//
			//  Book
			//
			iggFrameBook *mb = new iggFrameBook(this);
			this->AddLine(mb);
			//
			//  Main page
			//
			iggFrame *mbpage0 = new iggFrame(mb,4);
			mb->AddPage("Main",icon,mbpage0);
			{
				iggFrame *msb = new MarkerFrameBox(mbpage0);

				CurrentMarkerBox *mcm = new CurrentMarkerBox(msb);
				msb->AddLine(new iggWidgetTextArea("Current",msb),mcm);

				iggWidgetKeyComboBox *mcs = new iggWidgetKeyComboBox("",0,iMarker::KeyType(),msb);
				mcs->AddDependent(mcm);
				mcm->AddDependent(mbpage0);
				for(i=0; i<PointGlyphType::SIZE; i++)
				{
					mcs->InsertItem(iPointGlyph::GetName(i));
				}
				msb->AddLine(new iggWidgetTextArea("Type",msb),mcs);

				iggWidgetKeyColorSelection *mcc = new iggWidgetKeyColorSelection(iMarker::KeyColor(),msb);
				mcc->SetText("Set");
				msb->AddLine(new iggWidgetTextArea("Color",msb),mcc);

				msb->AddLine(new iggWidgetTextArea("Size",msb),new iggWidgetKeySizeSlider(1.0e-5,0,"",iMarker::KeySize(true),RenderMode::UseGlobal,msb));

				iggFrame *mcb = new iggFrame("Controls",mbpage0,1);
				iggFrameCreateDeleteButton *mcd = new iggFrameCreateDeleteButton(this,iMarker::Type(),false,"marker",mcb);
				mcd->AddDependent(mbpage0);
				mcb->AddLine(mcd);

				iggWidget *mmm = new iggWidgetKeyCheckBox("Show motion handles",iViewModule::KeyMarkerHandles(),mcb);
				mcb->AddLine(mmm);

				mcb->AddSpace(10);

				mbpage0->AddLine(msb,2,mcb,1);

				iggFrame *mfc = new iggFrame("Caption",mbpage0,1);
				iggWidget *mct = new iggWidgetKeyTextLineEdit(false,"",iMarker::KeyCaptionText(),RenderMode::UseGlobal,mfc);
				mfc->AddLine(mct);
				iggWidget *mmc = new MarkerMoveCaptionsButton("Rearrange captions",iViewModule::KeyMoveMarkerCaption(),mfc);
				mct->AddDependent(mmc);
				mfc->AddLine(mmc);
				mfc->AddSpace(10);

				iggFramePosition *mpos = new MarkerPositionFrame(mbpage0);
				mbpage0->AddLine(mpos,2,mfc,1);
				mmm->AddDependent(mpos);

				mbpage0->AddLine(new iggWidgetKeyCheckBox("Move objects interactively",iMarker::KeyInteractiveMove(),mbpage0),3);
				mbpage0->AddLine(new iggWidgetKeyCheckBox("Scale marker to keep size fixed",iMarker::KeyScaled(),mbpage0),3);

				mbpage0->AddSpace(10);
				mbpage0->SetColStretch(1,10);
				mbpage0->SetColStretch(3,3);
			}
			//
			//  Material page
			//
			iggFrame *mbpage1 = new iggFrame(mb,2);
			mb->AddPage("Material ",icon,mbpage1);
			{
				mbpage1->AddLine(new iggFrameMaterialProperties(false,false,iMarker::Type(),mbpage1));
				mbpage1->AddLine(new iggWidgetKeyFloatSlider(0.0f,1.0f,10,0,0,"Opacity",iMarker::KeyOpacity(),RenderMode::UseGlobal,mbpage1));

				iggWidget *mct = new iggWidgetKeyCheckBox("Captions are transparent",iViewModule::KeyMarkerCaptionsTransparent(),mbpage1);
				mct->SetExecuteFlags(ObjectOption::One|ModuleOption::One|RenderOption::All);
				mbpage1->AddLine(mct);

				mbpage1->AddSpace(10);
				mbpage1->SetColStretch(0,10);
				mbpage1->SetColStretch(1,3);
			}
			//
			//  Legend page
			//
			iggFrame *mbpage2 = new iggFrame(mb,2);
			mb->AddPage("Legend",icon,mbpage2);
			{
				mbpage2->AddLine(new iggWidgetKeyCheckBox("Show legend",iViewModule::KeyMarkerLegend(),mbpage2));
				iggWidgetKeyRadioBox *mlp = new iggWidgetKeyRadioBox(1,"Legend position",0,iViewModule::KeyMarkerLegendPosition(),mbpage2);
				mlp->InsertItem("Lower right corner");
				mlp->InsertItem("Lower left corner");

				mbpage2->AddLine(mlp);

				mbpage2->AddSpace(10);
				mbpage2->SetColStretch(1,10);
			}
			//
			//  Transform page
			//
			iggFrame *mbpage3 = new iggFrame(mb,2);
			mb->AddPage("Transform",icon,mbpage3);
			{
				iggFrame *mtb = new iggFrame("Scale",mbpage3,1);
				mtb->AddLine(new iggWidgetKeyFloatSlider(0.01f,100.0f,200,1,5,"X",iMarker::KeyTransform(),RenderMode::UseGlobal,mtb,3));
				mtb->AddLine(new iggWidgetKeyFloatSlider(0.01f,100.0f,200,1,5,"Y",iMarker::KeyTransform(),RenderMode::UseGlobal,mtb,4));
				mtb->AddLine(new iggWidgetKeyFloatSlider(0.01f,100.0f,200,1,5,"Z",iMarker::KeyTransform(),RenderMode::UseGlobal,mtb,5));
				mbpage3->AddLine(mtb);

				mtb = new iggFrame("Rotate",mbpage3,1);
				mtb->AddLine(new iggWidgetKeyFloatSlider(-180.0,180.0,360,0,5,"X",iMarker::KeyTransform(),RenderMode::UseGlobal,mtb,0));
				mtb->AddLine(new iggWidgetKeyFloatSlider(-180.0,180.0,360,0,5,"Y",iMarker::KeyTransform(),RenderMode::UseGlobal,mtb,1));
				mtb->AddLine(new iggWidgetKeyFloatSlider(-180.0,180.0,360,0,5,"Z",iMarker::KeyTransform(),RenderMode::UseGlobal,mtb,2));
				mbpage3->AddLine(mtb);

				mbpage3->AddSpace(10);
				mbpage3->SetColStretch(0,10);
				mbpage3->SetColStretch(1,3);
			}
		}
	};


	//
	//  Windows page
	// ************************************************
	//
	class WindowsPage : public iggFrameDC
	{

	public:
	
		WindowsPage(iggFrameBase *parent, iggPageView *page) : iggFrameDC(parent,3)
		{
			mPage = page;
			if(this->ImmediateConstruction()) this->CompleteConstruction();
		}

	protected:

		iggPageView *mPage;

		virtual void CompleteConstructionBody()
		{
			iggFrame *wfb = new iggFrame("Windows",this,1);
			WindowNumberComboBox *wnb = new WindowNumberComboBox(wfb,mPage);
			wfb->AddLine(wnb);
			CreateWindowButton *wcb1 = new CreateWindowButton(ModuleType::New,wfb);
			wcb1->AddDependent(wnb);
			wfb->AddLine(wcb1);
			CreateWindowButton *wcb2 = new CreateWindowButton(ModuleType::Copy,wfb);
			wcb2->AddDependent(wnb);
			wfb->AddLine(wcb2);
			CreateWindowButton *wcb3 = new CreateWindowButton(ModuleType::Clone,wfb);
			wcb3->AddDependent(wnb);
			wfb->AddLine(wcb3);
			DeleteWindowButton *wdb = new DeleteWindowButton(wfb);
			wdb->AddDependent(wnb);
			wfb->AddLine(wdb);
			wcb1->AddDependent(wdb);
			wcb2->AddDependent(wdb);
			wcb3->AddDependent(wdb);

			WindowStatusIcon *wsi = new WindowStatusIcon(this);
			wnb->AddDependent(wsi);
			wcb1->AddDependent(wsi);
			wcb2->AddDependent(wsi);
			wcb3->AddDependent(wsi);
			wdb->AddDependent(wsi);

			this->AddLine(wfb,wsi);

			this->AddLine(new CopyFromWindowComboBox(this));
			this->AddSpace(2);

			this->AddLine(new WidgetsControlAllWindowsCheckBox(this));
			this->AddLine(new WindowUnderFocusCurrentCheckBox(this));
			this->AddLine(new iggWidgetKeyCheckBox("Synchronize window interactors",iControlModule::KeySynchronizeInteractors(),this));
			this->AddLine(new iggWidgetKeyExecButton("Synchronize cameras",iControlModule::KeySynchronizeCameras(),this));

			this->AddSpace(10);
			this->SetColStretch(2,10);
		}
	};
};

using namespace iggPageView_Private;


iggPageView::iggPageView(iggFrameBase *parent) : iggPageMain(parent)
{
	const iImage *icon = iImageFactory::FindIcon("view.png");

	mWindowSizeDialog = new WindowSizeDialog(this->GetMainWindow());
	mWindowListDialog = new WindowListDialog(this->GetMainWindow());

	//
	//  Main page (always shown at start-up, hence no delayed construction)
	// ************************************************
	//
	iggFrame *page0 = new iggFrame(mBook,4);
	mBook->AddPage("Main",icon,page0);
	{
		page0->AddSpace(2);

		iggWidgetKeyTextComboBox *ur = new iggWidgetKeyTextComboBox(0,"Update rate",0,iViewModule::KeyUpdateRate(),page0);
		ur->InsertItem("Non-interactive");
		ur->InsertItem(" 3 frames/sec");
		ur->InsertItem("10 frames/sec");
		ur->InsertItem("30 frames/sec");
		ur->SetInvalidValue(0);
		page0->AddLine(ur,2);
		page0->AddSpace(2);

		iggFrame *pf = new iggFrame("Projection",page0,1);
		pf->AddLine(new iggWidgetKeyCheckBox("Parallel",iCamera::KeyParallelProjection(),pf));
		iggWidgetKeyCheckBox *pp = new iggWidgetKeyCheckBox("Perspective",iCamera::KeyParallelProjection(),pf);
		pf->AddLine(pp);
		pp->SetReverse(true);

		if(this->GetShell()->GetControlModule()->GetParallelManager()->GetMaxNumberOfProcessors() > 1)
		{
			iggWidget *peb = new ParallelExecutionBox(page0);
			page0->AddLine(pf,peb);
		}
		else
		{
			page0->AddLine(pf);
		}

		page0->AddLine(new iggWidgetKeyExecButton("Reset view",iCamera::KeyReset(),page0),new iggWidgetLaunchButton(mWindowSizeDialog,"Set window size",page0));
		page0->AddSpace(2);

		iggFrame *fb = new iggFrame("Features",page0,1);
		fb->AddLine(new iggWidgetKeyCheckBox("Bounding box",iViewModule::KeyBoundingBox(),fb));
		fb->AddLine(new ShowRecordLabelCheckBox(fb));
		fb->AddLine(new ShowRulerCheckBox(fb));
		fb->AddLine(new iggWidgetKeyCheckBox("Color bars",iViewModule::KeyColorBars(),fb));
		fb->AddLine(new iggWidgetKeyCheckBox("Alignment axes",iViewModule::KeyCameraAlignmentLabel(),fb));

		iggFrame *fb2 = new iggFrame(page0,1);
		iggWidgetKeyRadioBox *isb = new iggWidgetKeyRadioBox(1,"Interactor style",0,iViewModule::KeyInteractorStyle(),fb2);
		isb->InsertItem("Trackball");
		isb->InsertItem("Joystick");
		isb->InsertItem("Flight");
		isb->InsertItem("Keyboard");
		fb2->AddLine(isb);

		page0->AddLine(fb,fb2);

		fb = new iggFrame(page0,2);
		fb->AddLine(new iggWidgetTextArea("Wall paper",fb),new iggWidgetKeyFileNameLineEdit(true,"Load wallpaper image",this->GetShell()->GetEnvironment(Environment::Script),"Images (*.jpg *.jpeg *.pnm *.bmp *.png *.tif *.tiff)",true," ",true,iViewModule::KeyBackgroundImage(),fb));
		fb->AddLine(new iggWidgetTextArea("Image root",fb),new iggWidgetKeyFileNameLineEdit(false,"IFrIT Image Output",this->GetShell()->GetEnvironment(Environment::Image),"(*)",false," ",false,iViewModule::KeySnapshotRootName(),fb));
		fb->SetColStretch(0,0);
		fb->SetColStretch(1,3);
		fb2 = new iggFrame(page0,1);
		iggWidgetKeyColorSelection *bgc = new iggWidgetKeyColorSelection(iViewModule::KeyBackgroundColor(),fb2);
		if(this->GetMainWindow()->GetDialogImageComposer() != 0) bgc->AddDependent(this->GetMainWindow()->GetDialogImageComposer()->GetArea());
		fb2->AddLine(bgc);
		fb2->AddSpace(10);

		page0->AddLine(fb,2,fb2,1);

		page0->AddSpace(10);
		page0->SetColStretch(3,10);
	}

	//
	//  Scene page
	// ************************************************
	//
	iggFrame *page1 = new ScenePage(mBook);
	mBook->AddPage("Scene",icon,page1);

	//
	//  Animation page
	// ************************************************
	//
	iggFrame *page2 = new AnimationPage(mBook);
	mBook->AddPage("Animation",icon,page2);

	//
	//  Markers page
	// ************************************************
	//
	iggFrame *page3 = new MarkersPage(mBook);
	mBook->AddPage("Markers",icon,page3);

	//
	//  Windows page
	// ************************************************
	//
	iggFrame *page4 = new WindowsPage(mBook,this);
	mBook->AddPage("Windows",icon,page4);
}


iggPageView::~iggPageView()
{
	delete mWindowSizeDialog;
	delete mWindowListDialog;
}


void iggPageView::UpdateMarkerWidgets()
{
	if(mBook->GetPage(3) != 0) mBook->GetPage(3)->UpdateWidget();
}


iggFrame* iggPageView::CreateFlipFrame(iggFrameBase *parent) const
{
	iggFrame *tmp = new iggFrame(parent,3);
	tmp->AddSpace(1);
	tmp->AddLine(0,new ShowLogCheckBox(tmp),new ClearLogButton(tmp));
	tmp->AddSpace(1);
	tmp->SetColStretch(0,10);
	return tmp;
}

#endif
