#!/usr/bin/env python3
# -*- coding: iso-8859-13 -*-
#
# Autorius: Albertas Agejevas, 2003
# Koregavo: Laimonas Vbra, 2010-2016
#
# Veikia su Python v2.3+, v3.0+
#
"""
ispell-lt projekto/odyno rankis.
Suglaudia/suskliaudia pasikartojanius odius (suliejant j afiks
ymas, jei toki turi), o taip pat priedlinius veiksmaodius, pvz.:
    pa|eina, nu|eina, at|eina, ... -> eina/bef...

ir iveda surikiuot odi sra, tinkam galutiniam odynui.

odiai skliaudiami tik suderinamose (kalbos dali) grupse (dabar
tai: veiksmaodiai, bdvardiai ir lik).

Naudojimas:
    ./sutrauka odynas.txt > sutraukta.txt
    cat odynas.txt | ./sutrauka > sutraukta.txt

"""
import os
import sys
import locale
import fileinput
from itertools import chain


enc = "ISO8859-13"
loc = "lt_LT" + "." + enc

# Windows setlocale() nepriima POSIX lokals
if os.name is "nt":
    loc = "Lithuanian"

_setlocale_failed = False
try:
    locale.setlocale(locale.LC_COLLATE, loc)
except:
    _setlocale_failed = True
    sys.stderr.write(
        "Could not set locale '%s', default: '%s'. "
        "Won't be able to sort dictionary words correctly.\n"
        % (loc, locale.getdefaultlocale()))

# Nuo v2.4 set tipai built-in, o sets modulis deprecated nuo v2.6
if sys.version_info < (2, 4):
    from sets import Set
    set = Set

# Py2 ir Py3 dict iteratori suderinimas
if sys.version_info < (3,):
    items = dict.iteritems
else:
    items = dict.items


# global stat vars: constringed words and saved bytes count
c_wcount = 0
c_bsaved = 0

prefixes = (
    ("a", "ap"), ("a", "api"),
    ("b", "at"), ("b", "ati"),
    ("c", ""),
    ("d", "i"),
    ("e", "nu"),
    ("f", "pa"),
    ("g", "par"),
    ("h", "per"),
    ("i", "pra"),
    ("j", "pri"),
    ("k", "su"),
    ("l", "u"),
    ("m", "apsi"),
    ("n", "atsi"),
    ("o", "si"),
    ("p", "isi"),
    ("q", "nusi"),
    ("r", "pasi"),
    ("s", "parsi"),
    ("t", "persi"),
    ("u", "prasi"),
    ("v", "prisi"),
    ("w", "susi"),
    ("x", "usi"),
    )



def _stats(word, wflags, swflags, pverb=False):
    global c_wcount, c_bsaved
    # Statistika (sutaupyta odi ir vietos).
    #
    # Kiek sutaupoma vietos (bcount) suskliaudiant od:
    # odio ilgis + bendr ym kiekis + _papildomai_ 1 arba 2 baitai,
    # priklausomai nuo varianto:
    #   - kai odis be ym arba pried. veiksmaodis (pverb): '\n' (1)
    #   - visais kitais atvejais sutaupoma: '/', '\n' (2)
    c_wcount += 1

    if (pverb or not wflags):
        le = 2
    else:
        le = 1

    c_bsaved += len(word) + len(wflags & swflags) + le

def _msg(s, *args):
    if args:
        s = s % args
    sys.stderr.write(s)
    sys.stderr.flush()

def _progress(i, step=5000):
    if (i % step == 0):
        _msg('.')

def _sort(wlist):
    if _setlocale_failed:
        wlist.sort()
    elif sys.version_info < (2, 4):
        wlist.sort(locale.strcoll)
    elif sys.version_info < (3,):
        wlist.sort(cmp=locale.strcoll)
    elif sys.version_info >= (3,):
        from functools import cmp_to_key
        wlist.sort(key=cmp_to_key(locale.strcoll))


def sutrauka(lines, outfile=sys.stdout, myspell=True):
    i = 0
    adjes = {}
    verbs = {}
    words = {}
    wcount = 0

    # Skliaudiamj odi klass (pagal afiks ym rinkinius):
    vflags = set("TYEPRO")  # verb flags
    aflags = set("AB")      # adjective flags

    _msg("\n--- %s %s\nReading ", sys.argv[0], '-' * (55 - len(sys.argv[0])))

    for line in lines:
        _progress(lines.lineno())

        # Ignoruojamos tuios ir komentaro eiluts.
        line = line.split("#")[0]
        line = line.strip()
        if not line:
            continue

        wcount += 1

        # Eilut skeliama  od ir jo ym rinkin.
        sp = line.split("/")
        word = sp[0]
        if len(sp) > 1:
            wflags = set(sp[1])
        else:
            wflags = set()

        # Veiksmaodiai ir bdvardiai  atskirus dict.
        if vflags & wflags:
            d = verbs
        elif aflags & wflags:
            d = adjes
        else:
            d = words

        # odis pridedamas  dict arba jei jau yra -- suliejamos ymos
        swflags = d.get(word) # stored word flags
        if swflags is not None:
            _stats(word, wflags, swflags)
            swflags.update(wflags)
        else:
            d[word] = wflags


    _msg("\nProcessing ")

    # Priedlini veiksmaodi suskliaudimas.
    # XXX: dl skirtingo py2 ir py3 dict vidinio eilikumo, skiriasi ir
    # suglaudinimo rezultatas.
    # Neisprsta problema: priklausomai nuo to, kurie odiai ir kokiu
    # eilikumu irenkami, skliaudiant sudurtini pried. veiksmaodius,
    # i dict paalinamas skliaudiamasis odis ir tai vliau nebeleidia
    # suskliausti kit odi.
    # Pvz.:
    #    su|panainti -> pa|nainti/k -> nainti/fk
    # vs
    #    pa|nainti -> nainti/f;
    #    (vliau 'supanainti' nebesuskliaudiamas, nes nebra 'panainti')
    #
    # Norint vieningo rezultato su py2/py3, reikia surikiuoti sra:
    #    lverbs = list(verbs); lverbs.sort()
    # nors problema ilieka: algoritmas ne visai korektikai suskliaudia
    # sudurtini priedli veiksmaodius.
    for word in list(verbs):
        i += 1
        _progress(i)

        # odio afiks ym rinkinys.
        wflags = verbs[word]

        # Kiekvienam odyno odiui derinami/tikrinami visi priedliai.
        for pflag, pref in prefixes:

            if word.startswith(pref):
                # Jei pref sangrinis priedlis, tai odis atmetus paprastj
                # (nesangrin) priedl, pvz.: i{si}|urbia -> siurbia.
                # Kai toks odis yra odyne, tai situacija netriviali, nes
                # odyne yra trys odio formos: su priedliu, be priedlio
                # ir be sangrinio priedlio.  Tampa nebeaiku kok priedl
                # (sangrin ar ne) ir kokiam odiui pritaikyti; toki
                # odi savaime suskliausti nemanoma, pvz.:
                #     i{si}|urbia, siurbia, urbia (i|siurbia ar isi|urbia?)
                #     at{si}|joja, sijoja, joja;   (at|sijoja ar atsi|joja?)
                #
                # Kol kas tokie odiai neskliaudiami.
                if pref.endswith("si"):
                    # word without reflexive prefix part
                    wrp_word = word[len(pref)-2:]
                else:
                    wrp_word = None


                # odis be priedlio, pvz.: per|oko -> oko.
                # (word without prefix)
                wp_word = word[len(pref):]
                wp_wflags = verbs.get(wp_word)

                if (wp_wflags is not None and wrp_word not in verbs):
                    # Skliaudiant priedlinius veiksmaodius su /N /S /X
                    # afiks ymomis, dl ispell apribojimo jungiant afiksus,
                    # prarandamos kelios priedlins formos, pvz:
                    #
                    #   pavartyti/X  >  te|pa|vartyti, tebe|pa|vartyti,
                    #                   be|pa|vartyti, ...
                    # vs
                    #    vartyti/Xf  >  tevartyti, tebevartyti, bevartyti, ...
                    #
                    # Susitaikius su vykstaniu priedlini form:
                    #  [/N /S /X afiksai] {priedlis} odis
                    # praradimu, odynas suglaudinamas vir 50 kB.
                    #
                    # ARBA atvirkiai: siekiant, kad nebt praradim, kaip
                    # tik nereikt toki odi (jei priedlinis odis turi
                    # /S /X /N ymas) glaudinti.
                    _stats(word, wflags, wp_wflags, pverb=True)

                    # Suliejamos afiks ymos ir pridedama priedlio yma.
                    wp_wflags.update(wflags)
                    wp_wflags.add(pflag)

                    # odis sukliaustas (prie akninio odio sulietos
                    # ymos, pridta priedlio afikso yma).  Paaliname
                    # priedlin od i 'verbs' dict ir baigiame
                    # priedli cikl, nes priedliai unikals ir odio
                    # pradia nebegali sutapti su jokiu kitu priedliu.
                    del verbs[word]
                    break

    # beafiksini odi paalinimas, jei jie yra kitose afiksini klasse
    for word, flags in items(words.copy()):
        if (not flags and (word in verbs or word in adjes)):
            _stats(word, flags, set())
            # _msg("Deleting %s\n", word)
            del words[word]
    
    wlist = []
    NS = set('NS')
    for word, flags in chain(items(words), items(verbs), items(adjes)):
        if flags:
            # /S perdengia /N, todl abiej nereikia
            if NS < flags:
                flags.remove('N')
            fl = list(flags)
            fl.sort()
            word += "/" + "".join(fl)

        wlist.append(word + '\n')

    _sort(wlist)

    _msg(" done.\nWords before: %d, words after: %d.\n"
            "(words constringed: %d, bytes saved: %d)\n%s\n",
             wcount, len(wlist), c_wcount, c_bsaved, '-' * 60)

    # myspell'o odyno pradioje -- odi kiekis.
    if myspell:
        outfile.write(len(wlist) + '\n')

    outfile.writelines(wlist)



if __name__ == "__main__":
    outfile = sys.stdout
    # Nuo v2.5+ fileinput galima nurodyti openhook' (dekodavimas i
    # norimos koduots). Aktualu tik py3 (py2 dirba su byte strings;
    # perkodavimas  unikod nebtinas), taiau openhook'as neveikia
    # su stdin.
    if sys.version_info >= (3,):
        import io
        if not sys.argv[1:]:
            # jei nra argument, tai duomenys i stdin
            sys.stdin = io.TextIOWrapper(sys.stdin.buffer, encoding=enc)
        outfile = io.TextIOWrapper(sys.stdout.buffer, encoding=enc)
        _fileinput = fileinput.input(openhook=fileinput.hook_encoded(enc))
    else:
        _fileinput = fileinput.input()

    sutrauka(_fileinput, outfile=outfile, myspell=False)
