package latexDraw.parsers.svg;

import java.util.Vector;

import org.w3c.dom.DOMException;
import org.w3c.dom.NamedNodeMap;
import org.w3c.dom.Node;

/**
 * Defines an SVG named node map.<br>
 *<br>
 * This file is part of LaTeXDraw.<br>
 * Copyright (c) 2005-2008 Arnaud BLOUIN<br>
 *<br>
 *  LaTeXDraw is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.<br>
 *<br>
 *  LaTeXDraw is distributed without any warranty; without even the 
 *  implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
 *  PURPOSE. See the GNU General Public License for more details.<br>
 *<br>
 * 09/16/07<br>
 * @author Arnaud BLOUIN<br>
 * @version 0.1<br>
 */
public class SVGNamedNodeMap implements NamedNodeMap, Cloneable
{
	/** The set of nodes. @since 0.1 */
	protected Vector<SVGAttr> nnm;
	
	
	/**
	 * The constructor by default.
	 * @since 0.1
	 */
	public SVGNamedNodeMap()
	{
		nnm = new Vector<SVGAttr>();
	}
	
	
	
	public int getLength()
	{
		return nnm==null ? 0 : nnm.size();
	}



	
	public Node getNamedItem(String name)
	{
		if(nnm==null || name==null)
			return null;
		
		int i = 0, size = getLength();
		boolean found = false;
		
		while(i<size && !found)
			if(nnm.elementAt(i).getName().equals(name))
				found = true;
			else
				i++;
		
		if(found)
			return nnm.elementAt(i);
		
		return null;
	}


	

	public Node item(int index)
	{
		return nnm==null || index<0 || index>=getLength() ? null : nnm.elementAt(index);
	}



	public Node removeNamedItem(String name) throws DOMException
	{
		if(name==null)
			throw new DOMException(DOMException.NOT_FOUND_ERR, "name is null");
		
		int i = 0, size = getLength();
		boolean found = false;
		
		while(i<size && !found)
			if(nnm.elementAt(i).getName().equals(name))
				found = true;
			else
				i++;
		
		if(found)
			return nnm.remove(i);
		
		throw new DOMException(DOMException.NOT_FOUND_ERR, name);
	}



	public Node setNamedItem(Node node) throws DOMException
	{
		if(node==null)
			return null;
		
		Node attr = getNamedItem(node.getNodeName());
		
		if(attr==null)
		{
			nnm.add((SVGAttr)node);
			return null;
		}
		
		if(attr==node)
			return null;
		
		int index = nnm.indexOf(attr);
		nnm.remove(attr);
		nnm.add(index, (SVGAttr)node);
		
		return attr;
	}



	@Override
	public Object clone()
	{
		try
		{
			SVGNamedNodeMap clone = (SVGNamedNodeMap)super.clone();
			
			clone.nnm = new Vector<SVGAttr>();
			
			for(SVGAttr attr : nnm)
				clone.nnm.add((SVGAttr)attr.cloneNode(false));
			
			return clone;
		}
		catch(CloneNotSupportedException e) { return null; }
	}


	
	@Override
	public String toString()
	{
		StringBuffer str = new StringBuffer().append("{");//$NON-NLS-1$
		
		for(SVGAttr e : nnm)
			str.append(e.toString()).append(", ");//$NON-NLS-1$
		
		return str.append("}").toString();//$NON-NLS-1$
	}


	public Node getNamedItemNS(String namespaceURI, String localName) throws DOMException
	{ throw new DOMException(DOMException.INVALID_ACCESS_ERR, SVGDocument.ACTION_NOT_IMPLEMENTED); }

	public Node setNamedItemNS(Node arg) throws DOMException
	{ throw new DOMException(DOMException.INVALID_ACCESS_ERR, SVGDocument.ACTION_NOT_IMPLEMENTED); }

	public Node removeNamedItemNS(String namespaceURI, String localName) throws DOMException
	{ throw new DOMException(DOMException.INVALID_ACCESS_ERR, SVGDocument.ACTION_NOT_IMPLEMENTED); }



	/**
	 * @return the attributes.
	 * @since 0.1
	 */
	public Vector<SVGAttr> getAttributes()
	{
		return nnm;
	}



	@Override
	public boolean equals(Object obj)
	{
		if(obj==null || !(obj instanceof SVGNamedNodeMap))
			return false;
		
		SVGNamedNodeMap map = (SVGNamedNodeMap)obj;
		boolean ok = true;
		int i, size = getLength();
		
		if(size!=map.getLength())
			return false;
		
		for(i=0; i<size && ok; i++)
			ok = item(i).isEqualNode(map.item(i));
		
		return ok;
	}
}
