/**
 * Copyright (c) Members of the EGEE Collaboration. 2004-2010. 
 * See http://www.eu-egee.org/partners/ for details on the copyright
 * holders.  
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); 
 * you may not use this file except in compliance with the License. 
 * You may obtain a copy of the License at 
 * 
 *     http://www.apache.org/licenses/LICENSE-2.0 
 * 
 * Unless required by applicable law or agreed to in writing, software 
 * distributed under the License is distributed on an "AS IS" BASIS, 
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. 
 * See the License for the specific language governing permissions and 
 * limitations under the License.
 *
 *
 *  Authors:
 *  2009-
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     Mischa Sall\'e <msalle@nikhef.nl>
 *     David Groep <davidg@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *     <grid-mw-security@nikhef.nl> 
 *
 *  2007-2009
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     David Groep <davidg@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *
 *  2003-2007
 *     Martijn Steenbakkers <martijn@nikhef.nl>
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     David Groep <davidg@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *
 */


/*!
    \file   _lcas_utils.h
    \brief  Internal header for the LCAS utilities.
    \author Martijn Steenbakkers for the EU DataGrid.

    This header contains the declarations of the LCAS utility functions:
    -# lcas_fill_cred():
    -# lcas_release_cred():
    -# lcas_tokenize():
    \internal
*/

#ifndef _LCAS_UTILS_H
#define _LCAS_UTILS_H

/******************************************************************************
                             Include header files
******************************************************************************/
#ifndef NOGLOBUS
    #include <gssapi.h>
#endif /* NOGLOBUS */
#include "lcas_types.h"
#include "lcas_utils.h"

/******************************************************************************
 *                            Module definition
 *****************************************************************************/

/*! \name CREDENTIAL FUNCTIONS */
/*@{*/ /* Beginning of credential functions */
/******************************************************************************
Function:       lcas_fill_cred()
Description:    Fill cedential from dn and globus credential
Parameters:
                dn: distinguished name
                cred: globus credential
                plcas_cred: pointer to lcas_credential
Returns:        0: succes
                1: failure
******************************************************************************/
/*!
    \fn lcas_fill_cred(
        char * dn,
        gss_cred_id_t cred,
        lcas_cred_id_t * plcas_cred
        )
    \brief Fill cedential from distinghuished name and globus credential.

    The LCAS credential only differs from the GLOBUS credential by the extra
    entry for the dn. This allows (temporarily) the passed delegated GLOBUS credential
    to be empty.

    \param dn          distinguished name
    \param cred        GLOBUS credential
    \param plcas_cred  pointer to LCAS credential to be filled.
    \retval 0 succes.
    \retval 1 failure.
    \internal
*/
extern int lcas_fill_cred(
        char * dn,
#ifndef NOGLOBUS
        gss_cred_id_t cred,
#else
        void * cred,
#endif /* NOGLOBUS */
        lcas_cred_id_t * plcas_cred
);
/******************************************************************************
Function:       lcas_release_cred()
Description:    release lcas credential
Parameters:
                plcas_cred: pointer to lcas_credential
Returns:        0: succes
                1: failure
******************************************************************************/
/*!
    \fn lcas_release_cred(
        lcas_cred_id_t * plcas_cred
        )
    \brief Release the LCAS credential
    \param plcas_cred  pointer to LCAS credential to be released
    \retval 0 succes.
    \retval 1 failure.
    \internal
*/
extern int lcas_release_cred(
        lcas_cred_id_t * plcas_cred
);
/*@}*/ /* End of credential functions */

/******************************************************************************
        OTHER STUFF
******************************************************************************/
/*! \name OTHER FUNCTIONS */
/*@{*/ /* Beginning of other functions */
/******************************************************************************
Function:   lcas_tokenize() (in modified form from globus_gatekeeper_utils.c)

Description:
    Breakup the command in to args, pointing the args array
    at the tokens. Replace white space at the end of each
    token with a null. A token maybe in quotes. 

Parameters:
    command: The command line to be parsed.
    args:    A pointer to an array of pointers to be filled it
    n:       Size of the array, on input, and set to size used on output. 
    sep:     string of separating characters

Returns:
    0 on success. 
    -1 on to malloc
    -2 on to many args
    -3 on quote not matched
******************************************************************************/
/*!
    \fn lcas_tokenize(
        const char * command, 
        char ** args,
        int * n,
        char * sep
        )
    \brief Break the argument string up into tokens

    Breakup the command in to arguments, pointing the args array at the tokens.
    Replace white space at the end of each token with a null. A token maybe in quotes.
    (Copied (and modified) from GLOBUS gatekeeper.c)
    \param command the command line to be parsed
    \param args    pointer to an array of pointers to be filled
    \param n       size of the array, on input, and set to size used on output
    \param sep     string of separating characters
    \retval 0  succes
    \retval -1 malloc error
    \retval -2 too many args
    \retval -3 quote not matched
    \internal
*/
extern int lcas_tokenize(
        const char * command, 
        char ** args,
        int * n,
        char * sep
);
/*@}*/ /* End of other functions */

#endif /* _LCAS_UTILS_H */

/******************************************************************************
CVS Information:
    $Source: /srv/home/dennisvd/svn/mw-security/lcas/src/_lcas_utils.h,v $
    $Date: 2010-05-03 10:42:49 $
    $Revision: 2.7 $
    $Author: okoeroo $
******************************************************************************/
