/*
 * @(#)LineCountMeasureUTest.java
 *
 * Copyright (C) 2003 Matt Albrecht
 * groboclown@users.sourceforge.net
 * http://groboutils.sourceforge.net
 *
 *  Permission is hereby granted, free of charge, to any person obtaining a
 *  copy of this software and associated documentation files (the "Software"),
 *  to deal in the Software without restriction, including without limitation
 *  the rights to use, copy, modify, merge, publish, distribute, sublicense,
 *  and/or sell copies of the Software, and to permit persons to whom the 
 *  Software is furnished to do so, subject to the following conditions:
 *
 *  The above copyright notice and this permission notice shall be included in 
 *  all copies or substantial portions of the Software. 
 *
 *  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR 
 *  IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, 
 *  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL 
 *  THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER 
 *  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING 
 *  FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER 
 *  DEALINGS IN THE SOFTWARE.
 */

package net.sourceforge.groboutils.codecoverage.v2.module;

import java.io.IOException;

import junit.framework.Test;
import junit.framework.TestCase;
import net.sourceforge.groboutils.autodoc.v1.AutoDoc;
import net.sourceforge.groboutils.codecoverage.v2.BytecodeLoaderUtil;
import net.sourceforge.groboutils.codecoverage.v2.CCCreatorUtil;
import net.sourceforge.groboutils.codecoverage.v2.CreateMainClassHelper;
import net.sourceforge.groboutils.codecoverage.v2.IAnalysisModule;
import net.sourceforge.groboutils.codecoverage.v2.IAnalysisModuleUTestI;
import net.sourceforge.groboutils.codecoverage.v2.IMethodCode;
import net.sourceforge.groboutils.codecoverage.v2.compiler.ModifiedClass;
import net.sourceforge.groboutils.codecoverage.v2.compiler.ModifiedMethod;
import net.sourceforge.groboutils.codecoverage.v2.logger.TestLogger;
import net.sourceforge.groboutils.junit.v1.iftc.CxFactory;
import net.sourceforge.groboutils.junit.v1.iftc.InterfaceTestSuite;

import org.apache.bcel.Constants;
import org.apache.bcel.generic.InstructionConstants;
import org.apache.bcel.generic.ObjectType;
import org.apache.bcel.generic.PUSH;
import org.apache.bcel.generic.Type;


/**
 * Tests the LineCountMeasure class.
 *
 * @author    Matt Albrecht <a href="mailto:groboclown@users.sourceforge.net">groboclown@users.sourceforge.net</a>
 * @version   $Date: 2004/04/15 05:48:28 $
 * @since     January 22, 2003
 */
public class LineCountMeasureUTest extends TestCase
{
    //-------------------------------------------------------------------------
    // Standard JUnit Class-specific declarations
    
    private static final Class THIS_CLASS = LineCountMeasureUTest.class;
    private static final AutoDoc DOC = new AutoDoc( THIS_CLASS );
    
    public LineCountMeasureUTest( String name )
    {
        super( name );
    }


    //-------------------------------------------------------------------------
    // Tests
    
    public static class MyTestClass
    {
        public static void main( String[] args )
        {
            int i = 0; i += 10;
            System.out.println( ""+i );
        }
    }
    
    
    public void testAnalyze1() throws Exception
    {
        LineCountMeasure bcm = new LineCountMeasure();
        ModifiedClass mc = CCCreatorUtil.createModifiedClass(
            TestLogger.createPCL(), MyTestClass.class );
        ModifiedMethod mm = CCCreatorUtil.getMainModifiedMethod( mc );
        IMethodCode imc = createMethodCode( MyTestClass.class, mm, bcm );
        
        // populate the method with marks
        bcm.analyze( imc );
        
        // Generate the class from the modified bytecode, and run it.
        Class c = BytecodeLoaderUtil.loadClassFromBytecode(
            mc.getClassName(), mc.getModifiedClass() );
        TestLogger.reset();
        BytecodeLoaderUtil.runMain( c );
        assertEquals(
            "Did not mark every bytecode in method.",
            2,
            TestLogger.size() );
    }
    
    
    public void testAnalyze2() throws Exception
    {
        LineCountMeasure bcm = new LineCountMeasure();
        CreateMainClassHelper cmch = createHelloWorld();
        byte bytecode[] = cmch.getBytecode();
        Class cc = cmch.getGenClass();
        ModifiedClass mc = CCCreatorUtil.createModifiedClass(
            TestLogger.createPCL(), "test/HelloWorld.class",
            bytecode );
        ModifiedMethod mm = CCCreatorUtil.getMainModifiedMethod( mc );
        IMethodCode imc = createMethodCode( cc, mm, bcm );
        
        // populate the method with marks
        bcm.analyze( imc );
        
        // Generate the class from the modified bytecode, and run it.
        Class c = BytecodeLoaderUtil.loadClassFromBytecode(
            mc.getClassName(), mc.getModifiedClass() );
        TestLogger.reset();
        BytecodeLoaderUtil.runMain( c );
        assertEquals(
            "Somehow we found a line number.",
            0,
            TestLogger.size() );
    }
    
    
    
    //-------------------------------------------------------------------------
    // Helpers
    
    
    protected IMethodCode createMethodCode( Class c, ModifiedMethod mm,
            IAnalysisModule am )
    {
        return CCCreatorUtil.createIMethodCode( c, mm,
            CCCreatorUtil.createAnalysisModuleSet(
                new IAnalysisModule[] { am } ),
            0 );
    }
    
    
    protected CreateMainClassHelper createHelloWorld()
            throws IOException
    {
        CreateMainClassHelper cmch = new CreateMainClassHelper(
            "test.HelloWorld" );
        ObjectType p_stream = new ObjectType("java.io.PrintStream");
        cmch.il.append(
            cmch.factory.createFieldAccess("java.lang.System", "out", p_stream,
            Constants.GETSTATIC ) );
        cmch.il.append( new PUSH(cmch.cp, "Hello, world.")) ;
        cmch.il.append( cmch.factory.createInvoke("java.io.PrintStream",
            "println", Type.VOID, new Type[] { Type.STRING },
            Constants.INVOKEVIRTUAL ) );
        cmch.il.append(InstructionConstants.RETURN);
        
        cmch.close();
        return cmch;
    }
    
    
    //-------------------------------------------------------------------------
    // Standard JUnit declarations
    
    
    public static Test suite()
    {
        InterfaceTestSuite suite = IAnalysisModuleUTestI.suite();
        suite.addTestSuite( THIS_CLASS );
        suite.addFactory( new CxFactory( "A" ) {
            public Object createImplObject() throws IOException {
                return new LineCountMeasure();
            }
        } );
        
        return suite;
    }
    
    public static void main( String[] args )
    {
        String[] name = { THIS_CLASS.getName() };
        
        // junit.textui.TestRunner.main( name );
        // junit.swingui.TestRunner.main( name );
        
        junit.textui.TestRunner.main( name );
    }
    
    
    /**
     * 
     * @exception Exception thrown under any exceptional condition.
     */
    protected void setUp() throws Exception
    {
        super.setUp();

       
        // set ourself up
    }
    
    
    /**
     * 
     * @exception Exception thrown under any exceptional condition.
     */
    protected void tearDown() throws Exception
    {
        // tear ourself down
        
        
        super.tearDown();
    }
}

