#!/usr/bin/python
# -*-coding: UTF-8 -*-
# bbou@ac-toulouse.fr
# GPL license
# 17.11.2006 
# share.py

import pygtk
pygtk.require("2.0")
from gtk import *
import gtk.glade
import string
import re
import stat
import time

import sharesparser
import runner
import acls

#######################################################################
#	GLOBALS
#######################################################################

sysSetting={
	'connect':"smbclient //%SERVER%/%SHARE% -U %USER%%%PASSWORD% -c 'exit' 2> /dev/null",
	'connectRead':"smbclient //%SERVER%/%SHARE% -U %USER%%%PASSWORD% -c 'get ._sadms_' 2> /dev/null",
	'connectWrite':"smbclient //%SERVER%/%SHARE% -U %USER%%%PASSWORD% -c 'put _sadms_ ._sadms_%USER%_%STAMP%' 2> /dev/null",
	'connectDel':"smbclient //%SERVER%/%SHARE% -U %USER%%%PASSWORD% -c 'del ._sadms_%USER%*' 2> /dev/null",
	'getServer':"testparm -sv 2> /dev/null | grep 'netbios name' | awk '{print $4}'",
	'getSep':"testparm -sv 2> /dev/null | grep '^[[:space:]]*winbind separator[[:space:]]*=' | awk '{print $4}'",
	'getDomain':"testparm -sv 2> /dev/null | grep '^[[:space:]]workgroup[[:space:]]*=' | awk '{print $3}'",
	'getUidRange':"testparm -sv 2> /dev/null | grep '^[[:space:]]idmap uid[[:space:]]*=' | awk '{print $4}'",
	'getGidRange':"testparm -sv 2> /dev/null | grep '^[[:space:]]idmap gid[[:space:]]*=' | awk '{print $4}'",
	'docs':{'default':'file:///usr/share/doc/sadms-%s/%s',
		'redhat':'file:///usr/share/doc/sadms-%s/%s',
		'debian':'file:///usr/share/doc/sadms/%s',
		'suse':'file:///usr/share/doc/packages/sadms/%s',
		'mandriva':'file:///usr/share/doc/sadms-%s/%s'},
	'browser':{'default':'/usr/bin/firefox',
		'redhat':'/usr/bin/firefox',
		'debian':'/usr/bin/firefox',
		'suse':'/usr/bin/firefox',
		'mandriva':'/usr/bin/mozilla-firefox'},

	'getUsers':"getent passwd | grep -v '^$' | cut -f 1 -d ':' | sort",
	'getGroups':"getent group | grep -v '^$' | cut -f 1 -d ':' | sort",
	'signalSmb':'kill -HUP `pidof smbd` 2> /dev/null',
	'signalNmb':'kill -HUP `pidof nmbd` 2> /dev/null',
	'getCurrentUserId':"id -un 2> /dev/null",
	'getCurrentGroupId':"id -gn 2> /dev/null",
	'getUserId':"id -u \"%USER%\" 2> /dev/null",
	'getGroupId':"getent group \"%GROUP%\" 2> /dev/null | awk 'BEGIN{FS=\":\"}{print $3}'",
	'getOwnerGroup':"stat -c '%U:%G' \"%FSOBJECT%\"",
	'getDirMode':"ls -dl \"%FSOBJECT%\" | cut -f 1 -d ' '",
	'readFile':'cat "%FILE%"',
	'writeFile':'cat - > "%FILE%" <<EOF\n%TEXT%\nEOF',
	'getEnforcingSE':'sh -c \'if [ type getenforce 2> /dev/null && [ "`getenforce`" == "Enforcing" ]; then echo true; else echo false; fi\'',
	'exists':"sh -c \'if [ -e \"%FSOBJECT%\" ]; then echo true; else echo false; fi\'",
	'getVersion':"cat version | head -n 1",
	'getDistribution':'sh -c \'if [ -f /etc/debian_version ]; then echo "debian"; else if [ -f /etc/SuSE-release ]; then echo "suse"; else if [ -e /etc/mandrake-release -o -e /etc/mandriva-release ]; then echo "mandriva"; else if [ -e /etc/redhat-release ]; then echo "redhat"; else echo "default"; fi; fi; fi; fi\'',
}

#######################################################################
#	SHARE
#######################################################################

class ListView:
	
	USER=0
	GROUP=1

	IMAGE=0
	TEXT=1

	pixbufs=[]

	def __init__(self,listview,mode,subjecttype,source):

		self.listview=listview
		self.subjecttype=subjecttype
		self.imageOffset=0
		if source:
			self.imageOffset=2

		# header
		self.listview.set_headers_visible(False)

		# selection
		self.listview.get_selection().set_mode(mode)

		# model
		self.model=ListStore(gtk.gdk.Pixbuf,str)
		self.listview.set_model(self.model)

		# columns
		column=gtk.TreeViewColumn('')
		cell=gtk.CellRendererPixbuf()
		column.pack_start(cell,True)
		column.set_cell_data_func(cell,self.render)
		column.add_attribute(cell,'pixbuf',ListView.IMAGE)
		self.listview.append_column(column)

		column=gtk.TreeViewColumn('subject')
		cell=gtk.CellRendererText()
		column.pack_start(cell,True)
		column.set_cell_data_func(cell,self.render)
		column.add_attribute(cell,'text',ListView.TEXT)
		self.listview.append_column(column)
		
		# images
		if ListView.pixbufs==[]:
			ListView.pixbufs=self.setupImages()
		return

	def setupImages(self):
		imageFile=['user.png','group.png','user-hidden.png','group-hidden.png']
		pixbufs=[]
		for i in range(len(imageFile)):
        		pixbufs.append(gtk.gdk.pixbuf_new_from_file('pixmaps/'+imageFile[i]))
		return pixbufs

	# render callback
	def render(self,column,cell,model,i):
		r=model.get_path(i)
		toggle=r[0] % 2 == 0
		if toggle:
			cell.set_property('cell-background','lightGray')
		else:
			cell.set_property('cell-background','white')
		return

	def find(self,subject):
		for s in self.model:
			if s[ListView.TEXT]==subject:
				return True
		return False
	
	def clear(self):
		self.model.clear()
		return
	
	def set(self,subjects):
		for s in subjects:
			s=s.lower()
			self.model.append([ListView.pixbufs[self.subjecttype+self.imageOffset],s])
		return
	
	def put(self,subjects):
		for s in subjects:
			s=s.lower()
			if not self.find(s):
				self.model.append([ListView.pixbufs[self.subjecttype+self.imageOffset],s])
		return
	
	def putPrefix(self,s,isUser,prefix,inDomain):
		if s.startswith('builtin/'):
			return s
		if inDomain!=None:
			isInDomain=inDomain(s,isUser)
			if not isInDomain:
				return s
		return '%s%s' % (prefix,s)
	
	def get(self,prefix=None,indomain=None):
		subjects=[s[ListView.TEXT] for s in self.model]
		if prefix!=None:
			subjects=[self.putPrefix(s,self.subjecttype==ListView.USER,prefix,indomain) for s in subjects]
		return subjects
	
	def getSelection(self):
		selection=self.listview.get_selection()
		(model,pathlist)=selection.get_selected_rows()
		return [model.get_value(model.get_iter(path),ListView.TEXT) for path in pathlist]
	
	def select(self,subject):
		selection=self.listview.get_selection()
		i=self.model.get_iter_first()
		while i:
			if self.model.get_value(i,ListView.TEXT)==subject:
				selection.select_iter(i)
			i=self.model.iter_next(i)
		return

	def remove(self):
		selection=self.listview.get_selection()
		(model,pathlist)=selection.get_selected_rows()
		pathlist.reverse()
		for path in pathlist:
                        model.remove(model.get_iter(path))
		return
	
class ForceListView(ListView):
	
	XTRA=2

	def __init__(self,listview,mode,subjecttype,source,label):
		ListView.__init__(self,listview,mode,subjecttype,source)

		# model
		self.model=ListStore(gtk.gdk.Pixbuf,str,bool)
		self.listview.set_model(self.model)

		# header
		self.listview.set_headers_visible(True)

		# extra column
		column=gtk.TreeViewColumn(label)
		cell=gtk.CellRendererToggle()
		cell.set_property('activatable',True)
		cell.connect('toggled',self.toggle_acl_callback,(ForceListView.XTRA))
		column.pack_start(cell,False)
		column.set_cell_data_func(cell,self.render)
		column.add_attribute(cell,'active',ForceListView.XTRA)
		self.listview.append_column(column)
		return

	# toggle callback
	def toggle_acl_callback(self,cell,r,(c)):
		#print "path=%s,%s" % (r,c)
		self.model[r][c]=not self.model[r][c]
		return       

	def set(self,subjects):
		for s in subjects:
			forced=s[0]=='+'
			if forced:
				s=s[1:]
			self.model.append([ListView.pixbufs[self.subjecttype+self.imageOffset],s,forced])
		return

	def put(self,subjects):
		for s in subjects:
			forced=s[0]=='+'
			if forced:
				s=s[1:]
			if not self.find(s):
				self.model.append([ListView.pixbufs[self.subjecttype+self.imageOffset],s,forced])
		return

	def get(self,prefix=None,inDomain=None):
		tag={True:'+',False:''}
		if prefix!=None:
			subjects=['%s%s' % (tag[s[ForceListView.XTRA]],putPrefix(s[ListView.TEXT],self.subjecttype==ListView.USER,prefix,inDomain)) for s in self.model]
		else:
			subjects=['%s%s' % (tag[s[ForceListView.XTRA]],s[ListView.TEXT]) for s in self.model]			
		return subjects
	
class ListViews:
	
	def __init__(self,listview,listview0,mode,subjecttype,extended=False,label='xtra'):
		self.listview0=ListView(listview0,mode,subjecttype,True)
		if extended:
			self.listview=ForceListView(listview,mode,subjecttype,False,label)
		else:
			self.listview=ListView(listview,mode,subjecttype,False)
		return

	def get(self,prefix=None,indomain=None):
		return self.listview.get(prefix,indomain)
	
	def populate(self,subjects):
		self.listview0.set(subjects)
		return

	def put(self,subjects):
		self.listview.put(subjects)
		return

	def add(self):
		subjects=self.listview0.getSelection()
		if subjects==[]:
			return
		self.listview.put(subjects)
		return

	def addOne(self):
		subjects=self.listview0.getSelection()
		if subjects==[]:
			return
		self.listview.clear()
		self.listview.put(subjects)
		return

	def remove(self):
		self.listview.remove()
		return

class BatchConnectView:

	NONEPIX=0
	OKPIX=1
	ERRORPIX=2
	USERPIX=3

	# column info = rank, type, header, expand
	IMAGE=0
	USER=1
	PASSWORD=2
	RESULTCONNECT=3
	RESULTREAD=4
	RESULTWRITE=5
	OUTPUTCONNECT=6
	OUTPUTWRITE=7
	OUTPUTREAD=8

	columns=(
		[IMAGE,gtk.gdk.Pixbuf,'',False],
		[USER,str,'user',True],
		[PASSWORD,str,'password',True],
		[RESULTCONNECT,gtk.gdk.Pixbuf,'c',False],
		[RESULTREAD,gtk.gdk.Pixbuf,'r',False],
		[RESULTWRITE,gtk.gdk.Pixbuf,'w',False],
		[OUTPUTCONNECT,str,'connect',True],
		[OUTPUTREAD,str,'read',True],
		[OUTPUTWRITE,str,'write',True])

	pixbufs=[]

        def __init__(self,listview):

		# widget from glade
		self.listview=listview

		# selection
		self.listview.get_selection().set_mode(gtk.SELECTION_MULTIPLE)

		# model
		types=[t[1] for t in BatchConnectView.columns]
		self.model=gtk.ListStore(*types)
		self.listview.set_model(self.model)		
		
		# columns
		self.setupColumns()	

		# images
		if BatchConnectView.pixbufs==[]:
			BatchConnectView.pixbufs=self.setupImages()
                return
        
	def setupColumns(self):	
		types=[t[1] for t in BatchConnectView.columns]
		for rank,datatype,header,expand in BatchConnectView.columns:
			column=gtk.TreeViewColumn(header)
			if datatype==str:
				cell=gtk.CellRendererText()
				attr='text'
			elif datatype==bool:
				cell=gtk.CellRendererToggle()
				attr='active'
			elif datatype==gtk.gdk.Pixbuf:
				cell=gtk.CellRendererPixbuf()
				attr='pixbuf'
			else:
				cell=gtk.CellRendererText()
				attr='text'	
			column.pack_start(cell,expand)
			column.add_attribute(cell,attr,rank)
			column.set_cell_data_func(cell,self.render)
			self.listview.append_column(column)
		return

	def setupImages(self):
		imageFile=['grey.png','green.png','red.png','user.png']
		pixbufs=[]
		for i in range(len(imageFile)):
        		pixbufs.append(gtk.gdk.pixbuf_new_from_file('pixmaps/'+imageFile[i]))
		return pixbufs

	# render callback
	def render(self,column,cell,model,i):
		r=model.get_path(i)
		toggle=r[0] % 2 == 0
		if toggle:
			cell.set_property('cell-background','lightGray')
		else:
			cell.set_property('cell-background','white')
		return

	def clear(self):
		self.model.clear()

	def add(self,user,password):
		data=[self.pixbufs[BatchConnectView.USERPIX],user,password,BatchConnectView.pixbufs[BatchConnectView.NONEPIX],BatchConnectView.pixbufs[BatchConnectView.NONEPIX],BatchConnectView.pixbufs[BatchConnectView.NONEPIX],'','','']
		self.model.append(data)
		return

	def setConnect(self,i,result,output):
		imageIndex=BatchConnectView.ERRORPIX
		if result:
			imageIndex=BatchConnectView.OKPIX
		self.model[i][BatchConnectView.RESULTCONNECT]=BatchConnectView.pixbufs[imageIndex]
		self.model[i][BatchConnectView.OUTPUTCONNECT]=output
		return

	def setRead(self,i,result,output):
		imageIndex=BatchConnectView.ERRORPIX
		if result:
			imageIndex=BatchConnectView.OKPIX
		self.model[i][BatchConnectView.RESULTREAD]=BatchConnectView.pixbufs[imageIndex]
		self.model[i][BatchConnectView.OUTPUTREAD]=output
		return

	def setWrite(self,i,result,output):
		imageIndex=BatchConnectView.ERRORPIX
		if result:
			imageIndex=BatchConnectView.OKPIX
		self.model[i][BatchConnectView.RESULTWRITE]=BatchConnectView.pixbufs[imageIndex]
		self.model[i][BatchConnectView.OUTPUTWRITE]=output
		return

	def reset(self,i):
		for j in (BatchConnectView.RESULTCONNECT,BatchConnectView.RESULTREAD,BatchConnectView.RESULTWRITE):
			self.model[i][j]=BatchConnectView.pixbufs[BatchConnectView.NONEPIX]
		for j in (BatchConnectView.OUTPUTCONNECT,BatchConnectView.OUTPUTREAD,BatchConnectView.OUTPUTWRITE):
			self.model[i][j]=''
		return

#######################################################################
#	SHAREMANAGER
#######################################################################
	
class ShareManager:

	NONE=0
	OK=1
	ERROR=2

	VALID=2
	READ=3
	WRITE=4
	ADMIN=5
	FORCE=6
	FS=7

	pixbufs=[]

	def __init__(self,parent=None):
		self.parent=parent
		handlers={
			'on_destroy':			self.exit,
			'on_about_activate':		self.about,
			'on_ok_clicked':		self.exitOk,
			'on_cancel_clicked':		self.exit,
			'on_switch_page':		self.switchPage,
			'on_apply_clicked':		self.applyShare,
			'on_refresh_clicked':		self.forceRefresh,
			'on_connect_clicked':		self.connect,
			'on_batchConnect_clicked':	self.batchConnect,
			'on_closeBatchConnect_clicked':	self.closeBatchConnect,
			'on_loadBatchConnect_clicked':	self.loadBatchConnect,
			'on_runBatchConnect_clicked':	self.runBatchConnect,
			'on_resetBatchConnect_clicked':	self.resetBatchConnect,
			'on_share_browse_clicked':	self.browseFolder,
			'on_text_clicked':		self.getText,
			'on_help_clicked':		self.help,
			'on_validUsersAdd_clicked':	self.addValidUsers,
			'on_validUsersRemove_clicked':	self.removeValidUsers,
			'on_readUsersAdd_clicked':	self.addReadUsers,
			'on_readUsersRemove_clicked':	self.removeReadUsers,
			'on_writeUsersAdd_clicked':	self.addWriteUsers,
			'on_writeUsersRemove_clicked':	self.removeWriteUsers,
			'on_adminUsersAdd_clicked':	self.addAdminUsers,
			'on_adminUsersRemove_clicked':	self.removeAdminUsers,
			'on_forceUserAdd_clicked':	self.addForceUser,
			'on_forceUserRemove_clicked':	self.removeForceUser,
			'on_fsOwnerAdd_clicked':	self.addFsOwner,
			'on_fsOwnerRemove_clicked':	self.removeFsOwner,
			'on_validGroupsAdd_clicked':	self.addValidGroups,
			'on_validGroupsRemove_clicked':	self.removeValidGroups,
			'on_readGroupsAdd_clicked':	self.addReadGroups,
			'on_readGroupsRemove_clicked':	self.removeReadGroups,
			'on_writeGroupsAdd_clicked':	self.addWriteGroups,
			'on_writeGroupsRemove_clicked':	self.removeWriteGroups,
			'on_adminGroupsAdd_clicked':	self.addAdminGroups,
			'on_adminGroupsRemove_clicked':	self.removeAdminGroups,
			'on_forceGroupAdd_clicked':	self.addForceGroup,
			'on_forceGroupRemove_clicked':	self.removeForceGroup,
			'on_fsGroupAdd_clicked':	self.addFsGroup,
			'on_fsGroupRemove_clicked':	self.removeFsGroup,
			'on_acls_clicked':		self.acls,
			'on_share_path_changed':	self.pathChanged,
		}

		self.runner=runner.GtkRunner();
		self.uidRange=[]
		self.gidRange=[]

		self.widgets=gtk.glade.XML('share.glade')
		self.widgets.signal_autoconnect(handlers)
		self.dialog=self['share']
		self.authDialog=self['authDialog']
		self.batchConnectDialog=self['batchConnectDialog']
		self.aboutDialog=self['aboutDialog']

		self.shareName=self['shareName']
		self.sharePath=self['sharePath']
		self.shareComment=self['shareComment']
		self.shareReadOnly=self['readOnlyCheck']
		self.shareBrowseable=self['browseableCheck']
		self.shareGuestOk=self['guestOkCheck']
		self.fsPath=self['fsPath']
		self.fsStatusImage=self['fsStatusImage']
		self.allowAll=self['allowAllRadio']
		self.allowOnly=self['allowOnlyRadio']
		self.doNotWriteEmpty=self['doNotWriteEmptyRadio']
		self.writeEmpty=self['writeEmptyRadio']
		self.createMaskCheck=self['createMaskCheck']
		self.directoryMaskCheck=self['directoryMaskCheck']

		self.validUsers=ListViews(self['validUsers'],self['validUsers0'],gtk.SELECTION_MULTIPLE,ListView.USER)
		self.validGroups=ListViews(self['validGroups'],self['validGroups0'],gtk.SELECTION_MULTIPLE,ListView.GROUP)
		self.readUsers=ListViews(self['readUsers'],self['readUsers0'],gtk.SELECTION_MULTIPLE,ListView.USER)
		self.readGroups=ListViews(self['readGroups'],self['readGroups0'],gtk.SELECTION_MULTIPLE,ListView.GROUP)
		self.writeUsers=ListViews(self['writeUsers'],self['writeUsers0'],gtk.SELECTION_MULTIPLE,ListView.USER)
		self.writeGroups=ListViews(self['writeGroups'],self['writeGroups0'],gtk.SELECTION_MULTIPLE,ListView.GROUP)
		self.adminUsers=ListViews(self['adminUsers'],self['adminUsers0'],gtk.SELECTION_MULTIPLE,ListView.USER)
		self.adminGroups=ListViews(self['adminGroups'],self['adminGroups0'],gtk.SELECTION_MULTIPLE,ListView.GROUP)
		self.forceUser=ListViews(self['forceUser'],self['forceUser0'],gtk.SELECTION_SINGLE,ListView.USER)
		self.forceGroup=ListViews(self['forceGroup'],self['forceGroup0'],gtk.SELECTION_SINGLE,ListView.GROUP,extended=True,label='force if member')
		self.fsOwner=ListViews(self['fsOwner'],self['fsOwner0'],gtk.SELECTION_SINGLE,ListView.USER)
		self.fsGroup=ListViews(self['fsGroup'],self['fsGroup0'],gtk.SELECTION_SINGLE,ListView.GROUP)

		self.fsOwnerRead=self['fsOwnerReadCheck']
		self.fsOwnerWrite=self['fsOwnerWriteCheck']
		self.fsOwnerExec=self['fsOwnerExecCheck']
		self.fsGroupRead=self['fsGroupReadCheck']
		self.fsGroupWrite=self['fsGroupWriteCheck']
		self.fsGroupExec=self['fsGroupExecCheck']
		self.fsOtherRead=self['fsOtherReadCheck']
		self.fsOtherWrite=self['fsOtherWriteCheck']
		self.fsOtherExec=self['fsOtherExecCheck']

		self.batchConnectView=BatchConnectView(self['batchConnectView'])
		
		self.populate()

		# images
		if ShareManager.pixbufs==[]:
			ShareManager.pixbufs=self.setupImages()

		# index
		self.viewMap={
			ShareManager.VALID:[False,self.validUsers,self.validGroups],
			ShareManager.READ:[False,self.readUsers,self.readGroups],
			ShareManager.WRITE:[False,self.writeUsers,self.writeGroups],
			ShareManager.ADMIN:[False,self.adminUsers,self.adminGroups],
			ShareManager.FORCE:[False,self.forceUser,self.forceGroup],
			ShareManager.FS:[False,self.fsOwner,self.fsGroup]
		}
		return

	# widget access

	def __getitem__(self, key):
        	return self.widgets.get_widget(key)

	# images

	def setupImages(self):
		imageFile=['grey.png','green.png','red.png']
		pixbufs=[]
		for i in range(len(imageFile)):
        		pixbufs.append(gtk.gdk.pixbuf_new_from_file('pixmaps/'+imageFile[i]))
		return pixbufs

	# termination
	
	def exit(self,*options):
		if __name__ == "__main__":
			gtk.main_quit()
		else:
			self.dialog.destroy()	
		return
				
	def exitOk(self,*options):
		self.applyShare()
		self.exit()	
		return
			
	# switch page callback
	
	def switchPage(self,notebook,page,index,*options):
		#oldIndex=notebook.get_current_page()
		#oldPage=notebook.get_nth_page(oldIndex)
		if index>=2:
			loaded,userview,groupview=self.viewMap[index]
			if not loaded:
				userview.populate(self.users)
				groupview.populate(self.groups)
				self.viewMap[index][0]=True
		return

	# path change callback
	
	def pathChanged(self,w,*options):
		fsobject=w.get_text()
		#print fsobject
		if self.exists(fsobject):
			self.fsStatusImage.set_from_pixbuf(ShareManager.pixbufs[ShareManager.OK])

			# path
			self.fsPath.set_markup('<b>%s</b>' % fsobject)
			# owner
			owner,group=self.getOwnerGroup(fsobject)
			if owner!=None:
				self.fsOwner.put([owner])
			if group!=None:
				self.fsGroup.put([group])
			# perms
			ur,uw,ux,gr,gw,gx,wr,ww,wx=self.getMode(fsobject)
			self.fsOwnerRead.set_active(ur)
			self.fsOwnerWrite.set_active(uw)
			self.fsOwnerExec.set_active(ux)
			self.fsGroupRead.set_active(gr)
			self.fsGroupWrite.set_active(gw)
			self.fsGroupExec.set_active(gx)
			self.fsOtherRead.set_active(wr)
			self.fsOtherWrite.set_active(ww)
			self.fsOtherExec.set_active(wx)
		else:
			self.fsStatusImage.set_from_pixbuf(ShareManager.pixbufs[ShareManager.ERROR])

			# path
			self.fsPath.set_markup("<b><span foreground='#FF0000'>non-existent path</span></b>")
			# default owner
			user=self.runToString(sysSetting['getCurrentUserId'])[1].strip('\n\r ')
			group=self.runToString(sysSetting['getCurrentGroupId'])[1].strip('\n\r ')
			self.fsOwner.put([user])
			self.fsGroup.put([group])
			# default perms
			self.fsOwnerRead.set_active(True)
			self.fsOwnerWrite.set_active(True)
			self.fsOwnerExec.set_active(True)
			self.fsGroupRead.set_active(True)
			self.fsGroupWrite.set_active(False)
			self.fsGroupExec.set_active(True)
			self.fsOtherRead.set_active(True)
			self.fsOtherWrite.set_active(False)
			self.fsOtherExec.set_active(True)
		return

	# populate with users/groups

	def populate(self):
		prefix='%s%s' % (self.getDomain(),self.getSep())

		# users
		users=self.getUsers()
		self.users=[u.replace(prefix,'',1) for u in users]

		# groups
		groups=self.getGroups()
		self.groups=[g.replace(prefix,'',1) for g in groups]

		#for l in [self.validUsers,self.readUsers,self.writeUsers,self.forceUser,self.adminUsers,self.fsOwner]:
		#	l.populate(users)
		#for l in [self.validGroups,self.readGroups,self.writeGroups,self.forceGroup,self.adminGroups,self.fsGroup]:
		#	l.populate(groups)
		return	

	# put

	def put(self,share):

		# name
		self.shareName.set_text(share.name)
		# path, puts file system data via callback
		if share.props.has_key('path'):
			self.sharePath.set_text(share.props['path'])
		else:
			self.sharePath.set_text('<path>')
		# other
		if share.props.has_key('comment'):
			self.shareComment.set_text(share.props['comment'])
		if share.props.has_key('read only'):
			self.shareReadOnly.set_active(share.props['read only']=='Yes')
		else:
			self.shareReadOnly.set_active(True)
		if share.props.has_key('browseable'):
			self.shareBrowseable.set_active(share.props['browseable']=='Yes')
		else:
			self.shareBrowseable.set_active(True)
		if share.props.has_key('guest ok'):
			self.shareGuestOk.set_active(share.props['guest ok']=='Yes')
		else:
			self.shareGuestOk.set_active(False)

		# masks
		if share.props.has_key('create mask'):
			self.createMaskCheck.set_active(True)
			self.putCreateMask(share.props['create mask'])
		else:
			self.putCreateMask('0744')
			
		if share.props.has_key('directory mask'):
			self.directoryMaskCheck.set_active(True)
			self.putDirectoryMask(share.props['directory mask'])
		else:
			self.putDirectoryMask('0755')

		# users
		sep=self.getSep()
		domain=self.getDomain()
		prefix='%s%s' % (domain,sep)

		hasPerms=False
		for ul,gl,tag in [(self.validUsers,self.validGroups,'valid users'),
				(self.readUsers,self.readGroups,'read list'),
				(self.writeUsers,self.writeGroups,'write list'),
				(self.adminUsers,self.adminGroups,'admin users')]:
			ul.listview.clear()
			gl.listview.clear()
			if share.props.has_key(tag):
				users,groups=self.splitUsersGroups(share.props[tag])
				if users!=[]:
					users=[u.replace(prefix,'',1) for u in users]
					ul.put(users)
					hasPerms=True
				if groups!=[]:
					groups=[g.replace(prefix,'',1) for g in groups]
					gl.put(groups)
					hasPerms=True
		for ul,tag in [(self.forceUser,'force user')]:
			ul.listview.clear()
			if share.props.has_key(tag):
				users=self.splitSubjects(share.props[tag])
				if users!=[]:
					ul.put(users)
					hasPerms=True
		for gl,tag in [(self.forceGroup,'force group')]:
			gl.listview.clear()
			if share.props.has_key(tag):
				groups=self.splitSubjects(share.props[tag])
				if groups!=[]:
					gl.put(groups)
					hasPerms=True
		self.allowOnly.set_active(hasPerms)
		return	

	def putCreateMask(self,mask):
		mask=int(mask,8)
		ur,uw,ux,gr,gw,gx,wr,ww,wx=self.intToMode(mask)
		self['createMaskURCheck'].set_active(ur)
		self['createMaskUWCheck'].set_active(uw)
		self['createMaskUXCheck'].set_active(ux)
		self['createMaskGRCheck'].set_active(gr)
		self['createMaskGWCheck'].set_active(gw)
		self['createMaskGXCheck'].set_active(gx)
		self['createMaskWRCheck'].set_active(wr)
		self['createMaskWWCheck'].set_active(ww)
		self['createMaskWXCheck'].set_active(wx)
		return

	def putDirectoryMask(self,mask):
		mask=int(mask,8)
		ur,uw,ux,gr,gw,gx,wr,ww,wx=self.intToMode(mask)
		self['directoryMaskURCheck'].set_active(ur)
		self['directoryMaskUWCheck'].set_active(uw)
		self['directoryMaskUXCheck'].set_active(ux)
		self['directoryMaskGRCheck'].set_active(gr)
		self['directoryMaskGWCheck'].set_active(gw)
		self['directoryMaskGXCheck'].set_active(gx)
		self['directoryMaskWRCheck'].set_active(wr)
		self['directoryMaskWWCheck'].set_active(ww)
		self['directoryMaskWXCheck'].set_active(wx)
		return

	def intToMode(self,m):
		#print oct(m)
		ur=m & stat.S_IRUSR!=0
		uw=m & stat.S_IWUSR!=0
		ux=m & stat.S_IXUSR!=0
		gr=m & stat.S_IRGRP!=0
		gw=m & stat.S_IWGRP!=0
		gx=m & stat.S_IXGRP!=0
		wr=m & stat.S_IROTH!=0
		ww=m & stat.S_IWOTH!=0
		wx=m & stat.S_IXOTH!=0
		return ur,uw,ux,gr,gw,gx,wr,ww,wx

	# get

	def get(self):
		name=self.shareName.get_text()
		path=self.sharePath.get_text()
		if name=='' or path=='':
			return None,None,None
		shareData=self.getShareData(name,path)		
		fscommands=self.getFsCommands(path)
		return name,shareData,fscommands

	def getShareData(self,name,path):
		if name=='' or path=='':
			return None
		# empty switch
		writeEmpty=self.writeEmpty.get_active()

		# init
		toYesNo={True:'Yes',False:'No'}
		shareData=[]

		# header
		shareData.append('[%s]' % (name))

		# path, handle home special case
		if path!='<home>':
			shareData.append('\tpath = %s' % (path))
		# others
		shareData.append('\tcomment = %s' % (self.shareComment.get_text()))
		shareData.append('\tread only = %s' % (toYesNo[self.shareReadOnly.get_active()]))
		shareData.append('\tbrowseable = %s' % (toYesNo[self.shareBrowseable.get_active()]))
		shareData.append('\tguest ok = %s' % (toYesNo[self.shareGuestOk.get_active()]))

		# masks
		if self.createMaskCheck.get_active():
			shareData.append('\tcreate mask = %s' % self.getCreateMask())
		if self.directoryMaskCheck.get_active():
			shareData.append('\tdirectory mask = %s' % self.getDirectoryMask())
			
		# users
		sep=self.getSep()
		domain=self.getDomain()
		prefix='%s%s' % (domain,sep)
		if self.allowOnly.get_active():
			for ul,gl,tag in [(self.validUsers,self.validGroups,'valid users'),
					(self.readUsers,self.readGroups,'read list'),
					(self.writeUsers,self.writeGroups,'write list'),
					(self.adminUsers,self.adminGroups,'admin users')]:
				users=ul.get(prefix,self.inDomain)
				users=['"%s"' % u for u in users]
				groups=gl.get(prefix,self.inDomain)
				groups=['"@%s"' % g for g in groups]
				subjects=[]
				subjects.extend(users)
				subjects.extend(groups)
				if writeEmpty or subjects!=[]:
					subjects=','.join(subjects)
					#print '%s = %s' %( tag,subjects)
					shareData.append('\t'+tag+' = '+subjects)
			for ul,tag in [(self.forceUser,'force user')]:
				users=ul.get()
				users=['"'+u+'"' for u in users]
				subjects=users
				if writeEmpty or subjects!=[]:
					subjects=','.join(subjects)
					#print '%s = %s' %( tag,subjects)
					shareData.append('\t'+tag+' = '+subjects+'')
			for gl,tag in [(self.forceGroup,'force group')]:
				groups=gl.get()
				groups=['"'+g+'"' for g in groups]
				subjects=groups
				if writeEmpty or subjects!=[]:
					subjects=','.join(subjects)
					#print '%s = %s' %( tag,subjects)
					shareData.append('\t'+tag+' = '+subjects+'')
		return shareData		

	def getFsCommands(self,path):
		if path=='':
			return []
		if path=='<home>':
			return []
		# gather data
		ur=self.fsOwnerRead.get_active()
		uw=self.fsOwnerWrite.get_active()
		ux=self.fsOwnerExec.get_active()
		gr=self.fsGroupRead.get_active()
		gw=self.fsGroupWrite.get_active()
		gx=self.fsGroupExec.get_active()
		wr=self.fsOtherRead.get_active()
		ww=self.fsOtherWrite.get_active()
		wx=self.fsOtherExec.get_active()
		toR={True:'r',False:''}
		toW={True:'w',False:''}
		toX={True:'x',False:''}
		r=['r' for i in [ur,gr,wr] if i]

		# build
		fsCommands=[]
		if not os.path.isdir(path):
			fsCommands.append('mkdir -p "%s"' % path)
		owners=self.fsGroup.get()
		if owners!=[]:
			fsCommands.append('chown "%s" "%s"' % (self.fsOwner.get()[0],path))
		groups=self.fsGroup.get()
		if groups!=[]:
			fsCommands.append('chgrp "%s" "%s"' % (self.fsGroup.get()[0],path))
		fsCommands.append('chmod u=%s%s%s,g=%s%s%s,o=%s%s%s "%s"' % (toR[ur],toW[uw],toX[ux],toR[gr],toW[gw],toX[gx],toR[wr],toW[ww],toX[wx],path))
		fsCommands.append('touch "%s/._sadms_"' % path)		
		fsCommands.append('chmod ugo=r "%s/._sadms_"' % path)		
		return fsCommands

	def getCreateMask(self):
		ur=self['createMaskURCheck'].get_active()
		uw=self['createMaskUWCheck'].get_active()
		ux=self['createMaskUXCheck'].get_active()
		gr=self['createMaskGRCheck'].get_active()
		gw=self['createMaskGWCheck'].get_active()
		gx=self['createMaskGXCheck'].get_active()
		wr=self['createMaskWRCheck'].get_active()
		ww=self['createMaskWWCheck'].get_active()
		wx=self['createMaskWXCheck'].get_active()
		mode=self.modeToInt(ur,uw,ux,gr,gw,gx,wr,ww,wx)
		return oct(mode)

	def getDirectoryMask(self):
		ur=self['directoryMaskURCheck'].get_active()
		uw=self['directoryMaskUWCheck'].get_active()
		ux=self['directoryMaskUXCheck'].get_active()
		gr=self['directoryMaskGRCheck'].get_active()
		gw=self['directoryMaskGWCheck'].get_active()
		gx=self['directoryMaskGXCheck'].get_active()
		wr=self['directoryMaskWRCheck'].get_active()
		ww=self['directoryMaskWWCheck'].get_active()
		wx=self['directoryMaskWXCheck'].get_active()
		mode=self.modeToInt(ur,uw,ux,gr,gw,gx,wr,ww,wx)
		return oct(mode)

	def modeToInt(self,ur,uw,ux,gr,gw,gx,wr,ww,wx):
		mode=0
		if ur:
			mode=mode | stat.S_IRUSR
		if uw:
			mode=mode | stat.S_IWUSR
		if ux:
			mode=mode | stat.S_IXUSR
		if gr:
			mode=mode | stat.S_IRGRP
		if gw:
			mode=mode | stat.S_IWGRP
		if gx:
			mode=mode | stat.S_IXGRP
		if wr:
			mode=mode | stat.S_IROTH
		if ww:
			mode=mode | stat.S_IWOTH
		if wx:
			mode=mode | stat.S_IXOTH
		return mode

	def set(self,i,result,output):
		imageIndex=BatchConnectView.ERRORPIX
		if result:
			imageIndex=BatchConnectView.OKPIX
		self.model[i][BatchConnectView.RESULTCONNECT]=BatchConnectView.pixbufs[imageIndex]
		self.model[i][BatchConnectView.OUTPUTCONNECT]=output
		return


	# apply

	def forceRefresh(self,*options):
		self.refresh(self.shareName.get_text())
		return

	def refresh(self,sharename):
		p=sharesparser.SharesParser(runner.BasicRunner())
		conf=p.getConf()
		shares=p.parse(conf)
		if shares.has_key(sharename):
			share=shares[sharename]
			#share.printShare()
			self.put(share)
			return True
		return False

	# apply

	def applyShare(self,*options):
		# write samba config file
		share,shareData,fsCommands=self.get()
		if share!=None and shareData!=None:
			#self.printConf(shareData)
			lines=self.getConf()
			begin,end=self.clip(lines,share)
			if begin!=-1 and end!=-1:
				# section with same name
				#self.printConf(lines[begin:end])
				lines[begin:end]=shareData
			else:
				end=len(lines)
				lines[end:end]=['']
				begin=begin+1
				lines[end:end]=shareData
			#self.printConf(lines)
			self.setConf(lines)
		# apply fs commands
        	if fsCommands!=None:
			for cl in fsCommands:
				status,output=self.runToString(cl)
		# signal daemons
		status,output=self.signalNmb()
		#print 'hup nmb=%s %s' % (status,output)
		status,output=self.signalSmb()
		#print 'hup smb=%s %s' % (status,output)

		# parent
		if self.parent!=None:
			self.parent.refresh()
		return

	# commands

	def addValidUsers(self,*options):
		self.validUsers.add()
		return

	def removeValidUsers(self,*options):
		self.validUsers.remove()
		return

	def addReadUsers(self,*options):
		self.readUsers.add()
		return

	def removeReadUsers(self,*options):
		self.readUsers.remove()
		return

	def addWriteUsers(self,*options):
		self.writeUsers.add()
		return

	def removeWriteUsers(self,*options):
		self.writeUsers.remove()
		return

	def addAdminUsers(self,*options):
		self.adminUsers.add()
		return

	def removeAdminUsers(self,*options):
		self.adminUsers.remove()
		return

	def addForceUser(self,*options):
		self.forceUser.addOne()
		return

	def removeForceUser(self,*options):
		self.forceUser.remove()
		return

	def addFsOwner(self,*options):
		self.fsOwner.addOne()
		return

	def removeFsOwner(self,*options):
		self.fsOwner.remove()
		return

	def addValidGroups(self,*options):
		self.validGroups.add()
		return

	def removeValidGroups(self,*options):
		self.validGroups.remove()
		return

	def addReadGroups(self,*options):
		self.readGroups.add()
		return

	def removeReadGroups(self,*options):
		self.readGroups.remove()
		return

	def addWriteGroups(self,*options):
		self.writeGroups.add()
		return

	def removeWriteGroups(self,*options):
		self.writeGroups.remove()
		return

	def addAdminGroups(self,*options):
		self.adminGroups.add()
		return

	def removeAdminGroups(self,*options):
		self.adminGroups.remove()
		return

	def addForceGroup(self,*options):
		self.forceGroup.addOne()
		return

	def removeForceGroup(self,*options):
		self.forceGroup.remove()
		return

	def addFsGroup(self,*options):
		self.fsGroup.addOne()
		return

	def removeFsGroup(self,*options):
		self.fsGroup.remove()
		return

	def browseFolder(self,*options):
		path=self.sharePath.get_text()
		path=os.path.dirname(path)
		dialog=gtk.FileChooserDialog("Open folder...",None,gtk.FILE_CHOOSER_ACTION_SELECT_FOLDER,(gtk.STOCK_CANCEL,gtk.RESPONSE_CANCEL,gtk.STOCK_OPEN,gtk.RESPONSE_OK))
		dialog.set_default_response(gtk.RESPONSE_OK)
		if self.exists(path):
			dialog.set_current_folder(path)
		response = dialog.run()
		if response == gtk.RESPONSE_OK:
			self.sharePath.set_text(dialog.get_filename())
		elif response == gtk.RESPONSE_CANCEL:
			pass
		dialog.destroy()
		return

	def getText(self,*options):
		name,shareData,fscommands=self.get()
		text='%s\n\n[commands]\n\t%s\n' % ('\n'.join(shareData),'\n\t'.join(fscommands))
		dialog=gtk.MessageDialog(None,gtk.DIALOG_MODAL,gtk.MESSAGE_INFO,gtk.BUTTONS_CLOSE,text)
		result=dialog.run()
		dialog.destroy()
		return

	def help(self,*options):
		distribution=self.getDistribution()
		browser=sysSetting['browser'][distribution]
		urlbase=sysSetting['docs'][distribution]
		if distribution=="suse" or distribution=="debian":
			url=urlbase % ('share.html')
		else:
			url=urlbase % (self.getVersion(),'share.html')
		os.spawnv(os.P_NOWAIT,browser,[browser,url])
		return

	def getVersion(self):
		cl=sysSetting['getVersion']
		status,output=self.runToString(cl)
		return output

	def about(self,*options):
		self.aboutDialog.show()
		return

	#######################################################################
	#	CONNECT
	#######################################################################
	
	def connect(self,*options):
		response=self.authDialog.run()
		self.authDialog.hide()
		if response!=gtk.RESPONSE_OK:
			return
		user=self['authDialogUserEntry'].get_text()
		password=self['authDialogPasswordEntry'].get_text()
		server=self.getServer()
		share=self.shareName.get_text()

		status,output=self.connectToAs(server,share,user,password)

		if status:
			status='[OK]'
		else:
			status='[FAIL]'
		text='connect to //%s/%s as %s (%s)\n%s\n%s' % (server,share,user,password,status,output)
		dialog=gtk.MessageDialog(None,gtk.DIALOG_MODAL,gtk.MESSAGE_INFO,gtk.BUTTONS_CLOSE,text)
		result=dialog.run()
		dialog.destroy()
		return

	def batchConnect(self,*options):
		server=self.getServer()
		share=self.shareName.get_text()
		self['batchConnectInfoLabel'].set_text('//%s/%s' % (server,share))
		self.batchConnectDialog.show()
		return

	def closeBatchConnect(self,*options):
		self.batchConnectDialog.hide()
		return

	def resetBatchConnect(self,*options):
		for i in range(len(self.batchConnectView.model)):
			self.batchConnectView.reset(i)
		return

	def loadBatchConnect(self,*options):
		dialog=gtk.FileChooserDialog('Open',None,gtk.FILE_CHOOSER_ACTION_OPEN,(gtk.STOCK_CANCEL,gtk.RESPONSE_CANCEL,gtk.STOCK_OPEN,gtk.RESPONSE_OK))
		dialog.set_default_response(gtk.RESPONSE_OK)
		dialog.set_current_folder(dialog.get_current_folder()+'/settings')
		fileFilter=gtk.FileFilter()
		fileFilter.set_name('SADMS users files')
		fileFilter.add_mime_type('application/sadms-users')
		fileFilter.add_pattern('*.sadms-users')
		dialog.add_filter(fileFilter)
		fileFilter=gtk.FileFilter()
		fileFilter.set_name('All files')
		fileFilter.add_pattern('*')
		dialog.add_filter(fileFilter)
		response = dialog.run()
		if response==gtk.RESPONSE_OK:
			self.batchConnectView.clear()
			fn=dialog.get_filename()
			f=open(fn,'r')
			lines=f.readlines()
			for line in lines:
				if re.match('^$',line) or re.match('^#',line):
					continue
				line=line.strip('\n\r ')
				fields=line.split(':')
				user=fields[0]
				password=fields[1]
				self.batchConnectView.add(user,password)	
			f.close
		elif response==gtk.RESPONSE_CANCEL:
			pass
		dialog.destroy()
		return

	def runBatchConnect(self,*options):
		if self.getEnforcingSE():
			dialog=gtk.MessageDialog(None,gtk.DIALOG_MODAL,gtk.MESSAGE_INFO,gtk.BUTTONS_CLOSE,'SE linux is being enforced.\nSE linux may interfere with actual permissions.\nYou may temporarily turn it off with "setenforce Permissive"\nand revert to it with "setenforce Enforcing".')
			result=dialog.run()
			dialog.destroy()
		self.resetBatchConnect()
		server=self.getServer()
		share=self.shareName.get_text()
		for i in range(len(self.batchConnectView.model)):
			row=self.batchConnectView.model[i]
			user=row[BatchConnectView.USER]
			password=row[BatchConnectView.PASSWORD]
			status,output=self.connectToAs(server,share,user,password)
			self.batchConnectView.setConnect(i,status,output)
		for i in range(len(self.batchConnectView.model)):
			row=self.batchConnectView.model[i]
			user=row[BatchConnectView.USER]
			password=row[BatchConnectView.PASSWORD]
			status,output=self.connectToAsRead(server,share,user,password)
			self.batchConnectView.setRead(i,status,output)
		for i in range(len(self.batchConnectView.model)):
			row=self.batchConnectView.model[i]
			user=row[BatchConnectView.USER]
			password=row[BatchConnectView.PASSWORD]
			status,output=self.connectToAsWrite(server,share,user,password)
			self.batchConnectView.setWrite(i,status,output)
		return

	#######################################################################
	#	HELPERS
	#######################################################################
	
	def splitUsersGroups(self,subjects):
		if subjects=='':
			return [],[]
		subjects=subjects.split(',')
		subjects=[s.strip(' "\'') for s in subjects]
		users=[u for u in subjects if u[0]!='@']
		groups=[g[1:] for g in subjects if g[0]=='@']
		return users,groups

	def splitSubjects(self,subjects):
		subjects=subjects.split(',')
		subjects=[s.strip(' "\'') for s in subjects]
		return subjects

	def clip(self,lines,share):
		begin,end=(-1,-1)
		i=0
		for line in lines:
			if re.match('^\['+share+'\]$',line):
				begin=i
			elif begin!=-1 and end==-1 and re.match('^\[.*\]$',line):
				end=i
			i=i+1
					
		if begin!=-1 and end==-1:
			end=i

		# do not include comments or empty lines
		if begin!=-1 and end!=-1:
			for i in range(end-1,begin,-1):
				if re.match('^$',lines[i]) or re.match('^#',lines[i]) or re.match('^;',lines[i]):
					end=i
				else:
					break					
		return begin,end

	def signalSmb(self):
		cl=sysSetting['signalSmb']
	        status,output=self.runToString(cl)
		return status,output

	def signalNmb(self):
		cl=sysSetting['signalNmb']
	        status,output=self.runToString(cl)
		return status,output

	#######################################################################
	#	HELPERS
	#######################################################################
	
	def runToString(self,cl):
		status,output=self.runner.runToString(cl)
		return status,output

	def runToStringVerbose(self,cl):
		status,output=self.runner.runToStringVerbose(cl)
		return status,output

	def getShares(self,fsobject):
		cl=sysSetting['testparm']
	        status,output=self.runToString(cl)
		if not status:
			return ''
		return output.strip('\n\r ')

	def getUsers(self):
		cl=sysSetting['getUsers']
	        status,output=self.runToString(cl)
		if not status:
			return []
		output=output.strip('\n\r ')
		users=output.splitlines()
		return users
	
	def getGroups(self):
		cl=sysSetting['getGroups']
	        status,output=self.runToString(cl)
		if not status:
			return []
		output=output.strip('\n\r ')
		groups=output.splitlines()
		return groups
	
	def getOwnerGroup(self,fsobject):
		cl=sysSetting['getOwnerGroup']
		cl=cl.replace('%FSOBJECT%',fsobject)
		status,output=self.runToString(cl)
		if not status:
			return None,None
		output=output.strip('\n\r ')
		return output.split(":")
	
	def getMode(self,fsobject):
		cl=sysSetting['getDirMode']
		cl=cl.replace('%FSOBJECT%',fsobject)
		status,output=self.runToString(cl)
		if not status:
			return None,None,None,None,None,None,None,None,None
		ur=output[1]=='r'
		uw=output[2]=='w'
		ux=output[3] in ['x','X']

		gr=output[4]=='r'
		gw=output[5]=='w'
		gx=output[6] in ['x','X']
		wr=output[7]=='r'
		ww=output[8]=='w'
		wx=output[9] in ['x','X']
		return ur,uw,ux,gr,gw,gx,wr,ww,wx

	def exists(self,path):
		cl=sysSetting['exists']
		cl=cl.replace('%FSOBJECT%',path)
		status,output=self.runToString(cl)
		if not status:
			return False
		output=output.strip('\n\r ')
		return output=='true'
		
	def getEnforcingSE(self):
		cl=sysSetting['getEnforcingSE']
		status,output=self.runToString(cl)
		if not status:
			return False
		output=output.strip('\n\r ')
		return output=='true'
		
	def getServer(self):
		cl=sysSetting['getServer']
		status,output=self.runToString(cl)
		if status:
			return output.strip('\n\r ')
		return 'localhost'

	def getSep(self):
		cl=sysSetting['getSep']
		status,output=self.runToString(cl)
		if status:
			return output.strip('\n\r ')
		return ''

	def getDomain(self):
		cl=sysSetting['getDomain']
		status,output=self.runToString(cl)
		if status:
			return output.strip('\n\r ')
		return ''

	def inDomain(self,subject,isUser):
		if self.uidRange==[]:
			self.uidRange=self.getUidRange()
		if self.gidRange==[]:
			self.gidRange=self.getGidRange()

		if isUser:
			cl=sysSetting['getUserId']
			cl=cl.replace('%USER%',subject)
		else:
			cl=sysSetting['getGroupId']
			cl=cl.replace('%GROUP%',subject)
	        status,output=self.runToString(cl)
		if not status:
			return False
		try:
			subjectid=int(output.strip('\n\r '))
		except:
			return False;
		if isUser:
			domainrange=self.uidRange
		else:
			domainrange=self.gidRange
		return domainrange[0]<=subjectid<=domainrange[1]
	
	def getUidRange(self):
		cl=sysSetting['getUidRange']
	        status,output=self.runToString(cl)
		if not status:	
			return 10000,60000
		else:
			low,high=output.strip('\n\r ').split('-')
			return int(low),int(high)
		
	def getGidRange(self):
		cl=sysSetting['getGidRange']
	        status,output=self.runToString(cl)
		if not status:	
			return 10000,60000
		else:
			low,high=output.strip('\n\r ').split('-')
			return int(low),int(high)
		
	def getDistribution(self):
		cl=sysSetting['getDistribution']
		status,output=self.runToString(cl)
		if status:
			return output.strip('\n\r ')
		return 'default'

	def connectToAs(self,server,share,user,password):
		cl=sysSetting['connect']
		cl=cl.replace('%SERVER%',server)
		cl=cl.replace('%SHARE%',share)
		cl=cl.replace('%USER%',user)
		cl=cl.replace('%PASSWORD%',password)
		status,output=self.runToStringVerbose(cl)
		return status,output.strip('\n\r ')

	def connectToAsRead(self,server,share,user,password):
		cl=sysSetting['connectRead']
		cl=cl.replace('%SERVER%',server)
		cl=cl.replace('%SHARE%',share)
		cl=cl.replace('%USER%',user)
		cl=cl.replace('%PASSWORD%',password)
		status,output=self.runToStringVerbose(cl)
		return status,output.strip('\n\r ')

	def connectToAsDel(self,server,share,user,password):
		cl=sysSetting['connectDel']
		cl=cl.replace('%SERVER%',server)
		cl=cl.replace('%SHARE%',share)
		cl=cl.replace('%USER%',user)
		cl=cl.replace('%PASSWORD%',password)
		status,output=self.runToStringVerbose(cl)
		return status,output.strip('\n\r ')

	def connectToAsWrite(self,server,share,user,password):
		stamp=time.strftime('%Y%m%d-%H%M%S')
		cl=sysSetting['connectWrite']
		cl=cl.replace('%SERVER%',server)
		cl=cl.replace('%SHARE%',share)
		cl=cl.replace('%USER%',user)
		cl=cl.replace('%PASSWORD%',password)
		cl=cl.replace('%STAMP%',stamp)
		status,output=self.runToStringVerbose(cl)
		return status,output.strip('\n\r ')

	def getConf(self):
		cl=sysSetting['readFile']
		cl=cl.replace('%FILE%','/etc/samba/smb.conf')
		status,output=self.runToString(cl)
		if not status:
			return []
		output=output.strip('\n\r ')
		return output.splitlines()

	def setConf(self,lines):
		text='\n'.join(lines)
		cl=sysSetting['writeFile']
		cl=cl.replace('%TEXT%',text)
		cl=cl.replace('%FILE%','/etc/samba/smb.conf')
		status,output=self.runToString(cl)
		return

	def printConf(self,lines):
		print 'vvv'
		for line in lines:
			print line
		print '^^^'
		return

	#######################################################################
	#	HELPERS
	#######################################################################

	def acls0(self,*options):
		m=acls.AclManager()
		m.pathEdit.set_text(self.sharePath.get_text())
		m.refresh()
		return

	def acls(self,*options):
		os.spawnv(os.P_NOWAIT,'./acls.py',['./acls.py',self.sharePath.get_text()])
		return

#######################################################################
#	MAIN
#######################################################################

import sys
import os

if __name__ == "__main__":
	e=sys.argv[0]
	e=os.path.realpath(e)
	d=os.path.dirname(e)
	os.chdir(d)
	s=ShareManager()
	if len(sys.argv)>1:
		sharename=sys.argv[1]
		if not s.refresh(sharename):
			s.shareName.set_text(sharename)
			s.sharePath.set_text('/mypath')
			s.shareComment.set_text('description')
			s.shareReadOnly.set_active(True)
			s.shareBrowseable.set_active(True)
			s.shareGuestOk.set_active(False)	
			s.putCreateMask('0744')
			s.putDirectoryMask('0755')
	else:
		s.shareName.set_text('myshare')
		s.sharePath.set_text('/mypath')
		s.shareComment.set_text('description')
		s.shareReadOnly.set_active(True)
		s.shareBrowseable.set_active(True)
		s.shareGuestOk.set_active(False)
		s.putCreateMask('0744')
		s.putDirectoryMask('0755')
	gtk.main()

__author__='Bernard Bou <bou@ac-toulouse.fr>'
