/*
 *                            COPYRIGHT
 *
 *  sch-rnd - modular/flexible schematics editor - eeschema file format support
 *  Copyright (C) 2024..2025 Aron Barath
 *  Copyright (C) 2022..2024 Tibor 'Igor2' Palinkas
 *
 *  (Supported by NLnet NGI0 Entrust Fund in 2024)
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 31 Milk Street, # 960789 Boston, MA 02196 USA.
 *
 *  Contact:
 *    Project page: http://repo.hu/projects/sch-rnd
 *    contact lead developer: http://www.repo.hu/projects/sch-rnd/contact.html
 *    mailing list: http://www.repo.hu/projects/sch-rnd/contact.html
 */

struct netclass_decor
{
	const char* name;
	float height_norm; /* height in normal vector direction, in alien units */
	float width_norm; /* width in normal vector direction, in alien units */
	int (*render_decor)(read_ctx_t* const ctx, csch_cgrp_t* const dst,
		const struct netclass_decor* const decor, const float x, const float y,
		const int rot, const char* const stroke);
};

static int netclass_decor_render__round(read_ctx_t* const ctx,
	csch_cgrp_t* const dst, const struct netclass_decor* const decor,
	const float x, const float y, const int rot, const char* const stroke)
{
	if(!csch_alien_mkcircle(&ctx->alien, dst, x, y, decor->height_norm*0.5,
		stroke, NULL))
	{
		return -1;
	}

	return 0;
}

static int netclass_decor_render__dot(read_ctx_t* const ctx,
	csch_cgrp_t* const dst, const struct netclass_decor* const decor,
	const float x, const float y, const int rot, const char* const stroke)
{
	if(!csch_alien_mkcircle(&ctx->alien, dst, x, y, decor->height_norm*0.5,
		stroke, stroke))
	{
		return -1;
	}

	return 0;
}

static int netclass_decor_render__rectangle(read_ctx_t* const ctx,
	csch_cgrp_t* const dst, const struct netclass_decor* const decor,
	const float x, const float y, const int rot, const char* const stroke)
{
	float dx;
	float dy;

	if(rot==0 || rot==180)
	{
		dx = decor->width_norm * 0.5;
		dy = decor->height_norm * 0.5;
	}
	else
	{
		dx = decor->height_norm * 0.5;
		dy = decor->width_norm * 0.5;
	}

	if(!csch_alien_mkrect(&ctx->alien, dst, x-dx, y-dy, x+dx, y+dy, stroke,
		NULL))
	{
		return -1;
	}

	return 0;
}

static int netclass_decor_render__diamond(read_ctx_t* const ctx,
	csch_cgrp_t* const dst, const struct netclass_decor* const decor,
	const float x, const float y, const int rot, const char* const stroke)
{
	float dx;
	float dy;

	csch_chdr_t* poly = csch_alien_mkpoly(&ctx->alien, dst, stroke, NULL);

	if(rot==0 || rot==180)
	{
		dx = decor->width_norm * 0.5;
		dy = decor->height_norm * 0.5;
	}
	else
	{
		dx = decor->height_norm * 0.5;
		dy = decor->width_norm * 0.5;
	}

	csch_alien_append_poly_line(&ctx->alien, poly, x-dx, y, x, y+dy);
	csch_alien_append_poly_line(&ctx->alien, poly, x-dx, y, x, y-dy);
	csch_alien_append_poly_line(&ctx->alien, poly, x+dx, y, x, y+dy);
	csch_alien_append_poly_line(&ctx->alien, poly, x+dx, y, x, y-dy);

	return 0;
}

static const struct netclass_decor eeschema_netclass_decors[] =
{
	{ "round",     1.016,  1.016,  netclass_decor_render__round     },
	{ "dot",       0.8636, 0.8636, netclass_decor_render__dot       },
	{ "rectangle", 0.8128, 1.6256, netclass_decor_render__rectangle },
	{ "diamond",   1.016,  2.0066, netclass_decor_render__diamond   },
	{ 0 }
};

static const struct netclass_decor* eeschema_find_netclass_decor(
	const char* const name)
{
	const struct netclass_decor* decors = eeschema_netclass_decors;

	while(decors->name)
	{
		if(strcmp(decors->name, name)==0)
		{
			return decors;
		}

		++decors;
	}

	return NULL;
}
