/*
 * Copyright (C) 2013 Canonical, Ltd.
 *
 * This program is free software: you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License version 3, as published by
 * the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranties of MERCHANTABILITY,
 * SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Authored by: Ricardo Mendoza <ricardo.mendoza@canonical.com>
 */

// local
#include "taskcontroller.h"

// unity-mir
#include <logging.h>

// Qt
#include <QStringList>

// Process C++
#include <core/posix/process.h>
#include <core/posix/this_process.h>
#include <core/posix/linux/proc/process/oom_adj.h>
#include <core/posix/linux/proc/process/oom_score_adj.h>

// STL
#include <mutex>

// glib
#include <glib.h>

// std
#include <signal.h>
#include <unistd.h>

namespace unitymir
{

TaskController::TaskController(
        QObject* parent,
        const QSharedPointer<ApplicationController>& appController,
        const QSharedPointer<ProcessController>& processController) :
    QObject(parent),
    m_appController(appController),
    m_processController(processController)
{
    connect(m_appController.data(),
            &ApplicationController::applicationAboutToBeStarted,
            this,
            &TaskController::onApplicationAboutToBeStarted);

    connect(m_appController.data(),
            &ApplicationController::applicationStarted,
            this,
            &TaskController::onApplicationStarted);

    connect(m_appController.data(),
            &ApplicationController::applicationStopped,
            this,
            &TaskController::onApplicationStopped);

    connect(m_appController.data(),
            &ApplicationController::applicationFocusRequest,
            this,
            &TaskController::onApplicationFocusRequest);

    connect(m_appController.data(),
            &ApplicationController::applicationResumeRequest,
            this,
            &TaskController::onApplicationResumeRequest);

    connect(m_appController.data(),
            &ApplicationController::applicationError,
            this,
            &TaskController::onApplicationError);
}

TaskController::~TaskController()
{
}

bool TaskController::start(const QString& appId, const QStringList& arguments)
{
    DLOG("TaskController::start appId='%s'", qPrintable(appId));
    return m_appController->startApplicationWithAppIdAndArgs(appId, arguments);
}

bool TaskController::stop(const QString& appId)
{
    DLOG("TaskController::stop appId='%s'", qPrintable(appId));
    auto result = m_appController->stopApplicationWithAppId(appId);
    DLOG_IF(!result, "TaskController::stopApplication appId='%s' FAILED", qPrintable(appId));
    return result;
}

bool TaskController::appIdHasProcessId(const QString& appId, const quint64 pid)
{
    DLOG("TaskController::isApplicationPid appId='%s', pid=%lld", qPrintable(appId), pid);
    return m_appController->appIdHasProcessId(pid, appId);
}

bool TaskController::suspend(const QString& appId)
{
    DLOG("TaskController::suspend (this=%p, application=%p)", this, qPrintable(appId));
    pid_t pid = m_appController->primaryPidForAppId(appId);
    m_processController->oomController()->ensureProcessLikelyToBeKilled(pid);

    if (pid) {
        // We do assume that the app was launched by upstart and with that,
        // in its own process group. For that, we interpret the pid as pgid and
        // sigstop the complete process group on suspend.
        return m_processController->sigStopProcessGroupForPid(pid);
    } else {
        return false;
    }
}

bool TaskController::resume(const QString& appId)
{
    DLOG("TaskController::resume (this=%p, application=%p)", this, qPrintable(appId));
    pid_t pid = m_appController->primaryPidForAppId(appId);

    m_processController->oomController()->ensureProcessUnlikelyToBeKilled(pid);

    if (pid) {
        // We do assume that the app was launched by upstart and with that,
        // in its own process group. For that, we interpret the pid as pgid and
        // sigcont the complete process group on resume.
        return m_processController->sigContinueProcessGroupForPid(pid);
        return true;
    } else {
        return false;
    }
}

void TaskController::onApplicationAboutToBeStarted(const QString& id)
{
    Q_EMIT processStartReport(id, false);
}

void TaskController::onApplicationStarted(const QString& id)
{
    pid_t pid = m_appController->primaryPidForAppId(id);
    m_processController->oomController()->ensureProcessUnlikelyToBeKilled(pid);
}

void TaskController::onApplicationStopped(const QString& id)
{
    Q_EMIT processStopped(id, false);
}

void TaskController::onApplicationFocusRequest(const QString& id)
{
    pid_t pid = m_appController->primaryPidForAppId(id);
    m_processController->oomController()->ensureProcessUnlikelyToBeKilled(pid);
    Q_EMIT requestFocus(id);
}

void TaskController::onApplicationResumeRequest(const QString& id)
{
    Q_EMIT requestResume(id);
}

void TaskController::onApplicationError(const QString& id, ApplicationController::Error error)
{
    switch(error)
    {
    case ApplicationController::Error::APPLICATION_CRASHED:
        Q_EMIT processStopped(id, true);
        break;
    case ApplicationController::Error::APPLICATION_FAILED_TO_START:
        Q_EMIT processStartReport(id, true);
        break;
    }

    // Is this really the signal we want to emit?
    Q_EMIT requestResume(id);
}

} // namespace unitymir
