/*
 * Copyright (C) 2006-2008 the VideoLAN team
 *
 * This file is part of VLMa.
 * 
 * VLMa is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 * 
 * VLMa is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with VLMa. If not, see <http://www.gnu.org/licenses/>.
 *
 */

package org.videolan.vlma.common.medias;

import org.apache.log4j.Level;
import org.apache.log4j.Logger;
import org.videolan.vlma.common.programs.IVlProgram;

/**
 * This class implements the IVlMedia interface to represent a TNT channel.
 * 
 * @author cedric
 */

public class VlTNTChannel extends VlMedia {

    public static final String SAP_GROUP = "TNT";

    private static final Logger logger = Logger.getLogger(VlTNTChannel.class);

    private int frequency;

    private int sid;

    private String fullname;

    /**
     * Sets the channel program
     * 
     * @param program
     *            the program to set
     */
    public void setProgram(IVlProgram program) {
        super.setProgram(program);
        updateSapGroup();
    }

    /**
     * Updates the SAP group.
     * 
     */
    protected void updateSapGroup() {
        if (program != null)
            program.setSapGroup(SAP_GROUP);
    }

    /**
     * Gets the channel SID.
     * 
     * @return the channel SID
     */
    public int getSid() {
        return sid;
    }

    /**
     * Sets the channel SID.
     * 
     * @param sid
     *            the sid to set
     */
    public void setSid(int sid) {
        this.sid = sid;
    }

    /**
     * Gets the channel frequency.
     * 
     * @return the channel frequency (kHz)
     */
    public int getFrequency() {
        return frequency;
    }

    /**
     * Sets the channel frequency.
     * 
     * @param frequency
     *            the frequency to set (kHz)
     */
    public void setFrequency(int frequency) {
        this.frequency = frequency;
    }

    /**
     * Gets the channel full name which will be used in SAP announces.
     * 
     * @return the channel full name
     */
    public String getFullname() {
        return fullname;
    }

    /**
     * Sets the channel full name
     * 
     * @param fullname
     *            the full name to set
     */
    public void setFullname(String fullname) {
        this.fullname = fullname;
    }

    /**
     * Compares this object against the specified object. Returns true if and
     * only if, objects have the same name, frequency and SID.
     * 
     * @return true if the objects are the same; false otherwise.
     */
    public boolean equals(Object o) {
        if (o.getClass().equals(VlTNTChannel.class)) {
            VlTNTChannel c = (VlTNTChannel) o;
            logger.log(Level.DEBUG, "Comparaison de " + getName() + " et "
                    + c.getName());
            return (c.getFrequency() == this.frequency && c.getSid() == sid && c.name
                    .equals(this.name));
        }
        return false;
    }

    public boolean sameGroup(IVlMedia media) {
        if (media.getClass() == VlTNTChannel.class) {
            VlTNTChannel c = (VlTNTChannel) media;
            return (c.getFrequency() == this.frequency); // un
        }
        return false;
    }

    public boolean belongsToGroup(VlMediaGroup group) {
        if (!group.medias.isEmpty()) {
            return sameGroup(group.medias.get(0));
        }
        return false;
    }

    /**
     * Gets the hash code.
     * 
     * @return the hash code
     */
    public int hashCode() {
        return frequency + name.hashCode();
    }

    public int compareTo(IVlMedia other) {
        if (other == null) {
            return -1;
        }
        if (other.getClass().equals(VlSatChannel.class)) {
            return 1;
        }
        if (other.getClass().equals(VlTNTChannel.class)) {
            VlTNTChannel channel = (VlTNTChannel) other;
            if (channel.getFrequency() < frequency) {
                return 1;
            } else if (channel.getFrequency() > frequency) {
                return -1;
            } else {
                return name.compareTo(channel.getName());
            }
        }
        return -1;
    }

    public int getId() {
        return hashCode();
    }

}
